// Task.h: interface for the Task class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_TASK_H__5E26D5FE_BFF9_46AD_BB1D_624FA4AC84AA__INCLUDED_)
#define AFX_TASK_H__5E26D5FE_BFF9_46AD_BB1D_624FA4AC84AA__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <atask\TaskSynchronizationBase.h>

//////////////////////////////////////////////////////////////////////
// Each task has a priority
// The task with the highest priority at any given time is being executed
// by the system

enum Priority {
	idle_Priority,
	lowest_Priority,
	low_Priority,
	normal_Priority,
	high_Priority,
	highest_Priority,
	time_critical_Priority
};


//////////////////////////////////////////////////////////////////////
// A program implements concurrent tasks as specializations of the "Task" class
// The objects of the task (sub-)classes execute concurrently 
// Task objects can synchronize with each other and exchange data by calling
// each others "Entry" objects (see "Synchronization.h")
// 
// The following protected member functions must be overridden by each task 
// specialization:
// - body() implements the task's functionality 
// - priority() defines the task's base priority (normal_Priority by default)
// - stack_size() defines the task's stack size (default size by default)
// - virtual_sizeof() defines the size (in bytes) of the specialized task class
//   Returning anything other than "sizeof(task-class)" is considered naughty!

class Task  
{
public:
	bool activate();
	void terminate();
	bool has_activated();
	bool has_terminated();
	bool is_active() {return has_activated() && !has_terminated();}
	Priority get_priority();
	size_t get_size();
	Task();
	~Task();
protected:
	virtual void body() = 0;
	virtual Priority priority();
	virtual DWORD stack_size();
	virtual size_t virtual_sizeof() = 0;
private:
	static DWORD WINAPI thread_main(LPVOID this_task);
	HANDLE hThread;
	DWORD thread_id;
	TaskTermination term_object;
	friend class TaskTermination;
	friend class TaskTerminationMaster;
};


//////////////////////////////////////////////////////////////////////
// Global utility functions that apply to the thread currently running

typedef DWORD Milliseconds;
void delay(Milliseconds time);
// Delay execution of the current task by the specified amount of time
// Delay time is measured in milliseconds

Task* current_task();
// Returns a pointer to the currently running task, if any

typedef Task** TerminationGroup;
void terminate_all(TerminationGroup pt_table);
// The current task waits for all the tasks of the termination group to terminate 
// TerminationGroup is a NULL terminated array of pointers to Task


#endif // !defined(AFX_TASK_H__5E26D5FE_BFF9_46AD_BB1D_624FA4AC84AA__INCLUDED_)
