/*
 *   @(#) MPEG-I Video Decoder 1.0 Demicron (demicron@demicron.com)
 *
 *   P_MPEGApp.java   2002-08-20
 *
 *   Copyright (C) 2002  Demicron
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


import java.io.*;
import java.awt.*;

/**
 * MPEGApp is the control center for the system, video and audio decoders.
 * It provides function to init, stop, play, step, fforward the MPEG decoders
 */
public class P_MPEGApp implements Runnable
{
	// State Const

	/**
	 *   RESETED: when all the decoder is ready to start (e.g. after call stop() ),
	 * but the inputstream for the system decoder is closed, application should
	 * call setInputStream() to set the MPEG inputstream before call play().
	 */
	private static final int RESETED = 1;

	/**
	 *   PLAYING: Normal speed playing state.
	 */
	private static final int PLAYING = 2;

	/**
	 *   PAUSED: All the decoder thread is in wait state.
	 */
	private static final int PAUSED = 3;

	/**
	 *   FAST_FORW: MPEG player will skip all B-frames and has a speed of 3 * normal speed.
	 */
	private static final int FAST_FORW = 4;

	/**
	 *   System stream decoder
	 */
	private P_SystemDecoder systemDecoder;

	/**
	 *   Video stream decoder
	 */
	private P_VideoDecoder videoDecoder;

	/**
	 *   Audio stream decoder ( for future development )
	 */
	private P_AudioDecoder audioDecoder;

	/**
	 *   DisplayFrame implements the Displayable interface and is used
	 * to display the video data
	 */
	private P_DisplayFrame screen;

	/**
	 *   The inputstream where the MPEG data comes
	 */
	private InputStream mpegIn = null;

	/**
	 *   MPEG player's current state
	 */
	private int state;

	/**
	 *   Threads for different decoder, playerThread is used to wait
	 * for the first three to die and do the clean up.
	 */
	private Thread sysThread, videoThread, audioThread, playerThread;

	/**
	 *   Queues for the buffer exchange between decoders
	 */
	private P_BufferQueue systemQueue, videoQueue, audioQueue;

	/**
	 *   Exit Listener
	 */
	private class WindowExitListener extends java.awt.event.WindowAdapter {
		public void windowClosing(java.awt.event.WindowEvent e)
		{
			System.exit(0);
		}
	};

        private Dimension displaySize = null;
        public void displaySizeCalculated(int width, int height) {
          displaySize = new Dimension(width, height);
        }

        // Locked until first frame has been calculated in the movie
        public Dimension getDisplaySize() {
          while (displaySize==null) {
            try {
              Thread.currentThread().sleep(10);
            } catch (Exception e) {e.printStackTrace();}
          }
          return displaySize;
        }

	/**
	 *   Constructor
	 * @param	title		The title to display on the Window
	 */
	public P_MPEGApp(String title)
	{
		systemQueue = new P_BufferQueue();

		videoQueue = new P_BufferQueue();
		audioQueue = new P_BufferQueue();

		systemDecoder = new P_SystemDecoder(systemQueue, videoQueue, audioQueue);

		screen = new P_DisplayFrame(title, this);

		videoDecoder = new P_VideoDecoder(new P_BufferedBitStream(videoQueue, systemQueue, new P_ByteBuffer()), screen);
		audioDecoder = new P_AudioDecoder(new P_BufferedBitStream(audioQueue, systemQueue, new P_ByteBuffer()));
		state = RESETED;

	}



        public P_DisplayFrame getPlayerPanel() {
          return screen;
        }

	/**
	 *   If the MPEGApp is reseted.
	 */
	public boolean isReseted()
	{
		return (state == RESETED);
	}

	/**
	 *   set the InputStream after stop() or normal end of video stream(Note, stop()
	 * and normal end of video stream will close the inputstream so this method is used
	 * to add the inputstream to MPEG player for another play
	 */
	public void setInputStream(InputStream inputstream) throws IOException
	{
		if (mpegIn != null)
			mpegIn.close();
		mpegIn = inputstream;
		systemDecoder.setSystemStream(mpegIn);
	}

	/**
	 *   stop() will reset all the state in three decoder and close the inputstream
	 * controller should use setInputStream() to set another inputstream before start
	 * to play again.
	 */
	public void stop() throws IOException
	{
		if (state != RESETED) {
			if (state == PAUSED)
				play();
			systemDecoder.stop();
			try {
				playerThread.join();
			} catch (InterruptedException e) {
				System.out.println("Interrupted while waiting for player Thread to die");
			}
		}
	}

	/**
	 *   play() will start play if reseted , resume play if paused or fastforwarding
	 */
	public void play()
	{
		if (state == RESETED) {
			playerThread = new Thread(this);
			playerThread.start();
		} else {
			videoDecoder.setNormalPlay();
			if (state == PAUSED)
				videoDecoder.wakeup();
				// call the notify() in videoDecoder
		}
		state = PLAYING;
	}

	/**
	 *   start to FastForward only when paused or playing
	 */
	public void fastForward()
	{
		if (state == RESETED)
			return;
		if ((state == PAUSED) || (state == PLAYING)) {
			videoDecoder.setFastPlay();
			if (state == PAUSED)
				videoDecoder.wakeup();
		}
		state = FAST_FORW;
	}

	/**
	 *   decode one frame at one call
	 */
	public void step()
	{
		if (state == RESETED)
			return;
		else
			videoDecoder.wakeup();

		videoDecoder.setPause();
		state = PAUSED;
	}

	/**
	 *   One round of decoding
	 */
	public void run()
	{

		// Adjust the thread priority for better awt event respond
		// maybe we should use a ThreadGroup next time.
		Thread sysThread = new Thread(systemDecoder);
		sysThread.setPriority(Thread.MIN_PRIORITY);
		Thread videoThread = new Thread(videoDecoder);
		videoThread.setPriority(Thread.MIN_PRIORITY);
		Thread audioThread = new Thread(audioDecoder);
		audioThread.setPriority(Thread.MIN_PRIORITY);

		sysThread.start();
		videoThread.start();
		audioThread.start();

		// now wait for all the threads to die and perform the clean up
		try {
			sysThread.join();
			videoThread.join();
			audioThread.join();
		} catch (InterruptedException e) {
			System.out.println("Interrupted while waiting for threads to die:"+e);
		}

		systemDecoder.reset();
		videoDecoder.reset();
		audioDecoder.reset();
		try {
			mpegIn.close();
		} catch (IOException e) {
			System.out.println("IOException at playThread: "+e);
		}

		state = RESETED;
	}
}
