/* PhotoOrganizer - AlbumPane.java
 * Copyright (C) 1999-2001 Dmitriy Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  $Id: AlbumPane.java,v 1.8 2001/08/07 08:49:17 rogatkin Exp $
 */
package photoorganizer.renderer;

import java.awt.Cursor;
import java.awt.Point;
import java.awt.Container;
import java.awt.event.*;
import java.awt.image.*;
import java.awt.datatransfer.*;
import java.util.*;
import java.io.*;
import java.text.MessageFormat;
import java.awt.dnd.DropTargetAdapter;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetEvent;
//import java.awt.dnd.*;

import javax.swing.*;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import javax.swing.event.MouseInputAdapter;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeSelectionEvent;

import rogatkin.*;
import photoorganizer.*;
import photoorganizer.formats.*;
import photoorganizer.album.*;
import photoorganizer.courier.*;
import photoorganizer.directory.*;
import photoorganizer.media.*;

public class AlbumPane extends JTree implements ActionListener, TreeSelectionListener, Shuffler, IrdControllable {
	public static final String []IRD_NAVIGATIONS = new String[] {Resources.NAV_LEFT, Resources.NAV_RIGHT,
													  Resources.NAV_UP, Resources.NAV_DOWN,	Resources.NAV_COLAPSE,
			Resources.NAV_OPEN, Resources.NAV_SELECT, Resources.NAV_ACT};
	
	protected AlbumThumbnailsPanel thumbnails;
	protected PhotoCollectionPanel collectionpanel;
	private StatusBar statusbar;

	public AlbumPane(Controller controller) {
		this.controller = controller;
		access = new MediaAccess(controller);
		setModel(new AlbumModel(access));
		getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
		statusbar = (StatusBar)controller.component(Controller.COMP_STATUSBAR);
		addTreeSelectionListener(this);
		addMouseListener(new MouseInputAdapter() {
			public void mouseClicked(MouseEvent e) {
				if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0) {
					Point p = ((JViewport)getParent()).getViewPosition();
					new AlbumRightBtnMenu(AlbumPane.this).
					show(getParent(), e.getX()-p.x, e.getY()-p.y);
				}
			}
			});
		setEditable(true);
		
		setTransferHandler(new TransferHandler() {
			protected Transferable createTransferable(JComponent c) {
				
				if (c instanceof JTree) {
					JTree tree = (JTree) c;
					TreePath[] paths = tree.getSelectionPaths();
					return new TransferableTreeNode(paths, tree);
				}
				return null;
			}
			
			public int getSourceActions(JComponent c) {
				return COPY;
			}
			
			public boolean importData(JComponent comp, Transferable t) {				
				try {
					TreePath[] paths = ((TransferableTreeNode)t.getTransferData(t.getTransferDataFlavors()[0])).getTransferablePaths();
					int targetAlbum = access.getAlbumId(getSelectionPath());
					if (targetAlbum <= 0) {
						for (int i=0; i<paths.length; i++)
							paths[i] = null;
						return false;
					}
					int srcAlbum;
					for (int i=0; i<paths.length; i++) {
						srcAlbum = access.getAlbumId(paths[i]);
						if (srcAlbum <= 0 || srcAlbum == targetAlbum) {
							System.err.println("An attempt to include the album to itself, or undefined source.");
							paths[i] = null;
							continue;
						}
						//System.err.println(access.getNameOfAlbum(srcAlbum)+"-->"+access.getNameOfAlbum(targetAlbum));
						if (access.insertAlbumToAlbum(new int[]{targetAlbum}, srcAlbum) < 0) {
							JOptionPane.showMessageDialog(AlbumPane.this.controller.getMainFrame(), Resources.LABEL_ERR_COPYALBUM
																									+'\n'+access.getNameOfAlbum(srcAlbum)+"-->"+access.getNameOfAlbum(targetAlbum),
														  Resources.TITLE_ERROR, JOptionPane.ERROR_MESSAGE);
							continue;
						}
					}
					return true;
				} catch(Exception e) {
					e.printStackTrace();
				}
				return false;
			}
			
			public boolean canImport(JComponent comp, DataFlavor[] transferFlavors) {
				return true;
			}
			
			protected void exportDone(JComponent source, Transferable t, int action) {
				//System.err.println("Export done for "+action+" s "+source+" t "+t);
				if ((action & MOVE) != 0) 
					try {
						TreePath[] paths = ((TransferableTreeNode)t.getTransferData(t.getTransferDataFlavors()[0])).getTransferablePaths();
						for (int i=0,srcAlbum; i<paths.length; i++) {
							if (paths[i] != null)
								access.deleteAlbumFrom(new int[]{access.getAlbumId(paths[i].getParentPath())}, access.getAlbumId(paths[i]));
						}
					} catch(Exception e) {
						e.printStackTrace();
					}
				if ((action & COPY_OR_MOVE) != 0 && t != null)
					((AlbumModel)getModel()).fireTreeStructureChanged(this, new Object[]{getModel().getRoot()}, null, null);
				//super.exportDone(source, t, action);
			}
			
			});
		
		try {
			getDropTarget().addDropTargetListener( new DropTargetAdapter() {
				
				public void dragOver(DropTargetDragEvent dtde) {
					Point p = dtde.getLocation();
					TreePath dstPath = getClosestPathForLocation(p.x, p.y);
					if (dstPath != null)
						setSelectionPath(dstPath);
				}
				
				public void drop(DropTargetDropEvent dtde) {
				}				
				});
		} catch(TooManyListenersException tmle) {
			assert false;
		}
		setDragEnabled(true);
	}

	void setTumbnailsPanel(AlbumThumbnailsPanel thumbnails) {
		this.thumbnails = thumbnails;
	}

	void setCollectionPanel(PhotoCollectionPanel collection) {
		this.collectionpanel = collection;
	}

	public void actionPerformed(ActionEvent a) {
		String cmd = a.getActionCommand();
		boolean albumMove=false;
		if (cmd.equals(Resources.MENU_ADDTOCOLLECT)) {
			collectionpanel.add(access.getAlbumContents(access.getAlbumId(getSelectionPath())));
		} else if (cmd.equals(Resources.MENU_DELETE)) {
			access.deleteAlbumFrom(
								   new int[] {access.getAlbumId(getSelectionPath().getParentPath())},
								   access.getAlbumId(getSelectionPath()));
			invalidateTree();
		} else if (cmd.equals(Resources.MENU_DELETE_COMPLETLY)) {
			access.deleteAlbum(access.getAlbumId(getSelectionPath()));
			invalidateTree();
		} else if (cmd.equals(Resources.MENU_COPY_MOVE)) {
			final int album = access.getAlbumId(getSelectionPath());
			if (album > 0) {
				Serializer s = controller.getSerializer();
				final boolean albummove =
					Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.MOVETOFOLDER), 0) == 1;
				final String root = (String)s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.ALBUMROOT);
				new Thread(new Runnable() {
					public synchronized void run() {
						if(albummove)
							moveAlbum(album, new File(root!=null?root:"."));
						else
							copyAlbum(album, new File(root!=null?root:"."));
						statusbar.clearInfo();
						statusbar.clearProgress();
					}
					}, "Album copyer").start();
			}
		} else if (cmd.equals(Resources.MENU_SHOW)) {
			final int playAlbumId = access.getAlbumId(getSelectionPath());
			final PlaybackRequest request = new PlaybackRequest(controller.getSerializer());
			 new Thread(new Runnable() {
				 public void run() {
					 PlaybackProperties.doModal(controller, request);
					 if (request.matcher != null)
						 showAlbum(playAlbumId, request);
				 }}, "PlayAlbum").start();
			
		} else if (cmd.equals(Resources.MENU_RENAME)) {	
			startEditingAtPath(getSelectionPath());
		} else if (cmd.equals(Resources.MENU_PUBLISH_CURRENT)) {
			publishAlbum(access.getAlbumId(getSelectionPath()), false);
		} else if (cmd.equals(Resources.MENU_PUBLISH_ALL)) {
			publishAlbum(access.getAlbumId(getSelectionPath()), true);
		} else if (cmd.equals(Resources.MENU_PROPERTIES)) {
			new AlbumProperties(this, access.getAlbumId(getSelectionPath()), access);
		} else if (cmd.equals(Resources.MENU_EXPORTTOCSV)) {
			JOptionPane.showMessageDialog(this, Resources.LABEL_FOR_NEXT_VER);
		} else if (cmd.equals(Resources.MENU_EXPORTTODSK)) {
			// keep track previous exports and set start dir as previous was
			final int aid = access.getAlbumId(getSelectionPath());
			JDirectoryChooser dc = new JDirectoryChooser(new JFrame(), "/", null,
														 aid > 0?Resources.TITLE_DESTSELALBUM+access.getNameOfAlbum(aid):Resources.TITLE_DESTWHOLEALBUM,
														 Resources.LABEL_COPY,
														 Resources.TTIP_COPYALBUM);
			controller.setWaitCursor(getTopLevelAncestor(), false);
			if(dc.getDirectory() != null) {
				final File tarDir = new File(dc.getDirectory());
				new Thread(new Runnable() {
					public synchronized void run() {
							copyAlbum(aid, tarDir);
						statusbar.clearInfo();
						statusbar.clearProgress();
					}
					}, "Album copier").start();
			}
		} else if (cmd.equals(Resources.MENU_IMPORTCSV)) {
			JOptionPane.showMessageDialog(this, Resources.LABEL_FOR_NEXT_VER);
		} else if (cmd.equals(Resources.MENU_IMPORTDSK)) {
			final int aid = access.getAlbumId(getSelectionPath());
			if (aid < 0) {
				JOptionPane.showMessageDialog(controller.getMainFrame(), Resources.LABEL_ERR_NO_ALBUM_SELECTION,
											  Resources.TITLE_ERROR, JOptionPane.ERROR_MESSAGE);
				return;
				
			}
			JDirectoryChooser dc = new JDirectoryChooser(new JFrame(), "/", null,
														 Resources.TITLE_SRCALBUM,
														 Resources.LABEL_IMPORT,
														 Resources.TTIP_IMPORTALBUM);
			controller.setWaitCursor(getTopLevelAncestor(), false);
			String srcDir = dc.getDirectory();
			if(srcDir != null) {
				String albumRoot = null;
				Serializer s = controller.getSerializer();
				if (aid > 0 &&
					Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.USEALBUMFOLDER), 0) == 1) {
					albumRoot = (String)s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.ALBUMROOT);
					if (albumRoot != null) {
						// TODO: build a path from root to the current album
						int[] ap = access.getAlbumPath(aid);
						String sp = access.getNameOfAlbum(aid);
						for (int i=0; i<ap.length; i++)
							sp = access.getNameOfAlbum(ap[i])+File.separatorChar+sp;
						File file = new File(albumRoot, sp);
						file.mkdirs();
						albumRoot = file.toString();
						System.err.println("result path "+sp);
					}
				}
				importAlbum(aid, srcDir,
							albumRoot,
							Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.MOVETOFOLDER), 0) == 1);
			}			
		} else if (cmd.equals(Resources.MENU_REFRESH)) {
			invalidateTree();
		}
	}

	public AlbumSelectionDialog getSelectionDialog() {
		if (albumselection == null)
			createSelectionDialog();
		return albumselection;
	}

	final synchronized void createSelectionDialog() {
		if (albumselection == null)
			albumselection = new AlbumSelectionDialog(controller);
	}

	void copyAlbum(int album, File parent) {
		File []content = access.getAlbumContents(album);
		File albumdir;
		if (album != 0) {
			albumdir = new File(parent, access.getNameOfAlbum(album));
			albumdir.mkdir();
		} else 
			albumdir = parent;
		statusbar.displayInfo(Resources.INFO_COPYING+"->"+albumdir);
		statusbar.setProgress(content.length);
		for (int i = 0; i < content.length; i++)
			try {
				Controller.copyFile(content[i], new File(albumdir, content[i].getName()));
				statusbar.tickProgress();
			} catch(IOException ioe) {
				System.err.println("Exception "+ioe+" in copying "+content[i]);
			}
		int [] albums = access.getAlbumsId(album);
		for (int i = 0; i < albums.length; i++)
			copyAlbum(albums[i], albumdir);
	}

	void moveAlbum(int album, File parent) {
		File []content = access.getAlbumContents(album);
		File albumdir = new File(parent, access.getNameOfAlbum(album));
		albumdir.mkdir();
		statusbar.displayInfo(Resources.INFO_MOVING+"->"+albumdir);
		statusbar.setProgress(content.length);
		for (int i = 0; i < content.length; i++) {
			File oldname = new File(content[i], "");
			// below should be atransaction 
			if(content[i].renameTo(new File(albumdir, content[i].getName()))) {
				access.renamePictureTo(album, oldname.toString(), content[i].toString());
			} else
				System.err.println("File "+content[i]+" hasn't been renamed");
		}
		int [] albums = access.getAlbumsId(album);
		for (int i = 0; i < albums.length; i++)
			moveAlbum(albums[i], albumdir);
	}
	
	public void publishAlbum(final int albumId, final boolean child_also) {
		new Thread(new Runnable() {
			public void run() {
				try {
					Courier courier=null;
					// TODO: move the code to courier factory
					Serializer s = controller.getSerializer();
					switch(Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.PUBMODE), WebPublishOptionsTab.LOCAL)) {
						case WebPublishOptionsTab.FTP:
							courier = new FTPCourier(controller);
							break;
						case WebPublishOptionsTab.HTTP: // http
							courier = new HTTPCourier(controller);
							break;
						case WebPublishOptionsTab.EMAIL:
							new SendEmailFrame(controller, access.getAlbumContents(albumId));
							return;
						case WebPublishOptionsTab.XML_SVG: // SVG
							break;
						default:
							courier = new FileCourier(controller);
					}
					new HtmlProducer(controller).produce(courier, albumId, child_also);
				} catch(IOException ioe) {
					statusbar.flashInfo(Resources.INFO_ERR_WEBPUBLISHING);
					System.err.println("Exception in Web publishing "+ioe);
					ioe.printStackTrace();
				}
			}
			}, "Album publishing").start();
	}
	
	public void showAlbum(int albumId, PlaybackRequest request) {
		// TODO: consider different directions of traveling
		Object[] playList;
		if (request.shuffled) {
			playList = Shuffler.Instance.getShuffledList(new Object[] {new Integer(albumId)}, this);
		} else {
			int[] albumIds = access.getAlbumsId(albumId);
			for (int i=0; i<albumIds.length; i++)
				showAlbum(albumIds[i], request);
			playList = access.getAlbumContents(albumId);
		}
		
		controller.playMediaList(playList, request);
	}
	
	public void fill(List list, Object[] expandables) {
		for (int i=0; i < expandables.length; i++) {
			if (expandables[i] instanceof Integer)
				addShuffledAlbums(list, ((Integer)expandables[i]).intValue());
			else if (expandables[i] instanceof File)
				list.add(expandables[i]);
		}
	}
	
	protected void addShuffledAlbums(List list, int albumId) {
		Object []content = access.getAlbumContents(albumId);
		for (int i=0; i<content.length; i++) 
			list.add(content[i]);
		int[] albumIds = access.getAlbumsId(albumId);
		for (int i=0; i<albumIds.length; i++)
			addShuffledAlbums(list, albumIds[i]);
	}
	
	public void valueChanged(TreeSelectionEvent e) {
		controller.setWaitCursor(this, true);
		thumbnails.updateImages(access.getAlbumContents(access.getAlbumId(e.getPath())));
		controller.setWaitCursor(this, false);
	}

	public void addToAlbum(AbstractFormat[] formats, final TreePath[] paths) {
		Serializer s = controller.getSerializer();
		final boolean picmove =
			Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.MOVETOFOLDER), 0) == 1;
		final boolean maintainfld =
			Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.USEALBUMFOLDER), 0) == 1;
		final String root = (String)s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.ALBUMROOT);
		for (int i = 0; paths != null && i < paths.length; i++) {
			if(maintainfld) {
				Object[] _ps = paths[i].getPath();
				String target = root == null?("."+File.separatorChar):root;
				for (int k=1; k<_ps.length; k++) // we skip root element
					target += File.separator+_ps[k];
				File targetDir = new File(target);
				if (!targetDir.exists() && !targetDir.mkdirs())
					System.err.println("Album directory "+target+" can not be created, check album root location.");
				final String value = PhotoCollectionPanel.getRenameMask(this, s, " "+target);
				if (value == null)
					break;
				FileNameFormat fnf = new FileNameFormat(value, true);
				// TODO add work with real model to achieve corresponding update
				// Possible problem, a user in other thread can continue working on the model
				for (int j=0; j<formats.length; j++) {
					File file = new File(target, FileNameFormat.makeValidPathName(
																				  fnf.format(formats[j])));
					if (picmove && i == 0 && !access.belongsToAlbum(formats[j])) {
						if (!formats[j].renameTo(file))
							System.err.println("Problem in moving file "+formats[j].getFile()+" to "+file+" album.");
					} else {
						try {
							Controller.copyFile(formats[j].getFile(), file);
							formats[j] = MediaFormatFactory. createMediaFormat(file);
						} catch(IOException ioe) {
							System.err.println("Problem in copying file "+formats[j].getFile()+" to "+file+" album. ("+ioe+')');
						}
					}
				}
			}
			access.insertMediasToAlbum(access.getAlbumId(paths[i]), formats);
		}
	}

	void deletePicture(String name) {
		access.deletePicture(access.getAlbumId(getSelectionPath()), name);
	}

	void setCommentTo(String name, String comment) {
		access.setPictureComment(access.getAlbumId(getSelectionPath()), name, comment);
	}

	public String getCommentOf(String name) {
		return access.getPictureComment(access.getAlbumId(getSelectionPath()), name);
	}

	public void invalidateTree() {
		((AlbumModel)getModel()).
			fireTreeStructureChanged(this, new Object[]{getModel().getRoot()}, null, null);
		//getModel().reload();
	}

	public Access getAccess() {
		return access;
	}

	void importAlbum(int pai, String dir, String albumDir, boolean movePic) {
		String []content = new File(dir).list();
		Vector medias = new Vector(content.length);
		for(int i=0; i<content.length; i++) {
			File item = new File(dir, content[i]);
			if (item.isDirectory()) {				
				// create album 
				int naid = access.createAlbum(content[i]);
				access.insertAlbumToAlbum(new int[] {pai}, naid);
				File nodeDir = null;
				if (albumDir != null) {
					// find and create album folder, if needed
					nodeDir = new File(albumDir, content[i]);
					nodeDir.mkdirs();
				}
				importAlbum(naid, item.toString(), nodeDir==null?(String)null:nodeDir.toString(), movePic);
			} else {
				AbstractFormat af = MediaFormatFactory. createMediaFormat(item);
				if (af != null && af.isValid()) {
					if (albumDir != null) {
						File newLoc = new File(albumDir, content[i]);
						boolean succ = false;
						if(movePic)
							succ = item.renameTo(newLoc);
						else
							try {
								Controller.copyFile(item, newLoc);
								succ = true;
							} catch(IOException ioe) {
								System.err.println(ioe);
							}
						if(succ) {
							AbstractFormat naf = MediaFormatFactory. createMediaFormat(newLoc);
							if (naf != null && naf.isValid())
								af = naf;
						}
					}
					medias.addElement(af);
				}
			}
		}
		if (medias.size() > 0)
			access.insertMediasToAlbum(pai, (AbstractFormat[])medias.toArray(new AbstractFormat[medias.size()]));
	}
	
	// remote controllable
	public String getName() {
		return Resources.COMP_ALBUM_VIEWER;
	}
	
	public String toString() {
		return getName();
	}

	public Iterator	getKeyMnemonics() {
		return Arrays.asList(IRD_NAVIGATIONS).iterator();
	}
	
	public boolean doAction(String keyCode) {
		return false;
	}
	
	public void bringOnTop() {
		controller.selectTab(Resources.TAB_ALBUM);
	}
		
/*	void copyAlbumTo(String dir, int pai) {		
		int[] ai = access.getAlbumsId(pai);
		for(int i=0; i<ai.length; i++) {
			File nodeDir = new FIle(dir, access.getNameOfAlbum(ai[i]));
			nodeDir.mkdir();
			copyAlbumTo(nodeDir.toString(), ai[i]);
		}
		File[] a = access.getAlbumContents(pai);
		statusbar.displayInfo(Resources.INFO_COPYING+"->"+dir);
		statusbar.setProgress(a.length);
		for(int i=0; i<a.length; i++)
			try {
				Controller.copyFile(a[i], new File(dir, a[i].getName()));
				statusbar.tickProgress();
			} catch(IOException ioe) {
				System.err.println("Exception "+ioe+" in copying "+a[i]);
			}
	}*/

	class AlbumRightBtnMenu extends JPopupMenu {
		AlbumRightBtnMenu(ActionListener listener) {
			controller.getUiUpdater().notify(getSelectionPath() != null, UiUpdater.ALBUM_SELECTED);
			JMenuItem item;
			add(item = new JMenuItem(Resources.MENU_ADDTOCOLLECT));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			addSeparator();
			add(item = new JMenuItem(Resources.MENU_SHOW));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			addSeparator();
			add(item = new JMenuItem(Resources.MENU_COPY_MOVE));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			addSeparator();
			add(item = new JMenuItem(Resources.MENU_DELETE));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			add(item = new JMenuItem(Resources.MENU_DELETE_COMPLETLY));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			add(item = new JMenuItem(Resources.MENU_RENAME));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			addSeparator();
			JMenu menu = new JMenu(Resources.MENU_PUBLISH);
			menu.add(item = new JMenuItem(Resources.MENU_PUBLISH_CURRENT));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			menu.add(item = new JMenuItem(Resources.MENU_PUBLISH_ALL));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			add(menu);
			addSeparator();
			add(item = new JMenuItem(Resources.MENU_PROPERTIES));
			item.addActionListener(listener);
			item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.ALBUM_SELECTED));
			addSeparator();
			add(item = new JMenuItem(Resources.MENU_REFRESH));
			item.addActionListener(listener);
		}
	}

	private Controller controller;
	private AlbumSelectionDialog albumselection;

	MediaAccess access;
	
	static class TransferableTreeNode implements Transferable {
		public static DataFlavor TREE_NODE_FLAVOR = new DataFlavor(DataFlavor.javaJVMLocalObjectMimeType+";class=java.lang.String",
								  "Media Album");
		
		private static DataFlavor[] albumNodeFlavors = new DataFlavor[] {TREE_NODE_FLAVOR };
		
		TransferableTreeNode(TreePath[] treePaths, JTree tree) {
			this.treePaths = treePaths;
			this.tree = tree;
		}
		
		public DataFlavor[] getTransferDataFlavors() {
			return albumNodeFlavors;
		}
		
		public boolean isDataFlavorSupported(DataFlavor flavor) {
			return flavor.equals(albumNodeFlavors[0]);
		}
		
		public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
			if (flavor.equals(albumNodeFlavors[0]))
				return this;
			throw new UnsupportedFlavorException(flavor);
		}
		
		public TreePath [] getTransferablePaths() {
			return treePaths;
		}
		
		protected TreePath []treePaths;
		protected JTree tree;
	}
}