/* PhotoOrganizer - AdvancedImage
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package photoorganizer.jai;

import java.io.OutputStream;
import java.io.IOException;
import java.awt.image.RenderedImage;
import java.awt.Component;
import java.awt.Dimension;
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;

import javax.swing.Icon;

import com.sun.media.jai.codec.SeekableStream;
import com.sun.media.jai.codec.FileSeekableStream;
import com.sun.media.jai.codec.ImageDecoder;
import com.sun.media.jai.codec.ImageCodec;
import com.sun.media.jai.codec.JPEGEncodeParam;

import photoorganizer.formats.AdvancedImage;

public class AdvancedImageImpl implements AdvancedImage {

    public AdvancedImageImpl() {
       JAI.getDefaultInstance().getTileCache().setMemoryCapacity(0L);
    }

    public PlanarImage getImage(String filename) {
        // Use the JAI API unless JAI_IMAGE_READER_USE_CODECS is set
        if (System.getProperty("JAI_IMAGE_READER_USE_CODECS") == null) {
            try {
                return JAI.create("fileload", filename);
            } catch (Exception e) {
                reportError(e, "getImage");
            }
        } else {
            try {
                // Use the ImageCodec APIs
                SeekableStream stream = new FileSeekableStream(filename);
                String[] names = ImageCodec.getDecoderNames(stream);
                ImageDecoder dec =
                    ImageCodec.createImageDecoder(names[0], stream, null);
                RenderedImage im = dec.decodeAsRenderedImage();
                stream.close();
                return PlanarImage.wrapRenderedImage(im);
            } catch (Exception e) {
                reportError(e, "getImage");
            }
        }
        return null;
    }

    public boolean isValid(String filename) {
        try {
            // Use the ImageCodec APIs
            SeekableStream stream = new FileSeekableStream(filename);
            String[] names = ImageCodec.getDecoderNames(stream);
            stream.close();
            return names != null && names.length>0;
        } catch (Throwable e) {
            reportError(e, "isValid");
        }
        return false;
        /*try {
            return getImage(filename).getData() != null;
        } catch(Throwable e) {
            return false;
        }*/
    }

    protected void reportError(Throwable e, String msg) {
        System.err.println("JAI extension("+msg+"): "+e);
    }

    public Icon createIcon(String filename) {
        return new IconJAI(getImage(filename));
    }
    
    public Icon createThumbnailIcon(String filename, Dimension size) {
        return new IconJAI(IconJAI.createThumbnailImage(getImage(filename), size));
    }

    public void saveThumbnailImage(String filename, OutputStream os, Dimension size) throws IOException {
        if (os != null) {
            PlanarImage img = (PlanarImage)IconJAI.createThumbnailImage(getImage(filename), size);
            JPEGEncodeParam encodeParam = new JPEGEncodeParam();
            encodeParam.setQuality(0.75F);
            encodeParam.setWriteJFIFHeader(false);
            try {
                ImageCodec.createImageEncoder("JPEG", os, encodeParam).encode(img);
            } catch (Exception e) {
                reportError(e, "saveThumbnailImage");
            }
            os.close();
        }
    }
}