
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.util.java2;

import java.util.*;

/**
  * This really should be of type java.util.Map, but Java 1.2 is not 
  * out yet, so we can't subclass that. However all the operations on 
  * this object have been chosen so that it will be easy to convert to
  * the JDK/ODMG collection API once it is available. Only a subset of
  * that API is provided here, but eventually all of it will be available
  * as this will eventually be a subclass of Map.
  *
  */
public class HashMap implements Map
{
   private Hashtable myMap;
   private Vector myFriendMaps;
   private Class myKeyType = null;
   private Class myValueType = null;

   /** create a Map Object */
   public HashMap() {
      myMap = new Hashtable();
      myFriendMaps = new Vector();
   }

   /** 
     * Create a Map object specifying the allowed types for keys and 
     * values. Either can be set to null, in which case any type is 
     * allowed. 
     */
   public HashMap(Class keyType, Class valueType) {
      myMap = new Hashtable();
      myFriendMaps = new Vector();
      myKeyType = keyType;
      myValueType = valueType;
   }

   /** remove all values from this Map */
   final public void clear() {
      myMap.clear();
      myFriendMaps.removeAllElements(); 
   }

   /** return true if this context contains the supplied key 
     * @return whether the key is contained
     */
   final public boolean containsKey(Object key) {
      if (myMap.containsKey(key)) {
         return true; 
      } else {
	 Map eachMap;
         for (Enumeration enumMaps = myFriendMaps.elements(); 
	    enumMaps.hasMoreElements() ;) {
	    eachMap = (Map) enumMaps.nextElement();
	    if (eachMap.containsKey(key)) {
	       return true;
	    }
         }
         return false;
      } 
   }

   /** 
     * This performs a normal hashtable lookup, and returns the 
     * value for the supplied key if it exists, or null if it does not..
     */
   final public Object get(Object key) {
      Object value = myMap.get(key);
      if (value != null) {
         return value; 
      } else {
	 Map eachMap;
         for (Enumeration enumMaps = myFriendMaps.elements(); 
	    enumMaps.hasMoreElements() ;) {
	    eachMap = (Map) enumMaps.nextElement();
            value = eachMap.get(key);
            if (value != null) {
               return value;
            } 
         }
         return null;
      } 
   }

   /**
     * return true if the map contains no key-value mappings
     * @return boolean 
     */
   final public boolean isEmpty() {
      if (myMap.isEmpty() && myFriendMaps.isEmpty()) {
         return true;
      } else {
         return false; 
      }
   }

   /**
     * Put the supplied key and value into the Map. Both the key 
     * and value must obey the classtypes set by myKeyType and myValueType 
     * other wise a ClassCastException is thrown. Neither keys nor values  
     * may be null if they are null put does nothing and returns null.
     * <p>
     * @return the object that was under this key previously, or null
     */
   final public Object put(Object key, Object value) 
      throws ClassCastException 
   {

      // nothing will be added to the Map object if either key or value is null
      if ((key == null) || (value == null)) {
         return null;
      }
     
      // if myKeyType is equal to null all classes are exceptable for keys
      if (myKeyType != null) {
         if ((key != null) && (false == myKeyType.isInstance(key))) {
            throw new ClassCastException("Map Class: unexpected key class");
         }
      }

      // if myValueType is equal to null all classes are exceptable for keys
      if (myValueType != null) {
         if ((value != null) && (false == myValueType.isInstance(value))) {
            throw new ClassCastException("Map Class: unexpected value class");
         }
      }
      
      //System.out.println("key string: " + key); 
      return (myMap.put(key, value)); 
   }

   /**
     * Remove the supplied key from the map. 
     * @retrun removed key's value object
     */
   final public Object remove(Object key) 
   {
      Object value = myMap.remove(key);
      Object myReturnValue = null;
      myReturnValue = value; 

      Map eachMap;
      for (Enumeration enumMaps = myFriendMaps.elements(); 
	 enumMaps.hasMoreElements() ;) {
	 eachMap = (Map) enumMaps.nextElement();
         value = eachMap.remove(key);
         if (value != null) {
            myReturnValue = value;
         }
      }
      
      return myReturnValue;
   }


   /**
     * Inclue the supplied Map as a source of default values, in case 
     * the value we look for is not in our Map. Our Map will mask the 
     * contained Map--values will be found first locally. You can add
     * multiple sub-Maps, they will be searched for keys in the 
     * order in which they were added.
     */
   final public void include(HashMap includeHash) throws ClassCastException 
   {
      
      // making sure the user is not trying to include itself
      if (includeHash == this) {
         return;
      }

      // check that classtypes are the same
      if (this.myKeyType == null) {
          if (includeHash.myKeyType != null) {
             throw new ClassCastException("include(): expected same key class");
          } 
      }
      else if (!this.myKeyType.equals(includeHash.myKeyType)) {
         throw new ClassCastException("include(): expected same key class");
      }

      if (this.myValueType == null) {
          if (includeHash.myValueType != null) {
             throw new ClassCastException("include(): expected same key class");
          } 
      }
      else if (!this.myValueType.equals(includeHash.myValueType)) {
         throw new ClassCastException("include(): expected same value class");
      }

      this.myFriendMaps.addElement(includeHash); 
   }

   /** 
     * Remove the supplied Map so that it is no longer a source of 
     * defaults for this Map. 
     */
   final public void exclude(HashMap excludeHash) 
   {
      try {
         if (this.myFriendMaps.contains(excludeHash)) {
            this.myFriendMaps.removeElement(excludeHash);
         }
      } catch (NullPointerException e) {
         return; 
      }
   }
   
   /**
     * test harness
     */
   public static void main(String arg[])
   { 
      try {
	 Class stringType = Class.forName("java.lang.String");
	 Class intType = Class.forName("java.lang.Integer");
	 Integer twenty = new Integer(20);
	 Integer ten = new Integer(10);
	 Integer fifteen = new Integer(15);
	 Integer thirty = new Integer(30);
	 Integer twoHundred = new Integer(200);
	 Integer twoThousand = new Integer(2000);
	 String strRemove = new String("remove"); 
	 String strDelStringA = new String("delete string a"); 
	 String strStringA = new String("string a"); 
	 String strStringB = new String("string b"); 
	 String strStringC = new String("string c"); 

         HashMap noClassMapA, noClassMapB, noClassMapC, ClassMappedA, 
            ClassMappedB, ClassMappedC, ClassMappedD;
         noClassMapA = new HashMap();   
         noClassMapB = new HashMap();   
         noClassMapC = new HashMap();   
         ClassMappedA = new HashMap(stringType, intType);   
         ClassMappedB = new HashMap(stringType, intType);   
         ClassMappedC = new HashMap(stringType, stringType);   
         ClassMappedD = new HashMap(intType, intType);   
         
         // check if new class is empty
         System.out.println("Test: new noClassMapA is empty");      
	 if (!noClassMapA.isEmpty()) {
            System.out.println("Err: new map class should be empty");      
	 } else {
            System.out.println("verified: new noClassMapA class is empty");      
         }
         System.out.println("");      

         System.out.println("Test: new ClassMappedB is empty");      
	 if (!ClassMappedB.isEmpty()) {
            System.out.println("Err: new map class should be empty");      
	 } else {
            System.out.println("verified: new ClassMappedB class is empty");      
         }
         System.out.println("");      

         noClassMapA.put(strRemove, twenty);     // okay
         noClassMapA.put(strDelStringA, ten);    // okay
         noClassMapA.put(strStringA, thirty);    // okay
         noClassMapA.put(null, null);            // okay 

         // test put and containsKey fxn
         System.out.println("Test: added <string a, 30> to noClassMapA");      
         System.out.println("check: if <string a, 30> added");      
         if (!noClassMapA.containsKey(strStringA)) {
            System.out.println("Err: added key not found in Map");      
	 } else {
            System.out.print("found <string a>: value is ");      
            System.out.println(noClassMapA.get(strStringA));      
         }
         System.out.println("");      

         // test put and containsKey fxn
         System.out.println("Test: Delete Method");      
         System.out.println("check: if <remove, 10> added");      
         if (!noClassMapA.containsKey(strDelStringA)) {
            System.out.println("Err: added key not found in Map");      
	 } else {
            System.out.print("found remove: value is ");      
            System.out.println(noClassMapA.get(strDelStringA));      
         }
         System.out.println("");      

         // test if delete works
         System.out.println("deleting <delete string a, 10> added");      
         System.out.println("does string exist");      
         if (ten != (Integer) noClassMapA.get(strDelStringA)) {
            System.out.println("Err: string does not exist");      
	 } else {
            System.out.println("String exists and is 10");
         } 

         if (ten != (Integer) noClassMapA.remove(strDelStringA)) {
            System.out.println("Err: key not properly deleted");      
	 } 

         System.out.println("checking if <delete string a, 10> was deleted");      
         if (noClassMapA.containsKey(strDelStringA)) {
            System.out.println("Err: deleted key should no appear in Map");
	 } else {
            System.out.println("remove method behaved as expected.");
         }
         System.out.println("");      

         noClassMapB.put(null, null);                    // okay 
         noClassMapB.put(strRemove, twoHundred);              // okay 
         noClassMapB.put(strStringB, ten);     // okay

         noClassMapC.put(null, null);                    // okay 
         noClassMapC.put(strRemove, twoThousand);              // okay 
         noClassMapC.put(strStringC, fifteen);     // okay

         noClassMapA.include(noClassMapA);          // should break
         noClassMapA.include(noClassMapB);       
         noClassMapA.include(noClassMapC);       
      
         // test containsKey is checking in myFriendMap list 
         System.out.println("Test: include method hashB and containsKey");
         if (!noClassMapA.containsKey(strStringB)) {
            System.out.println("Err: key in included map should appear");
	 } else {
            System.out.print("included map key is found: (10) value is ");
            System.out.println(noClassMapA.get(strStringB));
         }
         System.out.println("");      

         System.out.println("Test: include method hashC and containsKey");
         if (!noClassMapA.containsKey(strStringC)) {
            System.out.println("Err: key in included map should appear");
	 } else {
            System.out.print("included map key is found: (15) value is ");
            System.out.println(noClassMapA.get(strStringC));
         }
         System.out.println("");      

         // test that remove removes from all
         System.out.println("Test: remove will remove from all myFriendMap");
         if (twoThousand != (Integer) noClassMapA.remove(strRemove)) {
            System.out.println("Err: remove should first come from THIS map");
	 } else {
            System.out.println("proper return object ie value = twoThousand");
         }

         // test containsKey is checking in myFriendMap list 
         System.out.println("make sure string has been properly removed");
         if (noClassMapA.containsKey(strRemove)) {
            System.out.println("Err: key in included map should appear");
	 } else {
            System.out.println("removed behave as expected with one list");
         }

         System.out.println("Make sure whole hashC has not been removed");
         if (!noClassMapA.containsKey(strStringC)) {
            System.out.println("Err: key in included map should appear");
	 } else {
            System.out.print("included map key is found: (15) value is ");
            System.out.println(noClassMapA.get(strStringC));
         }
         System.out.println("");      

         // test exclude method 
         noClassMapA.exclude(noClassMapB);       
         System.out.println("Test: exclude method");
         System.out.println("HashB should be removed");
         if (noClassMapA.containsKey(strStringB)) {
            System.out.println("string b key should be removed by exclude");
	 } else {
            System.out.println("proper behaviour string b key not found");
         }
         System.out.println("");      

         System.out.println("HashC should not be removed");
         if (!noClassMapA.containsKey(strStringC)) {
            System.out.println("Err: strStringC should not be removed");
	 } else {
            System.out.println("as expected string c key is still there");
         }
         System.out.println("");      
  
         // the rest of test tests to make sure that class types 
         // behave as expected tests includes and put only 
         System.out.println("Test: put and include with the wrong classtypes");
         ClassMappedA.put(strStringA, ten);            // okay 
         ClassMappedA.put(strRemove, twenty);             // okay 
	 
         ClassMappedB.put(strStringB, ten);            // okay
         ClassMappedB.put(strRemove, twoHundred);             // okay
         try {
            ClassMappedB.put(twenty , ten);                // FOUND CAST ERRORS
         } catch (ClassCastException e) {
            System.out.println("Got the expected exception putting wrong type.");
         }
  
         // testing put
         System.out.println("Testing proper classtype for put <remove, 20>");
         if (!ClassMappedB.containsKey(strRemove)) {
            System.out.println("Err: strRemove should have been added");      
         } else {
            System.out.println("Put behaved as Expected");      
         }
         System.out.println("");      
         
         // testing put
         System.out.println("Testing wrong classtype for put <20, 10>");
         if (ClassMappedB.containsKey(twenty)) {
            System.out.println("Err: twenty should not have been added");      
         } else {
            System.out.println("Put behaved as Expected");      
         }
         System.out.println("");      

         try {
            ClassMappedC.put(twenty, twoHundred);          // FOUND CAST ERRORS 
         } catch (ClassCastException e) {
            System.out.println("Got the expected exception putting wrong type.");
         }
  
         // testing put
         try {
            ClassMappedC.put(strRemove, twoThousand);      // FOUND CAST ERRORS 
         } catch (ClassCastException e) {
            System.out.println("Got the expected exception putting wrong type.");
         }
  
         // testing put
         ClassMappedC.put(strStringC, strStringC);     // okay 

         // testing put
         System.out.println("Testing wrong classtype for put <20, 200>");
         if (ClassMappedC.containsKey(twenty)) {
            System.out.println("Err: twenty should not have been added");      
         } else {
            System.out.println("Put behaved as Expected");      
         }
         System.out.println("");      

         // testing put
         System.out.println("Testing wrong classtype for put <remove, 2000>");
         if (ClassMappedC.containsKey(strRemove)) {
            System.out.println("Err: remove should not have been added");      
         } else {
            System.out.println("Put behaved as Expected");      
         }
         System.out.println("");      

         ClassMappedD.put(ten, twenty);                // okay 
         
         // testing clear 
         System.out.println("Test: clear with multiple list");
         ClassMappedA.clear();
  
         System.out.println("Testing For string in A and B");
         if (ClassMappedA.containsKey(strStringA)) {
            System.out.println("string a should not exist");      
         } else {
            System.out.println("clear behaved as Expected");      
         }
         if (ClassMappedA.containsKey(strStringB)) {
            System.out.println("string B should not exist");      
         } else {
            System.out.println("clear behaved as Expected");      
         }
         if (ClassMappedA.containsKey(strRemove)) {
            System.out.println("remove string should not exist");      
         } else {
            System.out.println("clear behaved as Expected");      
         }
         System.out.println("");      
  
  
       
      } catch (ClassCastException e) {
         System.out.println("FOUND CAST ERRORS");
      } catch (Exception e) {
         System.out.println("MAIN CAUGHT AN EXCEPTION:");
         e.printStackTrace();
      }  
            
   }

}

