
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.servlet;

import org.webmacro.util.java2.*;
import org.webmacro.broker.*;
import java.util.*;
import java.io.*;

import javax.servlet.*;
import javax.servlet.http.*;

import org.webmacro.util.*;
import org.webmacro.engine.*;
import org.webmacro.resource.*;

/**
  * A WebContext is the point of intersection between the back-end
  * Java code and the template script. Programmers are expected to
  * place objects needed by the template into the WebContext hashtable,
  * whereupon they become available in the template as a variable, with
  * the same name as their key in the hashtable.
  * <p>
  * At the simplest level, this means you can create variables by
  * adding them as strings: put("Title","Product Listings") creates
  * a variable in the template called $Title which evaluates to 
  * the string "Product Listings".
  * <p>
  * You can also put complex, bean-like, objects into the context
  * and rely on WebMacro's property introspection to analyze your
  * object and determine what sub-variables it can extract from
  * the object you deposit.
  * <p>
  * For example, if you put a Customer object into the WebContext
  * under the key "Buyer", then that creates a $Buyer variable
  * in the template script. The template can then use methods
  * on the Customer class via the introspector. So, $Buyer.Name
  * might be translated into Customer.getName(). See the 
  * PropertyOperator class for an explanation of the kinds of
  * introspection that WebMacro will performed. 
  * <p>
  * A majority of the methods of this class are intended as 
  * conveniences for the template writer. The getForm(String) method,
  * for example, creates a $Form variable in the template which 
  * can be used to access Form data: $Form.id translates into a 
  * call to WebContext.getForm("id").
  * <p>
  * It also contains a variety of properties populated with information 
  * about the current request, such as the session data, 
  * the cookies, and other information.
  * <p>
  * Please note that if you put values into the hashtable that have 
  * the same name as properties already declared here then your values 
  * will be hidden from the WebMacro script language--Property 
  * introspection will find the properties before checking the hashtable. 
  * <p>
  * If you absolutely must use the same names as the WebContext
  * a workaround is to declare your own hashtable and 
  * put them into that. Then put your hashtable into the context.
  * <p>
  * If your hashtable were called "data" and you had an object 
  * called "http" stuck in it, you could then access that as 
  * "$data.http" from the WebMacro script language.
  * <p>
  * If you want to use Property introspection from the Java side of 
  * things, you can do that using the getProperty method. This might
  * be convenient if you want to access things under exactly the same
  * name in Java as you do in the WebMacro script language.
  * <p>
  * @see org.webmacro.servlet.Reactor
  * @see org.webmacro.util.Property
  * @see org.webmacro.util.Map
  */
final public class WebContext extends HashMap
{

   // raw data fields that are always set, final, and available to the package

   /**
     * Log configuration errors, context errors, etc.
     */
   private final static Log _log = new Log("webcon","WebContext Messages");

   /**
     * Used to return dictionaries that reference nothing
     */
   final private static Dictionary _nullDictionary = new NullDictionary();

   /**
     * Broker from our reactor
     */
   final ResourceBroker broker_;              

   /**
     * The request for this http connect
     */
   final HttpServletRequest request_;

   /**
     * The response for this http connect 
     */
   final HttpServletResponse response_;

   // property interface fields that are lazily set, non-final, and private

   private Dictionary _form = null;        // form data
   private CGI_Impersonator _cgi = null;   // mimic cgi

   /**
     * Construct a new WebContext with all the appropriate information.
     * WebContext is constructed by Reactor, so this is package level.
     * <p>
     * @param req the servlet request object with info about the request
     * @param resp the servlet response object we need to fill in
     * @param requestType a constant, as defined in Reactor, naming req type
     */
   WebContext(
         final ResourceBroker broker,
         final HttpServletRequest req,
         final HttpServletResponse resp) 
   {
      broker_   = broker;
      request_  = req;
      response_ = resp;
   }

   /**
     * The HttpServletRequest object which contains information 
     * provided by the HttpServlet superclass about the Request.
     * Much of this data is provided in other forms later on;
     * those interfaces get their data from this object. 
     * In particular the form data has already been parsed.
     * <p>
     * @see HttpServletRequest
     * @see org.webmacro.util.Property
     */
   public final HttpServletRequest getRequest() { 
      return request_; 
   }

   /**
     * The HttpServletResponse object which contains information
     * about the response we are going to send back. Many of these
     * services are provided through other interfaces here as well;
     * they are built on top of this object.
     * <p>
     * @see HttpServletResponse
     * @see org.webmacro.util.Property
     */
   public final HttpServletResponse getResponse() { 
      return response_; 
   }


   /**
     * Get an HttpSession object associated with the current session. 
     * In this version this is the HttpSession returned by the JSDK, but
     * future versions of WebMacro may turn out to be smarter than the 
     * JSDK in tracking users, and therefore you should prefer to use 
     * this getSession method rather than getRequest().getSession(true), 
     * which for now is equivalent.
     */
   public final HttpSession getSession()  {
      return request_.getSession(true);
   }

   /**
     * This is a shortcut for HttpRequest.getParameterValues(field)[0]
     * you can thus use $Form.paramName as a variable in a WebMacro
     * property, which will return the first value associated with that 
     * field name.
     */
   public final String getForm(String field) { 
      try {
         return request_.getParameterValues(field)[0];
      } catch (NullPointerException e) {
         return null;
      }
   }

   /**
     * This is a shortcut for HttpRequest.getParameterValues(field)
     * you can thus use $Form.paramName as a variable in a WebMacro
     * property, which will return an array.
     */
   public final String[] getFormList(String field) { 
     return request_.getParameterValues(field);
   }

   /**
     * This object contains configuration data about the WebMacro
     * system, based on a file loaded at startup time. 
     * <p>
     * @see org.webmacro.util.Property
     * @see Config
     */
   public final String getConfig(String key) { 
      try {
         return (String) broker_.getValue(Config.TYPE,key);
      } catch (Exception e) {
         _log.exception(e);
         _log.error("Could not load config");
         return null;
      }
   }

   /**
     * This object contains information about users in the WebMacro
     * system.
     */
   public final ResourceMap getUsers() {
       try {
         return broker_.get(UserProvider.TYPE);
      } catch (Exception e) {
         _log.exception(e);
         _log.error("Could not load user map");
         return null;
      }
   }

   /**
     * This object provides information from the request header, but
     * does so under the standard CGI names. The names in request are
     * actually already pretty close to that--the real motivation for
     * having this object is to make the CGI names work in the 
     * WebMacro script language, which uses introspection. Via 
     * introspection getAUTH_TYPE is available as just "AUTH_TYPE",
     * which is identical to its name in CGI.
     * <p>
     * Many people may be more comfortable with the CGI names.
     * <p>
     * @see org.webmacro.util.Property
     */
   public final CGI_Impersonator getCGI() { 
      return (_cgi != null) ? _cgi : (_cgi = new CGI_Impersonator(request_));
   }


   /**
     * Search the cookie jar in the request and see if it contains the 
     * requested cookie; if so return it.
     */
   public final Cookie getCookie(final String cookieName) {
      Cookie[] jar = request_.getCookies();
      if (jar == null) {
         return null;
      }
      for(int i = 0; i < jar.length; i++) {
         if ((jar[i] != null) && (jar[i].getName().equals(cookieName))) 
         {
            return jar[i];
         }
      }
      return null;
   }

   /**
     * Create a new cookie with the supplied name and value and set it 
     * in the response header.
     */
   public final void setCookie(final String name, final String value) {
      Cookie c = new Cookie(name,value);
      response_.addCookie(c);
   }


   /**
     * This object provides access to extended services and other 
     * resources, which live in the org.webmacro.resource package,
     * or that you define, or obtain from a third party.
     */
   public final ResourceBroker getBroker() { 
      return broker_; 
   } 
}
