/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.registry;

import com.jniwrapper.UInt32;
import com.jniwrapper.ZeroTerminatedString;
import com.jniwrapper.util.EnumItem;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.Kernel32;

import java.util.*;

/**
 * Class RegistryKey represents a key level node in the Windows registry.
 *
 * @author Serge Piletsky
 */
public class RegistryKey extends EnumItem
{
    /**
     * Windows registry base key HKEY_CLASSES_ROOT.
     */
    public static final RegistryKey CLASSES_ROOT = new RegistryKey(0x80000000, "HKEY_CLASSES_ROOT");

    /**
     * Windows registry base key HKEY_CURRENT_USER.
     */
    public static final RegistryKey CURRENT_USER = new RegistryKey(0x80000001, "HKEY_CURRENT_USER");

    /**
     * Windows registry base key HKEY_LOCAL_MACHINE.
     */
    public static final RegistryKey LOCAL_MACHINE = new RegistryKey(0x80000002, "HKEY_LOCAL_MACHINE");

    /**
     * Windows registry base key HKEY_USERS.
     */
    public static final RegistryKey USERS = new RegistryKey(0x80000003, "HKEY_USERS");

    /**
     * Windows registry base key HKEY_PERFORMANCE_DATA.
     */
    public static final RegistryKey PERFORMANCE_DATA = new RegistryKey(0x80000004, "HKEY_PERFORMANCE_DATA");

    /**
     * Windows registry base key HKEY_CURRENT_CONFIG.
     */
    public static final RegistryKey CURRENT_CONFIG = new RegistryKey(0x80000005, "HKEY_CURRENT_CONFIG");

    /**
     * Windows registry base key HKEY_DYN_DATA.
     */
    public static final RegistryKey DYN_DATA = new RegistryKey(0x80000006, "HKEY_DYN_DATA");

    static final int DELETE = 0x00010000;
    static final int READ_CONTROL = 0x00020000;
    static final int WRITE_DAC = 0x00040000;
    static final int WRITE_OWNER = 0x00080000;
    static final int SYNCHRONIZE = 0x00100000;
    static final int STANDARD_RIGHTS_REQUIRED = 0x000F0000;
    static final int STANDARD_RIGHTS_READ = READ_CONTROL;
    static final int STANDARD_RIGHTS_WRITE = READ_CONTROL;
    static final int STANDARD_RIGHTS_EXECUTE = READ_CONTROL;
    static final int STANDARD_RIGHTS_ALL = 0x001F0000;
    static final int SPECIFIC_RIGHTS_ALL = 0x0000FFFF;
    static final int ACCESS_SYSTEM_SECURITY = 0x01000000;
    static final int MAXIMUM_ALLOWED = 0x02000000;
    static final int GENERIC_READ = 0x80000000;
    static final int GENERIC_WRITE = 0x40000000;
    static final int GENERIC_EXECUTE = 0x20000000;
    static final int GENERIC_ALL = 0x10000000;

    static final int KEY_QUERY_VALUE = 0x0001;
    static final int KEY_SET_VALUE = 0x0002;
    static final int KEY_CREATE_SUB_KEY = 0x0004;
    static final int KEY_ENUMERATE_SUB_KEYS = 0x0008;
    static final int KEY_NOTIFY = 0x0010;
    static final int KEY_CREATE_LINK = 0x0020;

    static final int KEY_READ = (STANDARD_RIGHTS_READ | KEY_QUERY_VALUE | KEY_ENUMERATE_SUB_KEYS | KEY_NOTIFY) & ~SYNCHRONIZE;
    static final int KEY_WRITE = (STANDARD_RIGHTS_WRITE | KEY_SET_VALUE | KEY_CREATE_SUB_KEY) & ~SYNCHRONIZE;
    static final int KEY_EXECUTE = KEY_READ & ~SYNCHRONIZE;
    static final int KEY_ALL_ACCESS = (STANDARD_RIGHTS_ALL | KEY_QUERY_VALUE | KEY_SET_VALUE | KEY_CREATE_SUB_KEY |
                                       KEY_ENUMERATE_SUB_KEYS | KEY_NOTIFY | KEY_CREATE_LINK) & ~SYNCHRONIZE;

    static final int REG_OPTION_RESERVED        = 0x00000000;
    static final int REG_OPTION_NON_VOLATILE    = 0x00000000;
    static final int REG_OPTION_VOLATILE        = 0x00000001;
    static final int REG_OPTION_CREATE_LINK     = 0x00000002;
    static final int REG_OPTION_BACKUP_RESTORE  = 0x00000004;
    static final int REG_OPTION_OPEN_LINK       = 0x00000008;
    static final int REG_LEGAL_OPTION           = REG_OPTION_RESERVED | REG_OPTION_NON_VOLATILE |
                                                  REG_OPTION_VOLATILE | REG_OPTION_CREATE_LINK |
                                                  REG_OPTION_BACKUP_RESTORE | REG_OPTION_OPEN_LINK;

    static final int NO_ERROR = 0;

    static final int MAX_PATH = 260;

    private String _path = "";
    private String _name = "";
    private Handle _handle = new Handle();
    private int _errorCode = NO_ERROR;
    private RegistryKeyValues _keyValues;

    private RegistryKey()
    {
        super(0);
    }

    private RegistryKey(int key, String name)
    {
        super(key);
        _name = name;
    }

    /**
     * Retrieves the name of the key.
     * @return The absolute (qualified) name of the key
     */
    public String getName()
    {
        return _name;
    }

    public String getPath()
    {
        return _path;
    }

    public String getAbsolutePath()
    {
        String path = getPath();
        StringBuffer result = new StringBuffer(path);
        if (path.length() > 0 && !path.endsWith("\\"))
        {
            result.append('\\');
        }
        result.append(getName());
        return result.toString();
    }

    protected void checkError()
    {
        if (_errorCode != NO_ERROR)
            throw new RegistryException(_errorCode);
    }

    protected void checkError(long errorCode)
    {
        _errorCode = (int)errorCode;
        checkError();
    }

    /**
     * Retrieves the count of subkeys at the base level, for the current key.
     * @return count of subkeys for the current key.
     */
    public int getSubKeyCount()
    {
        UInt32 subKeyCount = new UInt32();
        checkError(WinRegistry.queryInfoKey(getHandle(), null, null, subKeyCount, null, null, null, null, null));
        final int result = (int)subKeyCount.getValue();
        return result;
    }

    /**
     * Closes the key and flushes it to disk if the contents have been modified.
     */
    public void close()
    {
        checkError(WinRegistry.closeKey(getHandle()));
    }

    /**
     * Creates a new subkey or opens an existing subkey, with the write access as specified
     * @param subKey Name  of subkey to create
     * @param writable Set to true if you need write access to the key
     * @return Returns the subkey
     */
    public RegistryKey createSubKey(String subKey, boolean writable)
    {
        final RegistryKey result = new RegistryKey();
        checkError(WinRegistry.createKey(getHandle(),
                subKey,
                REG_OPTION_NON_VOLATILE,
                writable?(KEY_WRITE | KEY_READ):KEY_READ,
                result._handle));
        result._path = getAbsolutePath();
        result._name = subKey;
        return result;
    }

    /**
     * Creates a new subkey or opens an existing subkey
     * @param subKey
     * @return Returns the subkey
     */
    public RegistryKey createSubKey(String subKey)
    {
        return createSubKey(subKey, false);
    }

    /**
     * Deletes the specified subkey.
     * @param subKey Name of the subkey to delete
     */
    public void deleteSubKey(String subKey)
    {
        checkError(WinRegistry.deleteKey(getHandle(), subKey));
    }

    /**
     * Determines whether two Object instances are equal.
     * @param obj The Object to compare with the current Object.
     * @return true if the specified Object is equal to the current Object; otherwise, false
     */
    public boolean equals(Object obj)
    {
        if (obj == null || !(obj instanceof RegistryKey))
        {
            return false;
        }
        else
        {
            RegistryKey registryKey = (RegistryKey)obj;
            return getHandle().getValue() == registryKey.getHandle().getValue();
        }
    }

    /**
     * Writes all the attributes of the specified open registry key into the registry.
     */
    public void flush()
    {
        checkError(WinRegistry.flushKey(getHandle()));
    }

    /**
     * Retrieves List of strings that contains all the subkey names.
     * @return A list contains the names of the subkeys for the current key.
     */
    public List getSubKeyNames()
    {
        final int subKeyCount = getSubKeyCount();
        List result = new ArrayList(subKeyCount);
        if (subKeyCount > 0)
        {
            final Handle handle = getHandle();
            for (int i = 0, errorCode = NO_ERROR; errorCode == NO_ERROR; i++)
            {
                ZeroTerminatedString valueName = Kernel32.getInstance().stringParam("", MAX_PATH);
                errorCode = (int)WinRegistry.enumKeyEx(handle, i, valueName, new UInt32(MAX_PATH), null, null);
                if (errorCode == NO_ERROR)
                {
                    result.add(valueName.getValue());
                }
            }
        }
        return result;
    }

    /**
     * @return String representing the key.
     */
    public String toString()
    {
        return getName();
    }

    /**
     * Retrieves a specified subkey (as read-only)
     * @param name Name or path of subkey to open
     * @return subkey requested
     */
    public RegistryKey openSubKey(String name)
    {
        return openSubKey(name, false);
    }

    /**
     * Retrieves a specified subkey, with the write access as specified
     * @param name Name or path of subkey to open
     * @param writable Set to true if you need write access to the key
     * @return subkey requested
     */
    public RegistryKey openSubKey(String name, boolean writable)
    {
        RegistryKey result = new RegistryKey();
        try
        {
            checkError(WinRegistry.openKey(getHandle(),
                    name,
                    writable?(KEY_READ | KEY_WRITE):KEY_READ,
                    result._handle));
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        result._path = getAbsolutePath();
        result._name = name;
        return result;
    }

    /**
     * Returns a list of RegistryKey that are subkeys of the current RegistryKey.
     * @return Returns a list of RegistryKey that are subkeys
     *         of the current RegistryKey
     */
    public List getSubkeys()
    {
        List subkeyNames = getSubKeyNames();
        List result = new ArrayList(subkeyNames.size());
        for (Iterator i = subkeyNames.iterator(); i.hasNext();)
        {
            String name = (String)i.next();
            RegistryKey subkey = openSubKey(name);
            result.add(subkey);
        }
        return result;
    }

    /**
     * Returns RegistryKeyValues.
     * @return values map
     */
    public RegistryKeyValues values()
    {
        if (_keyValues == null)
        {
            _keyValues = new RegistryKeyValues(this);
        }
        return _keyValues;
    }

    /**
     * Returns handle of RegistryKey
     * @return handle of RegistryKey
     */
    public Handle getHandle()
    {
        if (_handle.isNull())
        {
            checkError(WinRegistry.openKey(new Handle(getValue()), "", KEY_READ, _handle));
        }
        return _handle;
    }

    private List _registryEventListeners = new LinkedList();
    private boolean _listening = false;

    public static final int REG_NOTIFY_CHANGE_NAME = 0x00000001;
    public static final int REG_NOTIFY_CHANGE_ATTRIBUTES = 0x00000002;
    public static final int REG_NOTIFY_CHANGE_LAST_SET = 0x00000004;
    public static final int REG_NOTIFY_CHANGE_SECURITY = 0x00000008;

    /**
     * Add Registry key event listener
     * @param listener
     */
    public void addRegistryEventListener(RegistryEventListener listener)
    {
        if (!_registryEventListeners.contains(listener))
            _registryEventListeners.add(listener);
    }

    /**
     * Removes Registry key event listener
     * @param listener
     */
    public void removeRegistryEventListener(RegistryEventListener listener)
    {
        _registryEventListeners.remove(listener);
    }

    /**
     * Notifies listeners about registry key event
     * @param event
     */
    protected void fireRegistryEvent(EventObject event)
    {
        List listeners;
        synchronized(this)
        {
            listeners = new LinkedList(_registryEventListeners);
        }
        for (Iterator i = listeners.iterator(); i.hasNext();)
        {
            RegistryEventListener listener = (RegistryEventListener) i.next();
            listener.handle(event);
        }
    }

    /**
     * Starts registry key change listening.
     * @param watchSubtree
     * @param filter
     */
    public void startChangeListening(final boolean watchSubtree, final int filter)
    {
        Thread listener = new Thread(new Runnable()
        {
            public void run()
            {
                _listening = true;
                while(_listening)
                {
                    checkError(WinRegistry.notifyChangeValue(getHandle(), watchSubtree, filter, new Handle(), false));
                    if (_listening)
                    {
                        fireRegistryEvent(new EventObject(this));
                    }
                }
            }
        });
        listener.start();
    }

    /**
     * Starts registry key change listening. It watches subtree and listen for all the changes.
     */
    public void startChangeListening()
    {
        int filter = REG_NOTIFY_CHANGE_NAME | REG_NOTIFY_CHANGE_ATTRIBUTES | REG_NOTIFY_CHANGE_LAST_SET | REG_NOTIFY_CHANGE_SECURITY;
        startChangeListening(true, filter);
    }

    /**
     * Stops registry key event listening.
     */
    public void stopChangeListening()
    {
        _listening = false;
        close();
    }
}
