package com.tolstoy.imagemeister;

import java.awt.*;

/**
A Panel which contains a scrolling grid of items:

<PRE>
ccccc|
ccccc|
ccccc|
ccccc|
</PRE>

<P>
This is used to implement the portion of ImageMeister in which the icons for files and folders is displayed.

<P>
Each item in the grid is called a "chic". There are methods to set the current number of chics, and to retrieve
the currently selected chic. 'numChics' holds the number of chics, and 'curChic' holds the index of the currently
selected chic.

<P>
When created, this object is passed an object implementing the GSPCallback interface. The methods of this
interface are called when the user double-clicks a chic, or presses a key. The paintItem() method of this
interface is called to draw each item.

<P>
Depending on the size of the Panel and the number of chics, a scrollbar is shown. When the size or number of
chics is changed, the scrollbar may be removed or added if necessary.

Copyright (c) 1998-2002 Samizdat Productions. All Rights Reserved. ImageMeister is a Trademark of Samizdat Productions.
*/

class GridScrollerPanel extends Panel {
	private static final String copyright="ImageMeister Copyright (c) 1998-2002 Samizdat Productions. All Rights Reserved. ImageMeister is a Trademark of Samizdat Productions.";

	private static final int		kDCTimeThreshold = 350;
	private static final int		kDCPixelThreshold = 20;

	private GSPCallback		callback;
	private Scrollbar		scrollBar;
	private long			dcLastTime;
	private int				chicW, chicH, offsetY, numChics, curChic,
							numRows, numCols, dcLastX, dcLastY;
//	private boolean			selectFlags[];

	private	Image			offscreen;
	private	boolean			bFirstTime;

	public GridScrollerPanel( GSPCallback cb, int cW, int cH ) {
		callback = cb;
		chicW = cW;
		chicH = cH;
		numChics = 0;
		curChic = -1;
		dcLastTime = -1;
		dcLastX = -5000;
		dcLastY = -5000;
		scrollBar = null;
		setLayout( new BorderLayout( 0, 0 ) );
		offsetY = 0;
		setBackground( Color.white );

		bFirstTime = true;
		offscreen = null;
	}
	
	public int getCurChic() {
		return curChic;
	}

	public void setNumItems( int n ) {
		Dimension	d;

		offsetY = 0;
		d = size();
		numChics = n;
		computeRC( d.width, d.height );
		repaint();
	}

	public void update( Graphics g ) {
		paint( g );
	}

	public void paint( Graphics g ) {
		Graphics	gg;
		Dimension	d;
		int			startChic, endChic, startY, which, xx, yy;

		d = size();
		
		if ( bFirstTime ) {
			bFirstTime = false;
			try {
				offscreen = createImage( d.width, d.height );
			}
			catch ( Exception e ) {
				offscreen = null;
			}
		}
		
		if ( offscreen != null ) {
			gg = offscreen.getGraphics();
			gg.clearRect( 0, 0, d.width, d.height );
		}
		else
			gg = g;

		startY = - ( offsetY % chicH );
		startChic = numCols * ( offsetY / chicH );
		endChic = numCols * ( ( offsetY + d.height ) / chicH ) + numCols - 1;
		if ( endChic >= numChics )
			endChic = numChics - 1;

		for ( yy = 0, which = startChic; yy < numRows && which <= endChic; yy++ )
			for ( xx = 0; xx < numCols && which <= endChic; xx++, which++ )
				callback.paintItem( gg, which, xx * chicW, startY + yy * chicH, chicW, chicH, ( which == curChic ) );

		if ( offscreen != null ) {
			g.drawImage( offscreen, 0, 0, this );
			gg.dispose();
		}
	}

	private void computeRC( int width, int height ) {
		numCols = width / chicW;
		if ( numCols < 1 )
			numCols = 1;

		numRows = ( numChics + numCols - 1 ) / numCols;
		if ( numRows < 1 )
			numRows = 1;
		
		if ( numRows * chicH < height ) {
			if ( scrollBar != null ) {
				remove( scrollBar );
				scrollBar = null;
			}
		}
		else {
			if ( scrollBar == null ) {
				scrollBar = new Scrollbar( Scrollbar.VERTICAL );
				offsetY = 0;
				add( "East", scrollBar );
			}
		}

		if ( scrollBar != null ) {			
			scrollBar.setValues( offsetY, height, 0, ( numRows * chicH ) );
			scrollBar.setPageIncrement( height / 2 );
			validate();
		}
	}

	public synchronized void reshape( int x, int y, int width, int height ) {
		try {
			if ( !bFirstTime ) {
				if ( offscreen != null ) {
					offscreen.flush();
					offscreen = null;
				}
				offscreen = createImage( width, height );
			}
		}
		catch ( Exception e ) {
			offscreen = null;
		}

		super.reshape( x, y, width, height );
		computeRC( width, height );
	}

	private boolean doKey( Event e ) {
		if ( e.key == 10 ) {
			if ( curChic != -1 )
				callback.itemDoubleClicked( curChic );
			return true;
		}
		else
			return callback.handleKeyPress( e, curChic );
		}

	private void doClick( Event e ) {
		int			row, col, which;
		boolean		bDoubleClick;

		col = e.x / chicW;
		row = ( e.y + offsetY ) / chicH;
		which = row * numCols + col;

		bDoubleClick = isDoubleClick( e );

		if ( which >= numChics || col >= numCols || row >= numRows )
			doOutsideClick();
		else if ( bDoubleClick )
			callback.itemDoubleClicked( which );
		else
			doInsideClick( which );
	}

	private boolean isDoubleClick( Event e ) {
		int				deltaX, deltaY;
		boolean			bRetVal;

		bRetVal = false;

		if ( e.clickCount == 2 )
			bRetVal = true;
		else {
			if ( e.when - dcLastTime < kDCTimeThreshold ) {
				deltaX = e.x - dcLastX;
				deltaY = e.y - dcLastY;
				if ( deltaX < 0 )
					deltaX = -deltaX;
				if ( deltaY < 0 )
					deltaY = -deltaY;
				if ( deltaX < kDCPixelThreshold && deltaY < kDCPixelThreshold )
					bRetVal = true;
			}
		}

		dcLastTime = e.when;
		dcLastX = e.x;
		dcLastY = e.y;
		
		return bRetVal;
	}

	public void doOutsideClick() {
		curChic = -1;
		repaint();
	}
	
	public void doInsideClick( int which ) {
		curChic = which;
		repaint();
	}

	public boolean handleEvent( Event e ) {
		if ( e.target == scrollBar ) {
			switch ( e.id ) {
				case Event.SCROLL_LINE_UP:
				case Event.SCROLL_LINE_DOWN:
				case Event.SCROLL_PAGE_UP:
				case Event.SCROLL_PAGE_DOWN:
				case Event.SCROLL_ABSOLUTE:
					offsetY = ( (Integer) e.arg ).intValue();
//					this.update( getGraphics() );
					repaint();
				break;
			}
			return true;
		}
		else if ( e.target == this ) {
			if ( e.id == Event.MOUSE_UP ) {
				doClick( e );
				return true;
			}
			else if ( e.id == Event.KEY_PRESS ) {
				if ( doKey( e ) )
					return true;
			}
		}
		
		return super.handleEvent( e );
	}

	public Dimension preferredSize() {
		return new Dimension( 400, 300 );
	}
}


