/****************************************************

	JConfig Source Code Distribution
	version: 2.1.0      written: Mon May 28 10:11:26 PDT 2001

	Unless otherwise noted, all contents
	Copyright (c) 1997,1998,1999,2000 Samizdat Productions.
	All Rights Reserved.
	All contents are subject to the terms of the
	JConfig Source Code License Agreement

****************************************************/

package com.jconfig.mac;

import com.jconfig.*;
import java.io.PrintStream;
import java.io.IOException;

/**
Implements the DiskFile interface on the Mac. See that interface for more information.
Most of the methods call methods in AppUtilsMRJ to obtain system information using native code.

@author Copyright (c) 1997,1998,1999,2000 Samizdat Productions. All Rights Reserved.
*/

class ResourceForkMRJ implements ResourceFork {
	private static final String copyrightString = "JConfig Copyright (c) 1997,1998,1999,2000 Samizdat Productions. All Rights Reserved.";

	private DiskFileMRJ		diskFile;
	private ResFileMRJ		resFileMRJ;

	public ResourceForkMRJ( DiskFileMRJ diskFile ) {
		this.diskFile = diskFile;
	}

/**
Returns the DiskFile object with which this resource fork is associated.
*/

	public DiskFile getDiskFile() {
		return diskFile;
	}

/**
Returns the raw resource fork of this file. Only valid on Mac.
The resource fork length must be less than Integer.MAX_VALUE.
If there is no resource fork, or if an error occurs, returns null.
*/

	public byte[] getRawResourceFork() throws ResourceForkException {
		long		resForkSize;
		int			theErr;
		byte		retData[];
		
		resForkSize = getResourceForkSize();

		if ( resForkSize <= 0 || resForkSize >= (long) Integer.MAX_VALUE )
			throw new ResourceForkException( "resForkSize out of range=" + resForkSize );

		retData = new byte[ (int) resForkSize ];

		theErr = AppUtilsMRJ.getRawResourceFork( 0,
													diskFile.getVRef(), diskFile.getParID(), diskFile.getPName(),
													retData );
		if ( theErr == ErrCodes.ERROR_NONE )
			return retData;
		
		retData = null;
		throw new ResourceForkException( "Can't read res fork, err=" + theErr );
	}

/**
Sets the raw resource fork of this file.
@param data contains the raw resource fork. Must have length >= 1 
*/

	public void setRawResourceFork( byte data[] ) throws ResourceForkException {
		int			theErr;

		theErr = AppUtilsMRJ.setRawResourceFork( 0, diskFile.getVRef(), diskFile.getParID(), diskFile.getPName(), data );

		if ( theErr != ErrCodes.ERROR_NONE )
			throw new ResourceForkException( "Can't set res fork, data.length=" +
												( data == null ? "null" : ( "" + data.length ) ) +
												", err=" + theErr );
	}

/**
Deletes the resource fork.
*/
	public void deleteResourceFork() throws ResourceForkException {
		int			theErr;

		theErr = AppUtilsMRJ.setForkLength( 0, AppUtilsMRJ.kSetForkLengthRSRC, diskFile.getVRef(), diskFile.getParID(), diskFile.getPName(), 0L );

		if ( theErr != ErrCodes.ERROR_NONE )
			throw new ResourceForkException( "Can't delete res fork, err=" + theErr );
	}

/**
Returns the size of this resource fork.
*/

	public long getResourceForkSize() throws ResourceForkException {
		long		sizes[];
		int			theErr;
		
		sizes = new long[ AppUtilsMRJ.kForkSizesLen ];

		theErr = AppUtilsMRJ.getForkSizes( diskFile.getVRef(), diskFile.getParID(), diskFile.getPName(), sizes );

		if ( theErr != ErrCodes.ERROR_NONE )
			throw new ResourceForkException( "Can't get res fork size, err=" + theErr );
		else
			return sizes[ AppUtilsMRJ.kForkSizesRsrcOffset ];
	}

	public void dumpInfo( PrintStream ps, String indent ) {
		ps.println( indent + "ResourceForkMRJ " + hashCode() +
								": vRef=" + diskFile.getVRef() +
								", parID=" + diskFile.getParID() +
								", name=" + diskFile.getName() );
		
		try {
			ps.println( indent + "  res fork size=" + getResourceForkSize() );
		}
		catch ( ResourceForkException e ) {
			ps.println( indent + "  can't get res fork size, e=" + e );
		}
	}

	public void openResources( int mode, int perms ) throws ResourceForkException, IOException {
		int			theErr;

		if ( mode != RESFORK_OPENEXISTING || perms != RESFORK_READONLY )
			throw new IllegalArgumentException( "bad mode=" + mode + " or perms=" + perms );

		if ( resFileMRJ == null ) {
			resFileMRJ = new ResFileMRJ( diskFile.getVRef(), diskFile.getParID(), diskFile.getPName(),
											ResFileMRJ.RESFORK_OPENEXISTING, ResFileMRJ.RESFORK_READONLY );
			theErr = resFileMRJ.open();
			if ( theErr != ErrCodes.ERROR_NONE ) {
				resFileMRJ = null;
				throw new ResourceForkException( "can't open res fork, err=" + theErr );
			}
		}
	}

	public byte[] getResource( int resName, int resID ) throws ResourceForkException {
		if ( resFileMRJ == null )
			throw new IllegalArgumentException( "must call openResources() first" );

		return resFileMRJ.getResource( resName, resID );
	}

	public void closeResources() throws ResourceForkException {
		if ( resFileMRJ == null )
			throw new IllegalArgumentException( "must call openResources() first" );

		resFileMRJ.close();

		resFileMRJ = null;
	}
}

