/*Data Access Library. v. 1.1.
 *Persistent object. 
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.eab.data;
import java.util.*;
import java.beans.*;
import java.sql.*;

/**<p>This abstract class
 *defines the standard interface for data access classes. It
 *provides interfaces for adding, deleting, retrieving, and
 *updating a row from a table. The Data Access Builder generates
 *classes that implement this class. </p>
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
*/
public abstract class PersistentObject implements Cloneable , java.io.Serializable , OnThreadActionExecutor ,DataAccessObject {
    private DatastoreJDBC _objDatastore;
    private boolean _isBusy;
    private PODataId _objDataId;
    private  boolean _isModified;
    private boolean _isAsync;
    private OnThreadActionDispatcher _disp;
    protected transient java.sql.PreparedStatement _stmt;

    protected DAPropertySupport _changes;
    protected transient POCompleteSupport _poSupport;

    protected transient boolean _sendingAttributeEvents;

    /**Default constructor.
     */
    public PersistentObject() {
	this(null);
    }

    /**This constructor creates a persistent object with an associated datastore.
     *@param ods - a datastore object.
    */
    public PersistentObject(DatastoreJDBC ods) {
	_objDatastore=ods;
	_changes = new DAPropertySupport(this);
	_poSupport = new POCompleteSupport(this);
    }

    public abstract void _executeAction(String action,Object[] params) throws Exception;

    public void _handleException(Exception ex) {
	ex.printStackTrace(System.err);
    }

    /**Use this method to register a listener for method complete events.
     *@param l - a POCompleteListener to listen for method complete events.
     */
    public void addPOCompleteListener(POCompleteListener l) {
	_poSupport.addPOCompleteListener(l);
    }

    public void addPropertyChangeListener(java.beans.PropertyChangeListener l) {
	_changes.addPropertyChangeListener(l);
    }

    /**This method cancels the SQL statement currently running on a
     * background thread.
     *@exception JProjects.eab.data.DAException with   message ID <samp>CANCEL_FAILED</samp> if it is unable to cancel the statement.
     */
    public void cancel() throws DAException {
	if ( !_onBackground() && isAsynchronous() && isBusy()) {
	    try {
		_stmt.cancel();
		_poSupport.fireCancelComplete();
	    } catch (SQLException ex) {
		throw new DAException(DAResource.CANCEL_FAILED,ex);
	    } finally {
		_setBusy(false);
	    }
	}
    }

    /**This method allows a class that implements PersistentObject to use the 
     *clone method.
     *@return an Object with the same attibute values as the current
     *PersistentObject.
     */
    public abstract Object clone();

    /**Implemented in subclass wich is generated by Data Access Builder.
     *Returns the datastore object associated with the current object,
     *unless it is <samp>null</samp>. If the object's datastore
     *is <samp>null</samp>, it returns the class datastore,
     *unless that is also <samp>null</samp>, in which case it
     *returns the application datastore (which may be <samp>null</samp>).
     *@return DatastoreJDBC object associated with the current object.
     */
    public abstract DatastoreJDBC getCurrentDatastore();

    /**This method, will be overridden in a subclass (such as <var>&lt;class&gt;
     *</var>, which is generated by Data Access Builder) to return the
     *DataId if the current object has a DataId.
     *@return the objects PODataId if it has an associated DataId,
     *or <samp>null</samp> if it does not.
     */
    public PODataId getObjectDataId() {
	return _objDataId;
    }

    /**This method returns the datastore object associated with the current
     *object, or <samp>null</samp> if one has not been associated with the object.
     *@return DatastoreJDBC object associated with the current object.
     */
    public DatastoreJDBC getObjectsDatastore() {
	return _objDatastore;
    }

    /**This method will be overridden in a subclass (such as <var>&lt;class&gt;
     *</var>, which is generated by Data Access Builder) if the current
     * object has a DataID (and is therefore retrievable).
     *@return a boolean value of <samp>true</samp> if the current object
     * has an associated DataId, or <samp>false</samp> if it does not.
     */
    public boolean hasDataId() {
	return _objDataId != null;
    }

    /**This method tells you whether methods are set to run on a background 
     *thread or not. By default, this setting is off.
     *@return boolean value of <samp>true</samp> if background thread
     * support is turned on, or <samp>false</samp> if it is not.
     */
    public boolean isAsynchronous() {
	return _isAsync;
    }

    /**This method tells you whether the object is running a method in the
     * background.
     *@return boolean value of <samp>true</samp> if the object is
     * currently running a method on a background thread, or <samp>false</samp> if it is not.
     */
    public boolean isBusy() {
	return _isBusy;
    }

    /**Use this method to determine whether any of the current PersistentObject's
     * properties has been changed.
     *@return boolean value of <samp>true</samp> if at least one of the
     * properties has changed, or <samp>false</samp> if none has changed.
     */
    public boolean isModified() {
	return _isModified;
    }

    /**This method, when implemented in a subclass (such as <var>&lt;class&gt;
     *</var>, which is generated by Data Access Builder), will tell you
     * whether the current object's class is read-only or not.
     *@return a boolean value of <samp>true</samp> if the current object
     * is read-only, or <samp>false</samp> if it is not.
     */
    public abstract boolean isReadOnly();

    /**Use this method to remove the method complete listener on the PersistentObject.
     *@param l - the POCompleteListener you wish to remove.
     */
    public void removePOCompleteListener(POCompleteListener l) {
	_poSupport.removePOCompleteListener(l);
    }

    public void removePropertyChangeListener(java.beans.PropertyChangeListener l) {
	_changes.removePropertyChangeListener(l);
    }

    /**This method lets you set whether the database methods are run on a
     * background thread. By default, this background thread support is turned off.
     *@param af - a boolean value of <samp>true</samp> to run methods on 
     *background thread, or <samp>false</samp> to turn background thread off.
     */
    public void setAsynchronous(boolean af) {
	_isAsync = af;
    }

    /**Use this method to change the isModified setting.
     *@param mf - a boolean value indicating whether an attribute
     * from the current PersistentObject has been
     * modified: <samp>true</samp> if at least one has
     * been modified, or <samp>false</samp> if none has.
     */
    public void setModified(boolean mf) {
	_isModified = mf;
    }

    /**This method will be overridden in a subclass (such as <var>&lt;class&gt;</var>,
     * which generated by Data Access Builder) to set the DataId
     *if the current object has a DataId.
     *@param di - new PODataId object for this one.
     */
    public void setObjectDataId(PODataId di) throws DAException {
	_objDataId = di;
    }


    /**This method lets you associate a datastore object with the current
     *persistent object. If you want the object to be
     *associated with no datastore, pass in <samp>null</samp>.
     *@param obDs - - the DatastoreJDBC object you want to associate
     *with the current object.
     */
    public void setObjectsDatastore(DatastoreJDBC obDs) {
	_objDatastore = obDs;
    }

    /**This method returns a string made up of the object's attributes.
     * Simply calls toString(".").
     *@return a String containing the attributes.
     */
    public String toString() {
	return toString(".");
    }

    /**This method, when implemented in a subclass (such as <var>&lt;class&gt;</var>,
     *which is generated by Data Access Builder), returns a
     *string made up of the object's attributes.
     *@param delim - a string used to separate the <samp>toString</samp>
     *representation of each attribute.
     *@return a String containing the attributes.
     */
    public abstract String toString(String delim);

    public abstract Object[] getAttributes();
    public abstract String[] getAttributeStrings();

    protected boolean _isSendingAttributeEvents() {
	return _sendingAttributeEvents;
    }

    protected void _setSendingAttributeEvents(boolean flag) {
	_sendingAttributeEvents = flag;
    }

    protected boolean _onBackground() {
	if ( _disp != null)
	    return _disp.onBackground();
	return false;
    }

    protected void _putOnBackgroundThread(String acnN, Object[] params) {
	_setBusy(true);
	_disp = new OnThreadActionDispatcher(this,acnN,params);
    }

    protected void _setBusy(boolean b) {
	_isBusy = b;
    }

    protected synchronized void firePropertyChange(String propName, Object oldV, Object newV) {
	_changes.firePropertyChange(propName,oldV,newV);
    }

}
