// LPING - a Win32 ping-like batch utility to wake-up ISDN lines.

// Copyright (C) 2000 Clem Dye

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#pragma pack(4)

#define WIN32_LEAN_AND_MEAN
#include <winsock2.h>
#include <stdio.h>
#include <stdlib.h>

#define ICMP_ECHO 8
#define ICMP_ECHOREPLY 0

#define ICMP_MIN 8 // Minimum 8 byte icmp packet (just the header).

// The IP header.
typedef struct iphdr {
	unsigned int h_len:4;          // Length of the header.
	unsigned int version:4;        // Version of IP.
	unsigned char tos;             // Type of service.
	unsigned short total_len;      // Total length of the packet.
	unsigned short ident;          // Unique identifier.
	unsigned short frag_and_flags; // Flags.
	unsigned char  ttl; 
	unsigned char proto;           // Protocol (TCP, UDP etc).
	unsigned short checksum;       // IP checksum.
	unsigned int sourceIP;
	unsigned int destIP;

}IpHeader;

// ICMP header.
typedef struct _ihdr {
  BYTE i_type;
  BYTE i_code; // Type sub code.
  USHORT i_cksum;
  USHORT i_id;
  USHORT i_seq;
  ULONG timestamp; // This is not the standard header, but we'll reserve space for time info.
}IcmpHeader;

#define STATUS_FAILED 1 // Was 0xFFFF.
#define DEF_PACKET_SIZE 32
#define MAX_PACKET 1024

#define xmalloc(s) HeapAlloc(GetProcessHeap(),HEAP_ZERO_MEMORY,(s))
#define xfree(p)   HeapFree (GetProcessHeap(),0,(p))

void fill_icmp_data(char *, int);
USHORT checksum(USHORT *, int);
void decode_resp(char *,int ,struct sockaddr_in *);
int pingit = 1;			// Used to terminate decode_resp loop at first successful ping response.
int first_time = 1;		// Used to determine if we should sleep when waiting for data.

void Usage(char *progname){
  
//  fprintf(stderr,"Usage: %s <host> [data_size] - data_size can be up to 1Kb.\n",progname); // See below.
	fprintf(stderr,"Usage: %s <host's IP address>\n",progname);
	ExitProcess(STATUS_FAILED);

}
int main(int argc, char **argv){

  WSADATA wsaData;
  SOCKET sockRaw;
  struct sockaddr_in dest,from;
//  struct hostent * hp;
  int bread,datasize;
  int fromlen = sizeof(from);
  int timeout = 1000; 
  char *dest_ip;
  char *icmp_data;
  char *recvbuf;
  unsigned int addr=0;
  USHORT seq_no = 0;
  ULONG ioctlarg;

  if (WSAStartup(MAKEWORD(2,1),&wsaData) != 0){
	fprintf(stderr,"WSAStartup failed. Error: %d.\n",GetLastError());
	ExitProcess(STATUS_FAILED);
  }

  if (argc <2 ) {
	Usage(argv[0]);
  }
  sockRaw = WSASocket (AF_INET,
					   SOCK_RAW,
					   IPPROTO_ICMP,
					   NULL,
					   0,
					   0);
  
  if (sockRaw == INVALID_SOCKET) {
	fprintf(stderr,"WSASocket() failed. Error: %d.\n",WSAGetLastError());
	ExitProcess(STATUS_FAILED);
  }
  bread = setsockopt(sockRaw,SOL_SOCKET,SO_RCVTIMEO,(char*)&timeout,sizeof(timeout));
  if(bread == SOCKET_ERROR) {
  	fprintf(stderr,"Failed to set receive timeout on socket. Error: %d.\n",WSAGetLastError());
	ExitProcess(STATUS_FAILED);
  }
  bread = setsockopt(sockRaw,SOL_SOCKET,SO_SNDTIMEO,(char*)&timeout,sizeof(timeout));
  if(bread == SOCKET_ERROR) {
  	fprintf(stderr,"Failed to set send timeout on socket. Error: %d.\n",WSAGetLastError());
	ExitProcess(STATUS_FAILED);
  }
  bread = ioctlsocket(sockRaw,FIONBIO,&ioctlarg);	// Set the non-blocking option on the socket. This is necessary so 
  if(bread == SOCKET_ERROR) {				// that unreachable hosts are handled properly (don't wait forever).
  	fprintf(stderr,"Failed to set non-blocking option on socket. Error: %d.\n",WSAGetLastError());
	ExitProcess(STATUS_FAILED);
  }
  memset(&dest,0,sizeof(dest));

//  hp = gethostbyname(argv[1]); 

//  if (!hp){
	addr = inet_addr(argv[1]);
//  }
//  if ((!hp) && (addr == INADDR_NONE) ) {
	if (addr == INADDR_NONE) {
	fprintf(stderr,"Unable to resolve address for target %s.\n",argv[1]);
	ExitProcess(STATUS_FAILED);
  }

//  if (hp != NULL)
//	  memcpy(&(dest.sin_addr),hp->h_addr,hp->h_length);
//  else
  	dest.sin_addr.s_addr = addr;

//  if (hp)
//	  dest.sin_family = hp->h_addrtype;
//  else
	  dest.sin_family = AF_INET;

  dest_ip = inet_ntoa(dest.sin_addr);

// if (argc >2) {		// Take the default packet size. We don't need/want anything special here.
//	datasize = atoi(argv[2]);
//	if (datasize == 0)
//	  datasize = DEF_PACKET_SIZE;
//  }
//  else
	  datasize = DEF_PACKET_SIZE;
	
  datasize += sizeof(IcmpHeader);  

  icmp_data = xmalloc(MAX_PACKET);
  recvbuf = xmalloc(MAX_PACKET);

  if (!icmp_data) {
	fprintf(stderr,"HeapAlloc failed. Error: %d.\n",GetLastError());
	ExitProcess(STATUS_FAILED);
  }
  
  memset(icmp_data,0,MAX_PACKET);
  fill_icmp_data(icmp_data,datasize);

  while(pingit) {
	int bwrote;
	
	((IcmpHeader*)icmp_data)->i_cksum = 0;
	((IcmpHeader*)icmp_data)->timestamp = GetTickCount();

	((IcmpHeader*)icmp_data)->i_seq = seq_no++;
	((IcmpHeader*)icmp_data)->i_cksum = checksum((USHORT*)icmp_data,datasize);

	bwrote = sendto(sockRaw,icmp_data,datasize,0,(struct sockaddr*)&dest,sizeof(dest));
	if (bwrote == SOCKET_ERROR){
	  if (WSAGetLastError() == WSAETIMEDOUT) {
	  	printf("Timed-out.\n");
		continue;
	  }
	  fprintf(stderr,"Send-to failed. Error: %d.\n",WSAGetLastError());
	  ExitProcess(STATUS_FAILED);
	}
	if (bwrote < datasize ) {
	  fprintf(stdout,"Wrote %d bytes. Datasize is %d bytes.\n",bwrote,datasize);
	}	
	bread = recvfrom(sockRaw,recvbuf,MAX_PACKET,0,(struct sockaddr*)&from,&fromlen);
	if (bread == SOCKET_ERROR){
	  if (WSAGetLastError() == WSAETIMEDOUT) {
	  	printf("Timed-out.\n");
		continue;
	  }
	  if (WSAGetLastError() == WSAEWOULDBLOCK) { // If we get this error, assume that the host is unreachable - maybe.
		  if (first_time == 0) {
				printf("Target appears to be currently unreachable.\n");				
				Sleep(5000); // Change this value as required.
		  }
		  if (first_time == 1) {		// This kludge is an attempt to get around the issue whereby a live 
				first_time = 0;		// host still ends-up falling through the WSAEWOULDBLOCK code. Not a
				Sleep(1000);		// 100% fix but better than earlier attempts (possibly)!
		  }
		continue;
	  }
	  fprintf(stderr,"Receive-from failed. Error: %d.\n",WSAGetLastError());
	  ExitProcess(STATUS_FAILED);
	}
	decode_resp(recvbuf,bread,&from);
  }
  return 0;
}

// The response is an IP packet. We must decode the IP header to locate the ICMP data.
void decode_resp(char *buf, int bytes,struct sockaddr_in *from) {

	IpHeader *iphdr;
	IcmpHeader *icmphdr;
	unsigned short iphdrlen;

	iphdr = (IpHeader *)buf;

	iphdrlen = iphdr->h_len * 4 ; // Number of 32-bit words * 4 = bytes.

	if (bytes  < iphdrlen + ICMP_MIN) {
		printf("Too few bytes received from %s.\n",inet_ntoa(from->sin_addr));
	}

	icmphdr = (IcmpHeader*)(buf + iphdrlen);

	if (icmphdr->i_type != ICMP_ECHOREPLY) {
		fprintf(stderr,"Non-echo type %d reply received.\n",icmphdr->i_type);
		pingit = 0; // Break the while loop in the calling routine, above.
		return;
	}
	if (icmphdr->i_id != (USHORT)GetCurrentProcessId()) {
		fprintf(stderr,"Received somebody else's data packet!\n");
		return;
	}
//	printf("%d bytes from %s:",bytes, inet_ntoa(from->sin_addr));
//	printf(" icmp_seq = %d. ",icmphdr->i_seq);
//	printf(" time: %d ms. ",GetTickCount()-icmphdr->timestamp);
	printf("\nTarget appears to be reachable.\n");
	printf("\a\a%d bytes received from address %s.\n",bytes, inet_ntoa(from->sin_addr));
	pingit = 0; // Break the while loop in the calling routine, above.
	return;
		
}

USHORT checksum(USHORT *buffer, int size) {

  unsigned long cksum=0;

  while(size >1) {
	cksum+=*buffer++;
	size -=sizeof(USHORT);
  }
  
  if(size ) {
	cksum += *(UCHAR*)buffer;
  }

  cksum = (cksum >> 16) + (cksum & 0xffff);
  cksum += (cksum >>16);
  return (USHORT)(~cksum);
}

// Helper function to fill in various stuff in our ICMP request.
void fill_icmp_data(char * icmp_data, int datasize){

  IcmpHeader *icmp_hdr;
  char *datapart;

  icmp_hdr = (IcmpHeader*)icmp_data;

  icmp_hdr->i_type = ICMP_ECHO;
  icmp_hdr->i_code = 0;
  icmp_hdr->i_id = (USHORT)GetCurrentProcessId();
  icmp_hdr->i_cksum = 0;
  icmp_hdr->i_seq = 0;
  
  datapart = icmp_data + sizeof(IcmpHeader);
  memset(datapart,'E', datasize - sizeof(IcmpHeader)); // Place some junk in the buffer.

}
