/* PhotoOrganizer 
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer.renderer;

import java.util.Locale;
import java.awt.event.*;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Color;
import javax.swing.*;
import javax.swing.border.*;

import rogatkin.*;
import photoorganizer.*;
import photoorganizer.formats.FileNameFormat;
import photoorganizer.directory.JDirectoryChooser;

public class MiscellaneousOptionsTab extends JPanel implements ActionListener, Persistentable {
    public final static String SECNAME = "PhotoCollectionOptions";
    public final static String DATEFORMAT = "DateFormat";
    public final static String TIMEFORMAT = "TimeFormat";
    public final static String TOOLBAR = "ToolBar";
    public final static String STATUSBAR = "StatusBar";
    public final static String MENUBAR = "MenuBar";
    public final static String TABPOS = "TabPosition";
    public final static String FC_FOLDER = "FlashCardDirectory";
    public final static String SHOWWARNDLG = "ShowWarnDlg";
    public final static String COUNTRY = "Country";
    public final static String LANGUAGE = "Language";
    public final static String SPLITVERT = "SplitVertically";
    // LIST_ONTOP, LIST_ONBOTTOM, LIST_ONLEFT, LIST_ONRIGHT
    final static Integer[] TABPOSTABLE = {new Integer(SwingConstants.TOP),
	new Integer(SwingConstants.BOTTOM), new Integer(SwingConstants.LEFT), new Integer(SwingConstants.RIGHT) };

    final static String formathelp = "<html>\n"+
	"<pre>\n"+
	" Symbol   Meaning                 Presentation        Example\n"+
	" ------   -------                 ------------        -------\n"+
	" G        era designator          (Text)              AD\n"+
	" y        year                    (Number)            1996\n"+
	" M        month in year           (Text &amp; Number)     July &amp; 07\n"+
	" d        day in month            (Number)            10\n"+
	" h        hour in am/pm (1~12)    (Number)            12\n"+
	" H        hour in day (0~23)      (Number)            0\n"+
	" m        minute in hour          (Number)            30\n"+
	" s        second in minute        (Number)            55\n"+
	" S        millisecond             (Number)            978\n"+
	" E        day in week             (Text)              Tuesday\n"+
	" D        day in year             (Number)            189\n"+
	" F        day of week in month    (Number)            2 (2nd Wed in July)\n"+
	" w        week in year            (Number)            27\n"+
	" W        week in month           (Number)            2\n"+
	" a        am/pm marker            (Text)              PM\n"+
	" k        hour in day (1~24)      (Number)            24\n"+
	" K        hour in am/pm (0~11)    (Number)            0\n"+
	" z        time zone               (Text)              Pacific Standard Time  \n"+
	" '        escape for text         (Delimiter)\n"+
	" ''       single quote            (Literal)           '\n"+
	" </pre>";

    public MiscellaneousOptionsTab(Controller controller) {
	this.controller = controller;
        setLayout(new GranulatedLayout(8, 6, 18, 2, 8, 8));
        JPanel canvas = new JPanel();
        canvas.setBorder(new TitledBorder(new EtchedBorder(), Resources.TITLE_SHOW));
        canvas.setOpaque(false);
        add(canvas, "0,0,8,4,16,8");
	add(cb_menu = new JCheckBox(Resources.LABEL_MENU), "1,1,1,1");
        add(cb_tool = new JCheckBox(Resources.LABEL_TOOLBAR), "3,1,1,1");
	add(new JLabel(Resources.LABEL_LOCALES, SwingConstants.RIGHT), "4,1");
        add(cb_country = new JComboBox(Resources.COUNTRIES), "5,1");
	add(cb_lang = new JComboBox(Resources.LANGUAGES), "6,1");
        add(cb_status = new JCheckBox(Resources.LABEL_SATUSBAR), "1,2,2,1");
        add(new JLabel(Resources.LABEL_TABPOS, SwingConstants.RIGHT), "4,2,1,1");
        add(mb_positions = new JComboBox(Resources.POSITIONS), "5,2,2,1");
	add(cb_warn = new JCheckBox(Resources.LABEL_SHOWWARNDLG), "0,4,3");
        ButtonGroup bg = new ButtonGroup();
        add(rb_split_vert = new JRadioButton(Resources.LABEL_SPLIT_VERT), "3,4,2");
        bg.add(rb_split_vert);
        add(rb_split_horz = new JRadioButton(Resources.LABEL_SPLIT_HORZ), "5,4,2");
        bg.add(rb_split_horz);
        JButton button;
	add(new JLabel(Resources.LABEL_CF_DIR), "0,6,2");
	add(tf_fc_folder = new JTextField(), "2,6,4");
	add(button = new JButton(Resources.CMD_BROWSE), "6,6,2");
        button.addActionListener(this);
        add(new JLabel(Resources.LABEL_DATEFMT), "0,7,2");
        add(tf_date = new JTextField(), "2,7,1,1");
        add(new JLabel(Resources.LABEL_TIMEFMT), "3,7,2,1,20");
        add(tf_time = new JTextField(), "5,7,1,1");
        add(button = new JButton(Resources.CMD_HELP), "6,7,2");
        button.addActionListener(this);
    }
    
    public void actionPerformed(ActionEvent a) {
        if (a.getActionCommand() == Resources.CMD_HELP) {
	    if (help_dialog == null) {
		help_dialog = new JDialog(new java.awt.Frame(), Resources.TITLE_HELP);
		help_dialog.getContentPane().setLayout(new BorderLayout());
		help_dialog.getContentPane().add(new JLabel(formathelp), BorderLayout.CENTER);
		JPanel p; 
		help_dialog.getContentPane().add(p = new JPanel(new FlowLayout(FlowLayout.RIGHT)), BorderLayout.SOUTH);
		JButton b; 
		p.add(b = new JButton(Resources.CMD_CLOSE));
		b.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			help_dialog.setVisible(false);
		    }
		});
		help_dialog.pack();
	    }
	    help_dialog.show();
        } else if (a.getActionCommand() == Resources.CMD_BROWSE) {
            JDirectoryChooser dc = new JDirectoryChooser(new JFrame(), tf_fc_folder.getText(), null);
            if (dc.getDirectory() != null)
                tf_fc_folder.setText(dc.getDirectory());
	}
    }

    public void load() {
        Serializer s = controller.getSerializer();
	tf_date.setText(s.arrayToString(s.getProperty(SECNAME, DATEFORMAT)));
	tf_time.setText(s.arrayToString(s.getProperty(SECNAME, TIMEFORMAT)));
        tf_fc_folder.setText(s.arrayToString(s.getProperty(SECNAME, FC_FOLDER)));
	cb_menu.setSelected(Serializer.getInt(s.getProperty(SECNAME, MENUBAR), 1) == 1);
	cb_status.setSelected(Serializer.getInt(s.getProperty(SECNAME, STATUSBAR), 1) == 1);
	cb_tool.setSelected(Serializer.getInt(s.getProperty(SECNAME, TOOLBAR), 0) == 1);
	cb_warn.setSelected(Serializer.getInt(s.getProperty(SECNAME, SHOWWARNDLG), 0) == 1);
	if (Serializer.getInt(s.getProperty(SECNAME, SPLITVERT), 1) == 1)
	    rb_split_vert.setSelected(true);
	else
	    rb_split_horz.setSelected(true);
	switch(Serializer.getInt(s.getProperty(SECNAME, TABPOS), SwingConstants.BOTTOM)) {
	case SwingConstants.TOP: 
	    mb_positions.setSelectedIndex(0);
	    break;
	case SwingConstants.BOTTOM: 
	    mb_positions.setSelectedIndex(1);
	    break;
	case SwingConstants.LEFT: 
	    mb_positions.setSelectedIndex(2);
	    break;
	case SwingConstants.RIGHT: 
	    mb_positions.setSelectedIndex(3);
	    break;
	}
	String a = (String)s.getProperty(SECNAME, COUNTRY);
	if (a == null)
	    a = Locale.getDefault().getCountry();
	cb_country.setSelectedItem(a);
	a = (String)s.getProperty(SECNAME, LANGUAGE);
	if (a == null)
	    a = Locale.getDefault().getLanguage();
	cb_lang.setSelectedItem(a);
    }
    
    public void save() {
        Serializer s = controller.getSerializer();
        FileNameFormat.datemask = tf_date.getText().trim();
        s.setProperty(SECNAME, DATEFORMAT, FileNameFormat.datemask);
        FileNameFormat.timemask = tf_time.getText().trim();
        s.setProperty(SECNAME, TIMEFORMAT, FileNameFormat.timemask);
        s.setProperty(SECNAME, FC_FOLDER, tf_fc_folder.getText());
        
	s.setProperty(SECNAME, MENUBAR, cb_menu.isSelected()?Resources.I_YES:Resources.I_NO);
	s.setProperty(SECNAME, STATUSBAR, cb_status.isSelected()?Resources.I_YES:Resources.I_NO);
	s.setProperty(SECNAME, TOOLBAR, cb_tool.isSelected()?Resources.I_YES:Resources.I_NO);
	s.setProperty(SECNAME, SHOWWARNDLG, cb_warn.isSelected()?Resources.I_YES:Resources.I_NO);
        s.setProperty(SECNAME, SPLITVERT, rb_split_vert.isSelected()?Resources.I_YES:Resources.I_NO);
	s.setProperty(SECNAME, TABPOS, TABPOSTABLE[mb_positions.getSelectedIndex()]);

	//if (getInputContext().selectInputMethod(new Locale(cb_lang.getSelectedItem(), cb_country.getSelectedItem()))) {
	    s.setProperty(SECNAME, COUNTRY, cb_country.getSelectedItem());
	    s.setProperty(SECNAME, LANGUAGE, cb_lang.getSelectedItem());
	//}
    }

    Controller controller;
    JTextField tf_date, tf_time, tf_fc_folder;
    JCheckBox  cb_menu, cb_status, cb_tool, cb_warn;
    JComboBox mb_positions, cb_country, cb_lang;
    JRadioButton rb_split_horz, rb_split_vert;
    JDialog help_dialog; 
}