/** \file WO_World.h
    Generic World that contains Objects.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_WO_WORLD
#define H_WO_WORLD

#include <WO_Object.h>


/** Generic Objects' World.  Provides a root to the objects' tree. */

class World : public World_Object
{
public:
   /** Construct a world. */
   World();

   /** this checks to make sure that any Object added to a World
       is derived from World_Object.  Returns -1 if dynamic_cast fails. */
   virtual long add(Object* O);

/** section:  Rendering functions */

   /** This method is called once before any rendering is done.  It should
       return non-zero to skip rendering. Default implementation calls 
       view.beginRendering().  */
   virtual long preRender(View& view);

   /** Subclases must override this to supply rendering. Default implemetation
       renders the child objects of the world. */
   virtual long render(View& view);

   /** This method is called once after all rendering is done. Default
       implemetation calls view.endRendering(). */
   virtual long postRender(View& view);

   /** This method is called once per object before it is rendered.
      If it returns a non-zero value, the object will not be rendered. */
   virtual long preObjectRender(World_Object* O, View& view);

   /** This method is called once per object after it is rendered. */
   virtual long postObjectRender(World_Object* O, View& view);

/** section:  Advancement/Movement functions */

   /** see Object::RETCODE in WO_Object.h for return values */

   /** Subclasses may override this to provide state changes.
       Default implemetation calls advance() for each child object of the world.
       Fraction gives the amount of time to provide change for (in seconds). */
   virtual long advance(float Fraction);

   /** This method is called once per object before it is advanced. */

   virtual long preObjectAdvance(World_Object* O, float Fraction);

   /** This method is called once per object after it is advanced */
   virtual long postObjectAdvance(World_Object* O, float Fraction);

   /** Gradual Advance.  This is will advance a given World instance by
       Fraction.  It is intended to be called once per frame.
       It has two purposes: 1) Call the world's advance routine at least
       twice per frame (this helps prevent collided objects from appearing
       overlapped on the screen). 2) Break the Fraction into manageable chunks.
       Precision gives the max fraction with which the World's advance code
       is to be called with.  Mutiple calls are made to World::advance when
       the Fraction remaining is greater than Precision.  Note that physics
       codes normally are more accurate with a lower Precision value, but
       there is a trade off.  If it is set too low, too much time will be
       spent in advance routines.  Note that World::advance will only be called
       with a fraction > EPSILON.
   */
   long advance(float Fraction, float Precision);

};

#endif // H_WO_WORLD