/** \file WO_Object.h
    Generic Object.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_WO_OBJECT
#define H_WO_OBJECT

#ifndef _CPPRTTI
#error Code must be compiled with RTTI  (Settings -> C++ -> Category: C++ Language -> Enable RTTI)
#endif 

#include <WO_View.h>
#include <WO_Event.h>
#include <DS_Hashtable.h>
#include <DS_Queue.h>

/** Generic object class.  Can represent any entity in the program,
    and is not tied to any environment.  An Object is "tree-like" in that it 
    may contain subobjects (children).
    All world objects should be subclasses of this one.
*/
class Object : public Integer_Containable
{
public:
  /** Construct an object. */
  Object();

  /** Destroys an object */
  virtual ~Object();

  /** Returns the object's ID. */
  virtual int getID();

  /** This method gets an object anywhere in the system by its ID number 
      (Use O->getID() to retrieve number).  This is useful, instead of keeping 
      pointers to objects, since objects may be deleted and then pointers are 
      invalid.  Return NULL if no Object with the given ID exists. */
  static Object* get(int ID);

  /** Get the object's parent. */
  virtual Object* getParent();

  /** Set a new parent.  If current parent is not NULL, the object is also
      removed from it. */
  virtual long    setParent(Object* newParent);

  /** Add a child object to this parent.
      Object must be dynamically allocated
      method returns the object's ID. */
  virtual long add(Object* O);

  /** Remove a child from this object.  2nd version removes the Nth child. 
      Note that removal does NOT delete the child.  Use scheduleDelete for this task.
      Returns 0 if successful (O is a a son of this, or index is valid).
  */
  virtual long remove(Object* O);
  virtual long remove(int index);

  /** Returns non zero if object has son objects */
  virtual long hasSons();

  /** return the Nth son of this object */
  virtual Object* getSon(int index);

  /** Returns a current enumeration of all son objects. */
  virtual Enumeration objects();

  /** Delete this object.  Deletions are postponed until doDeletions() is called.
      doDeletions does the actual work of deleting the Object.
      Note: that deletion is NOT recursive.  Child Objects must be
      explicitly deleted if desired. */
  void scheduleDelete(int doRemove=1);

  /** doDeletions() does the actual work of deleting objects.
      It should be called once per frame at the end of the
      processiong for that frame to carry out any postponed deletions. */
  static void doDeletions();

protected:
  Object    *m_Parent;
  Vector    *m_Sons;

private:

  static Hashtable s_AllObjects;  // tracks all Objects.
                                  //   Used for locating objects by their ID's.
  static Queue     s_Deletions;   // tracks Objects that are to be deleted.
};


class World;  // forward ref

/** A World_Object is a renderable, advancable entity that exists within a World.*/

class World_Object : public Object
{
public:
   World_Object();

   /** this checks to make sure that any Object added to a World_Object
       is derived from World_Object.  Returns -1 if dynamic_cast fails. */
   virtual long add(Object* O);

   /** return the world that this belongs to.
       a world will return itself. */
   World* getWorld();

   /** Set a new world.  If current world is not NULL, the object is also
       removed from it. */
   void setWorld(World* w);

   DS_BOOL isWorld() { return this == (World_Object*) m_World; }


/** section:  Rendering functions */

   /** Returns non zero if object wants son objects rendered. */
   virtual long renderSons();

   /** This method is called once before any rendering is done. It should
       return non-zero to skip rendering. */
   virtual long preRender(View& view);

   /** Subclases must override this to supply rendering. */
   virtual long render(View& view);

   /** This method is called once after all rendering is done. */
   virtual long postRender(View& view);


/** section:  Advancement/Movement functions */

   /** These flags can be returned by the advance(), preAdvance(), or postAdvance()
       functions to remove an object from its parent.  The rcDELETE flag will
       additionally schedule a deletion. */
   typedef 
      enum { rcOK=0, rcREMOVE=-1, rcDELETE=-2, rcSKIP=-3 } AdvRetCode;

   /** This method is called once before any object is advanced. */
   virtual long preAdvance(float Fraction);

   /** Subclasses may override this to provide state changes.
       Fraction gives the amount of time to provide change for (in seconds). */
   virtual long advance(float Fraction);

   /** This method is called once after any object is advanced. */
   virtual long postAdvance(float Fraction);

protected:
   World* m_World;
};

#endif // H_WO_OBJECT
