/** \file CON_Screen.h
    Screen object.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_CON_SCREEN
#define H_CON_SCREEN

#include <CON_Drawable.h>

/** Flags used to initialize the screen. */
enum InitFlags { NoAccel=1, SingleBuffer=2 };

/** This structure is used to get/set the gamma control for the display. */
class GammaControl
{
public:
  ushort Red[256];     // 0..65535 Ramp
  ushort Green[256];   // 0..65535 Ramp
  ushort Blue[256];    // 0..65535 Ramp
};

enum PixelFormat { BPP565, BPP555, BPP444, BPP32, OTHER };

/** This is an abstract class representing the 2D Screen.
    One of the subclasses:  FullScreen , WindowScreen should be created.
    Console automatically creates this object. */
class Screen : public virtual Drawable
{
public:
  /** Flips the pages at the end of rendering */
  virtual long flip() = 0;

  /** Check if a display mode is available
      Returns a non-zero value if the mode is available. */
  virtual int  isModeAvailable(int Width, int Height) = 0;

  /** Returns the available video memory for bitmaps. */
  virtual int  getFreeVideoMemory() = 0;

  /** Returns the pixel format of the display. */
  virtual PixelFormat getPixelFormat() = 0;

  /** Retrieve the gamma control settings for the screen.
      Returns NOT_SUPPORTED if hardware doesn't support gamma control. */
  virtual long getGammaControl(GammaControl& gc) = 0;

  /** Set the gamma control settings for the screen.
      Returns NOT_SUPPORTED if hardware doesn't support gamma control. */
  virtual long setGammaControl(GammaControl& gc) = 0;

  /** Uses setGammaControl to set a linear fade setting.
      All three colors must be in range of 0 - 1 */
  virtual long setGammaFade(float Red, float Green, float Blue) = 0;

  /** Return the primary surface for creating special effects.
      Care must be taken to return the surface to its original state (locks)
      to prevent internal inconsistencies.
      Return type should be cast to: LPDIRECTDRAWSURFACE4
  */
  virtual void* getDDrawPrimarySurface() = 0;
};

#endif // H_CON_SCREEN