/*
 * A program to faithfully clone directories
 *
 * Emmet P. Gray              US Army, HQ III Corps & Fort Hood
 * graye@hood-emh3.army.mil   Attn: AFZF-PW-ENV
 *                            Directorate of Public Works
 *                            Environmental Division
 *                            Fort Hood, TX 76544-5057
 */

#undef DEBUG
#ifdef DEBUG
#pragma option -f-
#else
#pragma option -f- -R- -v-
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dir.h>
#include <dirent.h>
#include <dos.h>
#include <io.h>
#include <errno.h>
#include <sys/stat.h>

#undef DEBUG
#define NAME "clone"
#define VERSION "v1.0"
#define DATED "3 May 97"

int optind = 1;
char *optarg;
int subdir, hidsys, attrib, modtime, prompt, quiet, files, dirs;
char to[MAXPATH], from[MAXPATH], tpath[MAXPATH], fpath[MAXPATH];

int
main(argc, argv)
int argc;
char *argv[];
{
   int c, getopt(int, char**, char*);
   void usage(void), traverse(char*);

   subdir = 0;
   hidsys = 0;
   attrib = 0;
   modtime = 0;
   prompt = 0;
   quiet = 0;
   files = 0;
   dirs = 0;

   while ((c = getopt(argc, argv, "shampq")) != EOF) {
      switch(c) {
         case 's':      /* include subdirectories */
            subdir++;
            break;
         case 'h':      /* include hidden and system files */
            hidsys++;
            break;
         case 'a':      /* retain attributes */
            attrib++;
            break;
         case 'm':      /* retain modification times */
            modtime++;
            break;
         case 'p':      /* prompt to overwrite existing files */
            prompt++;
            break;
         case 'q':      /* operate quietly */
            quiet++;
            break;
         default:
            usage();
            exit(1);
      }
   }

   if ((argc - optind) < 2) {
      fprintf(stderr, "Source and destination directories are required\n");
      usage();
      exit(1);
   }
   if ((argc - optind) > 2) {
      usage();
      exit(1);
   }

   strcpy(from, argv[optind]);
   strcpy(to, argv[optind+1]);

   if (!strchr(from, '\\') || !strchr(to, '\\')) {
      fprintf(stderr, "Use only absolute paths\n");
      usage();
      exit(1);
   }

   if (access(from, 0)) {
      fprintf(stderr, "Can't find source directory\n");
      usage();
      exit(1);
   }
   if (access(to, 0)) {
      fprintf(stderr, "Destination directory must already exist\n");
      usage();
      exit(1);
   }
                        /* here we go! */
   traverse(from);

   printf("%79.79s\r", "");
   printf("Copied %d files and %d directories\n", files, dirs);
   return(0);
}

/*
 * Traverse the directory tree
 */

void
traverse(dir)
char *dir;
{
   DIR *dirp;
   int attr, len, createdir(char*, char*), copyfile(char*, char*);
   char *s, *file;
   struct dirent *entry;
   static char cwd[MAXPATH] = {'\0'};

                        /* intialize CWD (first time only) */
   if (cwd[0] == '\0')
      strcpy(cwd, from);

   if ((dirp = opendir(dir)) == NULL)
      return;

   while ((entry = readdir(dirp)) != NULL) {
      file = entry->d_name;

                        /* skip the "." and ".." directory entries */
      if (!strcmp(file, ".") || !strcmp(file, ".."))
         continue;
                        /* reconstruct the "from" path */
      strcpy(fpath, cwd);
      if (fpath[strlen(fpath)-1] != '\\')
         strcat(fpath, "\\");
      strcat(fpath, file);

                        /* build the "to" path */
      len = strlen(from);
      if (from[len-1] != '\\')
         len++;
      strcpy(tpath, to);
      if (tpath[strlen(tpath)-1] != '\\')
         strcat(tpath, "\\");
      strcat(tpath, &fpath[len]);

                        /* include hidden and system files? */
      attr = _rtl_chmod(fpath, 0);
      if (!hidsys) {
         if (attr & FA_HIDDEN)
            continue;
         if (attr & FA_SYSTEM)
            continue;
      }
                        /* is a directory? */
      if (attr & FA_DIREC) {
         if (!subdir)
            continue;
                        /* create the destination directory */
#ifndef DEBUG
      if (!createdir(fpath, tpath))
         dirs++;
#endif
                        /* recursive call to traverse() */
         strcpy(cwd, fpath);
         traverse(cwd);
         continue;
      }
      if (!quiet)
         printf("%-79.79s\r", fpath);

                        /* copy the file */
      if (!copyfile(fpath, tpath))
         files++;
   }
   closedir(dirp);

   /*
    * Remove the last portion of the Current Working Directory.
    * Essentially the same as doing a "CD .."
    */
   if ((s = strrchr(cwd, '\\')) != NULL)
      *s = '\0';

   return;
}

/*
 * Copy a file.  Using small buffer sizes actually increase
 * performance over a network.
 */

int
copyfile(from_file, to_file)
char *from_file, *to_file;
{
   FILE *from_fp, *to_fp;
   int num, attr, temp;
   struct ftime ftime;
   void warning(void);
   char buf[BUFSIZ];

#ifdef DEBUG
   to_file = "nul";
#endif

   if ((from_fp = fopen(from_file, "rb")) == NULL) {
      fprintf(stderr, "Can't open \"%s\" for read\n", from_file);
      warning();
      return(1);
   }

   if (modtime)
      getftime(fileno(from_fp), &ftime);
   if (attrib)
      attr = _rtl_chmod(from_file, 0);

   if (prompt) {
      if (!access(to_file, 0)) {
         int doit, done;
         char ans[10];

         done = 0;
         doit = 0;
         while(!done) {
            printf("Overwrite \"%s\" [yes, No, all] : ", to_file);
            fgets(ans, 10, stdin);

            switch(toupper(ans[0])) {
               case 'A':
                  done++;
                  doit++;
                  prompt = 0;
                  break;
               case 'Y':
                  done++;
                  doit++;
                  break;
               case '\n':
               case 'N':
                  done++;
                  break;
               default:
                  fputc('\a', stderr);
            }
         }
         if (!doit) {
            fclose(from_fp);
            return(1);
         }
      }
   }
   /*
    * If the destination already exists, but is read only, then we have
    * to temporarily remove the read-only bit.  It's OK, since we're
    * probably gonna reset it after the file is copied
    */
   if ((temp = _rtl_chmod(to_file, 0)) >= 0) {
      if (temp & FA_RDONLY) {
         temp &= ~(FA_RDONLY);
         _rtl_chmod(to_file, 1, temp);
      }
   }

   if ((to_fp = fopen(to_file, "wb")) == NULL) {
      fprintf(stderr, "Can't open \"%s\" for write\n", to_file);
      warning();
      fclose(from_fp);
      return(1);
   }

   while ((num = fread(buf, sizeof(char), BUFSIZ, from_fp)) > 0) {
      if (fwrite(buf, sizeof(char), num, to_fp) != num) {
         fprintf(stderr, "Write error on \"%s\"\n", to_file);
         warning();
         fclose(from_fp);
         fclose(to_fp);
         return(1);
      }
   }

   if (modtime) {
      /*
       * Any writes after setftime() negate its effect, so the fflush()
       * is required here
       */
      fflush(to_fp);
      setftime(fileno(to_fp), &ftime);
   }
   fclose(from_fp);
   fclose(to_fp);
   if (attrib)
      _rtl_chmod(to_file, 1, attr);

   return(0);
}

/*
 * Create a directory
 */

int
createdir(from_dir, to_dir)
char *from_dir, *to_dir;
{
   int attr;
   void warning(void);

   if (attrib)
      attr = _rtl_chmod(from_dir, 0);

   if (mkdir(to_dir)) {
      int temp;
      /*
       * If the directory already exists, the error code returned by mkdir()
       * is the same as when there is a "bad path" error. So, to see if this
       * error is benign, we must check to see if the directory really exists
       */
      if ((temp = _rtl_chmod(to_dir, 0)) >= 0) {
         if (temp & FA_DIREC) {
                        /* if exists, just change attributes */
            if (attrib)
               _rtl_chmod(to_dir, 1, attr);
            return(0);
         }
      }
      fprintf(stderr, "Can't make directory \"%s\"\n", to_dir);
      warning();
      return(1);
   }
   if (attrib)
      _rtl_chmod(to_dir, 1, attr);
   return(0);
}

/*
 * The "usage" error message
 */

void
usage()
{
   fprintf(stderr, "%s, Version %s, Dated %s\n", NAME, VERSION, DATED);
   fprintf(stderr, "Usage: %s [-s] [-h] [-a] [-m] [-p] [-q] srcdir destdir\n", NAME);
   fprintf(stderr, "\t-s include subdirectories (including empty directories)\n");
   fprintf(stderr, "\t-h include hidden and system files\n");
   fprintf(stderr, "\t-a retain file attributes\n");
   fprintf(stderr, "\t-m retain file modification times\n");
   fprintf(stderr, "\t-p prompt to overwrite existing files\n");
   fprintf(stderr, "\t-q operate quietly\n");
   return;
}

/*
 * A simple warning message
 */

void
warning()
{
   char ans[10];

   if (!quiet) {
      fprintf(stderr, "\aPress Enter to continue");
      fgets(ans, 10, stdin);
   }
   return;
}

/*
 * The Unix-style option parsing routine
 */

int
getopt(argc, argv, opts)
int argc;
char *argv[];
char *opts;
{
   static int sp = 1;
   int c;
   char *cp;

   if (sp == 1) {
      if (optind >= argc || argv[optind][0] != '-' || argv[optind][1] == '\0')
         return(EOF);
      else if (strcmp(argv[optind], "--") == NULL) {
         optind++;
         return(EOF);
      }
   }
   c = argv[optind][sp];
   if (c == ':' || (cp = strchr(opts, c)) == NULL) {
      fprintf(stderr, "Illegal option '%c'\n", c);
      if (argv[optind][++sp] == '\0') {
         optind++;
         sp = 1;
      }
      return('?');
   }
   if (*++cp == ':') {
      if (argv[optind][sp + 1] != '\0')
         optarg = &argv[optind++][sp + 1];
      else if (++optind >= argc) {
         fprintf(stderr, "Option '%c' requires an argument\n", c);
         sp = 1;
         return('?');
      }
      else
         optarg = argv[optind++];
      sp = 1;
   }
   else {
      if (argv[optind][++sp] == '\0') {
         sp = 1;
         optind++;
      }
      optarg = NULL;
   }
   return(c);
}
