// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: gxbtree.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 08/22/2000 
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The gxBtree class is a disk-base B-tree used by various database 
applications for searching and sorting structured data and indexing 
associated objects or records stored in a data file. The gxBtree 
uses the 32/64-bit gxDatabase engine to perform all disk operations, 
supporting large files and proprietary file system interfaces. 
*/
// ----------------------------------------------------------- // 
#include <string.h>
#include "gxbtree.h"

gxBtree::gxBtree(DatabaseKeyB &key_type, BtreeNodeOrder_t order, gxClassID cid) 
{
  btree_header.node_order = order;
  btree_header.key_size = key_type.SizeOfDatabaseKey();
  btree_header.class_id = cid;
  header_in_sync = 0;
  f = 0;
}

gxBtree::~gxBtree()
{
  Close(); 
}

gxDatabaseError gxBtree::InitBtree(gxDatabase *fptr, int create, 
				   FAU header_address)
// Connect the Btree to the database engine and write a new Btree header if
// the "create" variable is true. Returns zero if successful or a non-zero 
// value to indicate a failure.
{
  f = fptr; 
  btree_header_address = header_address;

  if(create) {
    BtreeNode root_node(btree_header.key_size, btree_header.node_order); 
    FAU root_address = f->Alloc(root_node.SizeOfBtreeNode());

    // Check for block allocation errors
    if(root_address == (FAU)0) return f->GetDatabaseError();

    btree_header.root = root_address;
    btree_header.btree_height = (BtreeSize_t)1;
    btree_header.n_keys = (BtreeSize_t)0;
    btree_header.n_nodes = (BtreeSize_t)1;
    if(WriteBtreeHeader() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
    if(WriteNode(root_node, root_address) != gxDBASE_NO_ERROR) 
      return f->GetDatabaseError();
  }
  else {
    if(ReadBtreeHeader() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}


gxDatabaseError gxBtree::InitBtree(int create, FAU header_address)
// Connect the Btree to a header in the open database engine or write a
// new Btree header if the "create" variable is true. Returns zero if
// successful or a non-zero value to indicate a failure.
{
  btree_header_address = header_address;
  
  if(create) {
    BtreeNode root_node(btree_header.key_size, btree_header.node_order); 
    FAU root_address = f->Alloc(root_node.SizeOfBtreeNode());

    // Check for block allocation errors
    if(root_address == (FAU)0) return f->GetDatabaseError();

    btree_header.root = root_address;
    btree_header.btree_height = (BtreeSize_t)1;
    btree_header.n_keys = (BtreeSize_t)0;
    btree_header.n_nodes = (BtreeSize_t)1;
    if(WriteBtreeHeader() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
    if(WriteNode(root_node, root_address) != gxDBASE_NO_ERROR) 
      return f->GetDatabaseError();
  }
  else {
    if(ReadBtreeHeader() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

void gxBtree::Release()
// Reset the database file pointer to zero without closing the
// file or performing any other action. NOTE: This function is
// used to reset the file pointer when more then one object is
// referencing this file pointer and the file has been closed
// or the pointer has been deleted.
{
  f = 0;
}

gxDatabaseError gxBtree::Close()
// Close the Btree. Returns zero if successful or a non-zero 
// value to indicate a failure.
{
  if(f) {
    if(WriteBtreeHeader() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
    delete f; // Database engine closed by destructor call
    f = 0;              
  }
  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::Open(const char *fname, gxDatabaseAccessMode mode)
// Open an existing Btree file. Returns zero if successful or a non-zero 
// value to indicate a failure. NOTE: This function assumes that there is
// only one Btree in this file.
{
  if(Close() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
  f = new gxDatabase;
  if(!f) {
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }

  if(f->Open(fname, mode) != gxDBASE_NO_ERROR)
    return f->GetDatabaseError();

  return InitBtree(0, (FAU_t)f->FileHeaderSize());
}

gxDatabaseError gxBtree::Create(const char *fname, int num_trees)
// Create a new Btree file, specifying the number of trees this
// index will contain. NOTE: This create function only initialize
// the first tree header and will always connect the tree to the
// first tree header. Returns zero if successful or a non-zero
// value to indicate a failure.
{
  if(Close() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
  f = new gxDatabase;
  if(!f) {
    f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return f->GetDatabaseError();
#endif
  }
  
  FAU static_size = FAU_t(sizeof(gxBtreeHeader) * num_trees);
  if(f->Create(fname, static_size) != gxDBASE_NO_ERROR)
    return f->GetDatabaseError();

  // Connect to the first tree in the index file
  return InitBtree(1, (FAU_t)f->FileHeaderSize());
}

gxDatabaseError gxBtree::Open(const char *fname, FAU header_address,
			     gxDatabaseAccessMode mode)
// Open an existing Btree file, specifying the file address of the Btree
// header. Returns zero if successful or a non-zero value to indicate a 
// failure.
{
  if(Close() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
  f = new gxDatabase;
  if(!f) {
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }

  if(f->Open(fname, mode) != gxDBASE_NO_ERROR)
    return f->GetDatabaseError();

  return InitBtree(0, header_address);
}

gxDatabaseError gxBtree::WriteBtreeHeader()
// Write the Btree header. Returns zero if successful or 
// a non-zero value to indicate a failure.
{
  if(f->ReadyForWriting()) {
    if(f->Write(&btree_header, sizeof(gxBtreeHeader), btree_header_address) !=
       gxDBASE_NO_ERROR)
      return f->GetDatabaseError();
    header_in_sync = 1;
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::ReadBtreeHeader()
// Read the Btree header. Returns zero if successful or 
// a non-zero value to indicate a failure.
{
  if(f->Read(&btree_header, sizeof(gxBtreeHeader), btree_header_address) !=
     gxDBASE_NO_ERROR)
    return f->GetDatabaseError();

  header_in_sync = 1;
  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::Flush()
// Write the Btree header and flush any open disk buffers. Returns zero if 
// successful or a non-zero value to indicate a failure.

{
  if(WriteBtreeHeader() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
  return f->Flush();
}

size_t gxBtree::TotalNodeSize()
// Returns the total size of a single Btree node, including overhead.
{
  BtreeNode node(btree_header.key_size, btree_header.node_order);
  size_t node_size = node.SizeOfBtreeNode();
  node_size += f->BlockHeaderSize();
  
  // Adjust the node size according to the GXD revision letter.
  if((f->GetRevLetter() == '\0') || (f->GetRevLetter()  == ' ')) {
    return node_size;
  }
  else {
    node_size += sizeof(gxChecksum); // Rev 'A' and higher
    return node_size;
  }
}

gxDatabaseError gxBtree::WriteNode(const BtreeNode &node, FAU node_address)
// Write a Btree node to the specified file address. Returns zero if 
// successful or a non-zero value to indicate a failure.
{
  if(f->Write(&node.key_count, sizeof(node.key_count), node_address, 0, 0) 
     != gxDBASE_NO_ERROR)
    return f->GetDatabaseError();
  if(f->Write(&node.left_child, sizeof(node.left_child), gxCurrAddress, 0, 0) 
     != gxDBASE_NO_ERROR)
    return f->GetDatabaseError();
  
  // Optimize the write by only writing the exact number of keys
  if(node.key_count > (BtreeKeyCount_t)0) { // Ensure node has entries
    if(f->Write(node.key_entries, (node.key_size * node.key_count),
		gxCurrAddress, 0, 0) != gxDBASE_NO_ERROR)
     return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::ReadNode(BtreeNode &node, FAU node_address)
// Read a Btree node from the specified file address. Returns zero if 
// successful or a non-zero value to indicate a failure.
{
  if(f->Read(&node.key_count, sizeof(node.key_count), node_address) !=
     gxDBASE_NO_ERROR)
    return f->GetDatabaseError();
  if(f->Read(&node.left_child, sizeof(node.left_child)) != gxDBASE_NO_ERROR)
    return f->GetDatabaseError();

  // Optimize the read by only reading the exact number of keys
  if(node.key_count > (BtreeKeyCount_t)0) { // Ensure node has entries
    if(f->Read(node.key_entries, (node.key_size * node.key_count)) != 
       gxDBASE_NO_ERROR)
      return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::GrowNewRoot(DatabaseKeyB &key)
// Grow new root node. Returns zero if successful or a 
// non-zero value to indicate a failure.
{
  FAU old_root = btree_header.root;
  BtreeNode new_root_node(btree_header.key_size, btree_header.node_order);
  FAU root_address = f->Alloc(new_root_node.SizeOfBtreeNode());
  if(root_address == (FAU)0) f->GetDatabaseError();
  btree_header.n_nodes++; 
  btree_header.btree_height++;
  btree_header.root = root_address;
  header_in_sync = 0;
  new_root_node.left_child = old_root;
  new_root_node.InsertDatabaseKey(key, (BtreeKeyLocation_t)0);
  return WriteNode(new_root_node, root_address);
}

gxDatabaseError gxBtree::CollapseRoot()
// Collapse the current root node and shorten the tree. Returns zero 
// if successful or a non-zero value to indicate a failure.
{
  BtreeNode root_node(btree_header.key_size, btree_header.node_order);
  if(ReadNode(root_node, btree_header.root) != gxDBASE_NO_ERROR)
    return f->GetDatabaseError(); 

  // Ensure that the root node is empty and the height is greater then one
  if((root_node.IsEmpty()) && (btree_header.btree_height > (BtreeSize_t)1)) {
    FAU old_root = btree_header.root;
    btree_header.root = root_node.left_child;
    btree_header.n_nodes--; 
    btree_header.btree_height--;
    header_in_sync = 0;
    if(!f->Remove(old_root)) return f->GetDatabaseError(); 
  }
 
  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::CollapseRoot(BtreeNode &root_node)
// Collapse the specified root node after a read operation 
// and shorten the tree. Returns zero if successful or a 
// non-zero value to indicate a failure.
{
  // Ensure that the root node is empty and the height is greater then one
  if((root_node.IsEmpty()) && (btree_header.btree_height > (BtreeSize_t)1))  {
    FAU old_root = btree_header.root;
    btree_header.root = root_node.left_child;
    btree_header.n_nodes--; 
    btree_header.btree_height--;
    header_in_sync = 0;
    if(!f->Remove(old_root)) return f->GetDatabaseError(); 
  }
 
  return gxDBASE_NO_ERROR;
}

int gxBtree::Insert(DatabaseKeyB &key, DatabaseKeyB &compare_key,
		   int flush)
// Insert the specified key. Returns true if successful or false 
// if the key could not be inserted.
{
  if(InsertDatabaseKey(key, compare_key) != gxDBASE_NO_ERROR) return 0;

  // Ensure that the in memory buffers and the file data
  // stay in sync during multiple file access.
  if(flush) Flush();

  return 1;
}

int gxBtree::Find(DatabaseKeyB &key, DatabaseKeyB &compare_key, int test_tree)
// Find the specified key in the Btree. Returns true if successful
// or false if the key could not be found.
{
  // Ensure that the in memory buffers and the file data
  // stay in sync during multiple file access.
  if(test_tree) TestTree();
  
  FAU curr_address = btree_header.root;
  BtreeKeyLocation_t key_location;
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  while(curr_address != (FAU)0) {
    if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
    if(curr_node.FindKeyLocation(key, compare_key, key_location) == 0) {
      // Load the key data in the event that this is a partial look up
      curr_node.LoadKey(key, key_location); 
      return 1; // Found a matching node
    }
    curr_address = curr_node.GetBranch(key_location);
  }
  
  f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
  return 0;
}

int gxBtree::FindFirst(DatabaseKeyB &key, int test_tree)
// Find the first key in the Btree. Returns true if successful
// or false if the key could not be found.
{
  // Ensure that the in memory buffers and the file data
  // stay in sync during multiple file access.
  if(test_tree) TestTree();

  if(IsEmpty()) return 0;

  FAU curr_address = btree_header.root;
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
  
  while(curr_node.left_child != (FAU)0) {
    curr_address = curr_node.left_child;
    if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
  }
  if(curr_node.key_count <= 0) {
    f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
    return 0;
  }
  curr_node.LoadKey(key, (BtreeKeyLocation_t)0);
  return 1;
}

int gxBtree::FindNext(DatabaseKeyB &key, DatabaseKeyB &compare_key,
		     int test_tree)
// Find the next key after the specified key, passing back the key in the
// "key" variable. Returns true if successful or false if the key could not 
// be found.
{
  // Ensure that the in memory buffers and the file data
  // stay in sync during multiple file access.
  if(test_tree) TestTree();
  
  FAU curr_address = btree_header.root;
  FAU parent_address = (FAU)0;
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  BtreeNode parent_node(btree_header.key_size, btree_header.node_order);
  int rv;
  BtreeKeyLocation_t key_location, sav_location;
 
  while(curr_address != (FAU)0) { 
    if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
    rv = curr_node.FindKeyLocation(key, compare_key, key_location);
    if(rv == 0) { // Found matching key, search for the next key in sort order
      if(curr_node.GetBranch(key_location) != (FAU)0) {
        curr_address = curr_node.GetBranch(key_location);
        if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
	while(curr_node.left_child != (FAU)0) { // Get left most node
          curr_address = curr_node.left_child;
          if(ReadNode(curr_node, curr_address)!= gxDBASE_NO_ERROR) return 0;
	}
        if(curr_node.key_count <= (BtreeKeyCount_t)0) {
	  f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
	  return 0;
	}
        curr_node.LoadKey(key, (BtreeKeyLocation_t)0);
	return 1;
      }
      else {
        if(key_location < curr_node.key_count-1) {
          curr_node.LoadKey(key, key_location+1);
	  return 1;
	}
	if(parent_address != (FAU)0) { // Go back to the parent node
          if(ReadNode(parent_node, parent_address)!= gxDBASE_NO_ERROR) 
	    return 0;
          if(parent_node.key_count <= (BtreeKeyCount_t)0) {
	    f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
	    return 0;
	  }
          if(++sav_location >= parent_node.key_count) {
	    f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
	    return 0;
	  }
          parent_node.LoadKey(key, sav_location);
	  return 1;
	}
	return 0;
      }
    }
    if(key_location < curr_node.key_count-1) {
      parent_address = curr_address;
      sav_location = key_location;
    }
    curr_address = curr_node.GetBranch(key_location); // No match, keep walking
  }

  f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
  return 0;
}

int gxBtree::FindPrev(DatabaseKeyB &key, DatabaseKeyB &compare_key,
		     int test_tree)
// Find the previous key before the specified key, passing back the key in the
// "key" variable. Returns true if successful or false if the key could not 
// be found.
{
  // Ensure that the in memory buffers and the file data
  // stay in sync during multiple file access.
  if(test_tree) TestTree();
  
  FAU curr_address = btree_header.root;
  FAU parent_address = (FAU)0;
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  BtreeNode parent_node(btree_header.key_size, btree_header.node_order);
  int rv;
  BtreeKeyLocation_t key_location;
  BtreeKeyLocation_t sav_location = -1;

  while(curr_address != (FAU)0) {
    if(ReadNode(curr_node, curr_address)!= gxDBASE_NO_ERROR) return 0;
    rv = curr_node.FindKeyLocation(key, compare_key, key_location);
    if(rv == 0) {
      // Found matching key, search for the previous key in sort order 
      if (--key_location >= (BtreeKeyLocation_t)0) {
        while(curr_node.GetBranch(key_location) != (FAU)0) {
          curr_address = curr_node.GetBranch(key_location);
          if(ReadNode(curr_node, curr_address)!= gxDBASE_NO_ERROR) return 0;
          key_location = curr_node.key_count-1;
	}
        curr_node.LoadKey(key, key_location);
	return 1;
      }
      if(curr_node.left_child != (FAU)0) {
        curr_address = curr_node.left_child;
        if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
        key_location = curr_node.key_count-1;
        while(curr_node.GetBranch(key_location) != (FAU)0) {
          curr_address = curr_node.GetBranch(key_location);
          if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;
          key_location = curr_node.key_count-1;
	}
        curr_node.LoadKey(key, key_location);
	return 1;
      }
      if(parent_address != (FAU)0) { // Go back to the parent node
        parent_node.LoadKey(key, sav_location);
	return 1;
      }
      f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
      return 0;
    }
    else if (rv > 0) { // Save the parent node only if it branches to the right
      parent_address = curr_address;
      if(ReadNode(parent_node, parent_address)!= gxDBASE_NO_ERROR) return 0;
      sav_location = key_location;
    }
    curr_address = curr_node.GetBranch(key_location); // Keep walking 
  }
  f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
  return 0;
}

int gxBtree::FindLast(DatabaseKeyB &key, int test_tree)
// Find the last key in the Btree. Returns true if successful
// or false if the key could not be found.
{
  // Ensure that the in memory buffers and the file data
  // stay in sync during multiple file access.
  if(test_tree) TestTree();
  
  if (IsEmpty()) return 0;
  
  FAU curr_address = btree_header.root;
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) return 0;

  while(curr_node.GetBranch((BtreeKeyLocation_t)curr_node.key_count-1)) {
    curr_address = \
      curr_node.GetBranch((BtreeKeyLocation_t)curr_node.key_count-1);
    if(ReadNode(curr_node, curr_address)!= gxDBASE_NO_ERROR) return 0;
  }
  if(curr_node.key_count <= (BtreeKeyCount_t)0) return 0;
  curr_node.LoadKey(key, curr_node.key_count-1);
  return 1;
}

int gxBtree::Delete(DatabaseKeyB &key, DatabaseKeyB &compare_key, int flush)
// Delete the specified key and balance the Btree following the 
// deletion. Returns true if successful or false if the key could not 
// be deleted.

{
  if(DeleteDatabaseKey(key, compare_key) != gxDBASE_NO_ERROR) return 0;

  // Ensure that the Btree file header stays in sync
  // during multiple file access.
  if(flush) Flush();

  return 1;
}

int gxBtree::LazyDelete(DatabaseKeyB &key, DatabaseKeyB &compare_key,
			int flush)
// Delete the specified key without balancing the tree following the 
// deletion. Returns true if successful or false if the key could not 
// be deleted.
{
  if(LazyDeleteDatabaseKey(key, compare_key) != gxDBASE_NO_ERROR) return 0;

  // Ensure that the Btree file header stays in sync
  // during multiple file access.
  if(flush) Flush();

  return 1;
}

gxDatabaseError gxBtree::InsertDatabaseKey(DatabaseKeyB &key,
					   DatabaseKeyB &compare_key)
// Non-recursive insertion function used to insert a key into the Btree.
// Returns zero if successful or a non-zero value to indicate a failure.
{
  BtreeKeyLocation_t key_location;
  FAU curr_address;
  key.right_child = (FAU)0;
  curr_address = btree_header.root;
  BtreeStack node_stack;
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  int num_splits = 0;

  // Walk to the leaf where the key will be inserted and look duplicate keys
  while(curr_address != (FAU)0) {
    if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR)
      return f->GetDatabaseError();
    if(!node_stack.Push(curr_address)) {
      f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
      throw gxCDatabaseException();
#else
      return f->GetDatabaseError();
#endif
    }

    if(curr_node.FindKeyLocation(key, compare_key, key_location) == 0) {
      // Found a duplicate key
      node_stack.Clear();
      f->SetDatabaseError(gxDBASE_DUPLICATE_ENTRY);
#ifdef __CPP_EXCEPTIONS__
      throw gxCDatabaseException();
#else
      return f->GetDatabaseError();
#endif
    }
    
    // Continue the search until a node with no children is found
    curr_address = curr_node.GetBranch(key_location);
  }

  while(!node_stack.IsEmpty()) {
    curr_address = node_stack.Pop();
    if(num_splits > 0) {
      if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR)
	return f->GetDatabaseError();
      curr_node.FindKeyLocation(key, compare_key, key_location);
    }
    key_location++; // Move to the insertion key_location
    
    if(!curr_node.IsFull()) { // Add the key if the node is not full
      curr_node.InsertDatabaseKey(key, key_location);
      node_stack.Clear();
      if(WriteNode(curr_node, curr_address) != gxDBASE_NO_ERROR)
	return f->GetDatabaseError();
      btree_header.n_keys++;
      header_in_sync = 0;
      return gxDBASE_NO_ERROR;
    }
    else { // Split the node
      num_splits++;
      BtreeNode right_node(btree_header.key_size, btree_header.node_order);
      FAU right_address = f->Alloc(right_node.SizeOfBtreeNode());

      // Check for block allocation errors
      if(right_address == (FAU)0) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }
      btree_header.n_nodes++;
      header_in_sync = 0;
      
      BtreeKeyLocation_t split_location = btree_header.node_order/2;
      if(key_location < split_location) split_location--;
      curr_node.SplitNode(right_node, split_location);
      
      if(key_location > split_location) { // Insert key in the right node
	right_node.LoadKey(compare_key, (BtreeKeyLocation_t)0);
	right_node.DeleteDatabaseKey((BtreeKeyLocation_t)0);
        right_node.InsertDatabaseKey(key, key_location-split_location-1);
	
	// Pass the middle key up the tree and insert into its parent node
	memmove(key.db_key, compare_key.db_key, key.KeySize()); 
	key.right_child = compare_key.right_child;
      }
      else if(key_location < split_location) { // Insert key in the left node
        curr_node.InsertDatabaseKey(key, key_location);
	right_node.LoadKey(compare_key, (BtreeKeyLocation_t)0);
	
	// Pass the middle key up the tree and insert into its parent node
	memmove(key.db_key, compare_key.db_key, key.KeySize()); 
	key.right_child = compare_key.right_child;
	right_node.DeleteDatabaseKey((BtreeKeyLocation_t)0);
      }
      
      right_node.left_child = key.right_child;
      key.right_child = right_address;

      if(WriteNode(right_node) != gxDBASE_NO_ERROR) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }
      if(WriteNode(curr_node, curr_address) != gxDBASE_NO_ERROR) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }
      
      if(curr_address == btree_header.root) {
	node_stack.Clear();
	if(GrowNewRoot(key) != gxDBASE_NO_ERROR)
	  return f->GetDatabaseError();
	btree_header.n_keys++;
	header_in_sync = 0;
	return gxDBASE_NO_ERROR;
      }
    }
  }   

  // The insertion was not completed
  node_stack.Clear();
  f->SetDatabaseError(gxDBASE_INSERTION_ERROR);
#ifdef __CPP_EXCEPTIONS__
  throw gxCDatabaseException();
#endif
  return f->GetDatabaseError();
}

gxDatabaseError gxBtree::LazyDeleteDatabaseKey(DatabaseKeyB &key,
					       DatabaseKeyB &compare_key)
// Lazy (unbalanced) deletion method implemented specifically to support 
// concurrent database operations that are effected by the excessive node 
// splitting and merging that occurs during tree rotations. Returns zero 
// if successful or a non-zero value to indicate a failure. 
{
  BtreeKeyLocation_t key_location;
  FAU parent_address;
  FAU sibling_address;
  BtreeStack node_stack;
  BtreeNode parent_node(btree_header.key_size, btree_header.node_order);
  BtreeNode sibling_node(btree_header.key_size, btree_header.node_order);
  
  int found_key = 0;
  int num_siblings = 0;
  key.right_child = (FAU)0;
  parent_address = btree_header.root;

  while(parent_address != (FAU)0) {
    if(ReadNode(parent_node, parent_address) != gxDBASE_NO_ERROR)
      return f->GetDatabaseError();

    if(parent_node.FindKeyLocation(key, compare_key, key_location) == 0) {
      sibling_address = parent_node.GetBranch(key_location);  
      found_key = 1;
      break; // Found the specified key
    }
    
    // Continue the searching for the spcified key
    parent_address = parent_node.GetBranch(key_location);  
  }

  if(!found_key) {
    f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return f->GetDatabaseError();
#endif    
  }
  
  if(sibling_address == (FAU)0) { // This is leaf node 
    parent_node.DeleteDatabaseKey(key_location);
    if(WriteNode(parent_node, parent_address) != gxDBASE_NO_ERROR)
      return f->GetDatabaseError();
    btree_header.n_keys--;
    header_in_sync = 0;

    // Check the root node and collapse if empty
    if(CollapseRoot() != gxDBASE_NO_ERROR) return f->GetDatabaseError();
    return gxDBASE_NO_ERROR;
  }
  else { // This is a parent node
    num_siblings++;
    if(!node_stack.Push(sibling_address)) {
      f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
      throw gxCDatabaseException();
#else
      return f->GetDatabaseError();
#endif
    }
    
    if(ReadNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }
    
    // Follow the siblings to the parent node's left most child
    while(sibling_node.left_child != (FAU)0) {
      sibling_address = sibling_node.left_child;
      if(ReadNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }
      if(!node_stack.Push(sibling_address)) {
	node_stack.Clear();
	f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
	throw gxCDatabaseException();
#else
	return f->GetDatabaseError();
#endif
      }
      num_siblings++;
    }
    
    // Skip over the first entry
    sibling_address = node_stack.Pop();

    // This sibling is empty due to a prior deletion
    if(sibling_node.IsEmpty()) {

      // Remove the empty node
      if(!f->Remove(sibling_address)) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }

      btree_header.n_nodes--;
      header_in_sync = 0;
      num_siblings--;

      // Continue deleting the empty siblings
      while(!node_stack.IsEmpty()) {
	sibling_address = node_stack.Pop();
	if(ReadNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
	  node_stack.Clear();
	  return f->GetDatabaseError(); 
	}
	if(!sibling_node.IsEmpty()) break;
	
	if(!f->Remove(sibling_address)) {
	  node_stack.Clear();
	  return f->GetDatabaseError();
	}
	btree_header.n_nodes--;
	header_in_sync = 0;
	num_siblings--;
      }
    } 

    if(num_siblings == 0) { // Did not find any siblings containing keys
      parent_node.DeleteDatabaseKey(key_location);
      if(WriteNode(parent_node, parent_address) != gxDBASE_NO_ERROR) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }
      btree_header.n_keys--;
      header_in_sync = 0;

      // The root node is node empty
      if((parent_node.IsEmpty()) && (parent_address == btree_header.root)) {
	if(CollapseRoot(parent_node) != gxDBASE_NO_ERROR) {
	  node_stack.Clear();
	  return f->GetDatabaseError();
	}
      }
      return gxDBASE_NO_ERROR;
    }
  }

  // Take a the first key from the sibling and overwite the key
  // marked for deletion in parent node
  sibling_node.LoadKey(key, (BtreeKeyLocation_t)0);
  sibling_node.left_child = key.right_child;
  sibling_node.DeleteDatabaseKey((BtreeKeyLocation_t)0);
  key.right_child = compare_key.right_child; 
  parent_node.OverWriteDatabaseKey(key, key_location);
  if(WriteNode(parent_node, parent_address) != gxDBASE_NO_ERROR) {
    node_stack.Clear();
    return f->GetDatabaseError();
  }
  if(WriteNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
    node_stack.Clear();
    return f->GetDatabaseError();
  }
  btree_header.n_keys--;
  header_in_sync = 0;
  node_stack.Clear();
  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::DeleteDatabaseKey(DatabaseKeyB &key,
					   DatabaseKeyB &compare_key)
// Non-recursive balanced Btree delete function used to ensure that all
// nodes, with the exception of the root node, are at least half full 
// following a deletion. Returns zero if successful or a non-zero value 
// to indicate a failure.
{
  BtreeKeyLocation_t key_location;
  FAU curr_address;
  FAU sibling_address;
  BtreeStack node_stack;  
  BtreeNode curr_node(btree_header.key_size, btree_header.node_order);
  BtreeNode *leaf_node;
  BtreeNode parent_node(btree_header.key_size, btree_header.node_order);
  BtreeNode sibling_node(btree_header.key_size, btree_header.node_order);
  int found_key = 0;
  key.right_child = (FAU)0;
  curr_address = btree_header.root;

  if(!node_stack.Push(btree_header.root)) {
    node_stack.Clear();
    f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return f->GetDatabaseError();
#endif
  }
  
  // Search the Btree for the specified key
  while(curr_address != (FAU)0) {
    if(ReadNode(curr_node, curr_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }

    if(curr_node.FindKeyLocation(key, compare_key, key_location) == 0) {
      sibling_address = curr_node.GetBranch(key_location);  
      found_key = 1;
      break; // Found the specified key
    }
    
    // Push parent and grandparent addresses for rotations
    // that must be propogated up the tree
    if(!node_stack.Push(curr_address)) {
      node_stack.Clear();
      f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
      throw gxCDatabaseException();
#else
      return f->GetDatabaseError();
#endif
    }
    
    // Continue the searching for the key
    curr_address = curr_node.GetBranch(key_location);  
  }

  if(!found_key) { // The key was not found
    f->SetDatabaseError(gxDBASE_ENTRY_NOT_FOUND);
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return f->GetDatabaseError();
#endif    
  }

  if(sibling_address != (FAU)0) { // This is a parent node
    // Push the siblings parent node
    if(!node_stack.Push(curr_address)) {
      node_stack.Clear();
      f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
      throw gxCDatabaseException();
#else
      return f->GetDatabaseError();
#endif
    }
    if(ReadNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }
    
    // Follow the siblings to the parent node's left most child
    while(sibling_node.left_child != (FAU)0) {

      // Push all parent and grandparent nodes
      if(!node_stack.Push(sibling_address)) {
	node_stack.Clear();
	f->SetDatabaseError(gxDBASE_MEM_ALLOC_ERROR);
#ifdef __CPP_EXCEPTIONS__
	throw gxCDatabaseException();
#else
	return f->GetDatabaseError();
#endif
      }
      sibling_address = sibling_node.left_child;
      if(ReadNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
	node_stack.Clear();
	return f->GetDatabaseError();
      }
    }
    
    // Take a the first key from the sibling and overwite the key
    // marked for deletion in parent node
    sibling_node.LoadKey(key, (BtreeKeyLocation_t)0); // Promote this key
    sibling_node.DeleteDatabaseKey((BtreeKeyLocation_t)0);
    key.right_child = compare_key.right_child; 
    curr_node.OverWriteDatabaseKey(key, key_location);
    if(WriteNode(curr_node, curr_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }
    if(WriteNode(sibling_node, sibling_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }
    btree_header.n_keys--;
    header_in_sync = 0;
    parent_node.node_address = node_stack.Pop();
    leaf_node = &sibling_node;
    leaf_node->node_address = sibling_address;
  }
  else { // This is a leaf (node with no children)
    curr_node.DeleteDatabaseKey(key_location);
    if(WriteNode(curr_node, curr_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }
    btree_header.n_keys--;
    header_in_sync = 0;
    parent_node.node_address = node_stack.Pop();
    leaf_node = &curr_node;
    leaf_node->node_address = curr_address;
  }

  // Insert balance if the node does not contain the minimum number of entries
  if((leaf_node->HasFew()) && (leaf_node->node_address != btree_header.root)) {
    if(ReadNode(parent_node, parent_node.node_address) != gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }
    parent_node.FindKeyLocation(key, compare_key, key_location);
    if(InsertBalance(parent_node, key_location, compare_key) !=
       gxDBASE_NO_ERROR) {
      node_stack.Clear();
      return f->GetDatabaseError();
    }

    // Ensure that the node's parent contains the minimum number of entries
    // NOTE: The root node is allowed to have less then the minimum number
    // of entires and may be empty.
    if((parent_node.HasFew()) && 
       (parent_node.node_address != btree_header.root)) {
      while(!node_stack.IsEmpty()) {
	// Loop until all the grandparent nodes are balanced
	BtreeNode grandparent_node(btree_header.key_size, 
				   btree_header.node_order);
	grandparent_node.node_address = node_stack.Pop();
	if(ReadNode(grandparent_node, grandparent_node.node_address) !=
	   gxDBASE_NO_ERROR) {
	  node_stack.Clear();
	  return f->GetDatabaseError();
	}
        grandparent_node.FindKeyLocation(key, compare_key, key_location);
        if(InsertBalance(grandparent_node, key_location, compare_key) !=
	   gxDBASE_NO_ERROR) {
	  node_stack.Clear();
	  return f->GetDatabaseError();
	}
	if((grandparent_node.HasFew()) && 
	   (grandparent_node.node_address != btree_header.root)) {
	  continue;
	}
	else {
	  break;
	}
      }
    }
  }
    
  // Clear the node stack
  node_stack.Clear();
  
  // Check the root node and collapse if empty
  if(CollapseRoot() != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::InsertBalance(BtreeNode &parent_node, 
				       BtreeKeyLocation_t key_location,
				       DatabaseKeyB &compare_key)
// Internal function used to balance this parent's sibling node after 
// performing a deletion. Returns zero if successful or a non-zero value 
// to indicate a failure.
{
  FAU left_child;
  FAU right_child;
  BtreeNode left_node(btree_header.key_size, btree_header.node_order);
  BtreeNode right_node(btree_header.key_size, btree_header.node_order);
  
  if(key_location == (BtreeKeyLocation_t)-1) {
    // The parent node does not have a left child
    right_child = parent_node.GetBranch((BtreeKeyLocation_t)0);
    if(ReadNode(right_node, right_child) != gxDBASE_NO_ERROR) {
      return f->GetDatabaseError();
    }
    if(right_node.HasMany()) { // Take a key from the right child
      if(InsertLeftRotation(parent_node, (BtreeKeyLocation_t)0,
			    compare_key) != gxDBASE_NO_ERROR) {
	return f->GetDatabaseError();
      } 
    }
    else { 
      if(Merge(parent_node, (BtreeKeyLocation_t)0, compare_key) !=
	 gxDBASE_NO_ERROR) {
	return f->GetDatabaseError();
      }
    }
  }
  else if(key_location == parent_node.HighestKey()) {
    // The parent node does not have a right child
    left_child = parent_node.GetBranch(key_location-1);
    if(ReadNode(left_node, left_child) != gxDBASE_NO_ERROR) {
      return f->GetDatabaseError();
    }
    if(left_node.HasMany()) { // Take a key from the left child
      if(InsertRightRotation(parent_node, key_location, compare_key) !=
	 gxDBASE_NO_ERROR) {
	return f->GetDatabaseError();
      }
    }
    else {
      if(Merge(parent_node, key_location, compare_key) != gxDBASE_NO_ERROR) {
	return f->GetDatabaseError();
      }
    }
  }
  else {
    // The parent node has both left and right siblings
    left_child = parent_node.GetBranch(key_location-1);
    if(ReadNode(left_node, left_child) != gxDBASE_NO_ERROR) {
      return f->GetDatabaseError();
    }
    if(left_node.HasMany()) { // Take a key from the left child
      if(InsertRightRotation(parent_node, key_location, compare_key) !=
	 gxDBASE_NO_ERROR) {
	return f->GetDatabaseError();
      }
    }
    else {
      right_child = parent_node.GetBranch(key_location+1);
      if(ReadNode(right_node, right_child) != gxDBASE_NO_ERROR) {
	return f->GetDatabaseError();
      }
      if(right_node.HasMany()) { // Take a key from the right child
        if(InsertLeftRotation(parent_node, key_location+1, compare_key) !=
	   gxDBASE_NO_ERROR) {
	  return f->GetDatabaseError();
	}
      }
      else {
        if(Merge(parent_node, key_location, compare_key) != gxDBASE_NO_ERROR) {
	  return f->GetDatabaseError();
	}
      }
    }
  }
  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::InsertRightRotation(BtreeNode &parent_node, 
					     BtreeKeyLocation_t key_location,
					     DatabaseKeyB &compare_key)
// Internal function used by the gxBtree::InsertBalance() function to perform 
// a right rotation. Returns zero if successful or a non-zero value to
// indicate a failure.
{
  FAU right_child = parent_node.GetBranch(key_location);
  FAU left_child = parent_node.GetBranch(key_location-1);
  BtreeNode right_node(btree_header.key_size, btree_header.node_order);
  BtreeNode left_node(btree_header.key_size, btree_header.node_order);

  if(ReadNode(right_node, right_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }

  if(ReadNode(left_node, left_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  
  // Move the key from the parent node to the right most location of the
  // right child. 
  parent_node.LoadKey(compare_key, key_location);
  compare_key.right_child = right_node.left_child;
  right_node.InsertDatabaseKey(compare_key, (BtreeKeyLocation_t)0);


  // Move the right most key of the left child into the parent node.
  BtreeKeyLocation_t highest_key = left_node.HighestKey();
  right_node.left_child = left_node.GetBranch(highest_key);
  left_node.LoadKey(compare_key, highest_key);
  compare_key.right_child = right_child;
  parent_node.OverWriteDatabaseKey(compare_key, key_location);
  left_node.DeleteDatabaseKey(highest_key);

  if(WriteNode(parent_node, parent_node.node_address) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(WriteNode(right_node, right_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(WriteNode(left_node, left_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::InsertLeftRotation(BtreeNode &parent_node,
					    BtreeKeyLocation_t key_location,
					    DatabaseKeyB &compare_key)
// Internal function used by the gxBtree::InsertBalance() function to perform a 
// left rotation. Returns zero if successful or a non-zero value to indicate a 
// failure.
{
  FAU right_child = parent_node.GetBranch(key_location);
  FAU left_child = parent_node.GetBranch(key_location-1);
  BtreeNode right_node(btree_header.key_size, btree_header.node_order);
  BtreeNode left_node(btree_header.key_size, btree_header.node_order);
  if(ReadNode(right_node, right_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(ReadNode(left_node, left_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  
  // Move the key from the parent node to the left most location of the 
  // left child
  parent_node.LoadKey(compare_key, key_location);
  compare_key.right_child = right_node.left_child;
  left_node.AppendDatabaseKey(compare_key);

  // Move the move the right most key of the right child into the
  // parent node.
  right_node.left_child = right_node.GetBranch((BtreeKeyLocation_t)0);  
  right_node.LoadKey(compare_key, (BtreeKeyLocation_t)0);
  compare_key.right_child = right_child;
  parent_node.OverWriteDatabaseKey(compare_key, key_location);
  right_node.DeleteDatabaseKey((BtreeKeyLocation_t)0);

  if(WriteNode(parent_node, parent_node.node_address) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(WriteNode(right_node, right_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(WriteNode(left_node, left_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::Merge(BtreeNode &parent_node,
			       BtreeKeyLocation_t key_location,
			       DatabaseKeyB &compare_key)
// Internal function used by the gxBtree::InsertBalance() function to merge 
// the parents left and right siblings and remove the right sibling. Returns 
// zero if successful or a non-zero value to indicate a failure.
{
  FAU right_child = parent_node.GetBranch(key_location);
  FAU left_child = parent_node.GetBranch(key_location-1);
  BtreeNode right_node(btree_header.key_size, btree_header.node_order);
  BtreeNode left_node(btree_header.key_size, btree_header.node_order);
  if(ReadNode(right_node, right_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(ReadNode(left_node, left_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }

  // Move the key from the parent node to the left most location of the 
  // left child.
  parent_node.LoadKey(compare_key, key_location);
  compare_key.right_child = right_node.left_child;
  left_node.AppendDatabaseKey(compare_key);
  parent_node.DeleteDatabaseKey(key_location);

  // Merge the left and right siblings
  left_node.MergeNode(right_node);
  if(!f->Remove(right_child)) { // Remove the right child
    return f->GetDatabaseError();
  }
  btree_header.n_nodes--;
  header_in_sync = 0;
  
  if(WriteNode(parent_node, parent_node.node_address) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }
  if(WriteNode(left_node, left_child) != gxDBASE_NO_ERROR) {
    return f->GetDatabaseError();
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError gxBtree::ReInit(FAU header_address, int flush)
// Reconnect the Btree to a new tree header. If the flush variable is true,
// the file buffers will be flushed to disk before reconnecting the Btree.
// Returns zero if successful or a non-zero value to indicate a failure.
{
  if(flush) Flush();
  btree_header_address = header_address;
  return ReadBtreeHeader();          
}

gxDatabaseError gxBtree::ReInit(int flush)
// Reconnect the Btree header to ensure that the file data stays
// in sync during multiple file access and multiple instances of the 
// same application. If the flush variable is true, the file buffers
// will be flushed to disk before reconnecting the Btree. Returns 
// zero if successful or a non-zero value to indicate a failure.
{
  if(flush) Flush();
  return ReadBtreeHeader();          
}

int gxBtree::TestTree(int reinit)
// This function is used to ensure that the file data stays in 
// sync during multiple file access. Returns zero if no errors
// were encountered or an integer value corresponding to one of 
// the following values:
//
// 1 = Node order error 
// 2 = Key size error 
// 3 = Number of keys error
// 4 = Number of nodes error
// 5 = Btree height error
// 6 = Btree root address error
// 7 = Btree class ID error
// 8 = Error reading the Btree header
{
  int error_number = 0;
  gxBtreeHeader curr_btree_header;

  if(f->Read(&curr_btree_header, sizeof(gxBtreeHeader), 
	     btree_header_address) !=  gxDBASE_NO_ERROR) {
    return error_number = 8; // Error reading the btree header
  }

  if(curr_btree_header.node_order != btree_header.node_order) {
    if(reinit) ReInit();
    error_number = 1; // 1 = Node order error 
  }

  if(curr_btree_header.key_size != btree_header.key_size) {
    if(reinit) ReInit();
    error_number = 2; // 2 = Key size error 
  }

  if(curr_btree_header.n_keys != btree_header.n_keys) {
    if(reinit) ReInit();
    error_number = 3; // 3 = Number of keys error
  }

  if(curr_btree_header.n_nodes != btree_header.n_nodes) {
    if(reinit) ReInit();
    error_number = 4 ; // 4 = Number of nodes error
  }

  if(curr_btree_header.btree_height != btree_header.btree_height) {
    if(reinit) ReInit();
    error_number = 5; // 5 = Btree height error
  }

  if(curr_btree_header.root != btree_header.root) {
    if(reinit) ReInit();
    error_number = 6; // 6 = Btree root address error
  }

  if(curr_btree_header.class_id != btree_header.class_id) {
    if(reinit) ReInit();
    error_number = 7; // 7 = Btree class ID error
  }

  return error_number;
}

const char *gxBtree::DatabaseExceptionMessage()
// Returns a null terminated string that can
// be used to log or print a database exception.
{
  return gxDatabaseExceptionMessage(f->GetDatabaseError());
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
