// SQUASH25 by bill buckels 1995
// Scale a 256 color bitmap to half its original size
// using color averaging and optionally (if requested)
// a user defined color clipping algorithm.


/*
    25% area reduction fixed scale

    The Color Scaling Algorithm applies the following average sum
    to the RGB gun values in the image to scale the image:

    Sum - Divide by 4 = Average Color

    R  PIXEL
    A  -------|-------
    S  | RGB  +  RGB |
    T  ---+---|---+---     /4  = RGB
    E  | RGB  +  RGB |
    R  -------|-------

    Typically the color resolution of the result will be greater
    than the palette even with the fixed scale reduction.

    A color loss algorithm can be applied to the RGB gun values after
    scaling. Color resolution is lost by 2's compliments, by
    shifting the gun value right then left.

    This value is set to 0 by default but is user definable.
*/

#include <io.h>
#include <fcntl.h>
#include <malloc.h>
#include <dos.h>
#include <stdio.h>
#include <string.h>

// Bitmap Header structures
typedef struct tagRGBQUAD
{
    unsigned char    rgbBlue;
    unsigned char    rgbGreen;
    unsigned char    rgbRed;
    unsigned char    rgbReserved;
} RGBQUAD;

typedef struct tagBITMAPINFOHEADER
{
    unsigned long   biSize;
    unsigned long   biWidth;
    unsigned long   biHeight;
    unsigned        biPlanes;
    unsigned        biBitCount;
    unsigned long   biCompression;
    unsigned long   biSizeImage;
    unsigned long   biXPelsPerMeter;
    unsigned long   biYPelsPerMeter;
    unsigned long   biClrUsed;
    unsigned long   biClrImportant;
} BITMAPINFOHEADER;

// constants for the biCompression field
#define BI_RGB      0L
#define BI_RLE8     1L
#define BI_RLE4     2L

typedef struct tagBITMAPINFO
{
    BITMAPINFOHEADER bmiHeader;
    RGBQUAD          bmiColors[256];
} BITMAPINFO;


typedef struct tagBITMAPFILEHEADER
{
    unsigned        bfType;
    unsigned long   bfSize;
    unsigned        bfReserved1;
    unsigned        bfReserved2;
    unsigned long   bfOffBits;
} BITMAPFILEHEADER;


BITMAPFILEHEADER BitMapFileHeader;
BITMAPINFO       bmp;

// basic palette mapping globals
long bmpcount[256];
long outcount[256];

unsigned char rgbinfo[256][3];
unsigned char outinfo[256][3];
unsigned char outbuffer[400];

// flag for too many colors
int toomany=0;
int shifter=0;

// squash by 1/2 in the x and y axis
void halfsize(unsigned char *ptr1,unsigned char *ptr2,
              unsigned xres,FILE *fp2)
{
    unsigned x=xres/2,x1;
    unsigned red1, red2, red3, red4;
    unsigned green1, green2, green3, green4;
    unsigned blue1, blue2, blue3, blue4;
    unsigned red,green,blue;
    unsigned char outred, outgreen,outblue;
    unsigned char c;
    unsigned i,j,found,packet;

    // pad for double word boundaries
    packet=x;
    while((packet%4)!=0)packet++;
    i=0;
    for(x1=0;x1<x;x1++)
    {
       // reduce a 4 pixel area to 1 pixel
       c= ptr1[i];
       red1   = rgbinfo[c][0];
       green1 = rgbinfo[c][1];
       blue1  = rgbinfo[c][2];
       c= ptr2[i];
       red2   = rgbinfo[c][0];
       green2 = rgbinfo[c][1];
       blue2  = rgbinfo[c][2];
       i++;

       c= ptr1[i];
       red3   = rgbinfo[c][0];
       green3 = rgbinfo[c][1];
       blue3  = rgbinfo[c][2];
       c= ptr2[i];
       red4   = rgbinfo[c][0];
       green4 = rgbinfo[c][1];
       blue4  = rgbinfo[c][2];
       i++;

       red    =   (red1 + red2 + red3 + red4)>>2;
       green  =   (green1 + green2 + green3 + green4)>>2;
       blue   =   (blue1 + blue2 + blue3 + blue4)>>2;
       outred     = (unsigned char)red;
       outgreen   = (unsigned char)green;
       outblue    = (unsigned char)blue;
       if(shifter)
       {
       outred     = ((outred >> shifter)<<shifter);
       outgreen   = ((outgreen>>shifter)<<shifter);
       outblue    = ((outblue>>shifter)<<shifter);
       }
       found = 0;

       // check for an entry position in the current palette
       for(j=0;j<256;j++)
       {
         // if an unused entry use it... highwater mark
         if(outcount[j]<1L)
         {
           found++;
           outcount[j]++;
           outinfo[j][0] = outred;
           outinfo[j][1] = outgreen;
           outinfo[j][2] = outblue;
           break;
         }

         // if we have a match use that
         if(outred   == outinfo[j][0] &&
            outgreen == outinfo[j][1] &&
            outblue  == outinfo[j][2])
            {
             found++;
             outcount[j]++;
             outinfo[j][0] = outred;
             outinfo[j][1] = outgreen;
             outinfo[j][2] = outblue;
             break;
            }

       }
       // if more than 256 colors we have too many colors - don't bother
       // otherwise use the index position for this color
       if(!found)toomany = 1;
       else outbuffer[x1] = (unsigned char )j;

    }
    // regardless, write the file...
    fwrite(outbuffer,packet,1,fp2);

}

// command line help
char *usage[]={
"Scale a 256 color bitmap to half its original size using color averaging",
"and optionally (if requested) a user defined color clipping ratio.",
"The Color Scaling Algorithm applies the following average sum to the RGB",
"gun values in the image to scale the image:",
"",
" Sum - Divide by 4 = Average Color",
" R  PIXEL",
" A  -------|-------",
" S  | RGB  +  RGB |",
" T  ---+---|---+---     /4  = RGB",
" E  | RGB  +  RGB |",
" R  -------|-------",
"",
" Typically the color resolution of the result will be greater than the",
" palette even with the fixed scale reduction. A color loss algorithm can be",
" applied to the RGB gun values after scaling. Color resolution is lost by",
" 2's compliments, by shifting the gun value right then left.",
" This value is set to 0 by default but is user definable.",
"!"};

int main(int argc, char **argv)
{

    long colors;
    static unsigned char src1[800],src2[800];
    FILE *fp,*fp2;
    int status,i;
    unsigned x, bytesperline;
    unsigned char c;

    unsigned long hres,vres,seeksize;
    unsigned xres;

    puts("SQUASH25 Copyright (C) 1995 by Bill Buckels");
    if(argc==2 && argv[1][0] == '?')
    {
       for(i=0;usage[i][0]!='!';i++)puts(usage[i]);
       return 0;

    }
    if(argc<3)
    {
    puts(
   "Usage is : \"SQUASH25 [BMP256.IN] [BMP256.OUT] (optional-shift (1-8))\"");
   puts(
   "For Help : \"SQUASH25 ?\"");
   return 1;
    }

    if(argc==4)
    {
        shifter=atoi(argv[3]);
    }

    if((fp=fopen(argv[1],"rb"))==NULL)
    {
        perror(argv[1]);
        return 1;
    }

    // read the header stuff into the appropriate structures
    fread((char *)&BitMapFileHeader.bfType,
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp);

    status = 0;
    if(bmp.bmiHeader.biCompression!=BI_RGB)status++;
    if(bmp.bmiHeader.biWidth>800L)status++;
    if(bmp.bmiHeader.biHeight>600L)status++;
    if(bmp.bmiHeader.biPlanes!=1)status++;
    if(bmp.bmiHeader.biBitCount!=8)status++;

    if(status)
     {
        puts("Not A Supported 256 Color .BMP!");
        if(bmp.bmiHeader.biCompression!=BI_RGB)
          puts("Not an RGB Format .BMP!");
        if(bmp.bmiHeader.biWidth>800L)
          puts("Maximum Width of 800 pixels exceeded!");
        if(bmp.bmiHeader.biHeight>600L)
          puts("Maximum Height of 600 Rasters exceeded!");
        if(bmp.bmiHeader.biPlanes!=1)
          puts("File Has More Than 1 Plane of Image Data!");
        if(bmp.bmiHeader.biBitCount!=8)
          puts("File Must Be 8 Bits Per Pixel!");
        puts("Exiting!");
        fclose(fp);
        return 1;
     }

    if((fp2=fopen(argv[2],"wb"))==NULL)
    {
        perror(argv[2]);
        fclose(fp);
        return 1;
    }

    // palette stuff
    for(i=0;i<256;i++)
    {

      rgbinfo[i][0]=bmp.bmiColors[i].rgbRed;
      rgbinfo[i][1]=bmp.bmiColors[i].rgbGreen;
      rgbinfo[i][2]=bmp.bmiColors[i].rgbBlue;
      outinfo[i][0]=0;
      outinfo[i][1]=0;
      outinfo[i][2]=0;
    }

    // zero our colour count arrays
    memset((char *)&bmpcount[0],0,1024);
    memset((char *)&outcount[0],0,1024);
    memset(outbuffer,0,800);

    hres = bmp.bmiHeader.biWidth;
    vres = bmp.bmiHeader.biHeight;
    vres = (vres/2)*2;
    xres = (unsigned )hres;
    bytesperline = xres;
    // pad for double word boundaries
    while((bytesperline%4)!=0)bytesperline++;

    printf("Now Reading 256 Color Windows Bitmap : %s\n",argv[1]);
    printf("Height %ld\n",vres);
    printf("Width  %ld\n",hres);
    fwrite((char *)&BitMapFileHeader.bfType,
                 sizeof(BITMAPFILEHEADER),1,fp2);
    fwrite((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp2);

    seeksize =  0L;
    seeksize += BitMapFileHeader.bfOffBits;
    printf("       <- Scanline");
    for(i=0;i<vres;i++)
        {
        printf("\r%d",i+1);
        fseek(fp,seeksize,SEEK_SET);
        fread(src1,xres,1,fp);
        for(x=0;x<xres;x++)
        {
            c=src1[x];
            bmpcount[c]+=1;
        }
        seeksize += bytesperline;
        i++;
        if(i<vres)
        {
        printf("\r%d",i+1);
        fseek(fp,seeksize,SEEK_SET);
        fread(src2,xres,1,fp);
        for(x=0;x<xres;x++)
        {
            c=src2[x];
            bmpcount[c]+=1;
        }
        seeksize += bytesperline;
        }
        else
        {
         break;
        }
        halfsize(src1,src2,xres,fp2);
        if(toomany)break;

    }
    puts("");
    fclose(fp);
    fclose(fp2);
    colors=0;

    // do a colour count
    for(x=0;x<256;x++)
    {
      if(bmpcount[x]!=0L)colors++;
    }
    printf("Input File contains %ld colors.\n",colors);

    if(toomany)
    {
       puts("Not Enough Colors! Output File needs more than 256 colors!");
       puts("Try Running Again Using The Following Command Line :");
       printf("SQUASH25 %s %s %d\n",argv[1],argv[2],shifter+1);
       remove(argv[2]);
       return shifter+1;
    }
    colors=0L;
    for(x=0;x<256;x++)
    {
      if(outcount[x]!=0L)colors++;
    }

    if((fp2=fopen(argv[2],"rb+w"))==NULL)
    {
        perror(argv[2]);
        fclose(fp);
        return 1;
    }
    seeksize = filelength(fileno(fp2));

    // new palette stuff
    for(i=0;i<256;i++)
    {
      bmp.bmiColors[i].rgbRed =    outinfo[i][0];
      bmp.bmiColors[i].rgbGreen =  outinfo[i][1];
      bmp.bmiColors[i].rgbBlue =   outinfo[i][2];
      bmp.bmiColors[i].rgbReserved = 0;
    }
    // new header stuff
    BitMapFileHeader.bfOffBits = 0L;
    BitMapFileHeader.bfOffBits +=sizeof(BITMAPFILEHEADER);
    BitMapFileHeader.bfOffBits +=sizeof(BITMAPINFO);
    BitMapFileHeader.bfSize =  seeksize;
    seeksize -= BitMapFileHeader.bfOffBits;
    bmp.bmiHeader.biWidth         = hres/2;
    bmp.bmiHeader.biHeight        = vres/2;
    bmp.bmiHeader.biClrUsed       = 256;
    bmp.bmiHeader.biClrImportant  = colors;
    bmp.bmiHeader.biSizeImage     = seeksize;
    fseek(fp2,0L,SEEK_SET);
    fwrite((char *)&BitMapFileHeader.bfType,
                 sizeof(BITMAPFILEHEADER),1,fp2);
    fwrite((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp2);
    fclose(fp2);
    printf("Output File %s created! Done!\n",argv[2]);
    printf("Output File contains %ld colors.\n",colors);
    printf("Height %ld\n",bmp.bmiHeader.biHeight);
    printf("Width  %ld\n",bmp.bmiHeader.biWidth);

return 0;
}
