// ------------------------------------------------------------------------
// System       : MultiMedia Presentation
// Program      : MCIShow.c Version 3.0
//                (C) Copyright Bill Buckels 1996-1999.
//                All rights reserved.
// Description  : Simple Slideshow Presentation System
// Written by   : Bill Buckels
// Date Written : 1.1 March, 1996
//                1.2 March, 1996
//                1.3 July, 1996
// Purpose      : Script Driven MCI Playback and Bitmap File Presentation
// Revision     : 1.1 core functionality for bmp, avi, wav, and midi
// Revision     : 1.2 extended bmp and cdaudio
// Revision     : 1.3 added additional logic and windows help
// Version      : 3.0 general update
//
// Copyright Notice - This source code is the property of Bill Buckels.
//
// Portions of this source code were produced from the examples
// provided with the Windows 3.1 SDK and used by permission.
//
// Those portions are excluded from this agreement and are the property of
// their respective author.
//
// You are not prevented from Copyright against using those portions.
// However, for practical purposes the SDK derivatives are primarily
// conceptual and not identifiable, and are mentioned for information
// purposes only.
//
// If you are a registered user of MCIShow, you may use this source code
// for educational and non-profit use, including the production of
// shareware, freeware and other public spirited efforts, provided that you
// agree that Bill Buckels has no warranty obligations or liability for any
// use of this source code and the MCIShow program in its present or
// any modified form. You may not use this program or portions of this
// program to compete against its author in any way.
//
// You may also use this source code for commercial purposes if you have
// the author's permission.
//
// Contact: Bill Buckels
//          589 Oxford Street
//          Winnipeg, Mb, Cdn R3M 3J2
//          Email bbuckels@escape.ca
//          Website http://www.escape.ca/~bbuckels
//
// You have 30 days to register this program or you must remove it from
// your computer. Shareware Registration is $10.00 per copy. Payment
// by cheque or money order.
//
// Commercial users are not permitted to use the shareware version
// without the author's permission inless otherwise stated in the
// MCIShow Licence Agreement.
// ------------------------------------------------------------------------

// api includes
#include <windows.h>
#include <mmsystem.h>
#include <commdlg.h>

// msc runtime includes
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>
#include <share.h>
#include <conio.h>
#include <dos.h>
#include <direct.h>

// class NAME and title text
LPSTR lpMCIShow  ="MCISHOW";
LPSTR lpCopyRight="MCISHOW \251 V3.0 Copyright Bill Buckels 1996-1999";
BOOL  bRegistered=FALSE;

// menu constants include file
#include "mcishow.h"

BOOL     HELPACTIVE = FALSE;

// resource file element strings
LPSTR lpMyIcon    = "MyIcon";
LPSTR lpMyAcc     = "MCIShowAcc";
LPSTR lpMyPalette = "MyPalette";

// Debug Mode constants and flags
BOOL  bDebug  = FALSE;
LPSTR lpDebug = "debug";
LPSTR lpDebugMode = "MCIShow\251 SCRIPT DEBUG MESSAGE";


// Debug Mode Helper Functions
LONG FAR PASCAL DebugProc();
LONG FAR PASCAL DebugString(LPSTR,LPSTR);

// ------------------------------------------------------------------------
// Program Constant Definitions and Typedefs for MCISHOW.C
// ------------------------------------------------------------------------

// typedef for a signed int to match other typedefs used in windows.h
typedef int SINT;

// 2 typedefs required for AVI playback
typedef struct {
    DWORD   dwCallback;
    UINT    wDeviceID;
    WORD    wReserved0;
    LPSTR   lpstrDeviceType;
    LPSTR   lpstrElementName;
    LPSTR   lpstrAlias;
    DWORD   dwStyle;
    HWND    hWndParent;
    WORD    wReserved1;
} MCI_DGV_OPEN_PARMS;

typedef struct {
    DWORD   dwCallback;
    DWORD   dwReturn;
    DWORD   dwItem;
    DWORD   dwTrack;
    LPSTR   lpstrDrive;
    DWORD   dwReference;
} MCI_DGV_STATUS_PARMS;

// --------------------------------------------------------------------
// Threshold Constants, Boundary Constants, etc.
// --------------------------------------------------------------------

#define MINWIDTH  640                 // minimum window width
#define MINHEIGHT 480                 // minimum window height
#define MAXSCALE  10                  // maximum scale
#define MIDSIZE   257                 // medium buffer size
#define MIDMAX    (MIDSIZE-1)         // maximum allowable input
#define SMALLSIZE 129                 // small buffer size
#define SMALLMAX  (SMALLSIZE-1)       // maximum allowable input
#define MAXCUSTFILTER 60              // maximum size of custom filter buffer
#define DEVICESIZE 79                 // MCI device name buffer size (per SDK)
#define DEVICEMAX  (DEVICESIZE-1)     // maximum MCI device name

// ------ ordinal constants used for internal multimedia program control
#define WAVE 1                    // control constant for WAVE event
#define MIDI 2                    // control constant for MIDI event
#define AVI  3                    // control constant for AVI  event
#define PAUSE 4                   // control constant for PAUSE event
#define ALL 999                   // control constant to clobber all devices

// ------ string buffer operator constants
#define _FIRST_       0               // Base 0 - first string character
#define ASCIIZ        0               // asciiz string terminator
#define WHITESPACE    32              // clarion field padding
#define PERIOD        46              // file extension seperator

// used to retrieve filename from path name
#define SLASH         92              // ascii 92
#define COLON         ':'             // colon

// ------ standard constants for placeholder values or return value tests
#define SUCCESS       0               // (std return) integer 0 = success
#define FAILURE       1               // (std return) integer 1 = failure
#define NIL           0               // nothing
#define CENTRE        32767           // centre bmp on window
#define NOTHING       0L              // long integer arg/return
#define SOMETHING     1L              // long integer arg/return
#define FULL          1               // Full Scale constant
#define LOWEST        1               // Lowest Positive Number
#define FITBOX        1               // Fit BMP In a Box
#define FITWIDTH      2               // Scale by Width ratio
#define FITHEIGHT     3               // Scale by Height ratio

#define IDT_WAVE  100            // control constant for WAVE event timer
#define IDT_MIDI  200            // control constant for MIDI event timer
#define IDT_AVI   300            // control constant for AVI  event timer
#define IDT_PAUSETIMER 400       // control constant for PAUSE event timer
// since we don't use a timer for CD playback no timer ID is assigned

// Timer delays in milliseconds for Multimedia Playback completion checks.
// Adjust as necessary so we don't hog all windows' resources.
UINT WAVETIME     = (UINT)1500;    // checktime for wav completion
UINT MIDITIME     = (UINT)1500;    // checktime for mid completion
UINT AVITIME      = (UINT)4500;    // checktime for avi completion
UINT PAUSETIME    = (UINT)1000;    // 1 second callback time

// -----------------------------------------------------------------------
// Windows Bitmap Static Structure Declarations (based on windows.h)
// -----------------------------------------------------------------------

// allow up to 256 colors
typedef struct tagMYBITMAPINFO
{
    BITMAPINFOHEADER bmiHeader;
    RGBQUAD          bmiColors[256];
} MYBITMAPINFO;

typedef struct tagLOGPAL
{
    WORD         Version;
    WORD         Entries;
    PALETTEENTRY Entry[256];
}LOGPAL;


// windows api global variable declarations used throughout the program
HANDLE hInst;                   // handle to main program instance
HDC    hdcMain;                 // Device Contexts for Image Display
HICON  myicon;                  // Program Icon
HANDLE hAccTable;               // handle to accelerator table

// ------ Windows Bmp Specific Variable Declarations for Image Handling
GLOBALHANDLE hBmpHandle;        // Global Memory Block for the image
BYTE huge   *hpBmpBuffer;       // pointer to the image block
LONG         dwBmpBufSize;      // image block size
BITMAPFILEHEADER bmfhead;       // standard windows bitmap header declaration
MYBITMAPINFO     bmp;           // buffer for bmp header info
UINT     uiBmpWidth,uiBmpHeight;// current bitmap global metrics
UINT     uiBytesPerLine;        // current bitmap actual raster size
LPSTR   lpDefPalette = NULL;    // generic "true color" default palette ptr
HRSRC   hPaletteLoc  = NULL;    // default palette resource locator
HGLOBAL hDefPalette  = NULL;    // default palette resource handle
LOGPAL    myLogPal;             // current BMP logical palette ptr
HPALETTE  hpalCustom   = NULL;  // current palette handle
HPALETTE  hpalOriginal = NULL;  // original saved palette handle
BOOL bFirstPaint       = FALSE; // flag is set when an image is available
BOOL bSetPalette       = FALSE; // set a new palette
BOOL bGrayPalette      = FALSE; // use the general purpose "gray" palette
UINT uiScale           = FULL;  // load image full scale
UINT uiFit             = NIL;   // fit image to screen or dimensions
UINT uiFitWidth        = NIL;
UINT uiFitHeight       = NIL;
DWORD dwClearColor     = 0L;

// ------ Global variables for graphics display metrics
UINT  uiXwidth,  uiYheight;
UINT  uiXborder, uiYborder, uiYcaption;
LONG  dwXwidth,  dwYheight;
UINT  uiXoffset = CENTRE, uiYoffset = CENTRE;

// ------ Global General Purpose Throwaway Working Buffers
BYTE szErrorBuf[MIDSIZE]         ="\0";
BYTE szMessBuf[MIDSIZE]          ="\0";
BYTE szWorkBuf[MIDSIZE]          ="\0";

// ------ Global Path Names
BYTE szCustFilterSpec[MAXCUSTFILTER] ="\0";   // ofn custom filter
BYTE szPathName[MIDSIZE]             ="\0";   // ofn directory
BYTE szFileName[MIDSIZE]             ="\0";   // ofn filename
BYTE szCurrentBmp[SMALLSIZE]         ="\0";   // last succesfully loaded bmp

// ------  Global MCI File Names

BYTE szSlideShow[SMALLSIZE]      ="\0";
BYTE szBmpFile[SMALLSIZE]        ="\0";
BYTE szWaveFile[SMALLSIZE]       ="\0";
BYTE szMidiFile[SMALLSIZE]       ="\0";
BYTE szAviFile[SMALLSIZE]        ="\0";
BYTE szSaveShow[SMALLSIZE]       ="\0";

// ------ OpenFile Variable Declarations
LPSTR lpOFNTitle    ="Select/Enter Filename";
OPENFILENAME ofn;   // struct. passed to GetOpenFileName
LPSTR lpFilterSpec;

// ------ filter strings for file listings (expand as necessary)
BYTE szMCISpec[SMALLSIZE]
            = "MCI Files(*.MCI)\0*.MCI\0\0";

// ------- User Profile Sections and Keys

LPSTR lpMCISection  = "mci extensions";

LPSTR lpEventKey   = "event";
LPSTR lpWaveKey    = "wav";
LPSTR lpMidiKey    = "mid";
LPSTR lpAviKey     = "avi";
LPSTR lpCDKey      = "cd";
LPSTR lpBmpKey     = "bmp";

// menu capability not implemented
LPSTR lpMenuKey    = "menu";  
LPSTR lpCommand    = "command"; // commands are text,x1,y1,x2,y2,goto
LPSTR lpCaption    = "caption"; // captions are text,x1,y1

LPSTR lpPauseKey   = "pause";
LPSTR lpSecondsKey = "seconds";

LPSTR lpNameKey    = "name";
LPSTR lpPaletteKey = "usepalette";
LPSTR lpLeaveMidiOn= "playmidi";
LPSTR lpLeaveCDOn  = "playcd";
LPSTR lpAviStyle   = "usewindow";
LPSTR lpAviCentre  = "movewindow";
LPSTR lpXoffset    = "xorg";
LPSTR lpYoffset    = "yorg";
LPSTR lpScale      = "scale";    // scale=number, box, width, height
LPSTR lpBox        = "box";
LPSTR lpWidth      = "width";    // scale=box, scale=width
LPSTR lpHeight     = "height";   // scale=box, scale=height

// default palette not implemented yet
LPSTR lpGrayPalette= "default";

LPSTR lpDirection  = "direction";
LPSTR lpFrom       = "from";
LPSTR lpTo         = "to";
LPSTR lpEjectKey   = "eject";
LPSTR lpTrack      = "track";

// message box event
// OK, YES/NO, YES/NO/CANCEL
LPSTR lpMessage    = "message";
LPSTR lpTitle      = "title";
LPSTR lpText       = "text";
LPSTR lpYesKey     = "yes";
LPSTR lpNoKey      = "no";
LPSTR lpCancelKey  = "cancel";

// clear screen event
LPSTR lpClearKey   = "clear";
LPSTR lpRed        = "red";
LPSTR lpGreen      = "green";
LPSTR lpBlue       = "blue";

LPSTR lpPrevious   = "previous";
LPSTR lpGoto       = "goto";
LPSTR lpGosub      = "gosub";
LPSTR lpReturn     = "return";
LPSTR lpLoop       = "loop";
LPSTR lpEnd        = "end";

// commands to call and exit from external script...
// single level only... no stack
LPSTR lpCall    = "call";
LPSTR lpExit    = "exit";

// ------ MCI Playback Buffers and Control Variable Declarations

BYTE szWaveDevice[DEVICESIZE]="\0";         // MCI wave device name
BYTE szMidiDevice[DEVICESIZE]="\0";         // MCI midi device name
BYTE szAviDevice[DEVICESIZE]= "\0";         // MCI avi device name
BYTE szCDDevice[DEVICESIZE] = "cdaudio\0";  // MCI CD-ROM device name

// global device handles
UINT wWaveDeviceID=(UINT)0;
UINT wMidiDeviceID=(UINT)0;
UINT wAviDeviceID =(UINT)0;
UINT wCDDeviceID  =(UINT)0;

// current status of MCI events
BOOL bMidiPlaying  = FALSE;
BOOL bWavePlaying  = FALSE;
BOOL bAviPlaying   = FALSE;
BOOL bCDPlaying    = FALSE;

DWORD dwCDStartPosition=NOTHING;
DWORD dwCDStopPosition =NOTHING;

// Counter for pause events
UINT uiPauseTime   = (UINT)0;

// true if MCI device is installed
BOOL bPlayWave     = FALSE;
BOOL bPlayMidi     = FALSE;
BOOL bPlayAvi      = FALSE;
// not used for cd audio

// true if an MCI timer callback is currently being played-out
BOOL bWaveTimer    = FALSE;
BOOL bMidiTimer    = FALSE;
BOOL bAviTimer     = FALSE;
BOOL bPauseTimer   = FALSE;
// not used for cd audio

// global window handles used by some MCI core functions
HWND hWndAVI=NULL,hWndParent=NULL;

// Slideshow Related Status Flags
BOOL bSlideShow      =FALSE;     // A slideshow is in progress
BOOL bSlideShowPaused=FALSE;     // The SlideShow is Paused
BOOL bTimerPaused    =FALSE;     // The Timer is Paused
BOOL bWavePaused     =FALSE;     // Wave Playback is Paused
BOOL bMidiPaused     =FALSE;     // Midi PlayBack is Paused
BOOL bAviPaused      =FALSE;     // Avi PlayBack is Paused
BOOL bLeaveMidiOn    =FALSE;     // Leave Midi On During Avi Playback
BOOL bLeaveCDOn      =FALSE;     // Leave CD On During Playback
BOOL bAviCentre      =TRUE;      // Move Avi To Centre
BOOL bAviClear       =TRUE;      // Clear Screen before loading AVI

// Compact Disk Specific Flags (Script Invoked Options)
BOOL bCDPaused       =FALSE;     // CD player is not paused
BOOL bCDEject        =FALSE;     // CD eject is not required
BOOL bCDContinuous   =FALSE;     // CD play is not continuous

SINT iEventCounter   =NIL;       // event counter for slideshow playback
SINT iPreviousEvent  =NIL;
UINT iReturnEvent    =NIL;

SINT iSaveEvent      =NIL;
SINT iSavePrevious   =NIL;
SINT iSaveReturn     =NIL;
SINT iAviLoops       =NIL;

DWORD dwAviStyle     =WS_CHILD; // default AVI window style (no border)

LPSTR lpEject         = "Do you want to Eject the CD from the Drive?";

// some errors
LPSTR lpNoSlideShow   = "No SlideShow is Loaded!\n";
LPSTR lpInvalidEntry  = "SlideShow Entry not found!\n";
LPSTR lpInvalidDevice = "No MCI playback device available!\n";
LPSTR lpNonEvent      = "Unrecognized Event!";
LPSTR lpUncontext     = "Unable to create device context!\n";
LPSTR lpUnprimary     = "Unable to allocate primary memory block!\n";
LPSTR lpUnpalette     = "Unable to load default palette!\n";
LPSTR lpContinue      = "Unable to Continue...\n";
LPSTR lpUnmoving      = "Moving of Window is not allowed!\n";
LPSTR lpSecondPause   = " second pause!";
LPSTR lpMousePause    = "Paused for Mouse!";
LPSTR lpSlideDone     = "Slideshow Done!";

LPSTR lpUnError       = "Unknown MCI Error!\n"
                        "The MCI Device must be properly installed "
                        "and not in use by another application.\n"
                        "The MCI File must be a valid type.";

LPSTR lpDisable       = "Disable this device";
LPSTR lpUnstatus      = "Unknown MCI Error during status check!\n";
LPSTR lpCancel        = "Cancel current playback";
LPSTR lpUnToo         = "Unknown MCI Error!\n";
LPSTR lpCDError       = "A cdaudio device error occurred\n";

// bmp display errors
LPSTR lpOpenError     = "Problems Were Encountered Opening The File.\n";
LPSTR lpFormatError   = "Format is not supported...\n";
LPSTR lpTooLarge      = "File is too large to display...\n"
                        "Size %d x %d is beyond screen maximum %d x %d!";
LPSTR lpNoLock        = "Memory Lock Error!\n"
                        "Unable to globally lock a buffer!";
LPSTR lpLockRead      = "Unable to read image...";
LPSTR lpLockDisplay   = "Unable to display image...";

LPSTR lpNoWave = "An MCI Wave Player is not installed!\n";
LPSTR lpNoMidi = "An MCI Midi Player is not installed!\n";
LPSTR lpNoAvi  = "An MCI AVI Player is not installed!\n";

LPSTR lpMCIDisabled = "\nMCI playback requests for missing devices or "
                      "invalid filenames in an MCIShow script will "
                      "result in error messages during playback.\n";

// the about message text
LPSTR lpAbout = "MCIShow is a Simple Windows MCI "
                "MultiMedia SlideShow Application "
                "Written in Large Model Microsoft C Version 6.00a\n\n"
                "This program and its source code "
                "are distributed as Shareware. It is NOT FREE!\n\n"
                "The Shareware registration is $10 per copy and must be "
                "be paid within 30 days of installation... "
                "Payment by Cheque or Money Order. Discounts may apply. "
                "See the program documentation for details.\n\n"
                "This Shareware version of MCIShow may only be used "
                "by educational institutions and "
                "private individuals for non-profit purposes "
                "and community services, and by businesses not in the "
                "Software Industry unless otherwise noted.\n\n"
                "Special Commercial Licences and Custom Versions "
                "of this program "
                "can be also purchased from the author:\n\n"
                "\tBill Buckels\n"
                "\t589 Oxford Street\n"
                "\tWinnipeg, Mb, Cdn R3M 3J2\n\n"
                "\temail : bbuckels@escape.ca\n"
                "\twebsite : http://www.escape.ca/~bbuckels\n\n";


// function prototypes

// main and process functions
LONG FAR PASCAL MainWndProc (HWND,WORD,WORD,LONG);
LONG FAR PASCAL GetUserProfile();
SINT PASCAL WinMain(HANDLE,HANDLE,LPSTR,SINT);

// session handlers
LONG FAR PASCAL OpenSession(HWND);
BOOL FAR PASCAL SysHook(HWND,WORD);
LONG FAR PASCAL PaintMethod(HWND);
LONG FAR PASCAL CloseSession(HWND);
LONG FAR PASCAL AboutProc(HWND);
LONG FAR PASCAL HelpProc(HWND);

// MCI Session Handlers
BOOL FAR PASCAL StartSlideShow(HWND);
BOOL FAR PASCAL StopSlideShow(BOOL);
BOOL FAR PASCAL PauseSlideShow(HWND,BOOL);
BOOL FAR PASCAL ResumeSlideShow(HWND);
BOOL FAR PASCAL GetNextEvent(HWND);
BOOL FAR PASCAL SlideShowTimerProc(HWND,WORD);

// MultiMedia Core Functions
BOOL FAR PASCAL MediaPlay(LPSTR,UINT);
BOOL FAR PASCAL MediaStop(UINT);
BOOL FAR PASCAL IsMCIPlaying(UINT);
BOOL FAR PASCAL MCIPause(UINT);
BOOL FAR PASCAL MCIPlay(UINT);
BOOL FAR PASCAL MCIRewind(UINT);
BOOL FAR PASCAL CentreAvi();

// Music CD Playback Functions

BOOL FAR PASCAL MCICDPlay(UINT,UINT);     // starts cd play from slideshow
BOOL FAR PASCAL MCICDNext(SINT);          // plays next or previous track
BOOL FAR PASCAL OpenCDDevice();           // open the MCI cdaudio driver
BOOL FAR PASCAL CloseCDDevice();          // close the MCI cdaudio driver
BOOL FAR PASCAL PlayCD();                 // play the CD
BOOL FAR PASCAL StopCD();                 // stop CD, not the same as paused
BOOL FAR PASCAL IsCDPlaying();            // is the CD playing or stopped?
BOOL FAR PASCAL PauseCD();                // pause until asked to resume
DWORD FAR PASCAL GetNumberOfCDTracks();   // get the current CD info
DWORD FAR PASCAL GetCDStartPosition(UINT);// get track-based start position
BOOL FAR PASCAL SetCDTimeFormat();        // set time format to m/s/f
DWORD FAR PASCAL GetCurrentCDPosition();  // get the current position
DWORD FAR PASCAL GetCurrentCDTrack();     // get the current track
BOOL FAR PASCAL CDEject();                // eject the CD after play
BOOL FAR PASCAL CDErrorProc(DWORD);


// Graphics Core Functions
BOOL FAR PASCAL LoadPic(LPSTR,HWND);
UINT ScaleSrcToDest(UINT,UINT,UINT);
BOOL FAR PASCAL ShowPic();
SINT ClearScreen(HWND,HDC);

// Helper Functions
BOOL FAR PASCAL FileExists(LPSTR);
SINT InitOFNDialog(HWND);

// ------------------------------------------------------------------------
// code
// ------------------------------------------------------------------------

// ------------------------------------------------------------------------
// FUNCTION WinMain
//
// Windows Entry Point
// Every Windows Program Has One
// ------------------------------------------------------------------------
SINT PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdParam, SINT nCmdShow)
     {
     HWND        hWnd;
     MSG         msg ;
     WNDCLASS    wc ;
     WORD        wParam;

     hInst=hInstance;            // handle to instance

     if (!hPrevInstance)
          {
          myicon=LoadIcon(hInst,lpMyIcon);
          wc.style         = CS_HREDRAW | CS_VREDRAW ;
          wc.lpfnWndProc   = MainWndProc ;
          wc.cbClsExtra    = 0 ;
          wc.cbWndExtra    = 0 ;
          wc.hInstance     = hInstance ;
          wc.hIcon         = myicon;
          wc.hCursor       = LoadCursor (NULL, IDC_ARROW) ;
          wc.hbrBackground = GetStockObject (BLACK_BRUSH) ;
          wc.lpszMenuName  = lpMCIShow;
          wc.lpszClassName = lpMCIShow ;

          RegisterClass (&wc) ;
     }

     GetUserProfile();

     hAccTable = LoadAccelerators(hInst,lpMyAcc);

     // push the window up and back so the title bar is off the screen
     // this increases our display area but still allows normal menu access
     hWnd = CreateWindow (lpMCIShow,lpCopyRight,
              WS_MAXIMIZE|WS_MINIMIZEBOX|
              WS_CAPTION|WS_POPUPWINDOW,
              NIL-(uiXborder/2),NIL-((uiYcaption+uiYborder)/2),
              uiXwidth+uiXborder,uiYheight+uiYborder+uiYcaption,
              NULL, NULL, hInstance, NULL) ;

     uiXborder=uiXborder/2;
     uiYborder=uiYborder/2;

     ShowWindow (hWnd, nCmdShow) ;

     UpdateWindow (hWnd) ;
     SetCursor(LoadCursor(NULL,IDC_ARROW));

     while (GetMessage(&msg, NULL, NIL, NIL))
       {
         if (!TranslateAccelerator(hWnd, hAccTable, &msg))
            {
            TranslateMessage(&msg);
            DispatchMessage(&msg); 
            }
        }

     return msg.wParam;
}

// ------------------------------------------------------------------------
// FUNCTION GetUserProfile - mcishow version
//
// user profile information set-up function
// ------------------------------------------------------------------------
LONG FAR PASCAL GetUserProfile()
{
    UINT cx, cy;

    szErrorBuf[_FIRST_] = ASCIIZ;

    szWorkBuf[_FIRST_]  = ASCIIZ;
    GetProfileString(lpMCIShow,lpNameKey,"",szWorkBuf,MIDMAX);
    if(szWorkBuf[_FIRST_]!=ASCIIZ)
    {
      bRegistered = TRUE;
      wsprintf(szErrorBuf, "This Copy of MCIShow\251 is registered to %s "
              "for use under the conditions in the licence agreement.\n\n"
              "Thank you for registering.",szWorkBuf);
      MessageBox(GetFocus(),szErrorBuf,lpCopyRight,MB_ICONINFORMATION);
      szErrorBuf[_FIRST_]=ASCIIZ;
    }


    // MCI support for multimedia
    szWorkBuf[_FIRST_] = ASCIIZ;
    GetProfileString(lpMCISection,lpWaveKey,"",szWorkBuf,DEVICEMAX);
    if(szWorkBuf[_FIRST_]!=ASCIIZ)
    {
       bPlayWave = TRUE;
       strcpy(szWaveDevice,szWorkBuf);
    }
    else
    {
        strcat(szErrorBuf,lpNoWave);
    }

    szWorkBuf[_FIRST_] = ASCIIZ;
    GetProfileString(lpMCISection,lpMidiKey,"",szWorkBuf,DEVICEMAX);
    if(szWorkBuf[_FIRST_]!=ASCIIZ)
    {
       bPlayMidi = TRUE;
       strcpy(szMidiDevice,szWorkBuf);
    }
    else
    {
        strcat(szErrorBuf,lpNoMidi);
    }

    szWorkBuf[_FIRST_] = ASCIIZ;
    GetProfileString(lpMCISection,lpAviKey,"",szWorkBuf,DEVICEMAX);
    if(szWorkBuf[_FIRST_]!=ASCIIZ)
    {
       bPlayAvi = TRUE;
       strcpy(szAviDevice,szWorkBuf);
    }
    else
    {
        strcat(szErrorBuf,lpNoAvi);
    }

    // if expected devices
    if(szErrorBuf[_FIRST_]!=ASCIIZ)
    {

      strcat(szErrorBuf,lpMCIDisabled);
      MessageBox(GetFocus(),szErrorBuf,lpCopyRight,MB_ICONHAND);
      szErrorBuf[_FIRST_]=ASCIIZ;

    }

    // --------------------------------------------------------------------
    // use a Window Size and Buffer Size
    // appropriate for the display
    // --------------------------------------------------------------------
    // create a full screen window
    // allow for the window title and the borders
    // but make the window at least 640 x 480
    // --------------------------------------------------------------------

    uiXwidth  = GetSystemMetrics(SM_CXSCREEN);
    uiYheight  = GetSystemMetrics(SM_CYSCREEN);

    uiXborder  = GetSystemMetrics(SM_CXBORDER)*2;
    uiYborder  = GetSystemMetrics(SM_CYBORDER)*2;
    uiYcaption = GetSystemMetrics(SM_CYCAPTION)*2;

    if(uiXwidth < MINWIDTH)uiXwidth=MINWIDTH;
    if(uiYheight< MINHEIGHT)uiYheight=MINHEIGHT;

    // make sure we have enough memory to load 24 Bit true-color images
    dwXwidth = (LONG) uiXwidth;
    dwXwidth *= 3;                    // RGB triple
    while((dwXwidth%4)!=0)dwXwidth++; // margin of safety - 32 bit aligned
    dwYheight = (LONG) uiYheight;
    dwYheight += 1;                   // margin of safety - extra scanline
    dwBmpBufSize = dwXwidth * dwYheight;

return NOTHING;
}

// ------------------------------------------------------------------------
// FUNCTION MainWndProc - mcishow version
//
// Main Window Message Handler
// ------------------------------------------------------------------------
LONG FAR PASCAL MainWndProc (HWND hWnd, WORD message,
                             WORD wParam, LONG lParam)
{

     UINT wMCIDevice;

     switch (message)
     {
       case WM_TIMER      : if(SlideShowTimerProc(hWnd,wParam)==FALSE)break;
                            return NOTHING;

       case WM_CREATE     : return OpenSession(hWnd);

       case WM_SYSCOMMAND :
                            // Disable the screen saver seperately since
                            //   it gets in the way of avi and other display
                            // mmsystem.h says to return non-zero here
                            //   but the Windows 3.1 SDK volume 3 page 199
                            //   says to always always return 0L when a
                            //   system command is intercepted.
                            // Is there a clue here as to why some apps
                            //   hang windows when a screensaver kicks-in?
                            // I am returning a non-zero on a screensave
                            //   because I think the comment in mmsystem.h
                            //   is probably correct.
                            if(wParam==SC_SCREENSAVE)return SOMETHING;
                            if(SysHook(hWnd,wParam)==TRUE)return NOTHING;
                            break;

       case WM_COMMAND    :

               switch(wParam)
               {

                case IDM_START : if (bRegistered == TRUE)return NOTHING;
                                 return AboutProc(hWnd);
                                 // internal control

                case IDM_FIRST : StartSlideShow(hWnd);    // menu control
                                 return NOTHING;


                case IDM_PREVIOUS:
                                 // previous event - back-up 2 and
                                 // get next event
                                 if(iPreviousEvent > NIL )
                                 {
                                   iEventCounter=iPreviousEvent;
                                 }
                                 else
                                 {
                                    if(iEventCounter>NIL)iEventCounter--;
                                 }
                                 if(iEventCounter>NIL)iEventCounter--;

                case IDM_NEXT  : GetNextEvent(hWnd);      // menu control
                                 return NOTHING;

                case IDM_PAUSEMOUSE :                     // internal control
                                 PauseSlideShow(hWnd,FALSE);
                                 return NOTHING;

                case IDM_PAUSE : PauseSlideShow(hWnd,TRUE);
                                 return NOTHING;          // menu control

                case IDM_RESUME: ResumeSlideShow(hWnd);   // menu control
                                 return NOTHING;

                case IDM_STOP  : StopSlideShow(bSlideShow);
                                 // menu control
                                 return NOTHING;

                case IDM_MINIMIZE:
                                 // menu control
                                 // minimize window
                                 ShowWindow (hWnd, SW_SHOWMINIMIZED) ;
                                 return NOTHING;

                case IDM_QUIT  :                          // menu control
                                 if(StopSlideShow(TRUE)==TRUE)
                                       DestroyWindow(hWnd);
                                 return NOTHING;

                case IDM_ABOUT :
                                 AboutProc(hWnd);         // menu control
                                 return NOTHING;
                case IDM_HELP  :
                                 HelpProc(hWnd);         // menu control
                                 return NOTHING;


               }
               break;

       case WM_PAINT      : return PaintMethod(hWnd);
       case WM_DESTROY    :
                            if(StopSlideShow(TRUE)==FALSE)return NOTHING;
                            return CloseSession(hWnd);

        }

     return DefWindowProc (hWnd, message, wParam, lParam) ;
}

// ------------------------------------------------------------------------
// FUNCTION OpenSession - mcishow version
//
// set-up and start the session
// ------------------------------------------------------------------------
LONG FAR PASCAL OpenSession(HWND hWnd)
{
     WORD        wParam;

     // Set Initial Values
     // Send a Start Session Message if everything is OK
     // Otherwise Send a Quit Session Message
     hdcMain      =NULL;
     hDefPalette  =NULL;
     hBmpHandle   =NULL;
     szErrorBuf[_FIRST_] = ASCIIZ;
     hWndParent   =hWnd;

     if(wParam=GetSystemMenu(hWnd,FALSE))
     {
       DeleteMenu(wParam, SC_MOVE,        MF_BYCOMMAND);
       DeleteMenu(wParam, SC_SIZE,        MF_BYCOMMAND);
       DeleteMenu(wParam, SC_RESTORE,     MF_BYCOMMAND);
       DeleteMenu(wParam, SC_NEXTWINDOW,  MF_BYCOMMAND);
       DeleteMenu(wParam, SC_PREVWINDOW,  MF_BYCOMMAND);
     }

     if((hdcMain=GetDC(hWnd))==NULL)
     {
        wsprintf(szErrorBuf,"%s",lpUncontext);
     }
     else
     {
         if((hBmpHandle  = GlobalAlloc(GMEM_MOVEABLE,dwBmpBufSize))==NULL)
         {
         wsprintf(szErrorBuf,"%s",lpUnprimary);
         }
         else
         {
            wsprintf(szErrorBuf,"%s",lpUnpalette);
            if(hPaletteLoc=FindResource(hInst,lpMyPalette,RT_RCDATA))
            {
              if(hDefPalette=LoadResource(hInst,hPaletteLoc))
              {
               lpDefPalette=LockResource(hDefPalette);
               szErrorBuf[_FIRST_]=ASCIIZ;
              }
              else
              {
                hDefPalette=NULL;
              }
            }
         }

     }


     if(hdcMain == NULL || hBmpHandle == NULL ||  hDefPalette == NULL)
     {
          if(hdcMain    !=NULL)ReleaseDC(hWnd,hdcMain);
          if(hBmpHandle !=NULL)GlobalFree(hBmpHandle);
          if(hDefPalette!=NULL)
          {
              UnlockResource(hDefPalette);
              FreeResource(hDefPalette);
          }
          hDefPalette =NULL;
          hBmpHandle  =NULL;
          hdcMain     =NULL;
          MessageBox(hWnd,szErrorBuf,lpContinue,MB_ICONHAND);
          PostMessage(hWnd,WM_COMMAND,IDM_QUIT,NOTHING);
     }
     else
     {
        PostMessage(hWnd,WM_COMMAND,IDM_START,NOTHING);
     }

return NOTHING;
}

// ------------------------------------------------------------------------
// FUNCTION SysHook - mcishow version
//
// Intercept system messages in MainWndProc
// ------------------------------------------------------------------------
BOOL FAR PASCAL SysHook(HWND hWnd, WORD wParam)
{
    // moving of window is not allowed
    if((wParam&0xfff0)==SC_MOVE||wParam==SC_MOVE)
    {
       MessageBox(hWnd,lpUnmoving,lpCopyRight,MB_ICONHAND);
       return TRUE;
    }

    if((wParam&0xfff0)==SC_MINIMIZE||wParam==SC_MINIMIZE)StopSlideShow(TRUE);

return FALSE;
}

// ------------------------------------------------------------------------
// FUNCTION PaintMethod - mcishow version
//
// our paint method for the main window...
// restore the damaged area of the main window
// by redrawing the bitmap
// ------------------------------------------------------------------------
LONG FAR PASCAL PaintMethod(HWND hWnd)
{
    PAINTSTRUCT ps;

    BeginPaint(hWnd,&ps);
    if(bFirstPaint == TRUE)ShowPic();
    EndPaint(hWnd,&ps);

return NOTHING;
}

// ------------------------------------------------------------------------
// FUNCTION CloseSession - mcishow version
//
// clean-up and end the windows session
// invoked when the main window is destroyed
// ------------------------------------------------------------------------
LONG FAR PASCAL CloseSession(HWND hWnd)
{
   // Free all resources
   // Reset Initial Values
   bFirstPaint = FALSE;
   if(hBmpHandle !=NULL)GlobalFree(hBmpHandle);
   if(hDefPalette!=NULL)
     {
       UnlockResource(hDefPalette);
       FreeResource(hDefPalette);
     }
   if(hpalCustom!=NULL)
     {
       SelectPalette (hdcMain, hpalOriginal, NIL);
       DeleteObject(hpalCustom);
       hpalCustom=NULL;
     }
   if(hdcMain!=NULL)ReleaseDC(hWnd,hdcMain);
   hDefPalette = NULL;
   hBmpHandle  = NULL;
   hdcMain     = NULL;
   if(HELPACTIVE!=FALSE)WinHelp(hWnd,"MCISHOW.HLP",HELP_QUIT,0L);
   HELPACTIVE = FALSE;
   PostQuitMessage (NIL);

return NOTHING;
}

// ------------------------------------------------------------------------
// FUNCTION AboutProc
// ------------------------------------------------------------------------
LONG FAR PASCAL AboutProc(HWND hWnd)
{

    if(bSlideShow==TRUE)
    {
        // if a slideshow is running
        // don't stop.
        MessageBeep(MB_ICONEXCLAMATION);
    }
    else
    {
    MessageBox(hWnd,lpAbout,lpCopyRight,MB_ICONINFORMATION);
    }

return NOTHING;
}

// ------------------------------------------------------------------------
// FUNCTION HelpProc
// ------------------------------------------------------------------------
LONG FAR PASCAL HelpProc(HWND hWnd)
{

    if(bSlideShow==TRUE)
    {
        // if a slideshow is running
        // don't stop.
        MessageBeep(MB_ICONEXCLAMATION);
    }
    else
    {
    if(FileExists("MCISHOW.HLP")==TRUE)
    {
      HELPACTIVE = TRUE;
      HELPACTIVE = WinHelp(hWnd,"MCISHOW.HLP",HELP_CONTENTS,0L);
    }
    else
    {
      MessageBox(hWnd,lpAbout,lpCopyRight,MB_ICONINFORMATION);
     }
    }

return NOTHING;
}



// ------------------------------------------------------------------------
// FUNCTION DebugProc
// ------------------------------------------------------------------------
LONG FAR PASCAL DebugProc()
{

    if(bDebug==TRUE)
    {
       strcat(szMessBuf,"\n Continue?");
       if(MessageBox(hWndParent,szMessBuf,lpDebugMode,
                            MB_ICONQUESTION|MB_YESNO)!=IDYES)
       {
          StopSlideShow(TRUE);
          return SOMETHING;

       }
    }
return NOTHING;
}

// ------------------------------------------------------------------------
// FUNCTION DebugString
// ------------------------------------------------------------------------
LONG FAR PASCAL DebugString(LPSTR lpKeyStr,LPSTR lpValStr)
{
    if(bDebug==TRUE)
    {
      wsprintf(szErrorBuf,"%s=%s\n",lpKeyStr,lpValStr);
      strcat(szMessBuf,szErrorBuf);
    }

return NOTHING;
}


// ------------------------------------------------------------------------
// FUNCTION StartSlideShow - mcishow version
//
// Select a Slideshow and Start It Running
// ------------------------------------------------------------------------
BOOL FAR PASCAL StartSlideShow(HWND hWnd)
{
  StopSlideShow(TRUE);
  lpFilterSpec=(LPSTR)&szMCISpec[_FIRST_];        // select a slideshow
  InitOFNDialog(hWnd);
  if(!GetOpenFileName ((LPOPENFILENAME)&ofn))return FALSE;
  wsprintf(szSlideShow,"%s",ofn.lpstrFile);
  bSlideShow = TRUE;                              // yes, it's true
  PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);

return TRUE;

}

// ------------------------------------------------------------------------
// FUNCTION StopSlideShow - mcishow version
//
// Stop the slideshow if one is Running
// ------------------------------------------------------------------------
BOOL FAR PASCAL StopSlideShow(BOOL bInternal)
{
   bDebug = FALSE;

   if(hWndParent == NULL)return TRUE;

   if(bPauseTimer==TRUE && bTimerPaused==FALSE)
      KillTimer(hWndParent,IDT_PAUSETIMER);
   if(bMidiTimer==TRUE && bMidiPaused==FALSE)
      KillTimer(hWndParent,IDT_MIDI);
   if(bWaveTimer==TRUE && bWavePaused == FALSE)
      KillTimer(hWndParent,IDT_WAVE);
   if(bAviTimer==TRUE && bAviPaused == FALSE)
      KillTimer(hWndParent,IDT_AVI);
   MediaStop(ALL);

   bPauseTimer      = FALSE;
   bWaveTimer       = FALSE;
   bMidiTimer       = FALSE;
   bAviTimer        = FALSE;
   bSlideShow       = FALSE;
   bTimerPaused     = FALSE;
   bWavePaused      = FALSE;
   bMidiPaused      = FALSE;
   bAviPaused       = FALSE;
   bSlideShowPaused = FALSE;
   bLeaveMidiOn     = FALSE;
   iEventCounter    = NIL;
   iPreviousEvent   = NIL;
   iReturnEvent     = NIL;
   iSaveEvent       = NIL;
   iSavePrevious    = NIL;
   iSaveReturn      = NIL;
   iAviLoops        = NIL;
   uiPauseTime      = NIL;
   szCurrentBmp[_FIRST_]=ASCIIZ;
   szSaveShow[_FIRST_]  =ASCIIZ;
   dwClearColor     = 0L;
   ClearScreen(hWndParent,hdcMain);

   // if called from the menu when no slideshow is in progress
   //   mention it.
   // but reset the defaults anyway, since it does no harm.
   if(bInternal==FALSE)
   {
      MessageBox(hWndParent,lpNoSlideShow,lpCopyRight,MB_ICONHAND);
   }

return TRUE;
}

// ------------------------------------------------------------------------
// FUNCTION PauseSlideShow
//
// Pause the slideshow
// ------------------------------------------------------------------------
BOOL FAR PASCAL PauseSlideShow(HWND hWnd,BOOL bStopMidi)
{

    if(bSlideShow==FALSE)
    {
        MessageBox(hWnd,lpNoSlideShow,lpCopyRight,MB_ICONHAND);
        return FALSE;
    }

    if(bSlideShowPaused==TRUE)
    {
        if(bPlayMidi == TRUE && bMidiPlaying == TRUE &&
           bStopMidi == TRUE && bMidiPaused  == FALSE)
        {

          KillTimer(hWnd,IDT_MIDI);
          MCIPause(wMidiDeviceID);
          bMidiPaused = TRUE;
          if(bCDPlaying == TRUE && bStopMidi==TRUE && bCDPaused == FALSE)
          {
            PauseCD();
          }

        }
        else
        {
            if(bCDPlaying == TRUE && bStopMidi==TRUE && bCDPaused == FALSE)
            {
               PauseCD();
            }
            else
            {
               MessageBeep(MB_ICONEXCLAMATION);
            }
        }
        return FALSE;
    }

    bSlideShowPaused= TRUE;
    bWavePaused     = FALSE;
    bMidiPaused     = FALSE;
    bAviPaused      = FALSE;
    bTimerPaused    = FALSE;

    if(bPauseTimer==TRUE)
    {
        KillTimer(hWnd,IDT_PAUSETIMER);
        bTimerPaused = TRUE;
    }

    if(bPlayAvi == TRUE && bAviPlaying == TRUE)
    {
       KillTimer(hWnd,IDT_AVI);
       MCIPause(wAviDeviceID);
       bAviPaused = TRUE;
    }

    if(bPlayMidi == TRUE && bMidiPlaying == TRUE && bStopMidi==TRUE)
    {
       KillTimer(hWnd,IDT_MIDI);
       MCIPause(wMidiDeviceID);
       bMidiPaused = TRUE;
    }

    if(bPlayWave == TRUE && bWavePlaying == TRUE)
    {
       KillTimer(hWnd,IDT_WAVE);
       MCIPause(wWaveDeviceID);
       bWavePaused = TRUE;
    }

    if(bCDPlaying == TRUE && bStopMidi == TRUE)
    {
        PauseCD();
    }

return bSlideShowPaused;
}


// ------------------------------------------------------------------------
// FUNCTION ResumeSlideShow
//
// Resume a Paused SlideShow
// ------------------------------------------------------------------------
BOOL FAR PASCAL ResumeSlideShow(HWND hWnd)
{
    if(bSlideShow==FALSE)
    {
        MessageBox(hWnd,lpNoSlideShow,lpCopyRight,MB_ICONHAND);
        return FALSE;
    }

    if(bSlideShowPaused==FALSE)
    {
       MessageBeep(MB_ICONEXCLAMATION);
       return FALSE;
    }

    if(bMidiPaused==TRUE)
    {
       MCIPlay(wMidiDeviceID);
       SetTimer(hWnd,IDT_MIDI,MIDITIME, NULL);
    }

    if(bAviPaused==TRUE)
    {
       MCIPlay(wAviDeviceID);
       SetTimer(hWnd,IDT_AVI,WAVETIME, NULL);
    }

    if(bWavePaused==TRUE)
    {
       MCIPlay(wWaveDeviceID);
       SetTimer(hWnd,IDT_AVI,WAVETIME, NULL);

    }

    if(bTimerPaused==TRUE)
    {
       SetTimer(hWnd,IDT_PAUSETIMER,PAUSETIME, NULL);
    }

    if(bCDPaused==TRUE)
    {
        bCDPaused = FALSE;
        if(bCDPlaying==TRUE)PlayCD();
    }

    // if none of the timers are active, assume that we need to
    // get the next event from the slideshow script.
    if(bPauseTimer == FALSE && bWaveTimer == FALSE && bAviTimer  == FALSE)
    {
       PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
    }

    bTimerPaused     = FALSE;
    bWavePaused      = FALSE;
    bMidiPaused      = FALSE;
    bAviPaused       = FALSE;
    bCDPaused        = FALSE;
    bSlideShowPaused = FALSE;

return bSlideShowPaused;
}

// ------------------------------------------------------------------------
// FUNCTION GetNextEvent - mcishow version
//
//   get the next event in the slideshow.
//   return FALSE if done or if there is an error.
//
// Note: This function is way too big.
//       I kept this together in a single block for ease of
//       prototyping and debugging only.
//       It will probably be split into smaller handlers for
//       individual commands if MCIShow is finished and rewritten.
// ------------------------------------------------------------------------
BOOL FAR PASCAL GetNextEvent(HWND hWnd)
{
    // work variables
    BYTE szEventBuf[SMALLSIZE],szEventKey[SMALLSIZE];
    UINT wFrom, wTo, uiYes, uiNo, uiCancel;
    BOOL bSlash;
    SINT i, iDirectory, iDirection;

    if(bSlideShow==FALSE)
    {
        MessageBox(hWnd,lpNoSlideShow,lpCopyRight,MB_ICONHAND);
        return FALSE;
    }

    // --------------------------------------------------------------------
    // Set-Up the initial defaults.
    // --------------------------------------------------------------------
    iEventCounter++;           // increment the global event counter

    // --------------------------------------------------------------------
    // if the slideshow is paused
    // resume midi play and cancel other play
    // --------------------------------------------------------------------
    if(bMidiPaused == TRUE)
    {
      MCIPlay(wMidiDeviceID);
      SetTimer(hWnd,IDT_MIDI,MIDITIME, NULL);
           
    }
    else
    {
      if(bAviTimer==TRUE)KillTimer(hWnd,IDT_AVI);
      if(bWaveTimer==TRUE)KillTimer(hWnd,IDT_WAVE);
      if(bPauseTimer==TRUE)KillTimer(hWnd,IDT_PAUSETIMER);
    }

    if(bCDPaused == TRUE)PlayCD();   // assume for the moment that we can

    MediaStop(AVI);                  // exclusive events so just stop them
    MediaStop(WAVE);

    uiPauseTime  = NIL;
    bPauseTimer  = FALSE;
    bAviTimer    = FALSE;
    bWaveTimer   = FALSE;
    bTimerPaused = FALSE;
    bMidiPaused  = FALSE;
    bAviPaused   = FALSE;
    bCDPaused    = FALSE;
    bSlideShowPaused = FALSE;

    bLeaveMidiOn     = FALSE;
    bLeaveCDOn       = FALSE;
    iAviLoops        = NIL;

    // --------------------------------------------------------------------
    // get the next event
    // --------------------------------------------------------------------
    szEventBuf[_FIRST_] = ASCIIZ;
    szMessBuf[_FIRST_]  = ASCIIZ;
    wsprintf(szEventKey,"%s%d",lpEventKey,iEventCounter);
    GetPrivateProfileString(szEventKey,lpEventKey,"",szEventBuf,SMALLMAX,
                            szSlideShow);

    // --------------------------------------------------------------------
    // event=debug
    // modifiers
    //   yes/no
    //   default is no
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpDebug)==SUCCESS)
    {
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpDebug,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        bDebug = FALSE;
        // if it's not yes, it's no...
        // if it's no don't interrupt the show
        if(strcmpi(szEventBuf,lpYesKey)==NIL)
        {
           bDebug = TRUE;
           MessageBox(hWnd,"Entering Debug Mode...",
                            lpCopyRight,MB_ICONINFORMATION);
        }
        PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
        return TRUE;
    }
    DebugString("Current Event",szEventKey);
    DebugString(lpEventKey,szEventBuf);

    // --------------------------------------------------------------------
    // event=goto
    // modifiers
    //   goto=eventnumber
    //   force next event... skip over code blocks...
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpGoto)==SUCCESS)
    {
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpGoto,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpGoto,szEventBuf);
        if(DebugProc()==SOMETHING)return FALSE;
        if(szEventBuf[_FIRST_] != ASCIIZ)
        {
          iEventCounter=atoi(szEventBuf)-1;
        }
        PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
        return TRUE;
    }

    // --------------------------------------------------------------------
    // event=gosub
    // modifiers
    //   gosub=eventnumber
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpGosub)==SUCCESS)
    {
        szEventBuf[_FIRST_]=ASCIIZ;
        iReturnEvent = 0;    // reset return counter each gosub
                             // this helps ensure that unmatched returns
                             // will restart the show...
        GetPrivateProfileString(szEventKey,lpGosub,"",
                                 szEventBuf,SMALLMAX,szSlideShow);

        DebugString(lpGosub,szEventBuf);
        if(DebugProc()==SOMETHING)return FALSE;
        // nowhere to go...
        if(szEventBuf[_FIRST_] != ASCIIZ)
        {
          iReturnEvent = iEventCounter;    // save event counter
          iEventCounter=atoi(szEventBuf)-1;
        }
        PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
        return TRUE;
    }


    // --------------------------------------------------------------------
    // event=return
    // modifiers
    //   none
    //   returning from a gosub... check for return events...
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpReturn) == SUCCESS)
    {
      wsprintf(szEventBuf,"%d",iReturnEvent);
      DebugString(lpReturn,szEventBuf);
      if(DebugProc()==SOMETHING)return FALSE;
      iEventCounter=iReturnEvent;   // leave the last return loaded for now
      PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
      return TRUE;
    }

    // --------------------------------------------------------------------
    // event=call
    // modifiers
    //   name=mci script name
    //   call=event number
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpCall) == SUCCESS)
    {
         // get the file name for the mci file that we will call
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpNameKey,"",szEventBuf,SMALLMAX,
                                 szSlideShow);


         if(FileExists(szEventBuf)==TRUE)
         {
           // save the current show
           // we shall return on exit if we are not aborted...
           wsprintf(szSaveShow,"%s",szSlideShow);
           DebugString("save",szSaveShow);

           // windows wants a fully qualified path name for the
           // MCI included file...
           // if it already is entered this way, fine.
           // otherwise assume that the included file is in the
           // current directory.
           bSlash=FALSE; // assume current directory
           for(i=0;szEventBuf[i]!=ASCIIZ;i++)
           {
             if(szEventBuf[i]==SLASH||szEventBuf[i]==COLON)
             {
                bSlash=TRUE;break; // other directory
             }

           }
           // if we found any path modifiers, we need to use the
           //   literal path information...
           // otherwise the path implies the last saved slideshow
           // this will fail if calls are nested and directory paths
           //   are mixed and mismatched but we only
           //   support single level call conditions.
           if(bSlash == TRUE)
           {
             wsprintf(szSlideShow,"%s",szEventBuf);
           }
           else
           {
             // use the directory and drive for the last loaded
             // show... this is the default if no paths are given.
             iDirectory=0;   // strip the root from the filename...
             for(i=0;szSlideShow[i]!=0;i++)
             {
               if(szSlideShow[i]==SLASH || szSlideShow[i]==COLON)
                 iDirectory=i+1;
             }
             szSlideShow[iDirectory]=ASCIIZ;  // terminate the root name
             strcat(szSlideShow,szEventBuf);  // add the new file name
           }
           // display the fully qualified name during debug mode
           DebugString(lpNameKey,szSlideShow);

           szEventBuf[_FIRST_]=ASCIIZ;
           GetPrivateProfileString(szEventKey,lpCall,"",szEventBuf,SMALLMAX,
                                 szSaveShow);

           DebugString(lpCall,szEventBuf);
           iSaveEvent     = iEventCounter;     // store event counter
           iSavePrevious  = iPreviousEvent;
           iSaveReturn    = iReturnEvent;
           iEventCounter  = atoi(szEventBuf);

           // we always need to decrement the eventcounter
           // to normalize it because it always gets advanced
           // when we do the next event...
           if(iEventCounter<1)iEventCounter = NIL;
           else iEventCounter-=1;

           iPreviousEvent = NIL;
           iReturnEvent   = NIL;

           if(DebugProc()==SOMETHING)return FALSE;

         }
         else
         {
           MessageBox(hWnd,lpInvalidEntry,szEventBuf,MB_ICONHAND);
          }
         PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
         return TRUE;
    }

    // --------------------------------------------------------------------
    // event=exit
    // modifiers
    //   none
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpExit) == SUCCESS)
    {
         DebugString(lpExit,szSlideShow);
         wsprintf(szSlideShow,"%s",szSaveShow);
         DebugString(lpReturn,szSlideShow);

         iEventCounter   = iSaveEvent;
         iPreviousEvent  = iSavePrevious;
         iReturnEvent    = iSaveReturn;

         szSaveShow[_FIRST_] = ASCIIZ;
         iSaveEvent      = NIL;
         iSavePrevious   = NIL;
         iSaveReturn     = NIL;

         wsprintf(szEventBuf,"%d",iEventCounter);
         DebugString(lpGoto,szEventBuf);

         if(FileExists(szSlideShow)==TRUE)
         {
          if(DebugProc()==SOMETHING)return FALSE;
          PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
          return TRUE;
         }
         MessageBox(hWnd,szMessBuf,lpInvalidEntry,MB_ICONHAND);
         strcpy(szEventBuf,lpEnd);
    }


    // --------------------------------------------------------------------
    // event=clear
    // clear the screen and loop back for the next MCI event
    // modifiers
    //   previous=previous major event
    //   red=0-255
    //   green=0-255
    //   blue=0-255
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpClearKey)==SUCCESS)
    {
        dwClearColor = 0L;
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpBlue,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpBlue,szEventBuf);
        dwClearColor  |= (BYTE)atoi(szEventBuf);
        szEventBuf[_FIRST_]=ASCIIZ;
        dwClearColor <<= 8;
        GetPrivateProfileString(szEventKey,lpGreen,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpGreen,szEventBuf);
        dwClearColor  |= (BYTE)atoi(szEventBuf);
        dwClearColor <<= 8;
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpRed,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpRed,szEventBuf);
        dwClearColor  |= (BYTE)atoi(szEventBuf);
        ClearScreen(hWnd,hdcMain);
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        if(szEventBuf[_FIRST_]!=ASCIIZ)
        {
          iPreviousEvent = atoi(szEventBuf);
          DebugString(lpPrevious,szEventBuf);
        }
        if(DebugProc()==SOMETHING)return FALSE;

        PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
        return TRUE;
    }


    // --------------------------------------------------------------------
    // event=bmp
    // If we are loading a BMP we loop back for the next MCI event afterwards.
    // modifiers  (lots of them)
    //   usepalette=yes/no/default
    //   scale=   1-10, or box, width, height
    //   width=   scale=box, scale=width
    //   height=  scale=box, scale=height
    //   xorg=    x origin for load (default is centre)
    //   yorg=    y origin for load (default is centre)
    //   previous= previous major event
    //   playmidi=yes/no (yes=default)
    //   playcd=yes/no   (yes=default)
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpBmpKey)==SUCCESS)
    {
         szBmpFile[_FIRST_] = ASCIIZ;
         bSetPalette        = FALSE;
         bGrayPalette       = FALSE;
         uiXoffset          = CENTRE;
         uiYoffset          = CENTRE;
         uiScale            = FULL;
         uiFit              = NIL;
         uiFitWidth         = NIL;
         uiFitHeight        = NIL;
         bLeaveMidiOn       = TRUE;        // yes, it's true
         bLeaveCDOn         = TRUE;

         // get the bmp file name
         GetPrivateProfileString(szEventKey,lpNameKey,"",szBmpFile,SMALLMAX,
                                 szSlideShow);
         DebugString(lpNameKey,szBmpFile);

         // get the palette information
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpPaletteKey,"",
                                   szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpPaletteKey,szEventBuf);
         if(strcmpi(szEventBuf,lpYesKey)==SUCCESS)bSetPalette=TRUE;
         if(strcmpi(szEventBuf,lpGrayPalette)==SUCCESS)
             {
              bSetPalette =TRUE;
              bGrayPalette=TRUE;
              }

         // get the display origin coordinates
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpXoffset,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpXoffset,szEventBuf);
         if(szEventBuf[_FIRST_]!=ASCIIZ)uiXoffset=(UINT)atoi(szEventBuf);

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpYoffset,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpYoffset,szEventBuf);
         if(szEventBuf[_FIRST_]!=ASCIIZ)uiYoffset=(UINT)atoi(szEventBuf);

         // get the display scale,box,width,height,whatever
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpScale,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpScale,szEventBuf);

         uiScale=(UINT)atoi(szEventBuf);
         if(strcmpi(szEventBuf,lpBox)==NIL)uiFit    = FITBOX;
         if(strcmpi(szEventBuf,lpWidth)==NIL)uiFit  = FITWIDTH;
         if(strcmpi(szEventBuf,lpHeight)==NIL)uiFit = FITHEIGHT;

         if(uiFit==NIL)
         {
           if(uiScale==NIL || uiScale > MAXSCALE)uiScale = FULL;
         }
         else
         {
            szEventBuf[_FIRST_]=ASCIIZ;
            GetPrivateProfileString(szEventKey,lpWidth,"",
                                    szEventBuf,SMALLMAX,szSlideShow);
            DebugString(lpWidth,szEventBuf);
            uiFitWidth = (UINT)atoi(szEventBuf);
            szEventBuf[_FIRST_]=ASCIIZ;
            GetPrivateProfileString(szEventKey,lpHeight,"",
                                    szEventBuf,SMALLMAX,szSlideShow);
            DebugString(lpHeight,szEventBuf);
            uiFitHeight = (UINT)atoi(szEventBuf);
         }

         // are we supposed to turn the midi off now?
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpLeaveMidiOn,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpLeaveMidiOn,szEventBuf);
         if(strcmpi(szEventBuf,lpNoKey)==SUCCESS)bLeaveMidiOn=FALSE;

         // are we supposed to turn the cd off now?
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpLeaveCDOn,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpLeaveCDOn,szEventBuf);
         if(strcmpi(szEventBuf,lpNoKey)==SUCCESS)bLeaveCDOn=FALSE;

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         if(szEventBuf[_FIRST_]!=ASCIIZ)
         {
            iPreviousEvent = atoi(szEventBuf);
            DebugString(lpPrevious,szEventBuf);
         }

         // display the debug proc before going further
         if(DebugProc()==SOMETHING)return FALSE;

         // turn-off multimedia play as requested
         if(bLeaveMidiOn == FALSE)
         {
           if(bMidiTimer==TRUE)KillTimer(hWnd,IDT_MIDI);
           bMidiTimer = FALSE;
           MediaStop(MIDI);
         }

         if(bCDPlaying == TRUE && bLeaveCDOn == FALSE)StopCD();

         // load the picture and loop back for the next event
         if(FileExists(szBmpFile)==TRUE)
           LoadPic(szBmpFile,hWnd);
         else
           MessageBox(hWnd,lpInvalidEntry,szBmpFile,MB_ICONHAND);
         PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
         return TRUE;
    }

    // --------------------------------------------------------------------
    // event=pause
    // if we are pausing, we set a timer and complete it before
    // returning for another event. (1 second resolution)
    // modifiers
    //   seconds=  0=pause for mouse (default)
    //   previous= previous major event
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpPauseKey)==SUCCESS)
    {
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         if(szEventBuf[_FIRST_]!=ASCIIZ)
         {
           iPreviousEvent = atoi(szEventBuf);
           DebugString(lpPrevious,szEventBuf);

         }

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpSecondsKey,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         uiPauseTime = (UINT)atoi(szEventBuf);   // wait time in seconds
         // format the values for the debug proc
         // during the assignment of the callback
         if(uiPauseTime>NIL)
         {
           strcat(szEventBuf,lpSecondPause);
           bPauseTimer = TRUE;
         }
         else
         {
           uiPauseTime = NIL;
           strcpy(szEventBuf,lpMousePause);
         }

         DebugString(lpSecondsKey,szEventBuf);
         if(DebugProc()==SOMETHING)return FALSE;

         if(bPauseTimer==TRUE)SetTimer(hWnd,IDT_PAUSETIMER,PAUSETIME,NULL);
         else PostMessage(hWnd,WM_COMMAND,IDM_PAUSEMOUSE,NOTHING);
         return TRUE;
    }


    // --------------------------------------------------------------------
    // event=message
    // title=
    // text=
    // modifiers
    //   yes=     eventnumber
    //   no=      eventnumber
    //   cancel=  eventnumber
    //   previous=eventnumber
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpMessage)==SUCCESS)
    {
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         if(szEventBuf[_FIRST_]!=ASCIIZ)
         {
           iPreviousEvent = atoi(szEventBuf);
           DebugString(lpPrevious,szEventBuf);
         }

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpYesKey,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpYesKey,szEventBuf);
         uiYes = atoi(szEventBuf);

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpNoKey,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpNoKey,szEventBuf);
         uiNo = atoi(szEventBuf);

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpCancelKey,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpCancelKey,szEventBuf);
         uiCancel  = atoi(szEventBuf);

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpTitle,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpTitle,szEventBuf);
         if(DebugProc()==SOMETHING)return FALSE;

         szMessBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpText,"",
                                 szMessBuf,MIDMAX,szSlideShow);

         // if no yes or no, we use an ok style box
         if(uiNo < 1 && uiYes  < 1)
         {
            MessageBox(hWnd,szMessBuf,szEventBuf,MB_ICONINFORMATION);
            
         }
         else
         {

            // if no cancel, we use a yesno box
            if(uiCancel < 1)
            {
              if(MessageBox(hWnd,szMessBuf,szEventBuf,
                            MB_ICONQUESTION|MB_YESNO)==IDYES)
              {
                  if(uiYes>NIL)iEventCounter=uiYes-1;
              }
              else
              {
                if(uiNo>NIL)iEventCounter=uiNo-1;
              }
            }
            else
            {
              // otherwise we have a value for cancel
              // so we use that...
              i = MessageBox(hWnd,szMessBuf,szEventBuf,
                             MB_YESNOCANCEL | MB_ICONQUESTION);

              switch(i)
              {
                case IDCANCEL:
                      iEventCounter = uiCancel-1;
                      break;
                case IDNO:
                      if(uiNo>NIL)iEventCounter=uiNo-1;
                      break;
                case IDYES:
                      if(uiYes>NIL)iEventCounter=uiYes-1;
                      break;
              }
      

            }

         }
         PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
         return TRUE;

    }

    // --------------------------------------------------------------------
    // event=midi
    // midi play can take place during all events. avi play is a special
    // case and by default midi play is turned-off before avi play.
    // midi play can be optionally turned-off during wave play.
    // if we are playing a new midi file we will cancel previous midi play.
    // modifiers
    //   playcd=yes/no (default=no)
    //   previous= previous major event
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpMidiKey)==SUCCESS)
    {
          bLeaveCDOn         = FALSE;

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         if(szEventBuf[_FIRST_]!=ASCIIZ)
         {
           iPreviousEvent = atoi(szEventBuf);
           DebugString(lpPrevious,szEventBuf);

         }

          // read the keys and post the debug messages first if in
          // debug mode.

          szMidiFile[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpNameKey,"",szMidiFile,
                                  SMALLMAX,szSlideShow);
          DebugString(lpNameKey,szMidiFile);

          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpLeaveCDOn,"",
                                  szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpLeaveCDOn,szEventBuf);
          if(strcmpi(szEventBuf,lpYesKey)==SUCCESS)bLeaveCDOn=TRUE;
          if(DebugProc()==SOMETHING)return FALSE;

          // stop any playback that will interfere with the midi file
          if(bLeaveCDOn == FALSE && bCDPlaying == TRUE)StopCD();
          if(bMidiTimer==TRUE)KillTimer(hWnd,IDT_MIDI);
          bMidiTimer = FALSE;
          MediaStop(MIDI);

          if(FileExists(szMidiFile)==FALSE)
          {
             MessageBox(hWnd,lpInvalidEntry,szMidiFile,MB_ICONHAND);
          }
          else
          {
            // if they can't play the midi file
            // don't bother them about it.
            if(bPlayMidi == TRUE)
            {
              if(MediaPlay(szMidiFile,MIDI)==TRUE)
              {
                SetTimer(hWnd,IDT_MIDI,MIDITIME, NULL);
                bMidiTimer = TRUE;
              }
            }
          }
          PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
          return TRUE;
    }

    // --------------------------------------------------------------------
    // event=wav
    // wav file playback will last the duration of the wav file.
    // modifiers
    //   playmidi=yes/no (default is yes)
    //   playcd=yes/no   (default is yes)
    //   previous= previous major event
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpWaveKey)==SUCCESS)
    {
         // usually leave midi and cd on when playing wave
         bLeaveMidiOn = TRUE;
         bLeaveCDOn   = TRUE;

         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         if(szEventBuf[_FIRST_]!=ASCIIZ)
         {
           iPreviousEvent = atoi(szEventBuf);
           DebugString(lpPrevious,szEventBuf);

         }

         // add the filename to the debug string.
         szWaveFile[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpNameKey,"",szWaveFile,
                                 SMALLMAX,szSlideShow);
         DebugString(lpNameKey,szWaveFile);

         // turn midi off?
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpLeaveMidiOn,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpLeaveMidiOn,szEventBuf);
         if(strcmpi(szEventBuf,lpNoKey)==SUCCESS)bLeaveMidiOn=FALSE;

         // turn cd off?
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpLeaveCDOn,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpLeaveCDOn,szEventBuf);
         if(strcmpi(szEventBuf,lpNoKey)==SUCCESS)bLeaveCDOn=FALSE;

         // display debug messages now if in debug mode to avoid
         // trashing the global message buffers used by the debug proc.
         if(DebugProc()==SOMETHING)return FALSE;

         // turn-off other sound as required
         if(bLeaveCDOn   == FALSE && bCDPlaying == TRUE)StopCD();
         if(bLeaveMidiOn == FALSE)
         {
           if(bMidiTimer==TRUE)KillTimer(hWnd,IDT_MIDI);
           bMidiTimer = FALSE;
           MediaStop(MIDI);
         }

         // if the wave file is missing just complain
         // otherwise try to play it.
         if(FileExists(szWaveFile)==FALSE)
         {
           MessageBox(hWnd,lpInvalidEntry,szWaveFile,MB_ICONHAND);
         }
         else
         {
           // if they can play a wave file then wait until it finishes
           // otherwise jump to the next event.
           if(bPlayWave == TRUE)
           {
             if(MediaPlay(szWaveFile,WAVE)==TRUE)
             {
                SetTimer(hWnd,IDT_WAVE,WAVETIME, NULL);
                bWaveTimer = TRUE;
                return TRUE;
             }
           }
         }
         PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
         return TRUE;
    }

    // --------------------------------------------------------------------
    // event=avi
    // avi play will result in the screen being cleared unless clear=no
    // and will shut-off everything else unless playmidi is set to yes.
    // modifiers
    //   usewindow=yes/no  if yes use a window with a border and title
    //   movewindow=yes/no if no don't move the window to the centre
    //   clear=yes/no      if no don't clear the screen
    //                     use these color values to clear the screen
    //   red=0-255
    //   green=0-255
    //   blue=0-255
    //   playmidi=yes/no   if yes continue to play midi
    //   playcd=yes/no     if yes continue to play cd
    //   previous= previous major event
    //   loop=1
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpAviKey)==SUCCESS)
    {

        // avi playback controls
        bAviCentre       = TRUE;
        bAviClear        = TRUE;
        dwAviStyle       = WS_CHILD;
        bLeaveMidiOn     = FALSE;
        bLeaveCDOn       = FALSE;
        iAviLoops        = NIL;

        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
        if(szEventBuf[_FIRST_]!=ASCIIZ)
         {
           iPreviousEvent = atoi(szEventBuf);
           DebugString(lpPrevious,szEventBuf);
         }

        // get the avi file name
        szAviFile[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpNameKey,"",szAviFile,
                                           SMALLMAX,szSlideShow);
        DebugString(lpNameKey,szAviFile);

        // centre the avi window, default is yes
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpAviCentre,"",
                                szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpAviCentre,szEventBuf);
        if(strcmpi(szEventBuf,lpNoKey)==SUCCESS)bAviCentre = FALSE;

        // add a border and caption to avi child window style,
        // default is child window style
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpAviStyle,"",
                                szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpAviStyle,szEventBuf);
        if(strcmpi(szEventBuf,lpYesKey)==SUCCESS)
                            dwAviStyle = WS_CHILD|WS_BORDER|WS_CAPTION;

        // clear screen before avi playback, default is yes
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpClearKey,"",
                                szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpClearKey,szEventBuf);
        if(strcmpi(szEventBuf,lpNoKey)==SUCCESS)bAviClear=FALSE;

        // turn midi off before playing avi
        szEventBuf[_FIRST_]=ASCIIZ;
        GetPrivateProfileString(szEventKey,lpLeaveMidiOn,"",
                                szEventBuf,SMALLMAX,szSlideShow);
        DebugString(lpLeaveMidiOn,szEventBuf);
        if(strcmpi(szEventBuf,lpYesKey)==SUCCESS)bLeaveMidiOn=TRUE;

         // turn cd off?
         szEventBuf[_FIRST_]=ASCIIZ;
         GetPrivateProfileString(szEventKey,lpLeaveCDOn,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
         DebugString(lpLeaveCDOn,szEventBuf);
         if(strcmpi(szEventBuf,lpYesKey)==SUCCESS)bLeaveCDOn=TRUE;

         if(bAviClear==TRUE)
         {
          dwClearColor = 0L;
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpBlue,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpBlue,szEventBuf);
          dwClearColor  |= (BYTE)atoi(szEventBuf);
          dwClearColor <<= 8;
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpGreen,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpGreen,szEventBuf);
          dwClearColor  |= (BYTE)atoi(szEventBuf);
          dwClearColor <<= 8;
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpRed,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpRed,szEventBuf);
          dwClearColor  |= (BYTE)atoi(szEventBuf);

          ClearScreen(hWnd,hdcMain);
         }
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpLoop,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpLoop,szEventBuf);
          if(DebugProc()==SOMETHING)return FALSE;
          if((iAviLoops = atoi(szEventBuf))<1)iAviLoops=1;


         if(bLeaveCDOn   == FALSE && bCDPlaying == TRUE)StopCD();
         if(bLeaveMidiOn==FALSE)
         {
            if(bMidiTimer==TRUE)KillTimer(hWnd,IDT_MIDI);
            bMidiTimer = FALSE;
            MediaStop(MIDI);
         }

         if(FileExists(szAviFile)==FALSE)
         {
            MessageBox(hWnd,lpInvalidEntry,szAviFile,MB_ICONHAND);
         }
         else
         {

          if(bPlayAvi==TRUE)
           {
             if(MediaPlay(szAviFile,AVI)==TRUE)
              {
                if(bAviCentre==TRUE)CentreAvi();
                InvalidateRect(hWnd, NULL, FALSE);
                UpdateWindow(hWnd);
                MCIPlay(wAviDeviceID);
                SetTimer(hWnd,IDT_AVI,AVITIME,NULL);
                bAviTimer = TRUE;
                return TRUE;
             }
           }
         }
         iAviLoops = NIL;
         PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
         return TRUE;

    }

    // --------------------------------------------------------------------
    // event=cd
    // cd play can take place during all events.
    // the CD does not use a timer callback and once opened the device is
    // left on for the duration of the slideshow.
    // modifiers
    //   direction=      (+-tracks from current position)
    //   from=           (track 1... - 0 = start at 1 and loop)
    //   to=             (track 1... - 0 = loop)
    //   playmidi=yes/no (default=no)
    //   previous=previous major event
    // This has been pretty much added-in as an afterthought.
    // Since I wasn't writing a full featured CD player, I am dealing
    //   with only track specific or continuous CD playback.
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpCDKey)==SUCCESS)
    {
          bLeaveMidiOn = FALSE;
          wFrom        = NIL;
          wTo          = NIL;

          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpPrevious,"",
                                 szEventBuf,SMALLMAX,szSlideShow);
          if(szEventBuf[_FIRST_]!=ASCIIZ)
          {
           iPreviousEvent = atoi(szEventBuf);
           DebugString(lpPrevious,szEventBuf);
          }
          // get the starting track
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpDirection,"",
                                  szEventBuf,SMALLMAX,szSlideShow);

          iDirection=atoi(szEventBuf);
          DebugString(lpDirection,szEventBuf);

          // get the starting track
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpFrom,"",
                                  szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpFrom,szEventBuf);

          wFrom = (UINT)atoi(szEventBuf);
          if(wFrom<LOWEST)wFrom=LOWEST;

          // get the ending track if any
          // if its the same as the starting track, we only play 1 track
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpTo,"",
                                    szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpTo,szEventBuf);
          wTo = (UINT)atoi(szEventBuf);
          if(wTo<LOWEST)wTo=NIL;

          // if we are playing a midi file we may need to turn it off
          szEventBuf[_FIRST_]=ASCIIZ;
          GetPrivateProfileString(szEventKey,lpLeaveMidiOn,"",
                                  szEventBuf,SMALLMAX,szSlideShow);
          DebugString(lpLeaveMidiOn,szEventBuf);
          if(strcmpi(szEventBuf,lpYesKey)==SUCCESS)bLeaveMidiOn=TRUE;
          if(DebugProc()==SOMETHING)return FALSE;

          // stop any playback that will interfere with cd playback
          if(bCDPlaying == TRUE)StopCD();
          if(bLeaveMidiOn==FALSE)
          {
            if(bMidiTimer==TRUE)KillTimer(hWnd,IDT_MIDI);
            bMidiTimer = FALSE;
            MediaStop(MIDI);
          }
          // play the CD if we can
          bCDPlaying = FALSE;
          if(iDirection == NIL)
          {
            MCICDPlay(wFrom,wTo);
          }
          else
          {
            MCICDNext(iDirection);
          }
          PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
          return TRUE;
    }

    // --------------------------------------------------------------------
    // event=end
    // modifiers=none
    // check for the "end"
    // --------------------------------------------------------------------
    if(strcmpi(szEventBuf,lpEnd) == NIL)szEventBuf[_FIRST_]=ASCIIZ;

    if(szEventBuf[_FIRST_]!=ASCIIZ)
    {

        DebugString(lpNonEvent,"?");
        if(DebugProc()==SOMETHING)return FALSE;
        PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
        return TRUE;

    }

    // --------------------------------------------------------------------
    // if there are no additional valid events we are done.
    // --------------------------------------------------------------------

    DebugString(szSlideShow,lpSlideDone);
    DebugProc();
    PostMessage(hWnd,WM_COMMAND,IDM_STOP,NOTHING);

return FALSE;
}

// ------------------------------------------------------------------------
// FUNCTION SlideShowTimerProc - mcishow version
//
// Timer Callback Routines for MCI SlideShow Events
// ------------------------------------------------------------------------
BOOL FAR PASCAL SlideShowTimerProc(HWND hWnd, WORD wParam)
{

  switch(wParam)
   {

    case IDT_WAVE:

      KillTimer(hWnd,IDT_WAVE);
      if(bPlayWave == TRUE && bWavePlaying == TRUE)
      {
        if(IsMCIPlaying(WAVE)==TRUE && bSlideShow==TRUE)
        {
            SetTimer(hWnd,IDT_WAVE,WAVETIME, NULL);
            return TRUE;
        }
        MediaStop(WAVE);
      }
      bWaveTimer = FALSE;
      if(bSlideShow==TRUE)PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
      return TRUE;

    case IDT_MIDI:

      KillTimer(hWnd,IDT_MIDI);
      if(bPlayMidi == TRUE && bMidiPlaying == TRUE)
      {
        if(IsMCIPlaying(MIDI)==TRUE && bSlideShow==TRUE)
        {
            SetTimer(hWnd,IDT_MIDI,MIDITIME, NULL);
            return TRUE;
        }
        MediaStop(MIDI);
      }
      bMidiTimer = FALSE;
      return TRUE;

    case IDT_AVI:

      KillTimer(hWnd,IDT_AVI);
      if(bPlayAvi == TRUE && bAviPlaying == TRUE)
      {
        if(IsMCIPlaying(AVI)==TRUE && bSlideShow==TRUE)
        {
            SetTimer(hWnd,IDT_AVI,WAVETIME, NULL);
            return TRUE;
        }
        if(bSlideShow==TRUE && iAviLoops>1)
        {
          iAviLoops-=1;
          MCIRewind(wAviDeviceID);
          MCIPlay(wAviDeviceID);
          SetTimer(hWnd,IDT_AVI,WAVETIME, NULL);
          return TRUE;
        
        }
        MediaStop(AVI);
      }
      iAviLoops = NIL;
      bAviTimer = FALSE;
      if(bSlideShow==TRUE)PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
      return TRUE;

    case IDT_PAUSETIMER:

      KillTimer(hWnd,IDT_PAUSETIMER);
      if(bSlideShow == TRUE)
      {
        if(uiPauseTime>NIL)uiPauseTime-=1;
        if(uiPauseTime>NIL)
        {
            SetTimer(hWnd,IDT_PAUSETIMER,PAUSETIME, NULL);
            return TRUE;
        }
        bPauseTimer = FALSE;
        PostMessage(hWnd,WM_COMMAND,IDM_NEXT,NOTHING);
        return TRUE;

      }
      else
      {
        uiPauseTime = NIL;
        bPauseTimer = FALSE;
      }
      break;

    }
return FALSE;
}

// ========================================================================
// MULTIMEDIA CORE FUNCTIONS
//
// MCI Playback API Interface Functions
// Note: I do not use MCI callbacks because I consider tracking
//         these less straight forward than timer callbacks.
// ========================================================================

// ------------------------------------------------------------------------
// FUNCTION MediaPlay
//
// Play the Specified Multimedia File on the Specified MCI Device
// Unless it's an avi file then open it and let the dialog take over.
// ------------------------------------------------------------------------
BOOL FAR PASCAL MediaPlay(LPSTR lpMediaFileName, UINT wDeviceType)
{
   MCI_OVLY_OPEN_PARMS   mciOpenParms; // same as DGV_OPEN_PARMS
   MCI_GENERIC_PARMS     mciCloseParms;
   MCI_OVLY_WINDOW_PARMS mciWindow;    // same as DGV_WINDOW_PARMS
   MCI_DGV_STATUS_PARMS  mciStatus;    // 2 more fields than MCI_STATUS_PARMS
   MCI_PLAY_PARMS        mciPlayParms;

   DWORD dwReturn, dwFlags=NOTHING, dwOpenFlags=NOTHING;
   UINT wDeviceID=(UINT)NIL, ui;
   BOOL bPlaying = FALSE;
   LPSTR lpDevice = NULL, lpAviTitle = NULL;

   // return false if the file does not exist
   if(FileExists(lpMediaFileName)==FALSE)return FALSE;

   // ---------------------------------------------------------------------
   // if the device was previously disabled don't go any further
   //   otherwise cancel previous play if any
   //   nothing should be playing anyway
   // 1. if we are playing a wave file shut-off WAVE and AVI
   // 2. if we are playing a midi file shut off MIDI and AVI
   // 3. if we are playing an AVI file shut off WAVE and MIDI
   // ---------------------------------------------------------------------

   switch(wDeviceType)
   {
     case WAVE:

            if(bPlayWave == FALSE)return FALSE;
            MediaStop(AVI);
            MediaStop(wDeviceType);
            lpDevice = (LPSTR)&szWaveDevice[_FIRST_];
            break;

     case MIDI:

            if(bPlayMidi == FALSE)return FALSE;
            MediaStop(AVI);
            MediaStop(wDeviceType);
            lpDevice = (LPSTR)&szMidiDevice[_FIRST_];
            break;

     case AVI:

            if(bPlayAvi == FALSE)return FALSE;
            MediaStop(WAVE);
            if(bLeaveMidiOn!=TRUE)MediaStop(MIDI);
            MediaStop(wDeviceType);
            lpDevice = (LPSTR)&szAviDevice[_FIRST_];
            break;

     default:

            return FALSE;
   }

   // ---------------------------------------------------------------------
   // The video overlay or "old" MCI equivalents of the MCI DGV
   //   support are used where possible in order to maintain as much
   //   compatibility as I can with the Windows 3.1 sdk.
   // ---------------------------------------------------------------------

   mciCloseParms.dwCallback      = NOTHING;

   mciOpenParms.dwCallback       = NOTHING;
   mciOpenParms.lpstrElementName = lpMediaFileName;
   mciOpenParms.lpstrDeviceType  = lpDevice;
   mciOpenParms.wDeviceID        = NIL;
   mciOpenParms.lpstrAlias       = NULL;

   if(wDeviceType==AVI)
   {

     // Create a child window with a title bar
     mciOpenParms.dwStyle = dwAviStyle;
     mciOpenParms.hWndParent = hWndParent;
     dwOpenFlags = (DWORD)
     (MCI_OPEN_TYPE|MCI_OPEN_ELEMENT|MCI_OVLY_OPEN_PARENT|MCI_OVLY_OPEN_WS);
   }
   else
   {
     dwOpenFlags = (DWORD)
     (MCI_OPEN_TYPE|MCI_OPEN_ELEMENT);
   }

   dwReturn = mciSendCommand(NIL,MCI_OPEN,dwOpenFlags,
                             (DWORD)(LPVOID)&mciOpenParms);
   if(!dwReturn)
   {

      wDeviceID = mciOpenParms.wDeviceID;
      if(wDeviceType==AVI)
      {
            wAviDeviceID=wDeviceID;
            bPlaying    = TRUE;
            bAviPlaying = TRUE;

            // ------------------------------------------------------------
            // Put the avi filename in the window...
            // display only the avi base filename without the
            //   directory name... point past all ascii 92's & colons
            // ------------------------------------------------------------
            lpAviTitle = (LPSTR) &lpMediaFileName[_FIRST_];
            for(ui=NIL;lpMediaFileName[ui]!=ASCIIZ;ui++)
            {
                if(lpMediaFileName[ui]==SLASH||
                   lpMediaFileName[ui]==COLON)
                   lpAviTitle=(LPSTR) &lpMediaFileName[ui+1];
            }

            // show the playback window
            mciWindow.dwCallback = NOTHING;
            mciWindow.hWnd       = NULL;
            mciWindow.nCmdShow   = SW_SHOW;
            mciWindow.lpstrText  = lpAviTitle;

            mciSendCommand(wDeviceID,MCI_WINDOW,
                          MCI_OVLY_WINDOW_STATE,
                          (DWORD)(LPVOID)&mciWindow);

            // get the avi window handle
            mciStatus.dwItem = MCI_OVLY_STATUS_HWND;
            mciSendCommand(wDeviceID,
                           MCI_STATUS, MCI_STATUS_ITEM,
                           (DWORD)(LPVOID)&mciStatus);

            hWndAVI = (HWND)mciStatus.dwReturn;
      }
      else
      {
        mciPlayParms.dwCallback = NOTHING;
        mciPlayParms.dwFrom = mciPlayParms.dwTo = NIL;
        if(dwReturn = mciSendCommand(wDeviceID,MCI_PLAY,dwFlags,
                               (DWORD)(LPVOID)&mciPlayParms))
        {
         mciSendCommand(wDeviceID,MCI_CLOSE,NOTHING,
                        (DWORD)(LPVOID)&mciCloseParms);
         wDeviceID = (UINT)NIL;
         }
      }
   }

   if(!dwReturn)
   {
     bPlaying = TRUE;
     switch(wDeviceType)
     {
        case WAVE:
            wWaveDeviceID=wDeviceID;
            bWavePlaying =bPlaying;
            break;
        case MIDI:
            wMidiDeviceID=wDeviceID;
            bMidiPlaying =bPlaying;
            break;
     }
   }
   else
   {
        if(!mciGetErrorString(dwReturn,(LPSTR)szErrorBuf,MIDMAX))
                              wsprintf(szErrorBuf,lpUnError);

        wsprintf(szMessBuf,"%s\n\n%s (%s)?",
                           szErrorBuf,lpDisable,lpDevice);
        if(MessageBox(NULL,szMessBuf,lpMediaFileName,
                            MB_ICONQUESTION|MB_YESNO)==IDYES)
        {
            switch(wDeviceType)
            {
                case WAVE:
                           bPlayWave = FALSE;break;
                case MIDI:
                           bPlayMidi = FALSE;break;
                case AVI:
                           bPlayAvi  = FALSE;
                           iAviLoops = NIL;
                           break;
            }
        }
    }

return bPlaying;
}

// ------------------------------------------------------------------------
// FUNCTION MediaStop
//
// Stops MCI Play
// always returns FALSE
// ------------------------------------------------------------------------
BOOL FAR PASCAL MediaStop(UINT wDeviceType)
{

   switch(wDeviceType)
   {

   case WAVE:
            if(wWaveDeviceID)mciSendCommand(wWaveDeviceID,
                                            MCI_CLOSE,NOTHING,NULL);
            wWaveDeviceID=(UINT)NIL;
            bWavePlaying = FALSE;
            break;
   case MIDI:
            if(wMidiDeviceID)mciSendCommand(wMidiDeviceID,
                                            MCI_CLOSE,NOTHING,NULL);
            wMidiDeviceID=(UINT)NIL;
            bMidiPlaying = FALSE;
            break;
   case AVI:
           if(wAviDeviceID)mciSendCommand(wAviDeviceID,
                                          MCI_CLOSE,NOTHING,NULL);
           wAviDeviceID =(UINT)NIL;
           bAviPlaying  = FALSE;
           break;


   default:
           // kill the devices
           CloseCDDevice();
           if(wAviDeviceID)mciSendCommand(wAviDeviceID,
                                          MCI_CLOSE,NOTHING,NULL);
           if(wWaveDeviceID)mciSendCommand(wWaveDeviceID,
                                          MCI_CLOSE,NOTHING,NULL);
           if(wMidiDeviceID)mciSendCommand(wMidiDeviceID,
                                          MCI_CLOSE,NOTHING,NULL);
           wWaveDeviceID=(UINT)NIL;
           wMidiDeviceID=(UINT)NIL;
           wAviDeviceID =(UINT)NIL;
           wCDDeviceID  =(UINT)NIL;

           // reset the play status flags
           bMidiPlaying = FALSE;
           bWavePlaying = FALSE;
           bAviPlaying  = FALSE;
           bCDPlaying   = FALSE;

   }


return FALSE;
}

// ------------------------------------------------------------------------
// FUNCTION IsMCIPLaying - mcishow version
//
// returns TRUE if the specified device is playing
// returns FALSE if not
//
// .WAV, .MID, and .AVI MCI Playback Devices all return a
//   relative length and a status of playing when a device is
//   playing properly...
// if the MCI device is not playing and we have got this far
//   an error has occurred and we should just close the device
//   if it isn't closed already.
// if no play is occurring and the device is still open we
//   should still close it.
// ------------------------------------------------------------------------
BOOL FAR PASCAL IsMCIPlaying(UINT wDeviceType)
{
    MCI_DGV_STATUS_PARMS mciStatusParms;
    UINT wDeviceID=(UINT)NIL;
    DWORD dwMode,dwStart,dwLength,dwCompleted,dwErr;
    BOOL bResult =   TRUE;          // assume we're still playing
    LPSTR lpDevice = NULL;

    switch(wDeviceType)
    {
      case WAVE:
                    if(bPlayWave==FALSE)break;
                    if(bWavePlaying == FALSE)break;
                    wDeviceID = wWaveDeviceID;
                    wDeviceType   = WAVE;
                    lpDevice = (LPSTR)&szWaveDevice[_FIRST_];
                    break;
      case MIDI:

                    if(bPlayMidi==FALSE)break;
                    if(bMidiPlaying == FALSE)break;
                    wDeviceID = wMidiDeviceID;
                    wDeviceType   = MIDI;
                    lpDevice = (LPSTR)&szMidiDevice[_FIRST_];
                    break;

      case AVI:
                    if(bPlayAvi==FALSE)break;
                    if(bAviPlaying == FALSE)break;
                    wDeviceID = wAviDeviceID;
                    wDeviceType   = AVI;
                    lpDevice = (LPSTR)&szAviDevice[_FIRST_];
                    break;
   }

   if(!wDeviceID || bResult==FALSE)return FALSE;

   switch(wDeviceType)
   {

      default:

      // If we get an error when we try to get the device status
      //   for the mode of an open driver we don't want to continue
      // In the case of AVI play, we may have an active timer
      //   and we might bring the system down if we ignore the error
      //   perpetually.
      // Not certain about using the messagebox here but what the heck!
      mciStatusParms.dwItem = MCI_STATUS_MODE;
      dwErr= mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM,
                               (DWORD)(LPVOID) &mciStatusParms);
      if(dwErr)
      {
        if(!mciGetErrorString(dwErr,(LPSTR)szErrorBuf,MIDMAX))
                              wsprintf(szErrorBuf,lpUnstatus);

        wsprintf(szMessBuf,"%s\n\n%s (%s)?",
                           szErrorBuf,lpCancel,lpDevice);
        if(MessageBox(NULL,szMessBuf,lpCopyRight,
                            MB_ICONQUESTION|MB_YESNO)==IDYES)bResult = FALSE;
        break;
      }
      dwMode = mciStatusParms.dwReturn;

      mciStatusParms.dwItem = MCI_STATUS_POSITION;
      dwErr = mciSendCommand(wDeviceID,MCI_STATUS,
                               MCI_STATUS_ITEM|MCI_STATUS_START,
                               (DWORD)(LPVOID) &mciStatusParms);
      if(dwErr)break;
      dwStart = mciStatusParms.dwReturn;

      mciStatusParms.dwItem = MCI_STATUS_LENGTH;
      dwErr=mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM,
                               (DWORD)(LPVOID) &mciStatusParms);
      if(dwErr)break;
      dwLength = mciStatusParms.dwReturn;

      mciStatusParms.dwItem = MCI_STATUS_POSITION;
      dwErr = mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM,
                               (DWORD)(LPVOID) &mciStatusParms);
      if(dwErr)break;
      dwCompleted = mciStatusParms.dwReturn;

      if(dwCompleted == dwLength)bResult = FALSE;
      // maybe still playing... catch next time
      if(dwMode == ((DWORD)MCI_MODE_PLAY))bResult = TRUE;

   }
return bResult;
}

//-------------------------------------------------------------------------
// Function MCIPause
//
// Pause an MCI Device
//-------------------------------------------------------------------------
BOOL FAR PASCAL MCIPause(UINT wDeviceID)
{
    MCI_GENERIC_PARMS mciPauseParms;

    if(!wDeviceID)return FALSE;

    mciPauseParms.dwCallback = NOTHING;
    if(mciSendCommand(wDeviceID,MCI_PAUSE,NOTHING,
                               (DWORD)(LPVOID)&mciPauseParms))return FALSE;

return TRUE;
}

//-------------------------------------------------------------------------
// Function MCIPlay - mcishow version
//
// Playback function to start or resume play if MCI file is already loaded.
//-------------------------------------------------------------------------
BOOL FAR PASCAL MCIPlay(UINT wDeviceID)
{
    MCI_PLAY_PARMS  mciPlayParms;  // ok for avi. same as DGV_PLAY_PARMS
    DWORD dwResult  = NOTHING, dwFlags = NOTHING;

    if(!wDeviceID)return FALSE;

    mciPlayParms.dwCallback = NOTHING;
    mciPlayParms.dwFrom = mciPlayParms.dwTo = NIL;
    if(mciSendCommand(wDeviceID,MCI_PLAY,dwFlags,
                     (DWORD)(LPVOID)&mciPlayParms))return FALSE;
return TRUE;
}

//-------------------------------------------------------------------------
// Function MCIRewind - mcishow version
//
// Pause and rewind an MCI file if MCI file is already loaded.
//-------------------------------------------------------------------------
BOOL FAR PASCAL MCIRewind(UINT wDeviceID)
{
    MCI_DGV_STATUS_PARMS mciStatusParms;
    MCI_GENERIC_PARMS    mciPauseParms;

    if(!wDeviceID)return FALSE;

    // if device is playing, pause it first
    mciStatusParms.dwItem = MCI_STATUS_MODE;
    if(!mciSendCommand(wDeviceID,MCI_STATUS,MCI_STATUS_ITEM,
                      (DWORD)(LPVOID) &mciStatusParms))
    {
      if(mciStatusParms.dwReturn == ((DWORD)MCI_MODE_PLAY))
       {
          mciPauseParms.dwCallback = NOTHING;
          mciSendCommand(wDeviceID,MCI_PAUSE,NOTHING,
                               (DWORD)(LPVOID)&mciPauseParms);
       }
    }

 // now rewind it to the beginning
 if(mciSendCommand(wDeviceID,MCI_SEEK,MCI_SEEK_TO_START,(DWORD)(LPVOID)NULL))
       return FALSE;

return TRUE;
}

// ------------------------------------------------------------------------
// FUNCTION CentreAvi - mcishow version
//
// centre the AVI window in the parent window
// ------------------------------------------------------------------------
BOOL FAR PASCAL CentreAvi()
{
    RECT rcAvi, rcParent, rcBounds;
    MCI_OVLY_RECT_PARMS mciRect;

    // don't go further if we don't have both windows and a device handle
    if(hWndParent == NULL || hWndAVI == NULL || (!wAviDeviceID))
       return FALSE;

    // get the size of the parent window's client area 0,0,x,y
    GetClientRect(hWndParent, &rcParent);

    // get the size of the bounding frame for the AVI playback window
    mciSendCommand(wAviDeviceID, MCI_WHERE,
                  (DWORD)(MCI_OVLY_WHERE_SOURCE),
                  (DWORD)(LPVOID)&mciRect);

    CopyRect( &rcBounds, &mciRect.rc );

    rcAvi.left   = (rcParent.right/2) -(rcBounds.right/2);
    rcAvi.top    = (rcParent.bottom/2)-(rcBounds.bottom/2);
    rcAvi.right  = rcAvi.left + rcBounds.right;
    rcAvi.bottom = rcAvi.top  + rcBounds.bottom;

    // reposition the playback (child) window
    MoveWindow(hWndAVI, rcAvi.left, rcAvi.top,
                        rcBounds.right, rcBounds.bottom,FALSE);

return TRUE;

}


// ------------------------------------------------------------------------
// FUNCTION MCICDPlay
//
// High Level CD Player to handle script requests for CD Play
// CD Play is non-exclusive except of other CD play.
// ------------------------------------------------------------------------
BOOL FAR PASCAL MCICDPlay(UINT wStartTrack, UINT wStopTrack)
{
     BYTE bFrom, bTo;
     MCI_SET_PARMS  mciset;
     DWORD dwTracks;
     UINT   wTracks;
     BOOL   bRetVal;

     bCDPaused        = FALSE;              // not paused anymore
     bCDPlaying       = FALSE;              // not playing either
     bCDContinuous    = FALSE;              // certainly not looping
     dwCDStartPosition= NOTHING;            // start position is NOTHING
     dwCDStopPosition = NOTHING;            // stop position is NOTHING

     // open the CDDevice if it's not already open
     if(OpenCDDevice()==FALSE)return FALSE;

     if(wCDDeviceID == NIL || wStartTrack < LOWEST)return FALSE;

     // set the timing format to minute/second/frame
     if(SetCDTimeFormat()==FALSE)return FALSE;

     // get the total number of tracks for this CD
     if((dwTracks = GetNumberOfCDTracks())==NOTHING)return FALSE;

     // if the tracks are out of range return false
     wTracks = (UINT)dwTracks;
     if(wTracks<wStartTrack || wTracks<wStopTrack)return FALSE;

     dwCDStartPosition = GetCDStartPosition(wStartTrack);
     if(wStopTrack<1)
     {
        bCDContinuous = TRUE;
     }
     else
     {
        bCDContinuous = FALSE;
        wStopTrack+=1;
        if(wStopTrack>wTracks)wStopTrack=1;
        dwCDStopPosition = GetCDStartPosition(wStopTrack);
     }

     // 3 retries on continuous play
     bRetVal = PlayCD();
     if(bRetVal == FALSE && bCDContinuous == TRUE)
     {
       // maybe data at start of CD so try playing second track
       if(wStartTrack==1)
       {
          dwCDStartPosition = GetCDStartPosition(2);
     
       }
       else
       {  // maybe data at end of CD so try playing first track
          dwCDStartPosition = GetCDStartPosition(1);
     
       }
       bRetVal = PlayCD();
       if(bRetVal==FALSE)
       {
          dwCDStartPosition = GetCDStartPosition(wStartTrack);
       }
       bRetVal = PlayCD();

     }


return bRetVal;
}

// ------------------------------------------------------------------------
// FUNCTION MCICDNext
//
// Play Next or Previous Track
// ------------------------------------------------------------------------
BOOL FAR PASCAL MCICDNext(SINT iDirection)
{
     BYTE bFrom, bTo;
     MCI_SET_PARMS  mciset;
     DWORD dwTracks,dw;
     UINT   wStartTrack,wStopTrack;
     UINT   wTracks;
     SINT   i;
     BOOL   bRetVal;

     bCDPaused        = FALSE;              // not paused anymore
     bCDPlaying       = FALSE;              // not playing either
     bCDContinuous    = FALSE;              // certainly not looping
     dwCDStartPosition= NOTHING;            // start position is NOTHING
     dwCDStopPosition = NOTHING;            // stop position is NOTHING

     // open the CDDevice if it's not already open
     if(OpenCDDevice()==FALSE)return FALSE;

     if(wCDDeviceID == NIL)return FALSE;

     // set the timing format to minute/second/frame
     if(SetCDTimeFormat()==FALSE)return FALSE;

     // get the total number of tracks for this CD
     if((dwTracks = GetNumberOfCDTracks())==NOTHING)return FALSE;

     wTracks = (UINT)dwTracks;

     if((dw = GetCurrentCDTrack())==NOTHING)return FALSE;

     // if going forwards or backwards
     i = (SINT)dw;
     i+= iDirection;
     if(i>wTracks)i = 1;
     if(i<1)i=wTracks;

     wStartTrack = (UINT)i;

     dwCDStartPosition = GetCDStartPosition(wStartTrack);
     bCDContinuous = TRUE;

     // 3 retries on continuous play
     bRetVal = PlayCD();
     if(bRetVal == FALSE)
     {
       // maybe data at start of CD so try playing second track
       if(wStartTrack==1)
       {
          dwCDStartPosition = GetCDStartPosition(2);
     
       }
       else
       {  // maybe data at end of CD so try playing first track
          dwCDStartPosition = GetCDStartPosition(1);
     
       }
       bRetVal = PlayCD();
       // last try...
       if(bRetVal==FALSE)
       {
          dwCDStartPosition = GetCDStartPosition(wStartTrack);
       }
       bRetVal = PlayCD();

     }

return bRetVal;
}

// ------------------------------------------------------------------------
// FUNCTION OpenCDDevice
//   helper function called by MCICDPLay each time a request for CD play
//     is made.
//   the global CD device ID is assigned if not assigned yet but
//     no other global flags are set in this function.
//   returns TRUE if successful or if the device is already open and
//     FALSE if not.
// ------------------------------------------------------------------------
BOOL FAR PASCAL OpenCDDevice()
{

     MCI_OPEN_PARMS mciopen;
     DWORD dwRes;
     DWORD dwFlags;
     BOOL  bResult,bSaveDebug;

     // if the CD is already open just return TRUE
     if(wCDDeviceID)return TRUE;

     mciopen.lpstrDeviceType=(LPSTR)&szCDDevice[_FIRST_];
     dwFlags= MCI_OPEN_TYPE | MCI_OPEN_SHAREABLE;

     dwRes=mciSendCommand(NIL,MCI_OPEN,dwFlags,(DWORD)(LPSTR)&mciopen);

     // assign the CD device ID if no error
     // or display the error if there was an error
     if(dwRes==NOTHING)wCDDeviceID=mciopen.wDeviceID;

     bSaveDebug = bDebug;                   // always post an open error
     bDebug     = TRUE;                     // only post others if in debug
     bResult    = CDErrorProc(dwRes);       // returns TRUE if no error
     bDebug     = bSaveDebug;               // restore previous state
     return bResult;
}

// ------------------------------------------------------------------------
//  FUNCTION  CloseCDDevice
//    Closes the CD device if open and resets the Global ID and Flags
// ------------------------------------------------------------------------
BOOL FAR PASCAL CloseCDDevice()
{
     BOOL bResult=FALSE;

     StopCD();                              // stop and eject
     if(wCDDeviceID != NIL)
     {
     bResult=CDErrorProc(mciSendCommand(wCDDeviceID,MCI_CLOSE,NOTHING,NULL));
     wCDDeviceID      = (UINT)NIL;
     }
     return bResult;
}

// ------------------------------------------------------------------------
//
// FUNCTION  PlayCD
//
// This function starts the CD playing.
//
// The dwCDStartPosition and dwCDStopPosition global values are already set.
//   If CDPlay is Paused, CDPlay Resumes.
// ------------------------------------------------------------------------
BOOL FAR PASCAL PlayCD()
{
     MCI_OPEN_PARMS mciopen;
     MCI_PLAY_PARMS mciplay;
     MCI_SET_PARMS  mciset;
     BYTE bFrom, bTo;
     DWORD dwRes;
     DWORD dwFlags;

     if(wCDDeviceID == NIL)return FALSE;

     // play the CD
     mciplay.dwFrom=dwCDStartPosition;
     dwFlags=MCI_FROM;

     if(bCDContinuous!=TRUE)
     {
       dwFlags|=MCI_TO;
       mciplay.dwTo=dwCDStopPosition;
     }

     bCDPaused  = FALSE;
     bCDPlaying = TRUE;    // cheated here...

     return CDErrorProc(mciSendCommand(wCDDeviceID,
                        MCI_PLAY,dwFlags,(DWORD)(LPSTR)&mciplay));


}

// --------------------------------------------------------------------------
//  FUNCTION  StopCD
// --------------------------------------------------------------------------
BOOL FAR PASCAL StopCD()
{
     MCI_GENERIC_PARMS mcigeneric;
     DWORD dwFlags=NOTHING;
     BOOL  bResult    = FALSE;

     bCDPaused        = FALSE;              // not paused anymore
     bCDPlaying       = FALSE;              // not playing either
     bCDContinuous    = FALSE;              // certainly not looping
     dwCDStartPosition= NOTHING;            // start position is NOTHING
     dwCDStopPosition = NOTHING;            // stop position is nothing

     if(wCDDeviceID != NIL)
     {
       bResult = CDErrorProc(mciSendCommand(wCDDeviceID,MCI_STOP,
                             dwFlags,(DWORD)(LPSTR)&mcigeneric));
     }
     CDEject();
     return bResult;
}

// ------------------------------------------------------------------------
// FUNCTION IsCDPlaying
// ------------------------------------------------------------------------
BOOL FAR PASCAL IsCDPlaying()
{
     MCI_STATUS_PARMS mcistatus;
     DWORD dwFlags;
     DWORD dwRes;

     if(wCDDeviceID == NIL)return FALSE;

     dwFlags          = MCI_STATUS_ITEM;
     mcistatus.dwItem = MCI_STATUS_MODE;

     if(dwRes=mciSendCommand(wCDDeviceID,MCI_STATUS,dwFlags,
                          (DWORD)(LPSTR)&mcistatus))
                          {
                            return CDErrorProc(dwRes);
                          }

     if(mcistatus.dwReturn == MCI_MODE_STOP)StopCD();

return bCDPlaying;
}


// ------------------------------------------------------------------------
// FUNCTION PauseCD
// ------------------------------------------------------------------------
BOOL FAR PASCAL PauseCD()
{
     MCI_GENERIC_PARMS mcigeneric;
     DWORD dwFlags;

     if(IsCDPlaying()==FALSE)return FALSE;

     dwFlags=NOTHING;
     CDErrorProc(mciSendCommand(wCDDeviceID,
                             MCI_PAUSE,dwFlags,(DWORD)(LPSTR)&mcigeneric));

     // save the new start position
     dwCDStartPosition=GetCurrentCDPosition();

     bCDPaused = TRUE; // cheated here...
     return bCDPaused;
}

// ------------------------------------------------------------------------
//  FUNCTION GetNumberOfCDTracks
//
//    This function gets the number of tracks on the CD.
// ------------------------------------------------------------------------
DWORD FAR PASCAL GetNumberOfCDTracks()
{
     MCI_STATUS_PARMS mcistatus;
     DWORD dwFlags;
     DWORD dwRes;

     if(wCDDeviceID == NIL)return NOTHING;

     dwFlags           =MCI_STATUS_ITEM;
     mcistatus.dwItem = MCI_STATUS_NUMBER_OF_TRACKS;

     if(dwRes=mciSendCommand(wCDDeviceID,MCI_STATUS,dwFlags,
                            (DWORD)(LPSTR)&mcistatus))
                            {
                               CDErrorProc(dwRes);
                               mcistatus.dwReturn = NOTHING;
                             }
return mcistatus.dwReturn;
}

// ------------------------------------------------------------------------
// FUNCTION GetCDStartPosition
//
// Returns the start position for the specified track on the current CD
// ------------------------------------------------------------------------
DWORD FAR PASCAL GetCDStartPosition(UINT wTrack)
{

     MCI_STATUS_PARMS mcistatus;
     DWORD dwFlags;

     if(wCDDeviceID == NIL)return NOTHING;

     dwFlags           = MCI_STATUS_ITEM|MCI_TRACK;
     mcistatus.dwItem  = MCI_STATUS_POSITION;
     mcistatus.dwTrack = (DWORD)wTrack;

     if(CDErrorProc(mciSendCommand(wCDDeviceID,MCI_STATUS,dwFlags,
                                   (DWORD)(LPSTR)&mcistatus))==FALSE)
                            {
                               mcistatus.dwReturn = NOTHING;
                             }

return mcistatus.dwReturn;
}

// ------------------------------------------------------------------------
// FUNCTION GetCurrentCDTrack
// ------------------------------------------------------------------------
DWORD FAR PASCAL GetCurrentCDTrack()
{

     MCI_STATUS_PARMS mcistatus;
     DWORD dwFlags;

     if(wCDDeviceID == NIL)return NOTHING;

     dwFlags=MCI_STATUS_ITEM;
     mcistatus.dwItem = MCI_STATUS_CURRENT_TRACK;

     if(CDErrorProc(mciSendCommand(wCDDeviceID,MCI_STATUS,dwFlags,
                                   (DWORD)(LPSTR)&mcistatus))==FALSE)
                            {
                               mcistatus.dwReturn = NOTHING;
                             }

return mcistatus.dwReturn;
}

// ------------------------------------------------------------------------
// FUNCTION SetCDTimeFormat
//
// Set CD Time Format to Minute/Second/Frame
// Returns TRUE if Successful
// ------------------------------------------------------------------------
BOOL FAR PASCAL SetCDTimeFormat()
{
     MCI_SET_PARMS mciset;
     DWORD dwFlags;
     DWORD dwRes;

     if(wCDDeviceID == NIL)return FALSE;

     // set time format to minute/second/frame
     mciset.dwTimeFormat = MCI_FORMAT_MSF;
     dwFlags             = MCI_SET_TIME_FORMAT;

     return CDErrorProc(mciSendCommand(wCDDeviceID,MCI_SET,dwFlags,
                                       (DWORD)(LPSTR)&mciset));
}

// ------------------------------------------------------------------------
//  FUNCTION  GetCurrentCDPosition
//
//  This function gets the current track position playing on the CD
// ------------------------------------------------------------------------
DWORD FAR PASCAL GetCurrentCDPosition()
{
     MCI_STATUS_PARMS mcistatus;
     MCI_SET_PARMS    mciset;
     DWORD dwFlags;
     DWORD dwRes;

     if(wCDDeviceID == NIL)return NOTHING;

     dwFlags          = MCI_STATUS_ITEM;
     mcistatus.dwItem = MCI_STATUS_POSITION;

     if(CDErrorProc(mciSendCommand(wCDDeviceID,MCI_STATUS,dwFlags,
                                   (DWORD)(LPSTR)&mcistatus))==FALSE)
                                   {
                                     mcistatus.dwReturn = NOTHING;
                                   }

return mcistatus.dwReturn;
}

// ------------------------------------------------------------------------
// FUNCTION CDEject
//   Ejects the CD Once Per Play Session if requested to do so.
//   If the current CD hasn't already been ejected, ejects the current CD
//     and resets the Global Eject Request Completion Flag.
// ------------------------------------------------------------------------
BOOL FAR PASCAL CDEject()
{
     MCI_SET_PARMS mciset;
     DWORD dwFlags;

     // if no CD device ID or CD has already been ejected returns FALSE
     if(wCDDeviceID == NIL || bCDEject    == FALSE)
     {
        bCDEject = FALSE;
        return FALSE;
     }

     bCDEject = FALSE;          // reset the flag anyway, try only once
     if(MessageBox(NULL,lpEject,lpCopyRight,
                        MB_ICONQUESTION|MB_YESNO)==IDNO)return TRUE;

     dwFlags=MCI_SET_DOOR_OPEN;
     // post a message if an error or return TRUE if not
     return CDErrorProc(mciSendCommand(wCDDeviceID,MCI_SET,dwFlags,
                                       (DWORD)(LPSTR)&mciset));

}

// ------------------------------------------------------------------------
//  FUNCTION  CDErrorProc
//
//  CDErrorProc calls mciGetErrorString and displays an error message
//  returned by MCI if in DEBUG mode.
//
//  Returns TRUE if no Error, FALSE if Error
// ------------------------------------------------------------------------
BOOL FAR PASCAL CDErrorProc(DWORD dwReturn)
{
    if(dwReturn)
    {
      if(bDebug==TRUE)
      {
        if(!mciGetErrorString(dwReturn,(LPSTR)szErrorBuf,MIDMAX))
        {
          wsprintf(szErrorBuf,"%s",lpUnToo);
        }
        wsprintf(szMessBuf,"%s%s",lpCDError,szErrorBuf);
        MessageBox(NULL,szMessBuf,lpCopyRight,MB_ICONHAND);
      }
      return FALSE;
    }

return TRUE;
}

// ------------------------------------------------------------------------
// FUNCTION LoadPic - mcishow version
//
//   BMP Reader and Display function for MainWndProc
//   load a non compressed bitmap up to the size of the screen
//   and up to 16.7 million colors
// ------------------------------------------------------------------------
BOOL FAR PASCAL LoadPic(LPSTR lpNewBmp, HWND hWnd)
{

    FILE *fp;
    LPSTR lpPtr;
    DWORD dwBytesPerLine;
    SINT  k,j,i,x;
    BYTE  huge *hpTemp;
    UINT  uiHeight;

    if(lpNewBmp[_FIRST_]==ASCIIZ)return FALSE;

    if((fp=fopen(lpNewBmp,"rb"))==NULL)
    {
     MessageBeep(MB_ICONEXCLAMATION);
     MessageBox(hWnd,lpOpenError,lpNewBmp,MB_ICONEXCLAMATION);
     return FALSE;
    }

    bFirstPaint = FALSE;                             // disable paint method
    fread((LPSTR)&bmfhead.bfType,sizeof(BITMAPFILEHEADER),1,fp);
    fread((LPSTR)&bmp.bmiHeader.biSize,sizeof(MYBITMAPINFO),1,fp);

    lpPtr=(LPSTR)&bmfhead.bfType;
    // check header
    if(lpPtr[_FIRST_] != 'B' || lpPtr[1]  != 'M' ||
       bmp.bmiHeader.biPlanes   !=  1  ||              // normal BMP's
       bmp.bmiHeader.biBitCount >   24 ||
       bmp.bmiHeader.biCompression != BI_RGB)          // no DIB's or RLE's
        {
         fclose(fp);
         ClearScreen(hWnd,hdcMain);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(hWnd,lpFormatError,lpNewBmp,MB_ICONHAND);
         if(hpalCustom!=NULL)DeleteObject(hpalCustom);
         hpalCustom=NULL;
         szCurrentBmp[_FIRST_]=ASCIIZ;
         return FALSE;
        }

    uiBmpWidth  = (UINT)(LONG)bmp.bmiHeader.biWidth;
    uiBmpHeight = (UINT)(LONG)bmp.bmiHeader.biHeight;

    if(uiBmpWidth > uiXwidth || uiBmpHeight > uiYheight)
    {
         fclose(fp);
         ClearScreen(hWnd,hdcMain);
         wsprintf(szMessBuf,lpTooLarge,uiBmpWidth,uiBmpHeight,
                                       uiXwidth,uiYheight);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(hWnd,szMessBuf,lpNewBmp,MB_ICONHAND);
         if(hpalCustom!=NULL)DeleteObject(hpalCustom);
         hpalCustom=NULL;
         szCurrentBmp[_FIRST_]=ASCIIZ;
         return FALSE;
     }

    if(strcmpi(szCurrentBmp,lpNewBmp)==SUCCESS)
    {
      fclose(fp);
      ShowPic();
      bFirstPaint = TRUE;
      return bFirstPaint;
    }

    szCurrentBmp[_FIRST_]=ASCIIZ;
    dwBytesPerLine =  filelength(fileno(fp));
    dwBytesPerLine -= bmfhead.bfOffBits;
    dwBytesPerLine /= uiBmpHeight;
    uiBytesPerLine =  (UINT)dwBytesPerLine;

    // --------------------------------------------------------------------
    // based on image type, set j to the number of colors required to
    //   create the new palette from the palette entries in the bmp file.
    //   (we may not need this if we are reusing the previous palette.
    // set k to the image type (truecolor or normal).
    // adjust bytes per line if this may cause a problem.
    // --------------------------------------------------------------------
    j=bmp.bmiHeader.biBitCount;
    k=NIL;                            // k=24=24bit, k=0=normal
    switch(j)
    {
        case 4:  j=16;
                 break;
        case 8:  j=256;
                 uiBytesPerLine = uiBmpWidth;
                 while((uiBytesPerLine%4)!=NIL)uiBytesPerLine++;
                 break;
        case 24: j=256;
                 k=24;
                 uiBytesPerLine = uiBmpWidth*3;
                 while((uiBytesPerLine%4)!=NIL)uiBytesPerLine++;
                 break;
        default: j=2;
    }

    // if we don't have a palette yet let's get one
    // otherwise let them over-ride the palette check
    if(hpalCustom==NULL || bSetPalette==TRUE)
    {
        bSetPalette = TRUE;
        if(bGrayPalette==TRUE)
        {
            k=24;                   // default palette over-ride
            j=256;
        }
        if(hpalCustom!=NULL)DeleteObject(hpalCustom);
        myLogPal.Version = 0x300;
        myLogPal.Entries = j;
        x=NIL;
        for(i=NIL;i<j;i++)
        {
         switch(k)
         {
          case 24:
                myLogPal.Entry[i].peRed  =  lpDefPalette[x];x++;
                myLogPal.Entry[i].peGreen=  lpDefPalette[x];x++;
                myLogPal.Entry[i].peBlue =  lpDefPalette[x];x++;
                myLogPal.Entry[i].peFlags=  NIL;
                break;
          default:

                myLogPal.Entry[i].peRed  =  bmp.bmiColors[i].rgbRed;
                myLogPal.Entry[i].peGreen=  bmp.bmiColors[i].rgbGreen;
                myLogPal.Entry[i].peBlue =  bmp.bmiColors[i].rgbBlue;
                myLogPal.Entry[i].peFlags=  NIL;
         }
      }
      hpalCustom=CreatePalette((PLOGPALETTE )&myLogPal.Version);
    }

    uiHeight  = uiBmpHeight;

    if((hpBmpBuffer = GlobalLock(hBmpHandle))==NULL)
    {
        fclose(fp);
        ClearScreen(hWnd,hdcMain);
        MessageBox(hWnd,lpNoLock,lpLockRead,MB_ICONHAND);
        if(hpalCustom!=NULL)DeleteObject(hpalCustom);
        hpalCustom = NULL;
        return FALSE;
    }

    fseek(fp,(LONG)bmfhead.bfOffBits,SEEK_SET);
    hpTemp   = hpBmpBuffer;
    fread(hpTemp,uiBytesPerLine,uiHeight,fp);
    fclose(fp);
    GlobalUnlock(hBmpHandle);
    hpBmpBuffer = NULL;

    // if we are resetting the palette, we need to clear the screen
    // this is because of the ugly way the windows display reacts
    // to new palettes.
    if(bSetPalette==TRUE)ClearScreen(hWnd,hdcMain);
    if(ShowPic()!=TRUE)
    {
       if(bSetPalette==FALSE)ClearScreen(hWnd,hdcMain);
       MessageBox(hWnd,lpNoLock,lpLockDisplay,MB_ICONHAND);
       if(hpalCustom!=NULL)DeleteObject(hpalCustom);
       hpalCustom = NULL;
       return FALSE;
    }

    strcpy(szCurrentBmp,lpNewBmp);
    bFirstPaint = TRUE;

return bFirstPaint;
}


// ------------------------------------------------------------------------
// FUNCTION ScaleSrcToDest
//
// Helper function to scale a value from one ratio to another.
// ------------------------------------------------------------------------
UINT ScaleSrcToDest(UINT uiRetVal, UINT uiSrcUnits, UINT uiDestUnits)
{
    DWORD dwTemp      = (DWORD)uiRetVal;
    DWORD dwSrcUnits  = (DWORD)uiSrcUnits;
    DWORD dwDestUnits = (DWORD)uiDestUnits;


    if(uiRetVal < 1 || uiSrcUnits < 1 || uiDestUnits < 1)return NIL;

    dwTemp   *= 10L;              // good enough precision for pixel graphics
    dwTemp   *= dwDestUnits;      // multiply first
    dwTemp   =  dwTemp/dwSrcUnits;// then divide
    dwTemp   =  dwTemp/10;
    uiRetVal = (UINT)dwTemp;

return uiRetVal;
}

// ------------------------------------------------------------------------
// FUNCTION ShowPic - mcishow version
//
// BMP Display Function used by MainWndProc BMP Reader and Paint Method
// copy a file into view, scale as required.
// ------------------------------------------------------------------------
BOOL FAR PASCAL ShowPic()
{

   // center the BMP in the frame
   // unless the script has over-ride dimensions

   UINT cx=uiXborder;
   UINT cy=uiYborder;
   UINT uiDestWidth  = uiBmpWidth;
   UINT uiDestHeight = uiBmpHeight;

   // ---------------------------------------------------------------------
   // if scale is set to NIL scale to fit
   //   for aspect ratio fit, calculate the first time through,
   //   then change to a box fit.
   // ---------------------------------------------------------------------
   if(uiScale==NIL)
   {

     // if either dimension is eliminated or is out of range
     // then the screen dimension is used for that dimension
     if(uiFitHeight < 1 || uiFitHeight > uiYheight)uiFitHeight=uiYheight;
     if(uiFitWidth < 1  || uiFitWidth > uiXwidth)uiFitWidth=uiXwidth;

     switch(uiFit)
     {
          case FITHEIGHT:

            for(;;)
            {
              if(uiFitHeight == NIL)break;
              uiFitWidth = ScaleSrcToDest(uiBmpWidth,uiBmpHeight,uiFitHeight);
              if(uiFitWidth == NIL)break;
              if(uiFitWidth > uiXwidth)
              {
                uiFitHeight-=1;
                continue;
              }
              break;

            }
            break;

          case FITWIDTH:

            for(;;)
            {
              if(uiFitWidth == NIL)break;
              uiFitHeight = ScaleSrcToDest(uiBmpHeight,uiBmpWidth,uiFitWidth);
              if(uiFitHeight == NIL)break;
              if(uiFitHeight > uiYheight)
              {
                uiFitWidth-=1;
                continue;
              }
              break;

            }
     }

     uiFit = FITBOX;
     // assume we can fit into a box after we're done
     // but check to make sure
     if(uiFitWidth == NIL || uiFitHeight == NIL)
      {
        uiFit       = NIL;
        uiScale     = 1;
        uiFitWidth  = uiDestWidth;
        uiFitHeight = uiDestHeight;     // load normally if an error
     }
 
     uiDestWidth  = uiFitWidth;
     uiDestHeight = uiFitHeight;

   }
   else
   {
      // reduce by even factors if any value other than 1
      if(uiScale!=FULL)
      {
        if((uiDestWidth/uiScale)>NIL && (uiDestHeight/uiScale)>NIL)
        {
            uiDestWidth/=uiScale;
            uiDestHeight/=uiScale;
        }
      }

   }

   if(uiXoffset==CENTRE)cx+=((uiXwidth-uiDestWidth)/2);
   else cx+=uiXoffset;

   if(uiYoffset==CENTRE)cy+=((uiYheight-uiDestHeight)/2);
   else cy+=uiYoffset;

   // set the custom palette... save the windows palette
   if(hpalCustom!=NULL)
   {
   hpalOriginal=SelectPalette (hdcMain, hpalCustom, NIL);
   RealizePalette(hdcMain);
   }

   // lock the memory
   if((hpBmpBuffer = GlobalLock(hBmpHandle))==NULL)
   {
            if(hpalCustom!=NULL)
            {
              SelectPalette (hdcMain, hpalOriginal, NIL);
            }
            hpBmpBuffer  =NULL;
            return FALSE;
   }


   // draw the bitmap on the screen
   // scaled or unscaled
   if(uiScale==FULL)
   {
   SetDIBitsToDevice(hdcMain,cx,cy,
                             uiBmpWidth,uiBmpHeight,
                             NIL, NIL, NIL, uiBmpHeight,
                             (LPSTR)&hpBmpBuffer[_FIRST_],
                             (PBITMAPINFO)&bmp.bmiHeader.biSize,
                             DIB_RGB_COLORS);
   }
   else
   {
   StretchDIBits(hdcMain,cx,cy,
                              uiDestWidth,uiDestHeight,
                              0,0,
                              uiBmpWidth,uiBmpHeight,
                              (LPSTR)&hpBmpBuffer[_FIRST_],
                              (PBITMAPINFO)&bmp.bmiHeader.biSize,
                               DIB_RGB_COLORS,
                                  SRCCOPY);
   }
   // unlock the memory
   GlobalUnlock(hBmpHandle);

   // restore the windows palette
   if(hpalCustom!=NULL)
   {
   SelectPalette (hdcMain, hpalOriginal, NIL);
   }

   hpBmpBuffer=NULL;

return TRUE;
}

// ------------------------------------------------------------------------
// FUNCTION ClearScreen
//
// clear the client area of the window to the clearcolor.
// used in the main window.
// ------------------------------------------------------------------------
SINT ClearScreen(HWND hWnd,HDC hdcTemp)
{
    HBRUSH hPrevBrush, hClearBrush;// brush to clear client area
    RECT rcClientArea;   // data structure of xy coordinates for rectangle
    HPEN hPrevPen, hClearPen;      // pen to clear area


    bFirstPaint=FALSE;                         // reset image status
                                               // disable the paint method
    if(hdcTemp==NULL)return FAILURE;
    hClearBrush = CreateSolidBrush((COLORREF)dwClearColor);     // create brush
    hClearPen   = CreatePen(PS_SOLID,1,(COLORREF)dwClearColor); // create pen
    hPrevBrush = SelectObject(hdcTemp,hClearBrush); // select brush
    hPrevPen   = SelectObject(hdcTemp,hClearPen);   // select pen
    GetClientRect(hWnd,&rcClientArea);         // fetch bounding coordinates
    FillRect(hdcTemp,&rcClientArea,hClearBrush);    // filled rectangle
    SelectObject(hdcTemp,hPrevPen);                 // deselect pen
    SelectObject(hdcTemp,hPrevBrush);               // deselect brush
    DeleteObject(hClearBrush);                      // delete brush
    DeleteObject(hClearPen);                        // delete pen

return SUCCESS;                                // return to caller
}

// ------------------------------------------------------------------------
// FUNCTION FileExists - mcishow version
//
// returns TRUE if the pathname is valid
// returns FALSE if not
// ------------------------------------------------------------------------
BOOL FAR PASCAL FileExists(LPSTR lpszFileName)
{
   struct find_t wild_card;

   if(lpszFileName[_FIRST_]==ASCIIZ)return FALSE;
   if(_dos_findfirst(lpszFileName,_A_NORMAL,&wild_card)!=NIL)return FALSE;

return TRUE;
}

// ------------------------------------------------------------------------
// FUNCTION InitOFNDialog - mcishow version
//
// set-up function for the OpenFileName dialog
// fills in variant and non-variant fields of OPENFILENAME struct
// this function allows us to set the file mask for the type of
// file browses that we need throughout the program.
// ------------------------------------------------------------------------
SINT InitOFNDialog(HWND hWnd)
{
    ofn.lStructSize       = sizeof(OPENFILENAME);
    ofn.hwndOwner         = hWnd;
    ofn.lpstrFilter       = lpFilterSpec;
    ofn.lpstrCustomFilter = szCustFilterSpec;
    ofn.nMaxCustFilter    = MAXCUSTFILTER;
    ofn.nFilterIndex      = 1;
    ofn.lpstrFile         = szFileName;
    ofn.nMaxFile          = MIDMAX;
    ofn.lpstrInitialDir   = szPathName;
    ofn.Flags             = OFN_HIDEREADONLY|OFN_FILEMUSTEXIST;
    ofn.lpfnHook          = NULL;
    ofn.lpstrTitle        = lpOFNTitle;

return SUCCESS;
}
