unit Formenh;

{
  A component enhancing form's properties.
  Author: Nikolai Botev Botev
  For details see FORMENH.TXT

  DO NOT distribute this component without FORMENH.TXT. If you
  make any changes add a description of the work you've done in
  this comments section.

Version 1.0 - 6 June 1997
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Forms, ShellAPI;

type
  TSize = class(TPersistent)
  protected
    FX, FY: Integer;
  published
    property Width: Integer read FX write FX;
    property Height: Integer read FY write FY;
  end;

  EFormEnhanceError = class(Exception);
  TDropFilesEvent = procedure(Files: TStrings; const Pos: TPoint) of object;
  TFormEnhance = class(TComponent)
  private
    { Private declarations }
  protected
    { Protected declarations }
    OldWndProc: TFarProc;
    NewWndProc: Pointer;
    FAcceptFiles: Boolean;
    FBringToFront: Boolean;
    FMaxSize: TSize;
    FMinSize: TSize;
    FSavePlacement: Boolean;
    FSaveName: string;
    FSaveKey: string;
    FOnDropFiles: TDropFilesEvent;
    procedure HookWndProc(var Msg: TMessage);
    procedure SetAcceptFiles(Value: Boolean);
    procedure DefineProperties(Filer: TFiler); override;
    procedure WriteData(Writer: TWriter);
    procedure ReadData(Reader: TReader);
    procedure LoadPlacement;
    procedure DoSavePlacement;
    procedure Loaded; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure BringFormToFront;
  published
    { Published declarations }
    property AcceptFiles: Boolean read FAcceptFiles write SetAcceptFiles
      default True;
    property BringToFront: Boolean read FBringToFront write FBringToFront
      default True;
    property MaxSize: TSize read FMaxSize;
    property MinSize: TSize read FMinSize;
    property SavePlacement: Boolean read FSavePlacement write FSavePlacement
      default True;
    property SaveName: string read FSaveName write FSaveName;
    property SaveKey: string read FSaveKey write FSaveKey;
    property OnDropFiles: TDropFilesEvent read FOnDropFiles write FOnDropFiles;
  end;

procedure Register;

implementation

uses {$IFDEF WIN32} Registry {$ELSE} INIFiles {$ENDIF};

{$IFNDEF WIN32}
type
  TRegIniFile = class(TINIFile);
{$ENDIF}

procedure Register;
begin
  RegisterComponents('Samples', [TFormEnhance]);
end;

constructor TFormEnhance.Create(AOwner:TComponent);
var
  I, Instances: Integer;
begin
  inherited Create(AOwner);
  if not (Owner is TForm) then
    raise EFormEnhanceError.Create('Control parent must be a form!');
  Instances := 0;
  for I := 0 to Owner.ComponentCount - 1 do
    if (Owner.Components[I] is TFormEnhance) then
      Inc(Instances);
  if (Instances > 1) then
    raise EFormEnhanceError.Create('The form already contains a TFormEnhance component!');

  FAcceptFiles := True;
  FBringToFront := True;
  FMaxSize := TSize.Create;
  FMinSize := TSize.Create;
  FSavePlacement := True;
  {$IFDEF WIN32}
  FSaveName := '\Software\My Application';
  {$ELSE}
  FSaveName := 'MyINI.INI';
  {$ENDIF}
  FSaveKey := Owner.Name + ' Placement';
  { Hook parent }
  OldWndProc := TFarProc(GetWindowLong((Owner as TForm).Handle, GWL_WNDPROC));
  NewWndProc := MakeObjectInstance(HookWndProc);
  SetWindowLong((Owner as TForm).Handle, GWL_WNDPROC, LongInt(NewWndProc));
end;  { TFormEnhance.Create }

procedure TFormEnhance.Loaded;
begin
  DragAcceptFiles((Owner as TForm).Handle, FAcceptFiles);
  if not (csDesigning in ComponentState) and FSavePlacement then
    LoadPlacement;
end;  { TFormEnhance.Loaded }

destructor TFormEnhance.Destroy;
begin
  { Unhook parent }
  if (Owner <> nil) and Assigned(OldWndProc) then
    SetWindowLong((Owner as TForm).Handle, GWL_WNDPROC, LongInt(OldWndProc));
  if Assigned(NewWndProc) then
    FreeObjectInstance(NewWndProc);
  { Clean up }
  FMaxSize.Free;
  FMinSize.Free;
  inherited Destroy;
end;  { TFormEnhance.Destroy }

procedure TFormEnhance.LoadPlacement;
var
  Rect: TRect;
  Maximize: Boolean;
  Settings: TRegIniFile;
begin
  if (FSaveName = '') or (FSaveKey = '') then Exit;
  Settings := TRegIniFile.Create(FSaveName);
  try
    Rect := (Owner as TForm).BoundsRect;
    with Settings, Rect do begin
      Left := ReadInteger(FSaveKey, 'Left', Left);
      Top := ReadInteger(FSaveKey, 'Top', Top);
      Right := ReadInteger(FSaveKey, 'Right', Right);
      Bottom := ReadInteger(FSaveKey, 'Bottom', Bottom);
      Maximize := ReadBool(FSaveKey, 'Maximized',
        (Owner as TForm).WindowState = wsMaximized);
      { Make sure the window is entirely visible on the screen }
      if (Right > Screen.Width) then begin
        Dec(Left, (Right - Screen.Width));
        Right := Screen.Width;
      end;
      if (Bottom > Screen.Height) then begin
        Dec(Top, (Bottom - Screen.Height));
        Bottom := Screen.Height;
      end;
    end;
    (Owner as TForm).BoundsRect := Rect;
    if Maximize then
      (Owner as TForm).WindowState := wsMaximized;
  finally
    Settings.Free;
  end;
end;  { TFormEnhance.LoadPlacement }

procedure TFormEnhance.DoSavePlacement;
var
  Placement: TWindowPlacement;
  Settings: TRegIniFile;
begin
  if (FSaveName = '') or (FSaveKey = '') then Exit;
  Settings := TRegIniFile.Create(FSaveName);
  try
    Placement.length := SizeOf(Placement);
    GetWindowPlacement((Owner as TForm).Handle, @Placement);
    with Settings, Placement, rcNormalPosition do begin
      WriteInteger(FSaveKey, 'Left', Left);
      WriteInteger(FSaveKey, 'Top', Top);
      WriteInteger(FSaveKey, 'Right', Right);
      WriteInteger(FSaveKey, 'Bottom', Bottom);
      WriteBool(FSaveKey, 'Maximized', showCmd = SW_SHOWMAXIMIZED);
    end;
  finally
    Settings.Free;
  end;
end;  { TFormEnhance.DoSavePlacement }

procedure TFormEnhance.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('MinMaxSize', ReadData, WriteData, true);
end;

procedure TFormEnhance.WriteData(Writer: TWriter);
begin
  with Writer do begin
    WriteInteger(FMaxSize.Width);
    WriteInteger(FMaxSize.Height);
    WriteInteger(FMinSize.Width);
    WriteInteger(FMinSize.Height);
  end;
end;

procedure TFormEnhance.ReadData(Reader: TReader);
begin
  with Reader do begin
    FMaxSize.Width := ReadInteger;
    FMaxSize.Height := ReadInteger;
    FMinSize.Width := ReadInteger;
    FMinSize.Height := ReadInteger;
  end;
end;

procedure TFormEnhance.HookWndProc(var Msg: TMessage);
{$IFNDEF WIN32}
type
  UINT = Word;
{$ENDIF}
var
  I, Len, FileCount: Integer;
  FileName: {$IFDEF WIN32} ShortString {$ELSE} string {$ENDIF};
  Files: TStringList;
  Pos: TPoint;
begin
  with Msg do begin
    Result := CallWindowProc(OldWndProc, (Owner as TForm).Handle, Msg,
      wParam, lParam);
    case Msg of
    WM_DROPFILES: if not (csDesigning in ComponentState) then begin
      if FBringToFront then
        BringFormToFront;
      DragQueryPoint(wParam, Pos);
      Files := TStringList.Create;
      try
        FileCount := DragQueryFile(wParam, UINT(-1), nil, 0);
        for I := 0 to (FileCount - 1) do begin
          Len := DragQueryFile(wParam, I, @FileName[1], 255);
          FileName[0] := Char(Len);
          Files.Add(FileName);
        end;
        if (FileCount > 0) and Assigned(FOnDropFiles) then
          FOnDropFiles(Files, Pos);
      finally
        Files.Free;
      end;
    end;
    WM_GETMINMAXINFO: with PMinMaxInfo(lParam)^ do begin
      if (FMaxSize.Width <> 0) then ptMaxTrackSize.X := FMaxSize.Width;
      if (FMaxSize.Height <> 0) then ptMaxTrackSize.Y := FMaxSize.Height;
      if (FMinSize.Width <> 0) then ptMinTrackSize.X := FMinSize.Width;
      if (FMinSize.Height <> 0) then ptMinTrackSize.Y := FMinSize.Height;
    end;
    WM_DESTROY: if not (csDesigning in ComponentState) and FSavePlacement then
      DoSavePlacement;
    end;
  end;
end;  { TFormEnhance.HookWndProc }

procedure TFormEnhance.SetAcceptFiles(Value: Boolean);
begin
  if (Value <> FAcceptFiles) then begin
    FAcceptFiles := Value;
    if not (csDesigning in ComponentState) then
      DragAcceptFiles((Owner as TForm).Handle, FAcceptFiles)
  end;
end;  { TFormEnhance.SetAcceptFiles }

procedure TFormEnhance.BringFormToFront;
begin
{$IFDEF WIN32}
  SetForegroundWindow((Owner as TForm).Handle);
{$ELSE}
  { This is a very nice method to activate the window as it works under
    both Win3.1 and '95 unlike the SetWindowPos and ShowWindow API functions
    which actually do not help us under Win95 }
  SendMessage((Owner as TForm).Handle, WM_SYSCOMMAND, SC_HOTKEY,
    (Owner as TForm).Handle);
{$ENDIF}
end;  { TFormEnhance.BringFormToFront }

end.
