{***********************************************************************}
{ 	        HSoftware Components Collection                         }
{                                                                       }
{     	        Copyright (C) 1996 by Artem A. Berman                   }
{               E-mail: art@aber.kherson.ua                             }
{***********************************************************************}
{ These components can be freely used and distributed in commercial and }
{ private environments, provied this notice is not modified in any way. }
{ ----------------------------------------------------------------------}
{ Version 1.02. Date last modified:  06/20/96                           }
{ ----------------------------------------------------------------------}

unit ExtDlg;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Gauges, Forms, Dialogs, InpForm, ProgForm, ChdrForm;

type
  TIconInBox = (icNone, icError, icQuestion, icWarning, icInfo, icDefault, icCustom);

  ERangeError = class(Exception);

  TInputBox = class(TCommonDialog)
  private
    FCaption: string;
    FMessage: string;
    FPwd: Char;
    FIcon: TIcon;
    FIconType: TIconInBox;
  public
    Text: string;
    CustomIcon: TIcon;
    function Execute: Boolean; virtual;
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Caption: string read FCaption write FCaption;
    property BoxLabel: string read FMessage write FMessage;
    property PwdChar: Char read FPwd write FPwd;
    property IconType: TIconInBox read FIconType write FIconType;
  end;

  TProgressBox = class(TCommonDialog)
  private
    FLabel1,
    FLabel2,
    FCaption: string;
    FMaxValue,
    FMinValue,
    FProgress: integer;
    FCanCancelled: Boolean;
    FCancelAction: TNotifyEvent;
    FFont: TFont;
    FColor: TColor;
    FGaugeKind: TGaugeKind;
    procedure SetMinValue(Value: integer);
    procedure SetMaxValue(Value: integer);
    procedure SetProgress(Value: integer);
    procedure SetCaption(Value: string);
    procedure SetLabel1(Value: string);
    procedure SetLabel2(Value: string);
    procedure SetCanCancelled(Value: Boolean);
    procedure SetFont(Value: TFont);
    procedure SetColor(Value: TColor);
    procedure SetGaugeKind(Value: TGaugeKind);
    procedure CancelAction(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Show; virtual;
    procedure Close; virtual;
  published
    property FirstLabel: string read  FLabel1 write SetLabel1;
    property SecondLabel: string read  FLabel2 write SetLabel2;
    property Caption: string read FCaption write SetCaption;
    property MaxValue: integer read FMaxValue write SetMaxValue;
    property MinValue: integer read FMinValue write SetMinValue;
    property Progress: integer read FProgress write SetProgress;
    property CanCancelled: Boolean read FCanCancelled write SetCanCancelled;
    property Font: TFont read FFont write SetFont;
    property GaugeColor: TColor read FColor write SetColor;
    property GaugeKind: TGaugeKind read FGaugeKind write SetGaugeKind;  
    property OnCancelAction: TNotifyEvent read FCancelAction write FCancelAction;
  end;

  TChDirDialog = class(TCommonDialog)
  private
    FCaption,
    FDirectory: string;
  public
    constructor Create(aOwner: TComponent); override;
    function Execute: Boolean; virtual;
  published
    property Caption: string read FCaption write FCaption;
    property Directory: string read FDirectory write FDirectory;
  end;

procedure Register;

implementation


{TInputBox}

constructor TInputBox.Create(aOwner: TComponent);
begin
  FIcon := TIcon.Create;
  inherited Create(aOwner);
end;

destructor TInputBox.Destroy;
begin
  FIcon.Free;
  inherited Destroy;
end;

function TInputBox.Execute: Boolean;
begin
  InputForm := TInputForm.Create(Application);
  InputForm.Ctl3d := Ctl3d;
  with InputForm do
  begin
    try
      case FIconType of
        icCustom  : Image1.Picture.Icon := CustomIcon;
        icQuestion: FIcon.Handle := LoadIcon(0, IDI_QUESTION);
        icError   : FIcon.Handle := LoadIcon(0, IDI_HAND);
        icWarning : FIcon.Handle := LoadIcon(0, IDI_EXCLAMATION);
        icInfo    : FIcon.Handle := LoadIcon(0, IDI_ASTERISK);
        icDefault : FIcon.Handle := LoadIcon(0, IDI_APPLICATION);
        icNone    : begin
                      Edit1.Left := 25;
                      Label1.Left := 24;
                      Edit1.Width := 265;
                      FIcon.Handle := 0;
                    end;
      end;
      if FIconType <> icCustom then Image1.Picture.Icon := FIcon;
      Caption := FCaption;
      Label1.Caption := FMessage;
      Edit1.PasswordChar := FPwd;
      Edit1.Text := Text;
      Result := (ShowModal = IDOK);
    finally
      Text := Edit1.Text;
      Free;
    end;
  end;
end;

{TProgressBox}

constructor TProgressBox.Create(AOwner: TComponent);
begin
  inherited Create(aOwner);
  ProgressDlg := TProgressDlg.Create(Application);
  ProgressDlg.Button2.OnClick := CancelAction;
  FFont := TFont.Create;
  FFont.Name := 'MS Sans Serif';
  FFont.Size := 8;
  GaugeColor := clActiveCaption;
  GaugeKind := gkHorizontalBar;
  CanCancelled := True;
end;

destructor TProgressBox.Destroy;
begin
  FFont.Free;
  inherited Destroy;
end;

procedure TProgressBox.CancelAction(Sender: TObject);
begin
  if Assigned(FCancelAction) then FCancelAction(Self);
end;

procedure TProgressBox.SetMinValue(Value: integer);
begin
  if Value > FMaxValue then
  begin
    raise ERangeError.Create('Value of MinValue more that Value of MaxValue');
    exit;
  end;
  if Value <> FMinValue then FMinValue := Value;
  ProgressDlg.Gauge1.MinValue := FMinValue;
end;

procedure TProgressBox.SetMaxValue(Value: integer);
begin
  if Value < FMinValue then
  begin
    raise ERangeError.Create('Value of MinValue more that Value of MaxValue');
    exit;
  end;
  if Value <> FMaxValue then FMaxValue := Value;
  ProgressDlg.Gauge1.MaxValue := FMaxValue;
end;

procedure TProgressBox.SetProgress(Value: integer);
begin
  if Value > FMaxValue then
  begin
    raise ERangeError.Create('Value of Progress more that Value of MaxValue');
    exit;
  end;
  if Value <> FProgress then FProgress := Value;
  ProgressDlg.Gauge1.Progress := FProgress;
end;

procedure TProgressBox.SetCaption(Value: string);
begin
  if Value <> FCaption then FCaption := Value;
  ProgressDlg.Caption := FCaption;
end;

procedure TProgressBox.SetLabel1(Value: string);
begin
  if Value <> FLabel1 then FLabel1 := Value;
  ProgressDlg.Label1.Caption := FLabel1;
end;

procedure TProgressBox.SetLabel2(Value: string);
begin
  if Value <> FLabel2 then FLabel2 := Value;
  ProgressDlg.Label2.Caption := FLabel2;
end;

procedure TProgressBox.SetCanCancelled(Value: Boolean);
begin
  FCanCancelled := Value;
  if FCanCancelled then
  begin
    ProgressDlg.Height := 134;
    ProgressDlg.Button2.Visible := True;
  end else
  begin
    ProgressDlg.Height := 110;
    ProgressDlg.Button2.Visible := False;
  end;
end;

procedure TProgressBox.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
  ProgressDlg.Font := FFont;
end;

procedure TProgressBox.SetColor(Value: TColor);
begin
  FColor := Value;
  ProgressDlg.Gauge1.ForeColor := FColor;
end;

procedure TProgressBox.SetGaugeKind(Value: TGaugeKind);
begin
  if Value <> FGaugeKind then FGaugeKind := Value;
  ProgressDlg.Gauge1.Kind := FGaugeKind;
end;

procedure TProgressBox.Show;
begin
  ProgressDlg.Ctl3d := Ctl3d;
  ProgressDlg.Show;
end;

procedure TProgressBox.Close;
begin
  ProgressDlg.Close;
end;

{TChDirDialog}

constructor TChDirDialog.Create(aOwner: TComponent);
begin
  FDirectory := ExtractFilePath(ParamStr(0));
  FCaption := 'Browse Directory';
  inherited Create(aOwner);
end;

function TChDirDialog.Execute: Boolean;
begin
  ChDirForm := TChDirForm.Create(Application);
  try
    ChDirForm.DirectoryOutline1.Directory := FDirectory;
    ChDirForm.Caption := FCaption;
    ChDirForm.Ctl3d := Ctl3d;
    Result := (ChDirForm.ShowModal = IDOK);
  finally
    Directory := ChDirForm.DirectoryOutline1.Directory;
    ChDirForm.Free;
  end;
end;

procedure Register;
begin
  RegisterComponents('Dialogs', [TInputBox]);
  RegisterComponents('Dialogs', [TProgressBox]);
  RegisterComponents('Dialogs', [TChDirDialog]);
end;

end.