/*
 * The gutty-works of the calculator
 */

#include <owl\dialog.h>
#include <owl\gdiobject.h>
#include <owl\control.h>
#include <iostream.h>
#include <iomanip.h>
#include <strstrea.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <float.h>
#include "button.h"
#include "wrpn.h"
#include "function.h"

/*
 * Perform the function for the specified calculator key
 */

void
wrpn::calc_key(int key)
{
	int i, j, normal, store_it, over_flow;
	unsigned long int_part, mask, sign_bit_mask;
	char c, *p, buf[SCREEN_SIZE +1];
	double op2;

	static double current = 0.0;
	static char screen_buf[SCREEN_SIZE +1];
	static int last_operator = 0;
	static int position = 0;
	static int got_decimal = 0;
	static int got_exponent = 0;
	static TWO_PART two_part = NONE;
	static int compound = 0;

					/* is this a compound mouse event? */
	if (compound) {
		key += compound;
		compound = 0;
	}

	c = key & 0xff;
	normal = ((key > 0) && (key < 256)) ? 1 : 0;

					/* is this a two-part key? */
	if (two_part != NONE) {
		if (c == '\b') {
			two_part = NONE;
			return;
		}
		switch (two_part) {
			case STORE:
				if (!normal || !strchr("0123456789", c)) {
					cerr << '\a';
					return;
				}
				sto[c - '0'] = current;
				two_part = NONE;
				return;
			case RECALL:
				if (!normal || !strchr("0123456789", c)) {
					cerr << '\a';
					return;
				}
				current = sto[c - '0'];
				push(current);
				display(current, mode);
				two_part = NONE;
				return;
			case FLOAT_DIGITS:
				if (!normal || !strchr("0123456789", c)) {
					cerr << '\a';
					return;
				}
				decimal_places = c - '0';
				two_part = NONE;
				cm_view_float();
				return;
		}
	}

	store_it = 0;
	over_flow = 0;

					/* get incoming digits */
	switch (mode) {
		case FLOAT_MODE:
			if (key == CHS || key == EEX)
				store_it++;
			if (normal && strchr("0123456789.", c))
				store_it++;
			switch (key) {
				case '.':
					if (got_decimal || got_exponent) {
						cerr << '\a';
						return;
					}
					got_decimal++;
					break;
				case EEX:	/* exponent */
					if (!position || got_exponent) {
						cerr << '\a';
						return;
					}
					got_exponent++;
					c = 'e';
					break;
				case CHS:	/* change sign */
					// no pending input, so its an operator
					if (!position) {
						store_it = 0;
						break;
					}
					// contains an 'e', so toggle exponent
					if ((p = strchr(screen_buf, 'e')) != NULL) {
						p++;
						strcpy(buf, screen_buf);
						i = (int) (p - screen_buf);
						if (*p == '-') {
							strcpy(&buf[i], &screen_buf[i+1]);
							position--;
						}
						else {
							strcpy(&buf[i+1], &screen_buf[i]);
							buf[i] = '-';
							position++;
						}
						strcpy(screen_buf, buf);
						message(screen_buf, mode);
						return;
					}
					// just toggle the sign
					if (screen_buf[0] == '-') {
						strcpy(screen_buf, &screen_buf[1]);
						position--;
					}
					else {
						strcpy(&buf[1], screen_buf);
						buf[0] = '-';
						strcpy(screen_buf, buf);
						position++;
					}
					message(screen_buf, mode);
					return;
			}
			if (position >= MAX_FLOAT_DIGITS)
				over_flow++;
			break;
		case HEX_MODE:
			if (normal && strchr("0123456789abcdef", c))
				store_it++;
			if (position >= MAX_HEX_DIGITS)
				over_flow++;
			break;
		case OCT_MODE:
			if (normal && strchr("01234567", c))
				store_it++;
			if (position >= MAX_OCTAL_DIGITS)
				over_flow++;
			break;
		case DEC_MODE:
			if (normal && strchr("0123456789", c))
				store_it++;
			if (position >= MAX_DECIMAL_DIGITS)
				over_flow++;
			break;
		case BIN_MODE:
			if (normal && strchr("01", c))
				store_it++;
			if (position >= MAX_BINARY_DIGITS)
				over_flow++;
			break;
	}
					/* store digits */
	if (store_it) {
		if (over_flow) {
			cerr << '\a';
			return;
		}
		screen_buf[position++] = c;
		screen_buf[position] = '\0';
		message(screen_buf, mode);
		return;
	}
					/* process pseudo operators */
	switch (key) {
		case PB_ON:		/* time to quit? */
			CloseWindow(0);
			return;
		case PB_BSP:	/* do a backspace */
			if (position) {
				screen_buf[--position] = '\0';
				message(screen_buf, mode);
			}
			else
				display(current, mode);
			return;
		case PB_CTRL: 	/* the CTRL button */
			compound = CTRL;
			return;
		case PB_ALT:  	/* the ALT button */
			compound = ALT;
			return;
		default:
			break;
	}
					/* if you've got input, convert it */
	if (position) {
		current = convert(screen_buf, mode);

		/*
		 * This is a little weird... if the last operator was
		 * the <enter>, then replace (rather than push) the
		 * value on stack.
		 */
		if (last_operator == '\r')
			stack[0] = current;
		else
			push(current);

		position = 0;
		screen_buf[0] = '\0';
		got_decimal = 0;
		got_exponent = 0;
	}
					/* prepare the masks */
	mask = (1ul << word_size) -1;
	sign_bit_mask = (1ul << (word_size -1));

	/*
	 * OK, all pending input has been terminated, the
	 * input is on the stack, and we're ready to perform
	 * one of the operators
	 */
	switch (key) {
		case 0:			/* just update the display */
			break;
		case GSB:		/* goto subroutine*/
			message("GSB not implemented", FLOAT_MODE);
			return;
		case RUN_STOP:	/* run stop */
			message("R/S not implemented", FLOAT_MODE);
			return;
		case GTO:		/* goto */
			message("GTO not implemented", FLOAT_MODE);
			return;
		case SST:		/* single step */
			message("SST not implemented", FLOAT_MODE);
			return;
		case HEX:		/* hex */
			cm_view_hex();
			break;
		case ROL:		/* roll the stack */
			op2 = stack[0];
			for (i=0; i<STACK_SIZE -1; i++)
				stack[i] = stack[i +1];
			stack[STACK_SIZE -1] = op2;
			current = stack[0];
			break;
		case DEC:		/* decimal */
			cm_view_decimal();
			break;
		case X_Y:		/* exchange x y */
			op2 = pop();
			current = pop();
			push(op2);
			push(current);
			break;
		case STO:		/* store */
			two_part = STORE;
			break;
		case OCT:		/* octal */
			cm_view_octal();
			break;
		case RCL:		/* recall */
			two_part = RECALL;
			break;
		case BIN:		/* binary */
			cm_view_binary();
			break;
		case RETURN: 	/* return */
			push(current);
			break;
		case CHS:		/* change sign */
			current = pop() * -1.0;
			push(current);
			break;
		case DIVIDE: 	/* divide */
			op2 = pop();
			current = pop() / op2;
			push(current);
			if (mode != FLOAT)
				carry_bit = modf(current, &op2) ? 1 : 0;
			break;
		case MULTIPLY:	/* multiply */
			current = pop() * pop();
			push(current);
			break;
		case SUBTRACT:	/* subtract */
			if (mode != FLOAT) {
				int_part = dtoul(stack[0]) - dtoul(stack[1]);
				carry_bit = (int_part > mask) ? 1 : 0;
			}
			op2 = pop();
			current = pop() - op2;
			push(current);
			break;
		case ADD:		/* add */
			if (mode != FLOAT) {
				int_part = dtoul(stack[0]) + dtoul(stack[1]);
				carry_bit = (int_part > mask) ? 1 : 0;
			}
			current = pop() + pop();
			push(current);
			break;
		/*
		 * now for the CTRL "yellow" keys
		 */
		case SL:			/* shift left */
			int_part = dtoul(pop());
			carry_bit = (int_part & sign_bit_mask) ? 1 : 0;
			current = int_part << 1;
			push(current);
			break;
		case X_IND:		/* exchange x and (i) */
			message("x:(i) not implemented", FLOAT_MODE);
			return;
		case IND:		/* indirect register */
			message("(i) not implemented", FLOAT_MODE);
			return;
		case SR:			/* shift right */
			int_part = dtoul(pop());
			carry_bit = (int_part & 1) ? 1 : 0;
			current = int_part >> 1;
			push(current);
			break;
		case X_I:		/* exchange X and I */
			message("x:i not implemented", FLOAT_MODE);
			return;
		case I:			/* i register */
			message("i not implemented", FLOAT_MODE);
			return;
		case RL:			/* rotate left */
			int_part = dtoul(pop());
			carry_bit = (int_part & sign_bit_mask) ? 1 : 0;
			current = (int_part << 1) | carry_bit;
			push(current);
			break;
		case SHOW_HEX:	/* show hex */
			display(current, HEX_MODE);
			return;
		case CLR_PGRM:	/* clear program */
			message("clear PGRM not implemented", FLOAT_MODE);
			return;
		case RR:			/* rotate right */
			int_part = dtoul(pop());
			carry_bit = (int_part & 1) ? 1 : 0;
			int_part = (int_part >> 1);
			if (carry_bit)
				int_part |= sign_bit_mask;
			current = int_part;
			push(current);
			break;
		case SHOW_DEC:	/* show dec */
			display(current, DEC_MODE);
			return;
		case CLR_REG:	/* clear register */
			for (i=0; i<REGISTER_SIZE; i++)
				sto[i] = 0.0;
			break;
		case WSIZE:		/* word size */
			i = (int) pop();
			if (i < 1 || i > 32) {
				message("WSIZE out of range", FLOAT_MODE);
				return;
			}
			word_size = i;
							/* check or uncheck the menu */
			switch (word_size) {
				case 8:
					cm_options_8bit();
					break;
				case 16:
					cm_options_16bit();
					break;
				case 32:
					cm_options_32bit();
					break;
				default:	/* not one of the defaults */
					HMENU hwnd = Parent->GetMenu();
					CheckMenuItem(hwnd, CM_OPTIONS_8BIT, MF_UNCHECKED);
					CheckMenuItem(hwnd, CM_OPTIONS_16BIT, MF_UNCHECKED);
					CheckMenuItem(hwnd, CM_OPTIONS_32BIT, MF_UNCHECKED);
			}
			break;
		case RL_N:		/* rotate left "n" times */
			i = (int) pop();
			if (i < 1 || i > word_size ) {
				message("RLn out of range", FLOAT_MODE);
				return;
			}
			int_part = dtoul(pop());
			for (j=0; j<i; j++) {
				carry_bit = (int_part & sign_bit_mask) ? 1 : 0;
				int_part = (int_part << 1) | carry_bit;
			}
			current = int_part;
			push(current);
			break;
		case SHOW_OCT:	/* show oct */
			display(current, OCT_MODE);
			return;
		case CLR_PREFIX:/* clear prefix */
			two_part = NONE;
			return;
		case FLOAT:		/* float precision */
			two_part = FLOAT_DIGITS;
			return;
		case RR_N: 		/* rotate right "n" times */
			i = (int) pop();
			if (i < 1 || i > word_size ) {
				message("RRn out of range", FLOAT_MODE);
				return;
			}
			int_part = dtoul(pop());
			for (j=0; j<i; j++) {
				carry_bit = (int_part & 1) ? 1 : 0;
				int_part = (int_part >> 1);
				if (carry_bit)
					int_part |= sign_bit_mask;
			}
			current = int_part;
			push(current);
			break;
		case SHOW_BIN:	/* show bin */
			display(current, BIN_MODE);
			return;
		case WINDOW:	/* show window */
			// just silently ignore
			return;
		case MASKL:		/* mask left */
			i = (int) pop();
			if (i < 1 || i > word_size) {
				message("MASKL out of range", FLOAT_MODE);
				return;
			}
			int_part = (1ul << i) -1;
			int_part = (int_part << (word_size - i));
			current = int_part;
			push(current);
			break;
		case SB:			/* set bit */
			i = (int) pop();
			if (i < 1 || i > word_size) {
				message("SB out of range", FLOAT_MODE);
				return;
			}
			int_part = dtoul(pop());
			int_part |= (1ul << (i-1));
			current = int_part;
			push(current);
			break;
		case COMP_1S:	/* set 1's complement */
			cm_options_1s();
			break;
		case MEM:		/* display memory */
			message("MEM not implemented", FLOAT_MODE);
			return;
		case MASKR:		/* mask right */
			i = (int) pop();
			if (i < 1 || i > word_size) {
				message("MASKR out of range", FLOAT_MODE);
				return;
			}
			int_part = (1ul << i) -1;
			current = int_part;
			push(current);
			break;
		case CB:			/* clear bit */
			i = (int) pop();
			if (i < 1 || i > word_size) {
				message("CB out of range", FLOAT_MODE);
				return;
			}
			int_part = dtoul(pop());
			int_part &= ~(1ul << (i-1));
			current = int_part;
			push(current);
			break;
		case COMP_2S:	/* set 2's complement */
			cm_options_2s();
			break;
		case STATUS:{	/* display status */
			char buf[SCREEN_SIZE+1];
			ostrstream sb(buf, sizeof buf);

			sb << "word=" << word_size << ", ";
			sb << "comp=";
			switch (complement) {
				case ONES:
					sb << "1's, ";
					break;
				case TWOS:
					sb << "2's, ";
					break;
				case UNSGN:
					sb << "unsgn, ";
					break;
			}
			sb << "carry=" << (carry_bit ? "1" : "0") << ends;
			message(buf, FLOAT_MODE);
			return; }
		case RMD:		/* remainder (modulus) */
			int_part = dtoul(pop());
			int_part = dtoul(pop()) % int_part;
			carry_bit = (int_part) ? 1 : 0;
			current = int_part;
			push(current);
			break;
		case B_TEST:	/* test bit */
			message("B? not implemented", FLOAT_MODE);
			return;
		case COMP_UNSGN:/* set unsigned */
			cm_options_unsgn();
			break;
		case EEX:		/* exponent */
			// handled above as numeric input
			break;
		case XOR:		/* exclusive or */
			int_part = dtoul(pop()) ^ dtoul(pop());
			current = int_part;
			push(current);
			break;
		case AND:		/* and */
			int_part = dtoul(pop()) & dtoul(pop());
			current = int_part;
			push(current);
			break;
		case NOT:		/* not */
			int_part = (~dtoul(pop()) & mask);
			current = int_part;
			push(current);
			break;
		case OR:			/* or */
			int_part = dtoul(pop()) | dtoul(pop());
			current = int_part;
			push(current);
			break;
		/*
		 * And now for the ALT "blue" keys
		 */
		case LJ:			/* left justify */
			int_part = dtoul(pop());
			// find number of bits to shift
			for (i=0; i<word_size; i++) {
				if (int_part & (1ul << (word_size -i -1)))
					break;
			}
			int_part = int_part << i;
			push(int_part);
			current = i;
			push(current);
			break;
		case RTN:		/* return */
			message("RTN not implemented", FLOAT_MODE);
			return;
		case PGRM_RUN:	/* program/run  mode */
			message("P/R not implemented", FLOAT_MODE);
			return;
		case ASR:		/* arithmetic shift right */
			int_part = dtoul(pop());
			carry_bit = (int_part & 1) ? 1 : 0;
			if (int_part & sign_bit_mask) {
				int_part = (int_part >> 1);
				int_part |= sign_bit_mask;
			}
			else
				int_part = (int_part >> 1);
			current = int_part;
			push(current);
			break;
		case LBL:		/* label */
			message("LBL not implemented", FLOAT_MODE);
			return;
		case BST:		/* back step */
			message("BST not implemented", FLOAT_MODE);
			return;
		case RLC:		/* rotate left with carry */
			int_part = dtoul(pop());
			carry_bit = (int_part & sign_bit_mask) ? 1 : 0;
			int_part = (int_part << 1) | carry_bit;
			current = int_part;
			push(current);
			break;
		case DSZ:		/* decrement, skip on zero */
			message("DSZ not implemented", FLOAT_MODE);
			return;
		case ROL_UP:	/* roll stack up */
			op2 = stack[STACK_SIZE	-1];
			for (i=STACK_SIZE-1; i>0; i--)
				stack[i] = stack[i -1];
			stack[0] = op2;
			current = stack[0];
			break;
		case RRC:		/* rotate right with carry */
			int_part = dtoul(pop());
			carry_bit = (int_part & 1) ? 1 : 0;
			int_part = (int_part >> 1);
			if (carry_bit)
				int_part |= sign_bit_mask;
			current = int_part;
			push(current);
			break;
		case ISZ:		/* increment, skip on zero */
			message("ISZ not implemented", FLOAT_MODE);
			return;
		case PSE:		/* pause */
			message("PSE not implemented", FLOAT_MODE);
			return;
		case WINDOW_LEFT:/* shift window left */
			// just silently ignore
			return;
		case RLC_N:	 	/* rotate left with carry "n" times */
			i = (int) pop();
			if (i < 1 || i > word_size) {
				message("RLCn out of range", FLOAT_MODE);
				return;
			}
			int_part = dtoul(pop());
			for (j=0; j<i; j++) {
				carry_bit = (int_part & sign_bit_mask) ? 1 : 0;
				int_part = (int_part << 1) | carry_bit;
			}
			current = int_part;
			push(current);
			break;
		case SQRT:		/* square root */
			op2 = pop();
			// kinda odd that isn't handled as a FPE
			if (op2 < 0.0) {
				message("domain error", FLOAT_MODE);
				return;
			}
			current = sqrt(op2);
			push(current);
			if (mode != FLOAT)
				carry_bit = modf(current, &op2) ? 1 : 0;
			break;
		case CL_X:		/* clear x register */
			stack[0] = 0.0;
			current = 0.0;
			break;
		case WINDOW_RIGHT:/* shift window right */
			// just silently ignore
			return;
		case RRC_N:		/* rotate right with carry "n" times */
			i = (int) pop();
			if (i < 1 || i > word_size) {
				message("RRCn out of range", FLOAT_MODE);
				return;
			}
			int_part = dtoul(pop());
			for (j=0; j<i; j++) {
				carry_bit = (int_part & 1) ? 1 : 0;
				int_part = (int_part >> 1);
				if (carry_bit)
					int_part |= sign_bit_mask;
			}
			current = int_part;
			push(current);
			break;
		case INV:		/* inverse */
			current = 1.0 / pop();
			push(current);
			break;
		case LST_X:		/* last x register */
			current = last_x;
			push(current);
			break;
		case SUM_BITS:	/* sum bits */
			j = 0;
			int_part = dtoul(pop());
			for (i=0; i<word_size; i++)
				j += ((int_part & (1ul << i)) ? 1 : 0);
			current = j;
			push(current);
			break;
		case SF:			/* set flag */
			carry_bit = 1;
			break;
		case X_GE_Y:	/* x <= y */
			message("x<=y not implemented", FLOAT_MODE);
			return;
		case X_NE_Y:	/* x != y */
			message("x!=y not implemented", FLOAT_MODE);
			return;
		case ABS:		/* absolute value */
			current = fabs(pop());
			push(current);
			break;
		case CF:			/* clear flag */
			carry_bit = 0;
			break;
		case X_LT_ZERO:/* x < 0 */
			message("x<0 not implemented", FLOAT_MODE);
			return;
		case X_NE_ZERO:/* x != 0 */
			message("x!=0 not implemented", FLOAT_MODE);
			return;
		case DBLR:{		/* double remainder */
			unsigned long x, y, z;
			x = dtoul(pop());
			y = dtoul(pop());
			z = dtoul(pop());
			int_part = (y << word_size) + z;
			int_part = int_part % x;
			current = int_part;
			push(current);
			break;}
		case F_TEST:	/* test flag */
			message("F? not implemented", FLOAT_MODE);
			return;
		case X_GT_Y:	/* x > y */
			message("x>y not implemented", FLOAT_MODE);
			return;
		case X_EQ_Y:	/* x = y */
			message("x=y not implemented", FLOAT_MODE);
			return;
		case DBL_DIVIDE:{/* double divide */
			unsigned long x, y, z;
			x = dtoul(pop());
			y = dtoul(pop());
			z = dtoul(pop());
			int_part = ((y << word_size) + z) & mask;
			carry_bit = (int_part % x) ? 1 : 0;
			int_part = int_part / x;
			current = int_part;
			push(current);
			break;}
		case DBL_MULTIPLY:/* double multiply */
			int_part = dtoul(pop()) * dtoul(pop());
			push(int_part & mask);
			int_part = (int_part >> word_size);
			current = int_part;
			push(current);
			break;
		case X_GT_ZERO:/* x > 0 */
			message("x>0 not implemented", FLOAT_MODE);
			return;
		case X_EQ_ZERO:/* x = 0 */
			message("x=0 not implemented", FLOAT_MODE);
			return;
		default:
			cerr << '\a';
			return;
	}
	last_operator = key;

	/*
	 * Detect overflows in float mode only.  It's OK to overflow
	 * in the hex/dec/oct/bin modes.
	 */
	if (mode == FLOAT_MODE && fabs(current) >= DBL_MAX) {
		message("overflow", FLOAT_MODE);
		return;
	}
	if (mode == FLOAT_MODE && current != 0.0 && fabs(current) <= DBL_MIN) {
		message("underflow", FLOAT_MODE);
		return;
	}
					/* print the answer */
	display(current, mode);
	return;
}

/*
 * Push a number on the stack.  If the stack is full, the "top" value
 * is lost.
 */

void
wrpn::push(double f)
{
	for (int i=STACK_SIZE-1; i>0; i--)
		stack[i] = stack[i -1];
	stack[0] = f;
	return;
}

/*
 * Pop a value off the stack.  The "top" value is always preserved.
 */

double
wrpn::pop(void)
{
	double f;

	f = stack[0];
	last_x = f;
	for (int i=0; i<STACK_SIZE-1; i++)
		stack[i] = stack[i +1];
	return (f);
}

/*
 * Convert a double to an unsigned long, masking word size as appropriate
 */

unsigned long
wrpn::dtoul(double d)
{
	unsigned long ans, mask;

	mask = (1ul << word_size) -1;
	ans = (unsigned long) d;

	return (ans & mask);
}

/*
 * Convert an ascii "number" into a double.  Handles hex/oct/bin digits.
 */

double
wrpn::convert(char *buf, int mode)
{
	double f;

	switch (mode) {
		case DEC_MODE:
		case FLOAT_MODE:
			f = atof(buf);
			break;
		case HEX_MODE:
			f = (double) strtoul(buf, (char **) NULL, 16);
			break;
		case OCT_MODE:
			f = (double) strtoul(buf, (char **) NULL, 8);
			break;
		case BIN_MODE:
			f = (double) strtoul(buf, (char **) NULL, 2);
			break;
	}
	return (f);
}

/*
 * Convert a number into its binary bit pattern.  Returns a pointer to
 * a static buffer.
 */

char *
wrpn::bitpattern(unsigned long val)
{
	unsigned char *p;
	int i, j, bp, start;
	static char bitbuf[SCREEN_SIZE +1];

	p = (unsigned char *) &val;
	bp = 0;

	for (i = 3; i >= 0; i--) {
		for (j = 7; j >= 0; j--)
			bitbuf[bp++] = (p[i] & (1 << j)) ? '1' : '0';
		bitbuf[bp++] = ' ';
	}
	bitbuf[bp -1] = '\0';
	/*
	 * mask the output by returning just the part of the
	 * buffer that contains the number of bit (plus spaces)
	 */
	start = SCREEN_SIZE - word_size - ((word_size -1)/8);
	return(&bitbuf[start]);
}

/*
 * Format the answer for the "screen"
 */

void
wrpn::display(double f, int mode)
{
	unsigned long int_part;
	char buf[SCREEN_SIZE +1];

					/* convert double to integer */
	if (mode != FLOAT_MODE)
		int_part = dtoul(f);

	ostrstream screen(buf, sizeof buf);

	switch (mode) {
		case FLOAT_MODE:
			screen.precision(decimal_places);
			/*
			 * I want scientific notation only when "fixed" would
			 * run out of precision
			 */
			if (f == 0.0 || (fabs(f) < pow10(DBL_DIG) && fabs(f) > pow10(-decimal_places)))
				screen.setf(ios::fixed);
			else
				screen.setf(ios::scientific);
			screen << f << ends;
			break;
		case HEX_MODE:
			screen.setf(ios::hex);
			screen << int_part << ends;
			break;
		case OCT_MODE:
			screen.setf(ios::oct);
			screen << int_part << ends;
			break;
		case DEC_MODE: {
			long ans;
			unsigned long sign_bit_mask = 1ul << (word_size -1);
			unsigned long mask = (1ul << word_size) -1;
			/*
			 * The decimal mode is the only one which deals
			 * with the "complement" option
			 */
			ans = (long) int_part;
			switch (complement) {
				case ONES:
					if (int_part & sign_bit_mask)
						ans -= (long) mask;
					screen << ans << ends;
					break;
				case TWOS:
					if (int_part & sign_bit_mask)
						ans -= (long) (mask +1);
					screen << ans << ends;
					break;
				case UNSGN:
					screen << int_part << ends;
					break;
			}
			break; }
		case BIN_MODE:
			strcpy(buf, bitpattern(dtoul(f)));
			break;
	}
	message(buf, mode);
	return;
}
