;=======================================================================
; MODULE NAME:  StrFmt.ASM
; DEPENDENCIES: (None)
; LAST MOD ON:  9005.14
; PROGRAMMER:   Naoto Kimura
;
;     This is the assembly code for the routines in the StrUtil unit.
; Many of the routines were rewritten in the hopes that it will not
; only reduce the memory requirement, but also reduce the execution
; time.
;
;     This file should be assembled with Turbo Assembler.  If you need
; to use another assembler, then you should keep some things in mind.
;
; The TPASCAL memory model sets up automatically:
; * the standard Turbo Pascal entry code
;      push bp
;      mov bp,sp
; * the standard Turbo Pascal exit code
;      pop bp
;      ret n
; * the order of the arguments don't need to be reversed in the
;   assembly code.
;-----------------------------------------------------------------------
; 9001.20     Naoto Kimura
;             * Initial version created
;
; Modification history:
;
; 9005.05     Naoto Kimura
;             * Broke up assembler modules into separate files to try
;               to decrease overhead (although the overhead is pretty
;               small, it's always nice to make available whatever
;               memory the user can use).
; 9005.14     Naoto Kimura
;             * Added Strip to functions rewritten in assembler
;=======================================================================
.MODEL   TPASCAL
LOCALS

.CODE

;-----------------------------------------------------------------------
;FUNCTION Copies ( Original:String; Num:Byte ) : String
;
;     This function returns as many copies of a string concatenated
; together as requested.
;
; REGISTER USAGE:  AX,BX,CX,DX,DI,SI,ES	-- Destroyed
;-----------------------------------------------------------------------
Copies		PROC FAR Original:DWORD,Num:BYTE RETURNS Result:DWORD
		PUBLIC	Copies
		USES	ds
		xor	bx,bx		; init result string length
		mov	dl,[Num]	; find out number of times to
		or	dl,dl		; copy
		jz	@@Done
		cld			; forward string op
		xor	ax,ax
		lds	si,[Original]	; get addr of function parameter
		lodsb			; check length
		or	ax,ax
		jz	@@Done
		les	di,[Result]	; get addr of function result
		inc	di		; get past string length part
		mov	cx,ax
		mov	dh,cl
@@Froot:	rep movsb		; copy string
		add	bl,dh		; update length
		dec	dl		; decrement count
		or	dl,dl
		jz	@@Done
		lds	si,[Original]
		inc	si		; skip past length byte
		mov	cl,dh
		mov	ax,cx
		add	ax,bx
		cmp	ax,00FFH
		jl	@@Froot		; copy if still less than
		sub	ax,00FFH	; find out how many not to copy
		sub	cx,ax		; adjust count
		mov	al,cl
		rep movsb		; and copy
		add	bl,al
@@Done:		les	di,[Result]	; set result string length
		mov	[es:di],bl
		ret
Copies		ENDP

;-----------------------------------------------------------------------
;FUNCTION RightJustify ( S:string; Siz:byte; Pad:char ) : string
;
;     This function returns a string that has the string "S" right
; justified in a field of length "Siz" of the character "Pad".  If the
; string is longer than the field, the string will be truncated at the
; field width.
;
; REGISTER USAGE:  AX,CX,DI,SI,ES	-- Destroyed
;-----------------------------------------------------------------------
RightJustify	PROC FAR S:DWORD,Siz:BYTE,Pad:BYTE RETURNS Result:DWORD
		PUBLIC	RightJustify
		USES	DS
		les	di,[Result]	; get addr of function result
		cld			; forward string op
		xor	ax,ax
		mov	al,[Siz]	; get field size
		stosb			; store it
		mov	cx,ax
		mov	al,[Pad]	; get pad character
		rep stosb		; fill result string with pad char
		les	di,[Result]	; get addr of function result
		inc	di		; adjust to access 1st char
		lds	si,[S]		; get addr of function parameter
		lodsb			; get original string length
		mov	cl,[Siz]	; get field size
		cmp	ax,cx
		jge	@@Ok		; Ok  since (length(Orig) > Siz)
					; ax<cx  (length(orig) < Siz)
		xchg	cx,ax		; set count to be the lesser
		sub	ax,cx		; find adjustment
		add	di,ax		; and adjust destination
@@Ok:		rep movsb
		ret
RightJustify	ENDP

;-----------------------------------------------------------------------
;FUNCTION LeftJustify ( S:string; Siz:byte; Pad:char ) : String
;
;     This function returns a string that has the string "S"
; left justified in a field of length "Siz" of the character
; "Pad".  If the string is longer than the field, the string will
; be truncated at the field width.
;
; REGISTER USAGE:  AX,CX,DI,SI,ES	-- Destroyed
;-----------------------------------------------------------------------
LeftJustify	PROC FAR S:DWORD,Siz:BYTE,Pad:BYTE RETURNS Result:DWORD
		PUBLIC	LeftJustify
		USES	DS
		les	di,[Result]	; get addr of function result
		cld			; forward string op
		xor	ax,ax
		mov	al,[Siz]	; get field size
		stosb			; store it
		mov	cx,ax
		mov	al,[Pad]	; get pad character
		rep stosb		; fill result string with pad char
		lds	si,[S]		; get addr of function parameter
		lodsb			; get original string length
		les	di,[Result]	; get addr of function result
		inc	di		; adjust to access 1st char
		mov	cl,[Siz]	; get field size
		cmp	ax,cx
		jge	@@Ok		; Ok  since (length(Orig) > Siz)
		; ax<cx  (length(orig) < Siz)
		xchg	cx,ax		; set count to be the lesser
@@Ok:		rep movsb
		ret
LeftJustify	ENDP

;-----------------------------------------------------------------------
;FUNCTION Center ( S:string; Siz:byte; Pad:char ) : String
;
;     This function returns a string that has the string "S"
; centered in a field of length "Siz" of the character "Pad".  If
; the string is longer than the field, the string will be truncated at
; the field width.
;
; REGISTER USAGE:  AX,CX,DI,SI,ES	-- Destroyed
;-----------------------------------------------------------------------
Center		PROC FAR S:DWORD,Siz:BYTE,Pad:BYTE RETURNS Result:DWORD
		PUBLIC	Center
		USES	DS
		les	di,[Result]	; get addr of function result
		cld			; forward string op
		xor	ax,ax
		mov	al,[Siz]	; get field size
		stosb			; store it
		mov	cx,ax
		mov	al,[Pad]	; get pad character
		rep stosb		; fill result with pad char
		lds	si,[S]		; get addr of function parameter
		lodsb			; get original string length
		les	di,[Result]	; get addr of function result
		inc	di		; adjust to access 1st char
		mov	cl,[Siz]	; get field size
		cmp	ax,cx
		jge	@@Ok		; Ok  since (length(Orig) > Siz)
					; ax<cx  (length(orig) < Siz)
		xchg	cx,ax		; set count to be the lesser
		sub	ax,cx		; find adjustment
		shr	ax,1		; 1/2 the adjustment
		add	di,ax		; add adjustment to destination
@@Ok:		rep movsb
		ret
Center		ENDP

;-----------------------------------------------------------------------
;FUNCTION Reverse ( S:String ) : String
;
;     This function returns a copy of a string that is reversed.
;
; REGISTER USAGE:  AX,CX,DI,SI,ES	-- Destroyed
;-----------------------------------------------------------------------
Reverse		PROC FAR S:DWORD RETURNS Result:DWORD
		PUBLIC	Reverse
		USES	ds
		les	di,[Result]	; get addr of function result
		lds	si,[S]		; get addr of function parameter
		xor	ax,ax
		cld			; forward string op
		lodsb			; get length
		mov	[es:di],al	; copy length
		or	ax,ax		; null string ?
		jz	@@done
		mov	cx,ax
		add	di,cx
@@copy:		lodsb			; grab character forward
		mov	[es:di],al	; store character
		dec	di		; backward for destination
		loop	@@copy
@@done:		ret
Reverse		ENDP

;-----------------------------------------------------------------------
;FUNCTION Strip ( S:String;  Unwanted:String;  Location:Char ) : String
;
;     This function strips off unwanted characters from either the
; left, right or both ends of a string.
;
; REGISTER USAGE:  AX,CX,DX,DI,SI,ES	-- Destroyed
;-----------------------------------------------------------------------
Strip		PROC FAR S:DWORD,U:DWORD,L:BYTE RETURNS Result:DWORD
		PUBLIC	Strip
		USES	ds
		LOCAL	Skip:BYTE,EndSkip:BYTE
		mov	[Skip],0
		mov	[EndSkip],0
		mov	al,[L]
		cmp	al,'r'
		je	@@IsArr
		cmp	al,'R'
		je	@@IsArr
		cmp	al,'b'
		je	@@IsArr
		cmp	al,'B'
		jne	@@TestEll
@@IsArr:	sub	ax,ax
		mov	cx,ax
		lds	si,[S]
		mov	al,[si]		; get length of original
		mov	bx,ax
		add	si,bx		; move to end of original string
		les	di,[U]
		mov	cl,[es:di]	; get length of unwanted
		or	ax,cx
		jz	@@DoCopy	; if either are zero, then done
		mov	dx,di
@@RightStrip:	inc	di		; LOOP ; move past length byte
		std			;   backward string op
		lodsb			;   check char
		cld			;   forward string op
		repne scasb		;   scan
		jne	@@TestEll	; EXIT IF NOT Found
		inc	[EndSkip]	;   increment skip count
		dec	bx		;   decrease copy length
		or	bx,bx		;
		jz	@@DoCopy	; EXIT IF NullString
		mov	di,dx		;   reset search string
		mov	cl,[es:di]	;   get length again
		jmp	@@RightStrip	; ENDLOOP
@@TestEll:	mov	al,[L]
		cmp	al,'l'
		je	@@IsEll
		cmp	al,'L'
		je	@@IsEll
		cmp	al,'b'
		je	@@IsEll
		cmp	al,'B'
		jne	@@DoCopy
@@IsEll:	sub	cx,cx
		mov	ax,cx
		les	di,[U]
		mov	dx,di
		mov	cl,[es:di]	; get length of unwanted string
		lds	si,[S]
		lodsb			; get length of original string
		mov	bx,ax
		sub	bl,[Skip]	; adjust for skipped chars
		or	ax,cx
		jz	@@DoCopy	; if either is zero, then done
		cld			; forward string op
@@LeftStrip:	inc	di		; LOOP ;move past length byte
		lodsb			;   get char to check
		repne scasb		;   scan
		jne	@@DoCopy	; EXIT IF NOT Found
		inc	[Skip]		;   increment skip count
		dec	bx		;   decrease copy length
		or	bx,bx		;
		jz	@@DoCopy	; EXIT IF NullString
		mov	di,dx		;   reset search string
		mov	cl,[es:di]	;   get length again
		jmp	@@LeftStrip	; ENDLOOP
@@DoCopy:	lds	si,[S]
		les	di,[Result]
		sub	ax,ax
		sub	bx,bx
		lodsb
		mov	bl,[Skip]
		add	si,bx
		sub	al,bl
		sub	al,[EndSkip]
		stosb
		mov	cx,ax
		jcxz	@@Done
		cld			; forward string op
		rep movsb
@@Done:		ret
Strip		ENDP

END
