{ HLPTX.PAS : Create MAKEHELP input file from library (header) source

  title   : HLPTX
  version : 1.3
  date    : Sep 21,1994
  author  : J R Ferguson
  language: Borland Pascal v7.0 with Objects
  usage   : refer procedure Help
  remarks : Output to be processed with MAKEHELP for use with TpHelp unit
}

{$B-} { Short-circuit Boolean expression evaluation }
{$V-} { Relaxed var-string checking }
{$X+} { Extended syntax }

{$UNDEF OUTBUFHEAP}   { UNDEF to work around a BP 7.0 bug resulting in
                        erroneous file output }

program HLPTX;


uses DefLib, ArgLib, ChrLib, CvtLib, StpLib, StfLib, Dos, Objects;


const
  C_ProgIdn     = 'HLPTX';
  C_ProgVer     = '1.3';
  C_MsgFsp      = 'CON';
  C_OutExt      = 'HTX';
  C_InpExtI86   = 'I   ASM ';
  C_InpExtC     = 'H   C   CPP ';
  C_InpExtPas   = 'PAS INC ';
  C_InpExtTxt   = 'TXT DAT';

  C_IOBufSiz    = 4096;

  C_RefLParen   = '<';  { input reference left parenthesis }
  C_RefRParen   = '>';  { input reference right parenthesis }
  C_Ref1        = #004; { output reference topic marker }
  C_Ref2        = #005; { output reference name delimiter }

  { Error codes and messages: }
  C_ErrOK       = 0;
  C_ErrArg      = 1;
  C_ErrInp      = 2;
  C_ErrOut      = 3;

  C_ErrMsg      : array[C_ErrOK..C_ErrOut] of StpTyp =
 ('',
  '',
  'File not found : ',
  'Can''t open output : '
 );

type
  P_IOBuffer    = ^T_IOBuffer;
  P_TopicEntry  = ^T_TopicEntry;
  P_TopicList   = ^T_TopicList;
  P_MaskList    = ^T_MaskList;

  T_CharSet     = set of char;
  T_InpType     = (Inp_Ext, Inp_I86, Inp_C, Inp_Pas, Inp_Txt);

  T_IOBuffer    = array[1..C_IOBufSiz] of char;

  T_TopicEntry  = Object(TObject)
    Number      : integer;
    Ident       : StpPtr;
    Constructor Init(V_Number: integer; V_Ident: StpTyp);
    Destructor  Done; virtual;
  end;

  T_TopicList   = Object(TSortedCollection) { of P_TopicEntry }
    function    Compare(V_Key1, V_Key2: Pointer): integer; virtual;
    function    KeyOf(Item: Pointer): Pointer; virtual;
    function    SearchIdent(V_Ident: StpPtr): P_TopicEntry;
  end;

  T_MaskList    = Object(TCollection) { of input mask string pointers }
    procedure   FreeItem(Item: Pointer); virtual;
  end;

var
  ErrCod    : integer;
  OptType   : T_InpType;
  InpType   : T_InpType;
  DirInfo   : SearchRec;
  HasFile   : boolean;
  InpPath,
  InpFnm,
  OutFnm    : StpTyp;
  Inp,
  Out,
  Msg       : Text;
  InpBuf    : P_IOBuffer;
{$IFDEF OUTBUFHEAP}
  OutBuf    : P_IOBuffer;
{$ELSE}
  OutBuf    : T_IOBuffer;
{$ENDIF}
  InpOpen,
  OutOpen   : boolean;
  MaskList  : P_MaskList;
  TopicList : P_TopicList;
  TopicNbr  : integer;
  TopicIdn  : StpTyp;
  Pass      : integer;
  InEntry   : boolean;
  SkipLines : integer;

const
  IdnChr0I86 : T_CharSet = ['A'..'Z','a'..'z','.','_','@','$','?'];
  IdnChr1I86 : T_CharSet = ['A'..'Z','a'..'z','0'..'9','_','@','$','?'];
  IdnChrC    : T_CharSet = ['A'..'Z','a'..'z','0'..'9','_'];
  IdnChr0Pas : T_CharSet = ['A'..'Z','a'..'z'];
  IdnChr1Pas : T_CharSet = ['A'..'Z','a'..'z','0'..'9','_'];
  IdnChr0Txt : T_CharSet = ['A'..'Z','a'..'z','#','%'];
  IdnChr1Txt : T_CharSet = ['A'..'Z','a'..'z','0'..'9','_'];


{
--- General routines ---
}

procedure WrMsg(s: StpTyp); begin write  (Msg,s); end;
procedure WlMsg(s: StpTyp); begin writeln(Msg,s); end;

procedure Help;
  procedure wr(s: StpTyp); begin write  (s); end;
  procedure wl(s: StpTyp); begin writeln(s); end;
begin
wl('Create MAKEHELP input file from library (header) source');
wl('usage  : '+C_ProgIdn+' inpmask [...] [option [...]]');
wl('where  : inpmask = [path]filename.ext');
wl('                   wildcards are allowed in filename and ext');
wl('options: /P  input is Turbo Pascal unit source');
wl('         /C  input is C or C++ source or header file');
wl('         /A  input is Intel 80x86 declaration header file');
wl('         /T  input is text file (1-word topics starting in column 1)');
wl('         /H  send this help text to (redirected) output');
wl('If no /P, /C, /A or /T option is provided, the input extension decides');
wl('the file type as follows:');
wl('  Assembler  '+C_InpExtI86);
wl('  C or C++   '+C_InpExtC  );
wl('  Pascal     '+C_InpExtPas);
wl('  Text       '+C_InpExtTxt);
wl('The information of each source file is converted to a separate output');
wl('file in the current directory, having the same filename as the input');
wl('file and the extension "'+C_OutExt+'".');
wl('For text files, references are resolved automatically. For other files,');
wl('use "<" and ">" around references to topics within the same source.');
wl('To exclude lines, use ".hlptx skip n" immediately after a comment mark');
wl('starting in column 1. That line and the next n lines will be skipped.');
end;


{ --- T_TopicEntry methods --- }

Constructor T_TopicEntry.Init(V_Number: integer; V_Ident: StpTyp);
begin
  Inherited Init;
  Number:= V_Number;
  Ident := StpAlloc(V_Ident);
end;

Destructor  T_TopicEntry.Done;
begin
  StpFree(Ident);
  Inherited Done;
end;


{ --- T_TopicList methods --- }

function  T_TopicList.Compare(V_Key1, V_Key2: Pointer): integer;
begin Compare:= StpCmp(StpPtr(V_Key1)^,StpPtr(V_Key2)^); end;

function  T_TopicList.KeyOf(Item: Pointer): Pointer;
begin KeyOf:= P_TopicEntry(Item)^.ident; end;

function  T_TopicList.SearchIdent(V_Ident: StpPtr): P_TopicEntry;
var i: integer;
begin
  if Search(V_Ident,i) then SearchIdent:= At(i)
  else SearchIdent:= nil;
end;


{ --- T_MaskList --- methods --- }

procedure T_MaskList.FreeItem(Item: Pointer);
begin StpFree(StpPtr(Item)); end;


{
--- Command Line parsing routines ---
}


procedure ReadOpt(arg: StpTyp);
var NextOpt: boolean;
begin
  StpDel(arg,1,1);
  repeat
    if StpEmpty(arg) or (StpcRet(arg,1) = '/') then ErrCod:= C_ErrArg
    else begin
      NextOpt:= false;
      while (ErrCod = C_ErrOK) and not NextOpt and not StpEmpty(arg) do
      case StpcGet(arg) of
        'A': OptType:= Inp_I86;
        'C': OptType:= Inp_C;
        'P': OptType:= Inp_Pas;
        'T': OptType:= Inp_Txt;
        'H': ErrCod := C_ErrArg;
        '/': NextOpt:= true;
        else ErrCod := C_ErrArg;
      end;
    end;
  until (ErrCod <> C_ErrOK) or not NextOpt;
end;

procedure AddInpMask(mask: StpTyp);
begin MaskList^.Insert(StpAlloc(mask)); end;

procedure ReadArgs;
var i,n : ArgInd;
    arg : StpTyp;
begin
  GetArgs;
  i:= 0; n:= 0;
  while (ErrCod = C_ErrOK) and (i < ArgC) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    case StpcRet(arg,1) of
      '/' : ReadOpt(arg);
      else  begin AddInpMask(arg); Inc(n); end;
    end;
  end;
  if (ErrCod = C_ErrOK) and (n = 0) then ErrCod:= C_ErrArg;
end;


{
--- Low level I/O routines ---
}

procedure OpenInp;
begin
  Assign(Inp,InpPath + InpFnm);
  new(InpBuf); SetTextBuf(Inp,InpBuf^);
  {$I-} reset(Inp); {$I+}
  if IOresult <> 0 then ErrCod:= C_ErrInp else InpOpen:= true;
end;

procedure CloseInp;
begin if InpOpen then begin
  Close(Inp); dispose(InpBuf); InpOpen:= false;
end end;

procedure OpenOut;
begin
  Assign(Out,OutFnm);
{$IFDEF OUTBUFHEAP}
  new(OutBuf); SetTextBuf(Out,OutBuf^);
{$ELSE}
  SetTextBuf(Out,OutBuf);
{$ENDIF}
  {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then ErrCod:= C_ErrOut else OutOpen:= true;
end;

procedure CloseOut;
begin if OutOpen then begin
  Close(Out);
{$IFDEF OUTBUFHEAP}
  dispose(OutBuf);
{$ENDIF}
  OutOpen:= false;
end end;


{ --- High level I/O routines --- }


function GetReference(var Topic: P_TopicEntry; var Line: StpTyp): boolean;
var i: StpInd; c: char; Ident: StpTyp;
begin
  i:= 1; Topic:= nil;
  c:= StpcRet(Line,i);
  while (c<>#0) and (c<>C_RefRParen) and (not IsSpace(c)) do begin
    Inc(i);
    c:= StpcRet(Line,i);
  end;
  if c<>C_RefRParen then GetReference:= false
  else begin
    StpSub(Ident,Line,1,i-1);
    Topic:= TopicList^.SearchIdent(@Ident);
    if Topic = nil then GetReference:= false
    else begin StpDel(Line,1,i); GetReference:= true; end;
  end;
end;

procedure GetIdn(var Idn: StpTyp; Line: StpTyp);
var c: char; i: StpInd; ok: boolean;
begin case InpType of
  Inp_I86 : begin
              StpGtw(Idn,Line); StpRLS(Line); {skip 'public','global'}
              StpCreate(Idn); c:= StpcGet(Line);
              ok:= c in IdnChr0I86;
              while ok do begin
                StpcCat(Idn,c); c:= StpcGet(Line);
                ok:= c in IdnChr1I86;
              end;
            end;
  Inp_C   : begin
              StpBefore(Idn,Line,'('); StpRTS(Idn);
              i:= StpLen(Idn); c:= StpcRet(Idn,i);
              while (i>0) and (c in IdnChrC) do begin
                Dec(i); c:= StpcRet(Idn,i);
              end;
              StpDel(Idn,1,i);
              if (StpCmp(Idn,'_CLASSDEF') = 0) then
                Idn:= StfBefore(StfAfter(Line,'('),')');
            end;
  Inp_Pas : begin
              if StpUppPos(Line,'= OBJECT(') = 0 then
                StpGtw(Idn,Line); {skip 'procedure','function'}
              StpRLS(Line); StpCreate(Idn); c:= StpcGet(Line);
              ok:= c in IdnChr0Pas;
              while ok do begin
                StpcCat(Idn,c); c:= StpcGet(Line);
                ok:= c in IdnChr1Pas;
              end;
            end;
  Inp_Txt : StpGtw(Idn,Line);
end; end;

function EndTrigger(Line: StpTyp): boolean;
begin case InpType of
  Inp_Pas : EndTrigger:= StpUppNCmp(StfRLS(Line),'IMPLEMENTATION',14) = 0;
  else      EndTrigger:= false;
end; end;

function SkipCommand(Line: StpTyp): boolean;
var ok: boolean; tmp: StpTyp;
begin
  case InpType of
    Inp_I86 : ok:= (StpUppNCmp(Line, ';.HLPTX SKIP',12) = 0);
    Inp_C   : ok:= (StpUppNCmp(Line,'/*.HLPTX SKIP',13) = 0) or
                   (StpUppNCmp(Line,'//.HLPTX SKIP',13) = 0);
    Inp_Pas : ok:= (StpUppNCmp(Line, '{.HLPTX SKIP',12) = 0) or
                   (StpUppNCmp(Line,'(*.HLPTX SKIP',13) = 0);
    Inp_Txt : ok:= false;
  end;
  if ok then begin
    StpGtw(tmp,Line); StpGtw(tmp,Line); StpGtw(tmp,Line);
    SkipLines:= AtoIB(tmp,10);
    if SkipLines=0 then SkipLines:= 1;
  end;
  SkipCommand:= ok;
end;

function HasTopic(Line: StpTyp): boolean;
var i: StpInd;
begin case InpType of
  Inp_I86 :
    HasTopic:= ( StpUppNCmp(Line,'PUBLIC ',7) = 0 ) or
               ( StpUppNCmp(Line,'GLOBAL ',7) = 0 );
  Inp_C   :
    if ( StpUppNCmp(Line,'#IF'   ,3) = 0 ) or  { #if, #ifdef, #ifndef }
       ( StpUppNCmp(Line,'#EL'   ,2) = 0 ) or  { #else, #elif }
       ( StpUppNCmp(Line,'#ENDIF',6) = 0 ) or  { #endif }
       ( StpUppNCmp(Line,'/*'    ,2) = 0 ) or  { comment Line }
       ( StpUppNCmp(Line,'//'    ,2) = 0 ) or  { C++ comment Line }
       ( IsSpace(StpcRet(Line,1))        )     { not starting in col 1 }
    then HasTopic:= false
    else begin
      i:= StpPos(Line,'(');                    { look for function (macro) }
      HasTopic:=
       ( i > 0                           ) and { recognized by '(' }
       ( StpcRet(Line,i-1) in IdnChrC    );    { following a valid Idn char }
    end;
  Inp_Pas :
    HasTopic:= ( StpUppNCmp(Line,'PROCEDURE ',10) = 0 ) or
               ( StpUppNCmp(Line,'FUNCTION ' , 9) = 0 ) or
               ( StpUppPos (Line,'= OBJECT('    ) > 0 );
  Inp_Txt :
    HasTopic:= IsGraph(StpcRet(Line,1))
end; end;

procedure TxtLine(Line: StpTyp);
var Tmp, Wrd, NumberStr: StpTyp; c: char; Topic: P_TopicEntry;
  procedure SkipToWord;
  begin
    while not (c in [#0]+IdnChr0Txt) do begin
      StpcCat(Tmp,c); c:= StpcGet(Line);
    end;
  end;
  procedure GetWord;
  begin
    Wrd:= c; c:= StpcGet(Line);
    while c in IdnChr1Txt do begin
      StpcCat(Wrd,c); c:= StpcGet(Line);
    end;
  end;
begin {TxtLine}
  Tmp:= '';
  c:= StpcGet(Line);
  SkipToWord;
  while c <> #0 do begin {scan word}
    GetWord;
    Topic:= nil;
    if Wrd <> TopicIdn then Topic:= TopicList^.SearchIdent(@Wrd);
    if Topic = nil then StpCat(Tmp,Wrd)
    else begin
      ItoA(Topic^.Number,NumberStr);
      Tmp:= Tmp+C_Ref1+NumberStr+C_Ref2+Topic^.Ident^+C_Ref2;
    end;
    SkipToWord;
  end;
  writeln(Out,Tmp);
end;

procedure SrcLine(Line: StpTyp);
var i: StpInd; Tmp: StpTyp; Topic: P_TopicEntry; NumberStr: StpTyp;
begin
  Tmp:= ' ';
  i:= StpcPos(Line,C_RefLParen);
  if not ( (InpType = Inp_C) and (StpUppNCmp(Line,'#INCLUDE ',9) = 0) )
  then begin
    while i > 0 do begin
      StpCat(Tmp,StfSub(Line,1,i-1));
      StpDel(Line,1,i);
      if GetReference(Topic,Line) then begin
        ItoA(Topic^.Number,NumberStr);
        Tmp:= Tmp+C_Ref1+NumberStr+C_Ref2+Topic^.Ident^+C_Ref2;
      end
      else StpcCat(Tmp,C_RefLParen);
      i:= StpcPos(Line,C_RefLParen);
    end;
  end;
  StpCat(Tmp,Line);
  writeln(Out,Tmp);
end;

procedure NxtLine(Line: StpTyp);
begin case InpType of
  Inp_I86: if InEntry then begin
             if (TopicNbr = 1) then begin
               if ( StpUppNCmp(Line,'%NEWPAGE',8) = 0 ) or
                  ( StpUppNCmp(Line,'PAGE'    ,4) = 0 )
               then writeln(Out,'!page')
               else SrcLine(Line);
             end
             else if StpcGet(Line) = ';' then SrcLine(Line)
             else InEntry:= false;
           end;
  Inp_C  : SrcLine(Line);
  Inp_Pas: SrcLine(Line);
  Inp_Txt: if TopicNbr > 0 then TxtLine(Line);
end; end;


{
--- Main Line ---
}

procedure NewTopic;
begin case Pass of
  1: TopicList^.Insert(New(P_TopicEntry,Init(TopicNbr,TopicIdn)));
  2: begin writeln(Out,'!topic ',TopicNbr,' ',TopicIdn); InEntry:= true; end;
end; end;

procedure FileProcess;
var Line,FileIdn,Dummy: StpTyp; EndSource: boolean; p: StpInd;
begin
  StpBefore(FileIdn,InpFnm,'.');
  case Pass of
    1: WrMsg('  '+StfFill(InpFnm,' ',12)+' ');
    2: WlMsg('==> '+OutFnm);
  end;
  EndSource:= eof(Inp);
  if (InpType <> Inp_Txt) and not EndSource then begin
    Inc(TopicNbr); TopicIdn:= 'HEADER'; NewTopic;
    if Pass=2 then writeln(Out,'!index 1');
  end;
  while not EndSource do begin
    ReadLn(Inp,Line);
    if SkipLines>0 then Dec(SkipLines)
    else begin
      StpDetab(Line,Line,8); StpRTS(Line);
      if EndTrigger(Line) then EndSource:= true
      else if not SkipCommand(Line) then begin
        if HasTopic(Line)
        then begin
          Inc(TopicNbr); GetIdn(TopicIdn,Line); NewTopic;
          if Pass=2 then case InpType of
            Inp_Txt: TxtLine(' '+Line);
            else     SrcLine(Line);
          end;
        end
        else if Pass=2 then NxtLine(Line);
        EndSource:= Eof(Inp);
      end;
    end;
  end;
end;

procedure FileInit;
  function HasExtent(ext,mask: StpTyp): boolean;
  var i: StpInd;
  begin
    i:= StpPos(mask,ext);
    HasExtent:= (i > 0) and ((i-1) mod 4 = 0);
  end;
var ext: StpTyp;
begin {FileInit}
  Pass:= 1; SkipLines:= 0; TopicNbr:= 0; New(TopicList,Init(50,50));
  InpFnm:= DirInfo.Name; OpenInp;
  InpType:= OptType;
  StpAfter(ext,InpFnm,'.'); StpTrunc(ext,3); StpFill(ext,' ',3);
  if ext = C_OutExt then
    WlMsg('* '+StfFill(InpFnm,' ',12)+' skipped: already type '+C_OutExt)
  else begin
    if InpType = Inp_Ext then begin
      if      HasExtent(ext,C_InpExtI86) then InpType:= Inp_I86
      else if HasExtent(ext,C_InpExtC  ) then InpType:= Inp_C
      else if HasExtent(ext,C_InpExtPas) then InpType:= Inp_Pas
      else if HasExtent(ext,C_InpExtTxt) then InpType:= Inp_Txt;
    end;
    if InpType = Inp_Ext then
      WlMsg('* '+StfFill(InpFnm,' ',12)+' skipped: unknown type');
  end;
end;

procedure FileReInit;
begin
  Pass:= 2; SkipLines:= 0; TopicNbr:= 0;
  CloseInp; OpenInp;
  if ErrCod = C_ErrOK then begin
    OutFnm:= StfBefore(InpFnm,'.') + '.' + C_OutExt;
    OpenOut; writeln(Out,'!width 80');
  end;
end;

procedure FileTerm;
begin
  Dispose(TopicList,Done); TopicList:= nil;
  CloseInp; CloseOut;
  FindNext(DirInfo);
  HasFile:= DosError = 0
end;

procedure MaskInit;
var rc: integer; FullMask,Name,Ext: StpTyp;
begin
  HasFile:= false;
  FullMask:= FExpand(StpPtr(MaskList^.At(0))^); MaskList^.AtFree(0);
  FSplit(FullMask,InpPath,Name,Ext);
  WlMsg(FullMask);
  FindFirst(FullMask,Archive+ReadOnly,DirInfo);
  rc:= DosError;
  case rc of
    00 : HasFile:= true;
    02 : WlMsg('Directory not found : ' + FullMask);
    18 : WlMsg('No files : ' + FullMask);
    else WlMsg('Unexpexted error on FindFirst : ' + FullMask);
  end;
end;

procedure MaskTerm;
begin {empty} end;

procedure MainInit;
begin
  ErrCod  := C_ErrOK;
  OptType := Inp_Ext;
  InpOpen := false;
  OutOpen := false;
  TopicNbr := 0;
  TopicList:= nil;
  MaskList := new(P_MaskList,Init(10,10));
  Assign(Msg,C_MsgFsp); rewrite(Msg);
  writeln(C_ProgIdn+' v'+C_ProgVer);
  ReadArgs;
end;

procedure MainTerm;
begin
  WrMsg(C_ErrMsg[ErrCod]);
  case ErrCod of
    C_ErrOK  : ;
    C_ErrArg : Help;
    C_ErrInp : WlMsg(InpPath + InpFnm);
    C_ErrOut : WlMsg(OutFnm);
  end;
  Dispose(MaskList,Done);
  if TopicList<>nil then Dispose(TopicList,Done);
  CloseOut;
  close(Msg);
end;

begin { Main program }
  MainInit;
  while (ErrCod = C_ErrOK) and (MaskList^.Count > 0) do begin
    MaskInit;
    while HasFile and (ErrCod = C_ErrOK) do begin
      FileInit;
      if ErrCod = C_ErrOK then begin
        if InpType <> Inp_Ext then begin
          FileProcess; {pass 1: build topiclist}
          FileReInit;
        end;
        if ErrCod = C_ErrOK then begin
          if InpType <> Inp_Ext then FileProcess; {pass 2: write output}
          FileTerm;
        end;
      end;
    end;
    MaskTerm
  end;
  MainTerm;
end.
