
interrupts    segment at 0h                      ;interrupt table segment
              org 9h*4
keyboard_int  dw 2 dup (?)                       ;interrupt 9 vector
interrupts    ends

rom_bios_data segment at 40h                     ;ROM BIOS data area segment
              org 0h
com1base        dw      ?
com2base        dw      ?             ;base addresses for com1 and com2
              org 60h
cursor_mode   dw ?                      ;starting and ending cursor scan lines
rom_bios_data ends

rom           segment at 0F000h                  ;ROM segment
              org 0FFFEh
machine_id    db ?                ;ID byte identifies machine as PCjr or other
rom           ends

code          segment para public 'code'         ;code segment
              assume cs:code
              org 100h
begin:        jmp initialize                     ;goto initialization routine

com1    dw      ?
com2    dw      ?       ;local storage for com1 and com2 base addresses
adrs    dw      ?       ;current port address
byteout db      0       ;byte to be sent when rgt arrow key pressed

column_count  dw ?                               ;width of window in columns
cursor_type   dw ?                               ;cursor scan line definition
cursor_pos    dw ?      ;position of cursor on entry to program
setup_status  db 0              ;indicates if printer window is already active
display_mode  dw ?                               ;current crt display mode
vpage  equ this byte
page_no       dw ?                               ;current displayed page
attribute1    db 4Fh                             ;window attribute bytes
attribute2    db 70h

display_table db 2Dh,29h        ;display re-enable values for modes 2 and 3
video         dw 0B800h,0B900h,0BA00h,0BB00h     ;starting addresses of video
                                                 ;memory for CGA pages 0 - 3

mono_video    dw 0B000h                          ;video segment address for
                                                 ;monochrome adapter

old_kb_int              label dword
old_keyboard_int        dw 2 dup (?)             ;storage for old keyboard
                                                 ;interrupt vector

;----------------------------------------------------------------------------
;Text of the Printer Setup Menu window.
;After initialization, text and attribute bytes are combined and stored
;in the WINDOW_TEXT area, and this area is used to store the contents of
;the screen that underlie the window when the window is called up.
;----------------------------------------------------------------------------
window_buffer           label word
buffer_text             db 201,26 dup (205),187
                        db 186,'    COM1 UART REGISTERS   ',186
                        db 199,26 dup (196),182
                        db 186,' RX/TX reg*  =            ',186
                        db 186,' Intrpt enbl*=            ',186
                        db 186,' Intrpt ID   =            ',186
                        db 186,' Line ctl    =            ',186
                        db 186,' Modem ctl   =            ',186
                        db 186,' Line stat   =            ',186
                        db 186,' Modem stat  =            ',186
                        db 186,' *baud if MSB of LCR set  ',186
                        db 199,26 dup (196),182
                        db 186,'  PGup/dn slct com port   ',186
                        db 186,'  Up/down keys slct reg   ',186
                        db 186,'  Rgt arrow writes reg    ',186
                        db 186,'  Lft arrow reads reg     ',186
                        db 186,'  ESC to exit             ',186
                        db 186,'   BYTE TO SEND: 00 Hex   ',186
                        db 200,26 dup (205),188
                        db 532 dup (?)
;Storage area for the combination of text and attribute bytes that
;form the window image.
window_bytes  label byte
window_text   dw 532 dup (?)

;---------------------------------------------------------------------------
;Execution comes here, to the main body of the program, when an interrupt 9
;is generated from the keyboard.  Registers are saved, then the keypress is
;checked and compared to the key combination that activates the menu window.
;---------------------------------------------------------------------------
main          proc near
              sti                                ;enable software interrupts
              push ax                            ;save all registers
              push bx
              push cx
              push dx
              push si
              push di
              push ds
              push es
              push ax                   ;save ax for call to old routine
              in al,0A0h                         ;re-enable NMI on PCjr
              pop ax                             ;restore ax
              pushf          ;simulate interrupt call to old keyboard routine
              call old_kb_int                    ;call old routine
              mov ah,2                   ;check status of the shift keys
              int 16h
              cmp al,14         ;***** CTRL,ALT,LEFT SHFT = activate pop up
              je do_program                      ;yes, then skip exit routine
;Exit routine is the common point of exit for all routines in the program.
exit:         pop es
              pop ds
              pop di
              pop si
              pop dx
              pop cx
              pop bx
              pop ax
              iret

;Execution comes here when the proper key combination, Ctrl/Alt/LftShft, is
;pressed.  First task is to check whether or not the window is already open.
do_program:   push cs                   ;set es and ds to the code segment
              pop ds
              push cs
              pop es
              cmp setup_status,0                 ;is the window already open?
              jne exit                           ;yes, then ignore request
;----------------------------------------------------------------------------
;Check current video mode.  If it's mode 2, 3, or 7, then set the window
;status flag, store the mode number and page number, save the cursor type,
;and hide the cursor.  If any other display mode is active instead, ignore
;the request and exit.
;----------------------------------------------------------------------------
              mov ah,15                          ;get page and mode numbers
              int 10h                            ;al=mode, bh=page
              cmp al,2                  ;is crt now in an acceptable mode?
              je prog0                           ;yes, then continue
              cmp al,3
              je prog0
              cmp al,7
              je prog0
              jmp exit                           ;no, then ignore request
prog0:        mov setup_status,1                 ;set status flag to indicate
                                                 ;that window is active
              mov ah,0                           ;save mode number
              mov display_mode,ax
              push bx
              mov bl,bh                 ;save page number for color displays
              mov bh,0
              mov page_no,bx
              pop bx
              mov ah,3                           ;get cursor type
              int 10h
              mov cursor_type,cx                 ;save it
              mov cursor_pos,dx         ;save cursor position
              mov ah,1                           ;hide the cursor until later
              mov ch,20h
              int 10h
;Preparatory routines are completed.  Now open the window by first saving the
;contents of video memory beneath the window and then writing the window text
;directly to memory.
              mov bx,page_no             ;use bx as index into video segment
                                         ;address table
              cmp display_mode,7         ;manually adjust index for monochrome
                                         ;adapter
              jne prog1
              mov bx,4
prog1:        shl bx,1                           ;multiply bx by two since
                                                 ;table is made up of words
              mov ax,video[bx]                   ;read segment from table
              mov ds,ax                          ;ds set to video memory
              cmp display_mode,7                 ;skip disable if in mode 7
              je prog2
              call video_disable      ;turn display off for snow-free writing
prog2:        lea di,window_buffer         ;set di to buffer area to save
                                           ;screen contents
              mov ch,28               ;define window dimensions and location
              mov cl,19
              mov dh,2
              mov dl,26
              call video2mem          ;then transfer screen contents to buffer
              push ds                            ;set es to video memory
              pop es
              push cs                            ;reset ds to code segment
              pop ds
              lea si,window_text                 ;point si to window image
prog3:        mov ch,28                          ;define window region
              mov cl,19
              mov dh,2
              mov dl,26
              call mem2video            ;and write the window to the display
              mov byteout,0             ;clear output byte
              mov dx,51bh
              mov bh,vpage
              mov ah,2
              int 10h
              mov ax,093eh
              mov cx,1
              mov bl,attribute2
              int 10h                   ;mark rx/tx reg as current reg
              mov ax,com1
              mov adrs,ax               ;set current port address as com1 base
              or ax,ax
              jne ckdspmode
              mov ax,com2
              mov adrs,ax
ckdspmode:
              cmp display_mode,7                 ;skip enable if in mode 7
              je getkey
              call video_enable                  ;re-enable the video display
getkey:       mov ah,0                           ;get a keypress
              int 16h
              cmp al,0                           ;is it an extended code?
              jne isESC
              jmp extended_code
isESC:
              cmp al,27                          ;is it the ESC key?
              je fixwind                ;exit if it was
              sub al,'0'
              jb getkey1
              cmp al,9
              jbe hexentry
              and al,0dfh
              sub al,7
              jb getkey1                 ;ck for hex digit -- if it is
              cmp al,15                  ;put ASCII code in AH and binary
              jbe hexentry               ;value in AL and jmp to entry routine
getkey1:      call beep
              jmp getkey


hexentry:
              mov ah,byteout
              shl ah,1
              shl ah,1
              shl ah,1
              shl ah,1
              or al,ah
              mov byteout,al    ;update byte out value
              mov dx,132ch
              call dsphex       ;display new value
              jmp getkey

;----------------------------------------------------------------------------
;Execution comes here when the ESC key is pressed.  The window is refilled
;with its original contents, the cursor is restored, and control is handed
;back to the application program.
;----------------------------------------------------------------------------
fixwind:      cmp display_mode,7                 ;skip disable if in mode 7
              je prog4
              call video_disable                 ;turn off the display
prog4:        lea si,window_buffer               ;point si to the buffer area
              mov ch,28                          ;define the window
              mov cl,19
              mov dh,2
              mov dl,26
              call mem2video     ;and write the buffer contents to the display
              cmp display_mode,7                 ;skip enable if in mode 7
              je prog5
              call video_enable                  ;turn display back on
prog5:        mov bh,vpage
              mov dx,cursor_pos
              mov ah,2
              int 10h
              mov ah,1                           ;restore cursor
              mov cx,cursor_type
              int 10h
              mov setup_status,0                 ;reset window status
              jmp exit                           ;and exit
;
;An extended code has been entered...check its validity and goto the
;appropriate routine.
;

getkey2:
        jmp getkey1
extended_code:
        cmp ah,72      ;up arrow?
        jb  getkey2
        jne ispgdn
        dec adrs
        cmp byte ptr adrs,0f7h
        jne adrsok
        mov byte ptr adrs,0feh
        jmp short adrsok
ispgdn:
        cmp ah,81
        ja getkey2
        jne ispgup
        mov ax,com2
        or ax,ax
        je getkey2      ;no action if com2 not installed
        mov adrs,ax
        mov al,'2'
        jmp short port_set
ispgup:
        cmp ah,73
        jne isrgtarw
        mov ax,com1
        or ax,ax
        je getkey2      ;no action if com1 not installed
        mov adrs,ax
        mov al,'1'
port_set:
        push ax
        mov bh,vpage
        mov dx,0322h
        mov ah,2
        int 10h
        pop ax
        mov bl,attribute1
        mov cx,1
        mov ah,9
        int 10h
adrsok:
        mov dx,041bh
        mov cx,8
        push cx
clnup:
        pop cx
        dec cx
        jcxz donecln
        push cx
        inc dh
        mov bh,vpage
        mov ah,2
        int 10h
        mov ax,0920h
        mov bl,attribute2
        mov cx,1
        int 10h
        jmp short clnup
donecln:
        mov dh,byte ptr adrs
        sub dh,0f3h
        mov dl,27
        mov bh,vpage
        mov ah,2
        int 10h
        mov bl,attribute2
        mov ax,093eh
        mov cx,1
        int 10h
        jmp getkey
isrgtarw:
        cmp ah,77
        jne islftarw
        mov dx,adrs
        mov al,byteout
        out dx,al
        jmp getkey
islftarw:
        cmp ah,75
        jne isdnarw
        mov dx,adrs
        in al,dx
        sub dl,0f3h
        mov dh,43
        xchg dh,dl
        call dspbin
        jmp getkey
dodsphex:
        call dsphex
        jmp getkey
isdnarw:
        cmp ah,80
        je yesitis
        call beep
        jmp getkey
yesitis:
        inc adrs
        cmp byte ptr adrs,0ffh
        jne adrsok1
        mov byte ptr adrs,0f8h
adrsok1:
        jmp adrsok

main            endp
;

;display binary or hex in AL at row,col specified in DX

dsphex:
        push    ax
        mov     ah,2
        mov     bh,vpage
        int     10h             ;locate cursor at DX
        pop     ax
        mov     ah,al
        mov     cl,4
        shr     al,cl            ;shft high nibble down
        and     ah,0fh           ;mask off high bits to get low nibble
        add     ax,3030h
        cmp     al,'9'
        jna     cklownib
        add     al,7
cklownib:
        cmp     ah,'9'
        jna     dspit
        add     ah,7            ;convert to two hex digits
dspit:
        push    ax
        mov     ah,0eh
        int     10h
        pop     ax
        mov     al,ah
        mov     ah,0eh
        int     10h             ;display the digits
        ret                     ;back to caller

dspbin:
        mov     ah,24
        xchg    ah,al
        push    ax              ;fix up for dsploop
        mov     ah,2
        mov     bh,vpage
        int     10h             ;position the cursor
        mov     cx,4
        pop     ax
        call    dsplptop
        push    ax
        mov     al,' '
        mov     ah,0eh
        int     10h
        mov     cx,4
        pop     ax
dsplptop:
        push    ax
dsploop:
        pop     ax
        shl     ah,1
        push    ax
        rcl     al,1            ;al now = ascii 1 or 0 depending on bit
        mov     ah,0eh
        int     10h             ;dsplay binary character
        loop    dsploop
        pop     ax
        ret                     ;clean up stack and back to caller

;---------------------------------------------------------------------------
;VIDEO_ENABLE and VIDEO_DISABLE routines manipulate bit 3 of port 3D8h,
;the CGA Mode Control Register, to temporarily turn the display on or off.
;Since these routines write directly to hardware, they have no effect on
;other video adapters.
;---------------------------------------------------------------------------
;
video_disable proc near
              mov dx,3DAh                        ;read CGA status port
disable1:     in al,dx                  ;wait for vertical retrace to occur
              test al,8                          ;is bit 3 set?
              je disable1                        ;no, wait until it is
              mov dx,3D8h                        ;now disable the display
              mov al,25h      ;by clearing bit 3 of the Mode Control Register
              out dx,al
              ret
video_disable endp
;
video_enable  proc near
              mov dx,3D8h                        ;CGA Mode Control Register
              mov bx,display_mode        ;get value to re-enable display
              sub bx,2
              mov al,display_table[bx]
              out dx,al                          ;and send it to the port
              ret
video_enable  endp
;
;---------------------------------------------------------------------------
;VIDEO2MEM routine transfers the contents of a portion of video memory
;to a memory buffer for storage.
;Entry:       DS    - video segment
;             ES:DI - memory buffer
;             DH,DL - row and column of upper left corner of window
;             CH    - width of window in columns
;             CL    - number of lines in window
;---------------------------------------------------------------------------
;
video2mem     proc near
              mov al,ch                          ;store number of columns
              mov ah,0
              mov column_count,ax
              mov ch,0                           ;cx = number of lines
              push di                            ;save di
              call video_offset         ;get cell address of first character
              mov si,di                          ;put it in si
              pop di                             ;restore di
v2mem1:       push si                            ;save si for next line
              push cx                            ;save line count
              mov cx,column_count                ;set cx for call to WRITELN
              call writeln                       ;transfer one line
              pop cx                             ;restore saved registers
              pop si
              add si,160                         ;set si for next line address
              loop v2mem1               ;loop until all lines are done
              ret
video2mem     endp
;
;---------------------------------------------------------------------------
;MEM2VIDEO writes a selected area of memory to the video display.
;Entry:       DS:SI - memory buffer
;             ES    - video segment
;             DH,DL - row and column of upper left corner of window
;             CH    - width of window in columns
;             CL    - number of lines in window
;---------------------------------------------------------------------------
;
mem2video     proc near
              mov al,ch                          ;save number of columns
              mov ah,0
              mov column_count,ax
              mov ch,0                           ;cx = number of lines
              call video_offset                  ;get offset into video memory
mem2v1:       push di                            ;save video starting address
              push cx                            ;save line count
              mov cx,column_count                ;set cx for call to WRITELN
              call writeln                       ;transfer one line
              pop cx                             ;restore registers
              pop di
              add di,160                         ;set di for next display line
              loop mem2v1                        ;loop until done
              ret
mem2video     endp
;
;---------------------------------------------------------------------------
;VIDEO_OFFSET calculates the offset into video memory of a character cell.
;Entry:       DH,DL - row and column of cell (0-24,0-79)
;Exit:        DI    - offset address
;---------------------------------------------------------------------------
;
video_offset  proc near
              mov al,160
              mul dh                             ;row * 160
              shl dl,1                           ;column * 2
              mov dh,0                           ;byte to word
              add ax,dx                          ;(row *160)+(column*2)
              mov di,ax                          ;set offset in di
              ret
video_offset  endp
;
;---------------------------------------------------------------------------
;WRITELN subroutine copies a string of words from one memory location to
;another.  The CGA status port is not checked for vertical retrace status
;before transfer.
;Entry:       DS:SI - source
;             ES:DI - destination
;             CX    - number of words
;---------------------------------------------------------------------------
;
writeln       proc near
              cld                       ;clear for string instructions
write1:       movsw                              ;move one word
              loop write1                        ;loop until done
              ret
writeln       endp
;
;---------------------------------------------------------------------------
;BEEP uses the 8253 timer chip to emit a short beep thru the PC's speaker.
;---------------------------------------------------------------------------
;
beep          proc near
              mov al,182          ;notify 8253 that frequency data is coming
              out 67,al
              mov al,0                           ;send frequency (776.8 Hz)
              out 66,al
              mov al,6
              out 66,al
              in al,97                           ;activate speaker
              or al,3
              out 97,al
              mov cx,6000h                  ;time delay for sound duration
beep1:        loop beep1
              in al,97                           ;deactivate speaker
              and al,252
              out 97,al
              ret
beep          endp
;
;---------------------------------------------------------------------------
;MEM2MEM subroutine transfers a non-overlapping block of memory one byte
;at a time.
;Entry:       DS:SI - source
;             ES:DI - destination
;             CX    - number of bytes
;---------------------------------------------------------------------------
;
mem2mem       proc near
              cld
mem1:         movsb                              ;transfer one byte
              loop mem1                          ;and loop until done
              ret
mem2mem       endp
;
;---------------------------------------------------------------------------
;Initialization routine sets up the window image in the WINDOW_TEXT area,
;resets the CURSOR_MODE word if this is a PCjr, and saves and replaces the
;old keyboard interrupt vector.
;---------------------------------------------------------------------------
;
initialize    proc near
;
;Initialize the window text area by combining the text data with the attribute
;bytes and placing the conglomeration in the WINDOW_TEXT area.
;
              assume ds:code,es:code
              mov ah,15                          ;check the current video mode
              int 10h
              cmp al,7           ;if it's mode 7, then replace the attribute
              jne init0          ;bytes with ones appropriate for mono adapter
              mov attribute1,70h
              mov attribute2,07h
init0:        cld                  ;now combine the text and attribute bytes
              lea si,buffer_text                 ;point si to table of text
              lea di,window_bytes                ;and di to storage area
              mov cx,84            ;create first three lines by combining
              mov al,attribute1          ;text with attribute1 (112 words)
init1:        movsb                              ;text byte
              stosb                              ;attribute byte
              loop init1                 ;loop until all 112 words are done
              mov cx,14                          ;now do the next 11 lines
init2:        push cx              ;first attribute in each line is attribute1
              movsb
              stosb
              mov cx,26            ;next 26 attributes are attribute2
              mov al,attribute2
init3:        movsb
              stosb
              loop init3
              movsb
              mov al,attribute1     ;and the last in each line is attribute1
              stosb
              pop cx
              loop init2            ;loop until all 11 lines are done
              mov cx,56             ;create the last two lines just like
init4:        movsb                 ;the first three
              stosb
              loop init4
;
;Check the machine ID byte in ROM and if this is a PCjr, then reset the
;cursor and correct the CURSOR_MODE word at 0040:0060.
;
              mov ax,rom                         ;set ds to rom
              mov ds,ax
              assume ds:rom
              cmp machine_id,0FDh                ;is this a PCjr?
              jne init5                          ;no, then skip this routine
              mov ax,rom_bios_data               ;set ds to ROM BIOS data area
              mov ds,ax
              assume ds:rom_bios_data
              mov cursor_mode,0607h          ;reset the cursor mode indicator
              mov ah,1                       ;then physically reset the cursor
              mov cx,0607h
              int 10h
;
;Now save the old keyboard interrupt vector and replace it with the new one.
;
init5:        mov ax,rom_bios_data
              mov ds,ax
              assume ds:rom_bios_data
              mov ax,com1base
              mov es:com1,ax
              mov bx,ax
              mov ax,com2base
              mov es:com2,ax            ;save combase locations in our seg
              add ax,bx
              je errexit            ;error if no com1 or com2 installed
              mov ax,interrupts          ;set ds to the interrupt vector area
              mov ds,ax
              assume ds:interrupts
              mov ax,keyboard_int                ;save old vector
              mov old_keyboard_int,ax
              mov ax,keyboard_int[2]
              mov old_keyboard_int[2],ax
              cli                        ;disable all interrupts but NMI
              mov keyboard_int,offset main       ;and install new vector
              mov keyboard_int[2],cs
              sti                                ;re-enable interrupts
              mov dx,offset initialize   ;point dx to end of resident section
              int 27h                            ;terminate-but-stay-resident
errexit:
              call beep
              mov ah,4ch
              int 21h           ;exit with no action
initialize    endp
;
code          ends
              end begin
