#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include <process.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>

// defines for warnings
#define WR_ACMD      -5001
#define WR_LONGPRM   -5002
#define WR_NOOUT     -5003
#define WR_OVRWOLOG  -5004
#define WR_NOREDIR   -5005

// defines for errors
#define ER_NOPARAMS -1001
#define ER_BADOPT   -1002
#define ER_NOPARAM  -1003
#define ER_RCMD     -1004
#define ER_CRFILE   -1005
#define ER_APFILE   -1006
#define ER_DLFILE   -1007
#define ER_FORMAT   -1008
#define ER_NOSTART  -1009
#define ER_SPAWN    -1010
#define ER_INTERROR -1234

// variants of command
enum TCommand { cmStart, cmStop, cmRun };

// temp file name
char tempname[12] = "$TIMER$.TMP";

// months' lengths
char monthlen[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

FILE *f;			               // just file
TCommand cmd;			           // command
char quiet, overwrt, dates, nored; // quiet mode, overwrite log, save dates
char logname[128], cmdline[256];   // log file name, .exe command line
long time1, time2;		           // start & stop time
struct date date1, date2;	       // start & stop dates

// long numbers
class CLong
{
  char nums[20];  // digits
public:
  CLong(long);
  long get();
  CLong operator <<= (char);
  CLong operator >>= (char);
  CLong operator /= (long);
  CLong operator / (long);
  long operator % (long);
};

// constructor - set initial value
CLong::CLong(long num=0)
{
   char i=0;
   memset(nums, 0, 20); // clear (set all digits to 0)
   while (num)
   {
     nums[i]=num%10;	// put current digit
     // move to next digit
     num/=10;
     i++;
   }
}

// converts to long int
long CLong::get()
{
   char i;
   long res=0, mul=1;

   for (i=0; i<20; i++)
   {
      res+=nums[i]*mul; // add current digit
      mul*=10;		// move to next digit
   }

   return res;
}

// multiply by power of 10 and save
CLong CLong::operator <<= (char shift)
{
   for (char i=19; i>=0; i--)
     if (i>=shift)
       nums[i]=nums[i-shift];
     else nums[i]=0;

   return *this;
}

// divide by power of 10 with rounding and save
CLong CLong::operator >>= (char shift)
{
   char i;
   // rounding
   if (nums[shift-1]>4)  // first lost digit >=5
   {
      nums[shift]++;	 // increase last saved digit
      // correct all digits (0<=nums[i]<10)
      i=shift;
      while (nums[i]>9 && i<19) // this digit >9
      {
	       nums[i+1]++;  // increase next digit
	       nums[i]-=10;  // correct this digit
      }
   }
   // shift
   for (i=0; i<20; i++)
     if (i+shift<20)
       nums[i]=nums[i+shift];
     else nums[i]=0;

   return *this;
}

// divide and save
CLong CLong::operator /= (long val)
{
   char i;
   long d=0;

   for (i=19; i>=0; i--)
   {
       d*=10; d+=nums[i]; // read current digit
       nums[i]=d/val;	  // divide digit
       d%=val;		      // module
   }

   return *this;
}

// divide
CLong CLong::operator / (long val)
{
   char i;
   long d=0;
   CLong res;

   for (i=19; i>=0; i--)
   {
       d*=10; d+=nums[i]; // read current digit
       res.nums[i]=d/val; // divide digit
       d%=val;		      // module
   }

   return res;

}

// division module
long CLong::operator % (long val)
{
   char i;
   long d=0;

   for (i=19; i>=0; i--)
   {
       d*=10; d+=nums[i]; // read current digit
       d%=val;		      // module
   }

   return d;
}

// teach user
void Usage(void)
{
    printf("\nTimer version 1.22\n");
    printf("by Eugene Toder, 2001.\n");
    printf("Shows execution time. Can be used to profile programms.\n");
    printf("Precision is about 1/18 (0.05) of second.\n\n");
    printf("USAGE: timer [command] [options] [parameters]\n");
    printf("  where <command> is:\n");
    printf("	   A - start timer (no parameters),\n");
    printf("	   O - stop timer and show time, passed from start (no parameters),\n");
    printf("	   R - run program with specified arguments (parameter) and show\n");
    printf("	       execution time (default),\n");
    printf("	   H, ?, /H, /? - display this help.\n");
    printf("    and <options> are:\n");
    printf("	   /g - long times (execution time exceeds 24 hours),\n");
    printf("	   /q - quiet mode (don't show time on screen),\n");
    printf("	   /l<log_name> - write time to log file <log_name>,\n");
    printf("	   /n - not redirectable result output,\n");
    printf("	   /o - overwrite log file (default - append).\n\n");
    exit(1);
}

// show warnings
void ShowWarning(int warnum, char *p=NULL)
{
    printf("WARNING: ");
    switch (warnum)
    {
        case WR_ACMD:	  printf("Command A: %s option ignored.\n", p); break;
        case WR_LONGPRM:  printf("%s is too long. Truncated.\n", p); break;
        case WR_NOOUT:    printf("No output selected.\n"); break;
        case WR_OVRWOLOG: printf("/O option without log name.\n"); break;
        case WR_NOREDIR:  printf("/N and /Q used together.\n"); break;
        default:
	        printf("Unknown warning!\n");
    }
}

// show errors
void ShowError(int errnum, char *p=NULL)
{
    if (errnum==ER_SPAWN) printf("SPAWN ");
    printf("ERROR: ");
    switch (errnum)
    {
        case ER_NOPARAMS: printf("Not enough actual parameters!\n"); Usage();
        case ER_BADOPT:	  printf("Bad option '%s'!\n", p); Usage();
        case ER_NOPARAM:  printf("Command %s has no parameters!\n", p); break;
        case ER_RCMD:	  printf("Command R require 1 parameter!\n"); break;
        case ER_CRFILE:	  printf("Can't create file '%s'!\n", p); break;
        case ER_APFILE:	  printf("Can't append file '%s'!\n", p); break;
        case ER_DLFILE:	  printf("Can't delete file '%s'!\n", p); break;
        case ER_FORMAT:	  printf("File '%s' has wrong format!\n", p); break;
        case ER_NOSTART:  printf("Timer should be started first!\n"); break;
        case ER_SPAWN:	  printf("%s!\n", p); break;
        case ER_INTERROR: printf("Internal error!\n"); break;
        default:
    	    printf("Unknown error!\n");
    }
    exit(1);
}

// test for leap-year
inline int leapyear(int year)
{
    return (!(year%4) && year%100) || !(year%400);
}

// return day's number from start of the year
int daynum(struct date dt)
{
    int i, res=0;

    // set Feb's length
    if (leapyear(dt.da_year)) monthlen[1]=29; else monthlen[1]=28;
    // add all passed monthes
    for (i=0; i<dt.da_mon-1; i++)
      res+=monthlen[i];
    res+=dt.da_day; // add day

    return res;
}

// make resulting string
void WriteRes()
{
    char buf[80]; // buffer for resulting string

    long days=0;
    if (dates)    // long times mode
    {
        days=daynum(date2)-daynum(date1);  // difference between days' numbers
        // add difference between years
        while (date1.da_year<date2.da_year)
        {
	        date2.da_year--;
	        if (leapyear(date2.da_year)) days+=366; else days+=365;
        }
        if (days<0) ShowError(ER_FORMAT, tempname);
    }
    long time=time2-time1;	// calculate difference
    if (time<0) 		    // over midnight
    {
        time+=1573040L;		// correct time & days
        if (dates) days--;
    }
    CLong res(time);		// convert to long
    // divide by 18.206482 with precision of 0.01
    res<<=9;			// multiply by 10^9
    res/=18206482L;		// divide by 18206482
    res>>=1;			// divide by 10 with rounding
    // make string
    if (dates && days)
        sprintf(buf, "Time passed: %ld day%s and %02.2ld:%02.2ld:%02.2ld.%02.2ld", days, days!=1?"s":"" , (res/360000L).get(), res/6000%60, res/100%60, res%100);
    else
        sprintf(buf, "Time passed: %02.2ld:%02.2ld:%02.2ld.%02.2ld", (res/360000L).get(), res/6000%60, res/100%60, res%100);
    // write result
    if (!quiet)
    {
        if (nored)
           cprintf("%s\n", buf);
        else printf("%s\n", buf);
    }
    if (logname[0]) // write to log
    {
        char mode[2]; // open mode
        if (overwrt) mode[0]='w'; // overwrite log
      	   else mode[0]='a';	  // append log
        mode[1]=0;

        if ((f=fopen(logname, mode))==NULL) // open file
        {
	        // failed
	        if (overwrt) ShowError(ER_CRFILE, logname);  // failed create
	            else ShowError(ER_APFILE, logname);      // failed append
        }
        // write result
        fprintf(f, "%s\n", buf);
        fclose(f);
    }
}

// read start time [& date]
void readstart()
{
    if (dates) getdate(&date1);
    asm{
        // wait for time update (to increase precision)
        mov ax, 0x40
        mov es, ax
        mov di, 0x6C
        mov bl, es:[di] // read current time
    }
LOOP:
    asm{
        cmp bl, es:[di]
        je LOOP // while time hasn't changed
        // read new time
        mov dx, es:[di+2]
        mov bx, es:[di]
        // save new time
        mov word ptr time1, bx
        mov word ptr time1+2, dx
    }
}

// read stop time and date
void readstop()
{
    asm{
        mov ax, 0x40
        mov es, ax
        mov di, 0x6C
        // read time
        mov dx, es:[di+2]
        mov bx, es:[di]
        // save time
        mov word ptr time2, bx
        mov word ptr time2+2, dx
    }
    getdate(&date2);
}

void main(int argc, char *argv[])
{
     if (argc<2) ShowError(ER_NOPARAMS);    // no argvuments

     int i=2;
     // identify command
     switch (toupper(argv[1][0]))
     {
	     // start timer
         case 'A':
	         if (argv[1][1]) goto DEFAULT_CMD;
	         cmd=cmStart;
	         break;
	     // stop timer
	     case 'O':
	         if (argv[1][1]) goto DEFAULT_CMD;
	         cmd=cmStop;
	         break;
	     // run program
	     case 'R':
	         if (argv[1][1]) goto DEFAULT_CMD;
	         cmd=cmRun;
	         break;
	     // get help
	     case 'H':
	     case '?':
	         Usage();
	     case '-':
	     case '/':
	         if (argv[1][1]=='h' || argv[1][1]=='H' || argv[1][1]=='?')
             {
                if (argv[1][2]) goto DEFAULT_CMD;
                Usage();
             }
	     // default command
	     default:
DEFAULT_CMD:
	         cmd=cmRun;
             i=1;
     }
     // read options and parameters
     for (i; i<argc; i++)
         if (argv[i][0]=='-' || argv[i][0]=='/') // option
         {
	         switch (toupper(argv[i][1]))
             {
	             case 'Q': // quiet operations
	                 if (argv[i][2]) goto BAD_OPT;
	                 if (cmd==cmStart) ShowWarning(WR_ACMD, argv[i]); // ignore
	                 quiet=1;
	                 break;
	             case 'O': // overwrite log
	                 if (argv[i][2]) goto BAD_OPT;
	                 if (cmd==cmStart) ShowWarning(WR_ACMD, argv[i]); // ignore
	                 overwrt=1;
	                 break;
	             case 'G': // long time
 	                  if (argv[i][2]) goto BAD_OPT;
	                  dates=1;
	                  break;
	             case 'N': // not redirectable output
 	                  if (argv[i][2]) goto BAD_OPT;
                      if (cmd==cmStart) ShowWarning(WR_ACMD, argv[i]); // ignore
	                  nored=1;
	                  break;
	             case 'L': // log file name
	                  if (cmd==cmStart) ShowWarning(WR_ACMD, argv[i]); // ignore
	                  // save log name
	                  if (strlen(argv[i])>129) ShowWarning(WR_LONGPRM, "Log file name");
	                  strncpy(logname, argv[i]+2, 127);
	                  break;
	             default: // suxxx
BAD_OPT:
	                  ShowError(ER_BADOPT, argv[i]);
             }
         }
         else // parameters
         {
	         switch (cmd)
             {
	              case cmStart: ShowError(ER_NOPARAM, "A"); // A command
	              case cmStop:  ShowError(ER_NOPARAM, "O"); // O command
	              case cmRun: 				                // R command
	                  // save .exe file command line
                      int len=0;
                      for (; i<argc; i++)
                      {
                          strncpy(cmdline+len, argv[i], 255-len);
                          len+=strlen(argv[i]);
                          if (i<argc-1) cmdline[len++]=' ';
                          if (len>255)
                          {
                              ShowWarning(WR_LONGPRM, "Program command line");
                              break;
                          }
                      }
	                  break;
	              default: // contact Microsoft Product Support Servicies
	                  ShowError(ER_INTERROR);
	         }
	         break;
         }
     if (cmd!=cmStart && quiet && !logname[0]) // no output with R or O
	    ShowWarning(WR_NOOUT);
     if (quiet && nored)
        ShowWarning(WR_NOREDIR);
     if (overwrt && !logname[0])     // overwrite w/o log name
	    ShowWarning(WR_OVRWOLOG);
     if (cmd==cmRun && !cmdline[0])  // no .exe name with R
        ShowError(ER_RCMD);
     // work itself
     switch (cmd)
     {
         case cmStart:                       // start command
	         // create temp file
	         if ((f=fopen(tempname, "wb"))==NULL)
	             ShowError(ER_CRFILE, tempname); // failed
	         readstart();		             // get current time in TIME1
	         fwrite(&time1, 4, 1, f);        // write start time to temp file
	         if (dates)		                 // write date is necessary
	            fwrite(&date1, (sizeof(date1)), 1, f);
	         fclose(f);		                 // close file
	         break;
         case cmStop:
	         readstop();		             // get stop time in TIME2
	         // open temp file
	         if ((f=fopen(tempname, "rb"))==NULL)
	             ShowError(ER_NOSTART);      // failed
	         // read start time from temp file
	         if (fread(&time1, 4, 1, f)<1)
	             ShowError(ER_FORMAT, tempname); // failed
	         // try to read start date from temp file
	         if (fread(&date1, (sizeof(date1)), 1, f)<1)
             {
	             // failed
	             if (dates)                  // date required
		             ShowError(ER_FORMAT, tempname);
	             dates=0;		             // date not found
	         } else dates=1;	             // date found
	         WriteRes();		             // write result
	         fclose(f);		                 // close file
	         if (remove(tempname)<0)         // delete temp file
	            ShowError(ER_DLFILE, tempname); //failed
	         break;
         case cmRun:
	         readstart();		             // get current time in TIME1
	         // run program
	         if (system(cmdline)<0)
             {
	             switch (errno)              // failed
                 {
	                 case E2BIG: ShowError(ER_SPAWN, "Arg list too long"); break;
	                 case ENOENT: ShowError(ER_SPAWN, "Path or file name not found"); break;
	                 case ENOEXEC: ShowError(ER_SPAWN, "Exec format error"); break;
	                 case ENOMEM: ShowError(ER_SPAWN, "Not enough memory"); break;
	                 default: ShowError(ER_SPAWN, "Unknown");
	             }
	         }
	         readstop();		             // get stop time in TIME2
	         WriteRes();		             // write result
	         break;
         default: // contact Microsoft Product Support Servicies
	         ShowError(ER_INTERROR);
     }
}