page,128
;----------------------------------------------------------------------
;
; MIRFIX.ASM
;
; fixes a bug in MIRROR.COM
;
; compile:      MASM mirfix;
;               LINK mirfix;
;               EXE2BIN mirfix.exe mirfix.com
;               DEL mirfix.exe
;
; explanation:  MIRROR traps INT 21h to enable delete tracking.
;               When deleting a file with zero length, INT 21h returns
;               with interrupts clear. At times this may cause programs
;               to hang since no keyboard, timer, mouse, or any other
;               hardware interrupts can be processed with CLI.
;
; how it works: MIRFIX traps INT 21h. This interrupt trap works as follows:
;
;               Get the pushed flags (INT command works like PUSHF, CALL FAR)
;               Mask out the interrupt flag value and save it
;               Call the original interrupt
;               Merge the caller's interrupt flag to the flags register
;               Return from INT 21h
;
;               The fix procedure is written totally reentrant since we
;               do not use local variables during the call to original
;               INT 21h. (DOS or any other TSR could call INT 21h
;               recursively).
;
; how to load:  Load MIRFIX immediately after loading MIRROR /Tx in your
;               AUTOEXEC.BAT file.
;               MIRFIX can be kept quiet appending any command line
;               parameter, e.g.
;                      MIRFIX shutup
;               or     MIRFIX x
;----------------------------------------------------------------------

mirfix segment byte public 'CODE'
        assume cs:mirfix
        assume ds:mirfix

;----------------------------------------------------------------------
        ; COM style entry point
;----------------------------------------------------------------------

org     100h
startfix:
        jmp     fix_transient

;----------------------------------------------------------------------
        ; the TSR's resident data area
;----------------------------------------------------------------------

        int21   dd ?                   ; original interrupt address

        save_a  dw ?                   ; three word save areas
        save_b  dw ?
        save_c  dw ?


;----------------------------------------------------------------------
        ; INT 21h fix procedure (resident)
;----------------------------------------------------------------------

fixint proc far

        ; we need the AX register, so save it
        mov     cs:[save_a], ax

        ; get the return address off the stack
        pop     cs:[save_b]
        pop     cs:[save_c]

        ; get the flags off the stack
        pop     ax

        ; push the return address back
        push    cs:[save_c]
        push    cs:[save_b]

        pushf                          ; save the flags (AND operation)
        and     ax, 0200h              ; mask out int flag
        popf

        push    ax                     ; save iflag on stack

        mov     ax, cs:[save_a]        ; restore caller's AX

        ; now call original DOS interrupt
        pushf
        call    cs:[int21]

        ; we need AX and BX, so save them
        mov     cs:[save_a], ax        ; save AX and BX
        mov     cs:[save_b], bx

        pushf                          ; get the flags into AX
        pop     ax

        pop     bx                     ; retrieve saved int flag

        or      ax, bx                 ; merge it into flags

        push    ax                     ; back into the flags register
        popf

        mov     ax, cs:[save_a]        ; restore AX and BX
        mov     bx, cs:[save_b]

        ret                            ; the flags have already been removed

fixint endp


;----------------------------------------------------------------------
; MIRFIX transient part
;----------------------------------------------------------------------

author  db 13, 10
        db 'MIRFIX - written 1991 by E.Vogelsinger, EDP-Services, Vienna, Austria', 13, 10
        db 'Fixes a nasty bug in MIRROR.COM (1991 by Central Point Software).', 13, 10, '$'

fix_transient:
        mov     ax, cs
        mov     ds, ax

        mov     bx, 81h                ; point to command line

        mov     cl, byte ptr [bx-1]    ; get length
        sub     ch, ch
        jcxz    show_author

scan_cmd:
        cmp     byte ptr [bx], ' '     ; scan command line
        jz      @F                     ; for some non-blank entry
        cmp     byte ptr [bx], 09      ; to disable author notice
        jz      @F
        cmp     byte ptr [bx], 0Dh
        jz      show_author

        jmp     short go_resident      ; some non-blank entry has been found

@@:     inc     bx
        loop    scan_cmd

show_author:
        mov     dx, offset author
        mov     ah, 09h
        int     21h

go_resident:
        mov     ax, 3521h              ; get INT 21h address
        int     21h
        mov     word ptr [int21], bx
        mov     ax, es
        mov     word ptr [int21+2], ax

        mov     dx, offset fixint      ; hook into INT 21h
        mov     ax, 2521h
        int     21h


                                       ; calculate resident size
        mov     dx, offset author
        add     dx, 0Fh                ; make paragraphs
        mov     cx, 4
        shr     dx, cl

        mov     ax, 3100h              ; TSR
        int     21h

mirfix ends
end startfix    ; specify entry point
