//
// EXEPROP is a Windows 95 shell extension that adds a Dependencies page
// to the property sheets of Win32 EXE files.
//
 
#define INC_OLE2

#include <windows.h>
#include <initguid.h>
#include <shlobj.h>
#include <string.h>
#include "exeprop.h"

BOOL CALLBACK DependenciesDlgProc (HWND, UINT, WPARAM,  LPARAM);
void ListDependencies (char *, HWND);

//
// Global variables
//
UINT        g_cRefThisDll = 0;          // Reference count for this DLL
HINSTANCE   g_hInstance;                // Instance handle for this DLL

//
// DllMain is the DLL's entry point.
//
// Input parameters:
//   hInstance  = Instance handle
//   dwReason   = Code specifying the reason DllMain was called
//   lpReserved = Reserved (do not use)
//
// Returns:
//   TRUE if successful, FALSE if not
//

extern "C" int APIENTRY DllMain (HINSTANCE hInstance, DWORD dwReason,
    LPVOID lpReserved)
{
    if (dwReason == DLL_PROCESS_ATTACH)
        g_hInstance = hInstance;
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// In-process server functions

//
// DllGetClassObject is called by the shell to create a class factory object.
//
// Input parameters:
//   rclsid = Reference to class ID specifier
//   riid   = Reference to interface ID specifier
//   ppv    = Pointer to location to receive interface pointer
//
// Returns:
//   HRESULT code signifying success or failure
//

STDAPI DllGetClassObject (REFCLSID rclsid, REFIID riid, LPVOID *ppv)
{
    *ppv = NULL;

    //
    // Make sure the class ID is CLSID_ShellExtension. Otherwise, the class
    // factory doesn't support the object type specified by rclsid.
    //
    if (!IsEqualCLSID (rclsid, CLSID_ShellExtension))
        return ResultFromScode (CLASS_E_CLASSNOTAVAILABLE);
        
    //
    // Instantiate a class factory object.
    //
    CClassFactory *pClassFactory = new CClassFactory ();

    if (pClassFactory == NULL)
        return ResultFromScode (E_OUTOFMEMORY);

    //
    // Get the interface pointer from QueryInterface and copy it to *ppv.
    //
    HRESULT hr = pClassFactory->QueryInterface (riid, ppv);
    pClassFactory->Release ();
    return hr;
}

//
// DllCanUnloadNow is called by the shell to find out if the DLL can be
// unloaded. The answer is yes if (and only if) the module reference count
// stored in g_cRefThisDll is 0.
//
// Input parameters:
//   None
//
// Returns:
//   HRESULT code equal to S_OK if the DLL can be unloaded, S_FALSE if not
//

STDAPI DllCanUnloadNow (void)
{
    return ResultFromScode ((g_cRefThisDll == 0) ? S_OK : S_FALSE);
}

/////////////////////////////////////////////////////////////////////////////
// CClassFactory member functions

CClassFactory::CClassFactory ()
{
    m_cRef = 1;
    g_cRefThisDll++;
}

CClassFactory::~CClassFactory ()
{
    g_cRefThisDll--;
}

STDMETHODIMP CClassFactory::QueryInterface (REFIID riid, LPVOID FAR *ppv)
{
    if (IsEqualIID (riid, IID_IUnknown)) {
        *ppv = (LPUNKNOWN) (LPCLASSFACTORY) this;
        m_cRef++;
        return NOERROR;
    }

    else if (IsEqualIID (riid, IID_IClassFactory)) {
        *ppv = (LPCLASSFACTORY) this;
        m_cRef++;
        return NOERROR;
    }

    else {  
        *ppv = NULL;
        return ResultFromScode (E_NOINTERFACE);
    }
}

STDMETHODIMP_(ULONG) CClassFactory::AddRef ()
{
    return ++m_cRef;
}

STDMETHODIMP_(ULONG) CClassFactory::Release ()
{
    if (--m_cRef == 0)
        delete this;
    return m_cRef;
}

//
// CreateInstance is called by the shell to create a shell extension object.
//
// Input parameters:
//   pUnkOuter = Pointer to controlling unknown
//   riid      = Reference to interface ID specifier
//   ppvObj    = Pointer to location to receive interface pointer
//
// Returns:
//   HRESULT code signifying success or failure
//

STDMETHODIMP CClassFactory::CreateInstance (LPUNKNOWN pUnkOuter, REFIID riid,
    LPVOID FAR *ppvObj)
{
    *ppvObj = NULL;

    //
    // Return an error code if pUnkOuter is not NULL, because we don't
    // support aggregation.
    //
    if (pUnkOuter != NULL)
        return ResultFromScode (CLASS_E_NOAGGREGATION);

    //
    // Instantiate a shell extension object.
    //
    CShellExtension *pShellExtension = new CShellExtension ();

    if (pShellExtension == NULL)
        return ResultFromScode (E_OUTOFMEMORY);

    //
    // Get the interface pointer from QueryInterface and copy it to *ppvObj.
    //
    HRESULT hr = pShellExtension->QueryInterface (riid, ppvObj);
    pShellExtension->Release ();
    return hr;
}

//
// LockServer increments or decrements the DLL's lock count.
//
// Input parameters:
//   fLock = TRUE to increment lock count, FALSE to decrement it
//
// Returns:
//   HRESULT code signifying success or failure
//

STDMETHODIMP CClassFactory::LockServer (BOOL fLock)
{
    return ResultFromScode (E_NOTIMPL);
}

/////////////////////////////////////////////////////////////////////////////
// CShellExtension member functions

CShellExtension::CShellExtension ()
{
    m_cRef = 1;
    m_szFile[0] = 0;
    g_cRefThisDll++;
}

CShellExtension::~CShellExtension ()
{
    g_cRefThisDll--;
}

STDMETHODIMP CShellExtension::QueryInterface (REFIID riid, LPVOID FAR *ppv)
{
    if (IsEqualIID (riid, IID_IUnknown)) {
        *ppv = (LPUNKNOWN) (LPSHELLPROPSHEETEXT) this;
        m_cRef++;
        return NOERROR;
    }

    else if (IsEqualIID (riid, IID_IShellPropSheetExt)) {
        *ppv = (LPSHELLPROPSHEETEXT) this;
        m_cRef++;
        return NOERROR;
    }

    else if (IsEqualIID (riid, IID_IShellExtInit)) {
        *ppv = (LPSHELLEXTINIT) this;
        m_cRef++;
        return NOERROR;
    }
    
    else {
        *ppv = NULL;
        return ResultFromScode (E_NOINTERFACE);
    }
}

STDMETHODIMP_(ULONG) CShellExtension::AddRef ()
{
    return ++m_cRef;
}
 
STDMETHODIMP_(ULONG) CShellExtension::Release ()
{
    if (--m_cRef == 0)
        delete this;
    return m_cRef;
}

//
// IsPEFile examines a file and verifies that it is a Portable Executable.
//
// Input parameters:
//   pszFile = Pointer to file name
//
// Returns:
//   TRUE if Portable Executable, FALSE if not
//

BOOL CShellExtension::IsPEFile (char *pszFile)
{
    HANDLE hFile;
    OFSTRUCT ofs;
    IMAGE_DOS_HEADER dh;
    DWORD dwBytesRead;
    DWORD dwSignature;

    //
    // Open the file and read the DOS header.
    //
    if ((hFile = (HANDLE) OpenFile (pszFile, &ofs, OF_READ)) ==
        (HANDLE) HFILE_ERROR)
        return FALSE;
        
    if (!ReadFile (hFile, &dh, sizeof (dh), &dwBytesRead, NULL) ||
        (dwBytesRead < sizeof (dh))) {
        CloseHandle (hFile);
        return FALSE;
    }       

    //
    // Read the PE header signature and check for "PE\0\0".
    //
    SetFilePointer (hFile, dh.e_lfanew, NULL, FILE_BEGIN);

    if (!ReadFile (hFile, &dwSignature, sizeof (dwSignature), &dwBytesRead,
        NULL) || (dwBytesRead < sizeof (&dwSignature))) {
        CloseHandle (hFile);
        return FALSE;
    }       

    CloseHandle (hFile);
    return (dwSignature == 0x4550) ? TRUE : FALSE;
}

//
// AddPages is called by the shell to give property sheet shell extensions
// the opportunity to add pages to a property sheet before it is displayed.
//
// Input parameters:
//   lpfnAddPage = Pointer to function called to add a page
//   lParam      = lParam parameter to be passed to lpfnAddPage
//
// Returns:
//   HRESULT code signifying success or failure
//

STDMETHODIMP CShellExtension::AddPages (LPFNADDPROPSHEETPAGE lpfnAddPage,
    LPARAM lParam)
{
    PROPSHEETPAGE psp;
    HPROPSHEETPAGE hPage;
    
    //
    // Return now if the EXE file is not a Win32 Portable Executable.
    //
    if (!IsPEFile (m_szFile))
        return NOERROR;

    //
    // Fill in the PROPSHEETPAGE structure and create a property sheet page.
    //
    psp.dwSize = sizeof (psp);
    psp.dwFlags = PSP_USEREFPARENT | PSP_USETITLE;
    psp.hInstance = g_hInstance;
    psp.pszTemplate = "Dependencies";
    psp.pszTitle = "Dependencies";
    psp.pfnDlgProc = (DLGPROC) DependenciesDlgProc;
    psp.lParam = (LPARAM) &m_szFile;
    psp.pcRefParent = &g_cRefThisDll;

    hPage = ::CreatePropertySheetPage (&psp);

    //
    // Add the page to the property sheet.
    //
    if (hPage != NULL)
        if (!lpfnAddPage (hPage, lParam))
            DestroyPropertySheetPage (hPage);

    return NOERROR;
}
    
//
// ReplacePage is called by the shell to give control panel extensions the
// opportunity to replace control panel property sheet pages. It is never
// called for conventional property sheet extensions, so we simply return
// a failure code if called.
//
// Input parameters:
//   uPageID         = Page to replace
//   lpfnReplaceWith = Pointer to function called to replace a page
//   lParam          = lParam parameter to be passed to lpfnReplaceWith
//
// Returns:
//   HRESULT code signifying success or failure
//

STDMETHODIMP CShellExtension::ReplacePage (UINT uPageID,
    LPFNADDPROPSHEETPAGE lpfnReplaceWith, LPARAM lParam)
{
    return ResultFromScode (E_FAIL);
}

//
// Initialize is called by the shell to initialize a shell extension.
//
// Input parameters:
//   pidlFolder = Pointer to ID list identifying parent folder
//   lpdobj     = Pointer to IDataObject interface for selected object(s)
//   hKeyProgId = Registry key handle
//
// Returns:
//   HRESULT code signifying success or failure
//

STDMETHODIMP CShellExtension::Initialize (LPCITEMIDLIST pidlFolder,
    LPDATAOBJECT lpdobj, HKEY hKeyProgID)
{
    STGMEDIUM medium;
    FORMATETC fe = { CF_HDROP, NULL, DVASPECT_CONTENT, -1, TYMED_HGLOBAL };

    //
    // Fail the call if lpdobj is NULL.
    //
    if (lpdobj == NULL)
        return ResultFromScode (E_FAIL);

    //
    // Render the data referenced by the IDataObject pointer to an HGLOBAL
    // storage medium in CF_HDROP format.
    //
    HRESULT hr = lpdobj->GetData (&fe, &medium);
    if (FAILED (hr))
        return ResultFromScode (E_FAIL);

    //
    // If only one file is selected, retrieve the file name and store it in
    // m_szFile. Otherwise fail the call.
    //
    if (DragQueryFile ((HDROP) medium.hGlobal, 0xFFFFFFFF, NULL, 0) == 1) {
        DragQueryFile ((HDROP) medium.hGlobal, 0, m_szFile, sizeof (m_szFile));
        hr = NOERROR;
    }
    else
        hr = ResultFromScode (E_FAIL);

    ReleaseStgMedium (&medium);
    return hr;
}

/////////////////////////////////////////////////////////////////////////////
// Other functions

//
// DependenciesDlgProc is the dialog procedure for the Dependencies
// property sheet page.
//

BOOL CALLBACK DependenciesDlgProc (HWND hwnd, UINT uMessage, WPARAM wParam,
    LPARAM lParam)
{
    char *pszFile;

    switch (uMessage) {
    
    case WM_INITDIALOG:
        //
        // Open the EXE file and list its dependencies.
        //
        pszFile = (char *) (((LPPROPSHEETPAGE) lParam)->lParam);
        ListDependencies (pszFile, GetDlgItem (hwnd, IDD_LISTBOX));
        return TRUE;    

    case WM_NOTIFY:
        //
        // Respond FALSE to all notification messages.
        //    
        SetWindowLong (hwnd, DWL_MSGRESULT, FALSE);
        return TRUE;        
    }
    return FALSE;
}

//
// ListDependencies reads the module names from a Portable Executable's
// imported names table and copies them to a list box.
//
// Input parameters:
//   pszFile = Pointer to EXE file name
//   hwnd    = List box handle
//
// Returns:
//   Nothing
//

void ListDependencies (char *pszFile, HWND hwnd)
{
    HANDLE hFile, hFileMapping;
    LPVOID pBase;
    PIMAGE_FILE_HEADER pfh;
    PIMAGE_SECTION_HEADER psh;
    UINT i;
    BOOL bFound = FALSE;
    DWORD dwDelta;
    PIMAGE_IMPORT_DESCRIPTOR pid;
    char *pszModuleName;

    //
    // Open the file and map it to memory.
    //
    if ((hFile = CreateFile (pszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0)) == INVALID_HANDLE_VALUE)
        return;

    if ((hFileMapping = CreateFileMapping (hFile, NULL, PAGE_READONLY,
        0, 0, NULL)) == NULL) {
        CloseHandle (hFile);
        return;
    }

    if ((pBase = MapViewOfFile (hFileMapping, FILE_MAP_READ, 0, 0, 0)) ==
        NULL) {
        CloseHandle (hFileMapping);
        CloseHandle (hFile);
        return;
    }

    //
    // Find the section header for the .idata section.
    //
    pfh = (PIMAGE_FILE_HEADER) ((LPBYTE) pBase +
        ((PIMAGE_DOS_HEADER) pBase)->e_lfanew + sizeof (DWORD));

    psh = (PIMAGE_SECTION_HEADER) ((LPBYTE) pBase +
        ((PIMAGE_DOS_HEADER) pBase)->e_lfanew + sizeof (IMAGE_NT_HEADERS));

    for (i=0; i<pfh->NumberOfSections; i++) {
        if (strnicmp ((char *) psh->Name, ".idata",
            IMAGE_SIZEOF_SHORT_NAME) == 0) {
            bFound = TRUE;
            break;
        }
        else
            psh++;
    }

    if (!bFound) {
        UnmapViewOfFile (pBase);
        CloseHandle (hFileMapping);
        CloseHandle (hFile);
        return;
    }

    //
    // Loop through the import table copying module names to the list box.
    //
    dwDelta = psh->VirtualAddress - psh->PointerToRawData;
    pid = (PIMAGE_IMPORT_DESCRIPTOR) ((LPBYTE) pBase + psh->PointerToRawData);

    while (pid->Name != 0) {
        pszModuleName = (char *) ((LPBYTE) pBase + pid->Name - dwDelta);
        SendMessage (hwnd, LB_ADDSTRING, 0, (LPARAM) (LPCTSTR) pszModuleName);
        pid++;
    }

    //
    // Clean up and exit.
    //
    UnmapViewOfFile (pBase);
    CloseHandle (hFileMapping);
    CloseHandle (hFile);
}
