//========================================================================
// 
//  DIGIT.C - DLL that creates a Digit Class
//
//  (c) Douglas Boling 1992
//
//  For better readability, set tab stops to every 3 characters.
//
//========================================================================

#include <windows.h>
#include <custcntl.h>
#include <string.h>
#include "digit.h"

typedef struct {
	int	sValue;
	int	sBase;
} DIGITDATA;
typedef DIGITDATA *	PDIGITDATA;

//------------------------------------------------------------------------
// Function prototypes
//------------------------------------------------------------------------
BOOL FAR PASCAL DigitDlgFn (HWND, WORD, WORD, LONG);
long FAR PASCAL DigitWndFn (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL AboutDlgProc (HWND, WORD, WORD, LONG);

//------------------------------------------------------------------------
// Globals
//------------------------------------------------------------------------
HANDLE		hInstance;
BOOL			bDigitRegistered = FALSE;
HANDLE		hCtlStyle;
LPFNSTRTOID lpfnStr2ID;
LPFNIDTOSTR lpfnID2Str;


#define	NUMSTYLES   4
struct {
	DWORD	dwStyle;
	char	szLabel[20];
} ssStyleTags [NUMSTYLES] = {
                     {DIGS_LED, DIGS_LEDSTR},
                     {DIGS_BLANK, DIGS_BLANKSTR},
                     {DIGS_COLON, DIGS_COLONSTR},
                     {DIGS_DECIMAL, DIGS_DECIMALSTR}
                    };

//------------------------------------------------------------------------
// LibMain (DigitInit) - Control initialization routine
//------------------------------------------------------------------------
int FAR PASCAL LibMain (HANDLE hInst, WORD wDataSeg, WORD wHeapSize,
	                     LPSTR lpszCmdLine) {
   WNDCLASS	wndclass;
	hInstance = hInst;

	if (!bDigitRegistered) {
      wndclass.style          = CS_HREDRAW | CS_VREDRAW | CS_GLOBALCLASS;
      wndclass.lpfnWndProc    = (FARPROC) DigitWndFn;
      wndclass.cbClsExtra     = 0;
      wndclass.cbWndExtra     = sizeof (PDIGITDATA);
      wndclass.hInstance      = hInstance;
      wndclass.hIcon          = NULL;
      wndclass.hCursor        = LoadCursor (NULL, IDC_ARROW);
      wndclass.hbrBackground  = GetStockObject (BLACK_BRUSH);
      wndclass.lpszMenuName   = NULL;
      wndclass.lpszClassName  = DIGITCLASS;
      
      bDigitRegistered = RegisterClass (&wndclass);
   }
	return bDigitRegistered;
}
//------------------------------------------------------------------------
// WEP - DLL termination routine
//------------------------------------------------------------------------
int FAR PASCAL WEP (int nParameter) {

	if (bDigitRegistered) {
		UnregisterClass (DIGITCLASS, hInstance);
		bDigitRegistered = FALSE;
	}
	return 1;
}

//------------------------------------------------------------------------
// DigitInfo - Returns basic info about control
//------------------------------------------------------------------------
HANDLE FAR PASCAL DigitInfo () {
			   
	HANDLE 		hCtlInfo;
	LPCTLINFO	lpCtlInfo;

	hCtlInfo = GlobalAlloc (GHND | GMEM_SHARE, sizeof (CTLINFO));
	if (hCtlInfo != 0) {

		lpCtlInfo = (LPCTLINFO) GlobalLock (hCtlInfo);
		lpCtlInfo->wVersion = 0x10;
		lpCtlInfo->wCtlTypes = 1;
		lstrcpy (lpCtlInfo->szClass, DIGITCLASS);
		lstrcpy (lpCtlInfo->szTitle, "Digit - Copyright 1992 Douglas Boling");

		lpCtlInfo->Type->wType = 0;
		lpCtlInfo->Type->wWidth = 16;
		lpCtlInfo->Type->wHeight = 24;
		lpCtlInfo->Type->dwStyle = WS_CHILD;
		lstrcpy (lpCtlInfo->Type->szDescr, "Digit Control");

		GlobalUnlock (hCtlInfo);
	}
	return hCtlInfo;
}

//------------------------------------------------------------------------
// DigitStyle - Displays the Style Dialog box (in Dialog editor) for the
//              control.
//------------------------------------------------------------------------
BOOL FAR PASCAL DigitStyle (HWND hWnd, HANDLE hCtlStyle, 
                            LPFNSTRTOID lpfnStrToId,
									 LPFNIDTOSTR lpfnIdToStr) {
	WORD			rc;
	FARPROC		lpfnDigitDlgFn;

	lpfnStr2ID = lpfnStrToId;
	lpfnID2Str = lpfnIdToStr;

	lpfnDigitDlgFn = MakeProcInstance (DigitDlgFn, hInstance);
	rc = DialogBoxParam (hInstance, "Style", hWnd, lpfnDigitDlgFn, 
	                     (LONG) hCtlStyle);
	FreeProcInstance (lpfnDigitDlgFn);

	if (rc == 0) hCtlStyle = 0;
	return hCtlStyle;
}

//------------------------------------------------------------------------
// DigitDlgFn - The Dialog box proc for the Style dialog box
//------------------------------------------------------------------------
BOOL FAR PASCAL DigitDlgFn (HWND hWnd, WORD message, WORD wParam, 
                            LONG lParam) {

	static		DWORD		dwMyStyle;

	LPCTLSTYLE	lpCtlStyle;
	WORD			wMyCtlID, rc;
	DWORD			dwRC;
	char			szTemp[64];

	switch (message) {

		case WM_INITDIALOG:

			hCtlStyle = (HANDLE) lParam;
			lpCtlStyle = (LPCTLSTYLE) GlobalLock (hCtlStyle);
			//
			// Different from Docs.  It appears that user defined
			// styles should be in the low word.
			//
			dwMyStyle = lpCtlStyle->dwStyle; 
			wMyCtlID = lpCtlStyle->wId;

			rc = lpfnID2Str (wMyCtlID, szTemp, sizeof (szTemp));
			if (rc) 
				SetDlgItemText (hWnd, IDD_CTLID, szTemp);
			else
				SetDlgItemInt (hWnd, IDD_CTLID, wMyCtlID, FALSE);

			if (dwMyStyle & DIGS_BLANK)
				CheckDlgButton (hWnd, IDD_BLANK, TRUE);
			else
				CheckDlgButton (hWnd, IDD_BLANK, FALSE);

			if (dwMyStyle & DIGS_COLON)
				CheckDlgButton (hWnd, IDD_COLON, TRUE);
			else
				CheckDlgButton (hWnd, IDD_COLON, FALSE);

			if (dwMyStyle & DIGS_DECIMAL)
				CheckDlgButton (hWnd, IDD_DECIMAL, TRUE);
			else
				CheckDlgButton (hWnd, IDD_DECIMAL, FALSE);

			if (dwMyStyle & DIGS_LED)
				CheckRadioButton (hWnd, IDD_LED, IDD_LCD, IDD_LED);
			else
				CheckRadioButton (hWnd, IDD_LED, IDD_LCD, IDD_LCD);

			GlobalUnlock (hCtlStyle);

			return TRUE;

		case WM_COMMAND:
			switch (wParam) {

				case IDD_ABOUT:
					{
						FARPROC	lpfnAboutDlgFn;

						lpfnAboutDlgFn = MakeProcInstance (AboutDlgProc, hInstance);
						DialogBox (hInstance, "About", hWnd, lpfnAboutDlgFn);
						FreeProcInstance (lpfnAboutDlgFn);
					}
					break;

				case IDD_LED:
				case IDD_LCD:
					CheckRadioButton (hWnd, IDD_LED, IDD_LCD, wParam);
					break;

				case IDOK:

					lpCtlStyle = (LPCTLSTYLE) GlobalLock (hCtlStyle);

					GetDlgItemText (hWnd, IDD_CTLID, szTemp, sizeof (szTemp));
					dwRC = lpfnStr2ID (szTemp);

					if (LOWORD (dwRC)) 
						lpCtlStyle->wId = HIWORD (dwRC);
					else {
						GlobalUnlock (hCtlStyle);
						break;
					}					
				
					dwMyStyle = 0;
					if (IsDlgButtonChecked (hWnd, IDD_BLANK))
						dwMyStyle |= DIGS_BLANK;
					else
						dwMyStyle &= ~DIGS_BLANK;

					if (IsDlgButtonChecked (hWnd, IDD_COLON))
						dwMyStyle |= DIGS_COLON;
					else
						dwMyStyle &= ~DIGS_COLON;

					if (IsDlgButtonChecked (hWnd, IDD_DECIMAL))
						dwMyStyle |= DIGS_DECIMAL;
					else
						dwMyStyle &= ~DIGS_DECIMAL;

					if (IsDlgButtonChecked (hWnd, IDD_LED))
						dwMyStyle |= DIGS_LED;
					else
						dwMyStyle &= ~DIGS_LED;

					lpCtlStyle->dwStyle &= 0xffff0000;
					lpCtlStyle->dwStyle |= dwMyStyle;

					GlobalUnlock (hCtlStyle);

					EndDialog (hWnd, TRUE);
					break;
					
				case IDCANCEL:
					EndDialog (hWnd, FALSE);
					break;
			}
			return TRUE;

		case WM_CLOSE:
			EndDialog (hWnd, FALSE);
			return TRUE;
	}
	return FALSE;
}

//------------------------------------------------------------------------
// DigitFlags - Translates the class style flags provided into a
//              corresponding text string for the .RC file.
//------------------------------------------------------------------------
WORD FAR PASCAL DigitFlags (DWORD dwFlags, LPSTR lpStyle, WORD wMaxString) {

	WORD	wChars, i, sLen;

	lpStyle[0] = 0;
	wChars = 0;

	for (i = 0; i < NUMSTYLES; i++) {
		if (dwFlags & ssStyleTags[i].dwStyle) {
			sLen = strlen (ssStyleTags[i].szLabel);
			if (wChars) {
				if (wMaxString > wChars + sLen + 3) {
					lstrcat (lpStyle, " | ");
					lstrcat (lpStyle, ssStyleTags[i].szLabel);
					sLen += 3;
				} else {
					return 0;
				}
			} else {
				if (wMaxString > wChars + sLen) {
					lstrcpy (lpStyle, ssStyleTags[i].szLabel);
				} else {
					return 0;
				}
			}
			wChars += sLen;
		}
	}
	return lstrlen (lpStyle);
}

//------------------------------------------------------------------------
//Routines used to draw LED segments
//------------------------------------------------------------------------
void DrawTop (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*3 + wPenWidth, yUnit*1 + wPenWidth);
	LineTo (hdc, xUnit*6 + wPenWidth, yUnit*1 + wPenWidth);
}
void DrawMid (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*3 + wPenWidth, yUnit*6 + wPenWidth);
	LineTo (hdc, xUnit*6 + wPenWidth, yUnit*6 + wPenWidth);
}
void DrawBot (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*3 + wPenWidth, yUnit*11 + wPenWidth);
	LineTo (hdc, xUnit*6 + wPenWidth, yUnit*11 + wPenWidth);
}
void DrawRTop (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*7 + wPenWidth, yUnit*2 + wPenWidth);
	LineTo (hdc, xUnit*7 + wPenWidth, yUnit*5 + wPenWidth);
}
void DrawRBot (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*7 + wPenWidth, yUnit*7 + wPenWidth);
	LineTo (hdc, xUnit*7 + wPenWidth, yUnit*10 + wPenWidth);
}
void DrawLTop (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*2 + wPenWidth, yUnit*2 + wPenWidth);
	LineTo (hdc, xUnit*2 + wPenWidth, yUnit*5 + wPenWidth);
}
void DrawLBot (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	MoveTo (hdc, xUnit*2 + wPenWidth, yUnit*7 + wPenWidth);
	LineTo (hdc, xUnit*2 + wPenWidth, yUnit*10 + wPenWidth);
}
void DrawDecimal (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {

	Ellipse (hdc, xUnit-wPenWidth/2, yUnit*11-wPenWidth/2, 
	              xUnit+wPenWidth/2, yUnit*11+wPenWidth/2);
}
void DrawColon (HDC hdc, WORD xUnit, WORD yUnit, WORD wPenWidth) {
	Ellipse (hdc, xUnit-wPenWidth/2, yUnit*4-wPenWidth/2, 
	              xUnit+wPenWidth/2, yUnit*4+wPenWidth/2);
	Ellipse (hdc, xUnit-wPenWidth/2, yUnit*8-wPenWidth/2, 
	              xUnit+wPenWidth/2, yUnit*8+wPenWidth/2);
}

//------------------------------------------------------------------------
// DigitWndFn - The window proc for the control
//------------------------------------------------------------------------
long FAR PASCAL DigitWndFn (HWND hWnd, WORD message, WORD wParam, 
                            LONG lParam) {
	HANDLE 		hData;
	PDIGITDATA	pData;

	HDC			hdc;
	PAINTSTRUCT	ps;
 	RECT			rect;
	int			i;

	if (message != WM_CREATE) 
		hData = GetWindowWord (hWnd, 0);
		
	switch (message) {

		case WM_CREATE:
			hData = LocalAlloc (LHND, sizeof (DIGITDATA));
			pData = (PDIGITDATA) LocalLock (hData);
			pData->sValue = 0;
			pData->sBase = 10;
			SetWindowWord (hWnd, 0, hData);
			
//		   MessageBeep(0);
			
			LocalUnlock (hData);
			return 0;

		case DIGM_SET:
			pData = (PDIGITDATA) LocalLock (hData);

			if (pData->sValue != (int) wParam) {
				pData->sValue = wParam; 

				if (pData->sValue < 0) pData->sValue = 0;

				if (pData->sValue >= pData->sBase)
					pData->sValue = pData->sBase - 1;
				InvalidateRect (hWnd, NULL, FALSE);
			}
			LocalUnlock (hData);	
			return 0;
			
		case DIGM_GET:
			pData = (PDIGITDATA) LocalLock (hData);
			i = pData->sValue;
			LocalUnlock (hData);	
			return i;

		case DIGM_GETBASE:
			pData = (PDIGITDATA) LocalLock (hData);
			i = pData->sBase;
			LocalUnlock (hData);	
			return i;
				  			
		case DIGM_INC:
			pData = (PDIGITDATA) LocalLock (hData);
			pData->sValue++;
			if (pData->sValue >= pData->sBase) {
				pData->sValue = 0;
				PostMessage (GetParent (hWnd), WM_COMMAND,
				             GetWindowWord (hWnd, GWW_ID), 
								 MAKELONG (hWnd, DIGN_CARRY)); 
			}
			LocalUnlock (hData);
			InvalidateRect (hWnd, NULL, FALSE);
			return 0;

			
		case DIGM_DEC:
			pData = (PDIGITDATA) LocalLock (hData);
			pData->sValue--;
			if (pData->sValue < 0) {
				pData->sValue = pData->sBase - 1;
				PostMessage (GetParent (hWnd), WM_COMMAND,
				             GetWindowWord (hWnd, GWW_ID), 
								 MAKELONG (hWnd, DIGN_BORROW)); 
			}
			LocalUnlock (hData);
			InvalidateRect (hWnd, NULL, FALSE);
			return 0;


		case DIGM_SETBASE:
			pData = (PDIGITDATA) LocalLock (hData);
			if (wParam > 16)
				pData->sBase = 16;
			else
				if (wParam <= 0)
					pData->sBase = 2;
				else
					pData->sBase = wParam;
			pData->sValue %= pData->sBase;
			LocalUnlock (hData);
			return 0;


		case DIGM_BLANK:
			pData = (PDIGITDATA) LocalLock (hData);
			if (wParam) 
				SetWindowLong (hWnd, GWL_STYLE, 
				               GetWindowLong (hWnd, GWL_STYLE) | DIGS_BLANK);
			else
				SetWindowLong (hWnd, GWL_STYLE, 
				               GetWindowLong (hWnd, GWL_STYLE) & ~DIGS_BLANK);
			LocalUnlock (hData);
			InvalidateRect (hWnd, NULL, FALSE);
			return 0;



		case WM_LBUTTONUP:
			PostMessage (GetParent (hWnd), WM_COMMAND,
			             GetWindowWord (hWnd, GWW_ID), 
							 MAKELONG (hWnd, DIGN_LCLICK)); 
			break;


		case WM_RBUTTONUP:
			PostMessage (GetParent (hWnd), WM_COMMAND,
			             GetWindowWord (hWnd, GWW_ID), 
							 MAKELONG (hWnd, DIGN_RCLICK)); 
			break;


		case WM_PAINT:
			{
				HPEN		hPen;
				HBRUSH	hBrush;
				WORD		xUnit, yUnit, wPenWidth;

				pData = (PDIGITDATA) LocalLock (hData);
				hdc = BeginPaint (hWnd, &ps);
				GetWindowRect (hWnd, &rect);

				xUnit = (rect.right-rect.left)/8;
				yUnit = (rect.bottom-rect.top)/12;
				wPenWidth = max (2, (rect.right-rect.left)/32);

				if (GetWindowLong (hWnd, GWL_STYLE) & DIGS_LED) 
					hBrush = GetStockObject (BLACK_BRUSH);
				else 
					hBrush = GetStockObject (GRAY_BRUSH);

				SelectObject (hdc, GetStockObject (NULL_PEN));
				SelectObject (hdc, hBrush);
				SelectObject (hdc, GetStockObject (NULL_PEN));
				SelectObject (hdc, hBrush);
				Rectangle (hdc, 0, 0, rect.right, rect.bottom);

				//
				//If blank set, don't write number
				//
				if (GetWindowLong (hWnd, GWL_STYLE) & DIGS_BLANK) {
					EndPaint (hWnd, &ps);
					return 0;
				}		
				//
				//Get color for segments
				//
				if (GetWindowLong (hWnd, GWL_STYLE) & DIGS_LED) 
					hPen = CreatePen (PS_SOLID, wPenWidth, RGB (255, 0, 0));
				else									  
					hPen = CreatePen (PS_SOLID, wPenWidth, RGB (0,0,0));
				SelectObject (hdc, hPen);
				//
				//Draw colon or decimal if necessary
				//
				if (GetWindowLong (hWnd, GWL_STYLE) & DIGS_COLON) 
					DrawColon (hdc, xUnit, yUnit, wPenWidth);

				if (GetWindowLong (hWnd, GWL_STYLE) & DIGS_DECIMAL) 
					DrawDecimal (hdc, xUnit, yUnit, wPenWidth);
				//
				//Write proper segments for each number
				//
				switch (pData->sValue) {

					case 0:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 1:
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 2:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 3:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 4:
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 5:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 6:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 7:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 8:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 9:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 10:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 11:
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 12:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 13:
						DrawRTop (hdc, xUnit, yUnit, wPenWidth);
						DrawRBot (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 14:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;

					case 15:
						DrawTop (hdc, xUnit, yUnit, wPenWidth);
						DrawMid (hdc, xUnit, yUnit, wPenWidth);
						DrawLBot (hdc, xUnit, yUnit, wPenWidth);
						DrawLTop (hdc, xUnit, yUnit, wPenWidth);
						break;
				}
				DeleteObject (SelectObject (hdc, GetStockObject (BLACK_PEN)));
				EndPaint (hWnd, &ps);
			}
			LocalUnlock (hData);
			return 0;

		case WM_DESTROY:
			LocalFree (hData);
			break;
	}
	return DefWindowProc (hWnd, message, wParam, lParam);
}

//------------------------------------------------------------------------
// About box dialog procedure
//------------------------------------------------------------------------
BOOL FAR PASCAL AboutDlgProc (HWND hwnd, WORD message, WORD wParam, 
                               LONG lParam) {
	switch (message) {
		case WM_COMMAND:
			switch (wParam) {
				case IDOK:
					EndDialog (hwnd, 1);
					return TRUE;
			}
			break;

		case WM_CLOSE:
			EndDialog (hwnd, 0);
			return TRUE;
	}
	return FALSE;	
}

