/******************************************************************************
* SBzrEval.c - Bezier surfaces handling routines - evaluation routines.	      *
*******************************************************************************
* Written by Gershon Elber, Mar. 90.					      *
******************************************************************************/

#include "cagd_loc.h"

/******************************************************************************
* Evaluate the given tensor product Bezier surface at a given point, by       *
* extracting an isoparamteric curve along u and evaluating v in it.	      *
*									      *
*		u -->							      *
*     +----------------------+						      *
*     |P0		 Pi-1|						      *
*   V |Pi		P2i-1|	Parametric space orientation - control mesh.  *
*    ||			     |						      *
*    v|Pn-i		 Pn-1|						      *
*     +----------------------+						      *
*									      *
******************************************************************************/
CagdRType *BzrSrfEvalAtParam(CagdSrfStruct *Srf, CagdRType u, CagdRType v)
{
    CagdRType *Pt;
    CagdCrvStruct
	*IsoCrv = BzrSrfCrvFromSrf(Srf, u, CAGD_CONST_U_DIR);

    Pt = BzrCrvEvalAtParam(IsoCrv, v);

    CagdCrvFree(IsoCrv);

    return Pt;
}

/******************************************************************************
* Extract an isoline curve out of the given tensor product Bezier surface.    *
* Operations should prefer the CONST_U_DIR, in which the extraction is	      *
* somewhat faster if it is possible.					      *
******************************************************************************/
CagdCrvStruct *BzrSrfCrvFromSrf(CagdSrfStruct *Srf, CagdRType t,
							CagdSrfDirType Dir)
{
    CagdCrvStruct *Crv;
    CagdBType
	IsNotRational = !CAGD_IS_RATIONAL_SRF(Srf);
    int i, j, CrvOrder, VecLen,
	MaxCoord = CAGD_NUM_OF_PT_COORD(Srf -> PType);
    CagdRType *CrvP, *SrfP;

    switch (Dir) {
	case CAGD_CONST_U_DIR:
	    Crv = BzrCrvNew(CrvOrder = Srf -> VLength, Srf -> PType);
	    VecLen = Srf -> ULength;

	    for (i = IsNotRational; i <= MaxCoord; i++) {
		CrvP = Crv -> Points[i];
		SrfP = Srf -> Points[i];
		for (j = 0; j < CrvOrder; j++) {
		    *CrvP++ = BzrCrvEvalVecAtParam(SrfP, CAGD_NEXT_U(Srf),
								VecLen, t);
		    SrfP += CAGD_NEXT_V(Srf);
		}
	    }
	    break;
	case CAGD_CONST_V_DIR:
	    Crv = BzrCrvNew(CrvOrder = Srf -> ULength, Srf -> PType);
	    VecLen = Srf -> VLength;

	    for (i = IsNotRational; i <= MaxCoord; i++) {
		CrvP = Crv -> Points[i];
		SrfP = Srf -> Points[i];
		for (j = 0; j < CrvOrder; j++) {
		    *CrvP++ = BzrCrvEvalVecAtParam(SrfP, CAGD_NEXT_V(Srf),
								VecLen, t);
		    SrfP += CAGD_NEXT_U(Srf);
		}
	    }
	    break;
	default:
	    FATAL_ERROR(CAGD_ERR_DIR_NOT_CONST_UV);
	    break;
    }
    return Crv;
}

/******************************************************************************
* Extract an isoline curve out of the given mesh row/col.		      *
* The provided (zero based) Index specifies which row/col Index to extract.   *
******************************************************************************/
CagdCrvStruct *BzrSrfCrvFromMesh(CagdSrfStruct *Srf, int Index,
							CagdSrfDirType Dir)
{
    CagdCrvStruct *Crv;
    CagdBType
	IsNotRational = !CAGD_IS_RATIONAL_SRF(Srf);
    int i, j, CrvOrder,
	MaxCoord = CAGD_NUM_OF_PT_COORD(Srf -> PType);
    CagdRType *CrvP, *SrfP;

    switch (Dir) {
	case CAGD_CONST_U_DIR:
	    if (Index + 1 > Srf -> ULength)
		FATAL_ERROR(CAGD_ERR_INDEX_NOT_IN_MESH);

	    Crv = BzrCrvNew(CrvOrder = Srf -> VLength, Srf -> PType);

	    for (i = IsNotRational; i <= MaxCoord; i++) {
		CrvP = Crv -> Points[i];
		SrfP = Srf -> Points[i] + Index * CAGD_NEXT_U(Srf);
		for (j = 0; j < CrvOrder; j++) {
		    *CrvP++ = *SrfP;
		    SrfP += CAGD_NEXT_V(Srf);
		}
	    }
	    break;
	case CAGD_CONST_V_DIR:
	    if (Index + 1 > Srf -> VLength)
		FATAL_ERROR(CAGD_ERR_INDEX_NOT_IN_MESH);

	    Crv = BzrCrvNew(CrvOrder = Srf -> ULength, Srf -> PType);

	    for (i = IsNotRational; i <= MaxCoord; i++) {
		CrvP = Crv -> Points[i];
		SrfP = Srf -> Points[i] + Index * CAGD_NEXT_V(Srf);;
		for (j = 0; j < CrvOrder; j++) {
		    *CrvP++ = *SrfP;
		    SrfP += CAGD_NEXT_U(Srf);
		}
	    }
	    break;
	default:
	    FATAL_ERROR(CAGD_ERR_DIR_NOT_CONST_UV);
	    break;
    }
    return Crv;
}

