/*-------------------------------------------------------------------------

  display.c - Screen display code for the moder.lst viewing program.

  Copyright (c) 1993 Rhys Weatherley

  Permission to use, copy, modify, and distribute this material
  for any purpose and without fee is hereby granted, provided
  that the above copyright notice and this permission notice
  appear in all copies, and that the name of Rhys Weatherley not be
  used in advertising or publicity pertaining to this
  material without specific, prior written permission.
  RHYS WEATHERLEY MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR
  SUITABILITY OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED
  "AS IS", WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.

  Revision History:
  ================

   Version  DD/MM/YY  By  Description
   -------  --------  --  --------------------------------------
     1.0    08/09/93  RW  Original Version of display.c

-------------------------------------------------------------------------*/

#ifdef	__TURBOC__
#ifndef	MSDOS
#define	MSDOS
#endif
  #pragma warn -pro
#endif

#include "display.h"

#ifdef	MSDOS

/* Use Borland C++'s conio routines to access the screen */
#include <conio.h>
#include <bios.h>
#include <string.h>
#define	DL_CLEAR()	(textattr (normalattr), clrscr ())
#define	DL_NORMAL()	(textattr (normalattr))
#define	DL_INVERSE()	(textattr (inverseattr))
#define	DL_TITLE()	(textattr (titleattr))
#define	DL_GOTOXY(x,y)	(gotoxy ((x)+1, (y)+1))
#define	DL_PUTCH(c)	(putch ((c)))
#define	DL_CLREOL()	(clreol ())
#define	DL_INSLINE()	(insline ())
#define	DL_DELLINE()	(delline ())
#define	DL_REFRESH()
#define	DL_KEY_UP	case 0x4800:
#define	DL_KEY_DOWN	case 0x5000:
#define	DL_KEY_LEFT	case 0x4B00:
#define	DL_KEY_RIGHT	case 0x4D00:
#define	DL_KEY_HOME	case 0x4700:
#define	DL_KEY_END	case 0x4F00:
#define	DL_KEY_PGUP	case 0x4900:
#define	DL_KEY_PGDN	case 0x5100:

#else	/* !MSDOS */

/* Use the Unix curses library to access the screen */
#include <curses.h>
#define	DL_CURSES
static	WINDOW	*dl_screen;
#define	DL_CLEAR()	(wstandend (dl_screen), wclear (dl_screen))
#define	DL_NORMAL()	(wstandend (dl_screen))
#define	DL_INVERSE()	(wstandout (dl_screen))
#define	DL_TITLE()	(standout ())
#define	DL_GOTOXY(x,y)	(wmove (dl_screen, (y), (x)))
#define	DL_PUTCH(c)	(waddch (dl_screen, (c)))
#define	DL_CLREOL()	(wclrtoeol (dl_screen))
#define	DL_INSLINE()	(winsertln (dl_screen))
#define	DL_DELLINE()	(wdeleteln (dl_screen))
#define	DL_REFRESH()	(wrefresh (dl_screen))
#define	DL_KEY_UP
#define	DL_KEY_DOWN
#define	DL_KEY_LEFT
#define	DL_KEY_RIGHT
#define	DL_KEY_HOME
#define	DL_KEY_END
#define	DL_KEY_PGUP
#define	DL_KEY_PGDN

#endif	/* !MSDOS */

#define	DL_CTRL(x)	((x) & 0x1F)

/*
 * Global data for this module.
 */
static	int	width, height;
#ifdef	MSDOS
static	int	normalattr, inverseattr, titleattr, boxattr;
static	int	oldattr, oldmode;
#endif	/* MSDOS */
#ifdef	DL_CURSES
static	char	bottomline[BUFSIZ];
#endif	/* DL_CURSES */

/*
 * Initialise the display device.
 */
void	InitDisplay ()
{
#ifdef	MSDOS
  struct text_info ti;
  gettextinfo (&ti);
  oldattr = ti.attribute;
  oldmode = ti.currmode;
  if (ti.currmode == BW40 || ti.currmode == BW80)
    {
      ti.currmode = BW80;
      normalattr = 0x0F;
      inverseattr = 0x70;
      titleattr = 0x70;
      boxattr = 0x07;
    } /* then */
   else if (ti.currmode == MONO)
    {
      normalattr = 0x07;
      inverseattr = 0x70;
      titleattr = 0x70;
      boxattr = 0x0F;
    } /* then */
   else
    {
      ti.currmode = C80;
      normalattr = 0x1F;
      inverseattr = 0x20;
      titleattr = 0x74;
      boxattr = 0x1E;
    } /* else */
  textmode (ti.currmode);
  textattr (normalattr);
  clrscr ();
  gettextinfo (&ti);
  width = ti.screenwidth;
  height = ti.screenheight;
  _wscroll = 0;
  _setcursortype (_NOCURSOR);
  window (1, 2, width, height - 1);
  height -= 2;
#endif	/* MSDOS */
#ifdef	DL_CURSES
  initscr ();
  dl_screen = subwin (stdscr, LINES - 2, COLS, 1, 0);
  if (dl_screen == NULL)
    {
      endwin ();
      fprintf (stderr, "Could not create subwindow for list display purposes\n");
      exit (1);
    } /* if */
  idlok (dl_screen, TRUE);
  scrollok (dl_screen, FALSE);
  nonl ();
  cbreak ();
  noecho ();
  width = COLS;
  height = LINES - 2;
#endif	/* DL_CURSES */
} /* InitDisplay */

/*
 * Clean up the display device.
 */
void	TermDisplay ()
{
#ifdef	MSDOS
  textmode (oldmode);
  textattr (oldattr);
  clrscr ();
  _setcursortype (_NORMALCURSOR);
#endif	/* MSDOS */
#ifdef	DL_CURSES
  delwin (dl_screen);
  clear ();
  refresh ();
  nl ();
  nocbreak ();
  echo ();
  endwin ();
#endif	/* DL_CURSES */
} /* TermDisplay */

/*
 * Initialise a display list.
 */
void	InitList (list, param, draw, num, title, titlelen, fileonly)
DisplayList *list;
void	*param;
void	(*draw) ();
int	num, titlelen, fileonly;
DrawPtr	title;
{
  list -> param = param;
  list -> draw = draw;
  list -> num = num;
  list -> title = title;
  list -> titlelen = titlelen;
  list -> fileonly = fileonly;
  list -> current = 0;
  list -> top = 0;
} /* InitList */

/*
 * Draw a list of characters.  This is called by a
 * list's "draw" function.
 */
void	DrawChars (str, len)
DrawPtr	str;
int	len;
{
#ifdef	MSDOS
  int x = wherex () - 1;
#endif	/* MSDOS */
#ifdef	DL_CURSES
  int x, y;
  getyx (dl_screen, y, x);
#endif	/* DL_CURSES */
  if (len >= (width - x))
    len = width - x - 1;
  while (len-- > 0)
    DL_PUTCH (*str++);
} /* DrawChars */

/*
 * Draw a single item in the list.
 */
static	void	DrawItem (list, item)
DisplayList	*list;
int		item;
{
  DL_GOTOXY (0, item - list -> top);
  if (item == list -> current && !(list -> fileonly))
    DL_INVERSE ();
   else
    DL_NORMAL ();
  if (item < list -> num)
    {
      if (!(list -> fileonly))
        DL_PUTCH (' ');
      (*(list -> draw)) (list, list -> param, item);
    } /* if */
  DL_CLREOL ();
} /* DrawItem */

/*
 * Show a screen-full of list items.
 */
static	void	ShowList (list)
DisplayList	*list;
{
  int ht = height;
  int current = list -> top;
  int y;
  if (ht > (list -> num - current))
    ht = list -> num - current;
  DL_CLEAR ();
  for (y = 1; y <= ht; ++y)
    DrawItem (list, current++);
  DL_REFRESH ();
} /* ShowList */

/*
 * Move the selection bar to a new item.
 */
static	void	MoveItem (list, item, page)
DisplayList	*list;
int		item, page;
{
  int oldcurrent;
  if (item < 0)
    item = 0;
  if (item >= list -> num)
    item = list -> num - 1;
  if (list -> current == item)
    return;
  oldcurrent = list -> current;
  list -> current = item;
  DrawItem (list, oldcurrent);
  if (item >= list -> top && item < (list -> top + height))
    DrawItem (list, item);
   else if (item == list -> top - 1 && !page)
    {
      /* Scroll the screen down to add another line at the top */
      DL_GOTOXY (0, 0);
      DL_INSLINE ();
      --(list -> top);
      DrawItem (list, item);
    } /* then */
   else if (item == (list -> top + height) && !page)
    {
      /* Scroll the screen up to add another line at the bottom */
      DL_GOTOXY (0, 0);
      DL_DELLINE ();
      ++(list -> top);
      DrawItem (list, item);
    } /* then */
   else if (item < list -> top)
    {
      /* Scroll a full page up, or go right to the start */
      if (item == 0)
        list -> top = 0;
       else
        {
          list -> top -= height;
          if (list -> top < 0)
            list -> top = 0;
	} /* else */
      ShowList (list);
    } /* then */
   else
    {
      /* Scroll a full page down, or go right to the end */
      if (item == (list -> num - 1) && !page)
        {
	  list -> top = list -> num - height;
	  if (list -> top < 0)
	    list -> top = 0;
	} /* then */
       else
        {
	  list -> top += height;
	  if (list -> top >= list -> num)
	    list -> top = list -> num - 1;
	} /* else */
      ShowList (list);
    } /* else */
  DL_REFRESH ();
} /* MoveItem */

/*
 * Move the whole list up or down as a file is
 * being displayed rather than a selection list.
 */
static	void	MoveFile (list, newtop)
DisplayList	*list;
int		newtop;
{
  if (newtop < 0)
    newtop = 0;
  if (newtop >= list -> num)
    newtop = list -> num - 1;
  if (newtop == list -> top)
    return;
  if (newtop == list -> top - 1)
    {
      DL_GOTOXY (0, 0);
      DL_INSLINE ();
      --(list -> top);
      DrawItem (list, newtop);
    } /* then */
   else if (newtop == list -> top + 1)
    {
      DL_GOTOXY (0, 0);
      DL_DELLINE ();
      ++(list -> top);
      DrawItem (list, list -> top + height - 1);
    } /* then */
   else
    {
      list -> top = newtop;
      ShowList (list);
    } /* else */
  list -> current = list -> top;
  DL_REFRESH ();
} /* MoveFile */

/*
 * Move through a display list by a delta amount.
 */
static	void	MoveDelta (list, delta, page)
DisplayList	*list;
int		delta, page;
{
  if (list -> fileonly)
    MoveFile (list, list -> top + delta);
   else
    MoveItem (list, list -> current + delta, page);
} /* MoveDelta */

#ifdef	DL_CURSES

/*
 * Clear an entire line using the current attribute.  This is
 * needed because "clrtoeol()" always uses the normal attribute,
 * but we want to clear an entire line with the standout attribute.
 */
static	void	ClearLine ()
{
  int wid = COLS - 1;
  while (wid-- > 0)
    addch (' ');
} /* ClearLine */

#endif	/* DL_CURSES */

/*
 * Display a list of items and wait for a selection.
 * Returns -1 if the list was exited.
 */
int	SelectDisplay (list)
DisplayList *list;
{
  int key, quit;
#ifdef	MSDOS
  window (1, 1, width, height + 2);
  DL_GOTOXY (0, 0);
  DL_TITLE ();
  DL_CLREOL ();
  DL_GOTOXY ((width - list -> titlelen) / 2, 0);
  DrawChars (list -> title, list -> titlelen);
  window (1, 2, width, height + 1);
#endif	/* MSDOS */
#ifdef	DL_CURSES
  char *temp;
  move (0, 0);
  DL_TITLE ();
  ClearLine ();
  addch (' ');
  move (0, (COLS - list -> titlelen) / 2);
  key = list -> titlelen;
  temp = list -> title;
  while (key-- > 0)
    addch (*temp++);
  refresh ();
#endif	/* DL_CURSES */
  ShowList (list);
  quit = 0;
  while (!quit)
    {
#ifdef	MSDOS
      key = bioskey (0);
      if ((key & 255) != 0)
        key &= 255;
#endif	/* MSDOS */
#ifdef	DL_CURSES
      move (LINES - 1, COLS - 1);
      refresh ();
      key = getchar ();
#endif	/* DL_CURSES */
      switch (key)
        {
	  case '?':	return (-2);
	  case 0x0D: case ' ':
	  		if (!(list -> fileonly))
	  		  return (list -> current);
			if (key == 0x0D)
			  MoveDelta (list, 1, 0);
			 else
			  MoveDelta (list, height, 1);
			break;
#ifndef	DL_CURSES
	  /* Esc is too much of a hassle in curses, so leave it out */
	  case 0x1B:
#endif	/* DL_CURSES */
	  case 'q': case 'Q': case 'x': case 'X':
	  		quit = 1; break;
	  case 'j': case 'J': case DL_CTRL('J'): case '+': case '=':
	  DL_KEY_DOWN DL_KEY_RIGHT
	  		MoveDelta (list, 1, 0); break;
	  case 'k': case 'K': case DL_CTRL('K'): case '-': case '_':
	  DL_KEY_UP DL_KEY_LEFT
	  		MoveDelta (list, -1, 0);  break;
	  case 'h': case 'H': case '^': DL_KEY_HOME
	  		MoveDelta (list, -(list -> current), 0); break;
	  case 'e': case 'E': case '$': DL_KEY_END
	  		MoveDelta (list, list -> num - list -> current - 1, 0);
			break;
	  case 'u': case 'U': case 'b': case 'B': case DL_CTRL('U'):
	  case DL_CTRL('B'): DL_KEY_PGUP
	  		MoveDelta (list, -height, 1); break;
	  case 'd': case 'D': case 'f': case 'F': case DL_CTRL('D'):
	  case DL_CTRL('F'): DL_KEY_PGDN
	  		MoveDelta (list, height, 1); break;
#ifdef	DL_CURSES
	  case DL_CTRL('R'): case DL_CTRL('L'):
			wrefresh (curscr); break;
#endif	/* DL_CURSES */
	} /* switch */
    } /* while */
  return (-1);
} /* SelectDisplay */

/*
 * Set the bottom line of the screen to a certain string.
 */
void	SetBottomLine (str)
char	*str;
{
#ifdef	MSDOS
  window (1, 1, width, height + 2);
  DL_GOTOXY (0, height + 1);
  DL_TITLE ();
  DL_CLREOL ();
  DL_GOTOXY ((width - strlen (str)) / 2, height + 1);
  cputs (str);
  window (1, 2, width, height + 1);
#endif	/* MSDOS */
#ifdef	DL_CURSES
  if (str != bottomline)
    strcpy (bottomline, str);
  move (LINES - 1, 0);
  DL_TITLE ();
  ClearLine ();
  move (LINES - 1, (COLS - strlen (str)) / 2);
  addstr (str);
  refresh ();
#endif	/* DL_CURSES */
} /* SetBottomLine */

/*
 * Report an error to the screen and wait for a keypress.
 */
void	ReportError (msg, name, namelen)
char	*msg;
DrawPtr	name;
int	namelen;
{
#ifdef	MSDOS
  int boxwid = namelen + strlen (msg) + 4;
  int x1 = ((width - boxwid) / 2) + 1;
  int y1 = ((height + 2) / 2) - 2;
  int x2 = x1 + boxwid - 1;
  int y2 = y1 + 4;
  int temp;
  window (x1, y1, x2, y2);
  DL_NORMAL ();
  clrscr ();
  window (1, 1, width, height + 2);
  textattr (boxattr);
  gotoxy (x1, y1);
  putch (213);
  for (temp = x1 + 1; temp < x2; ++temp)
    putch (205);
  putch (184);
  for (temp = y1 + 1; temp < y2; ++temp)
    {
      gotoxy (x1, temp);
      putch (179);
      gotoxy (x2, temp);
      putch (179);
    } /* for */
  gotoxy (x1, y2);
  putch (212);
  for (temp = x1 + 1; temp < x2; ++temp)
    putch (205);
  putch (190);
  DL_NORMAL ();
  gotoxy (x1 + 2, y1 + 2);
  cputs (msg);
  while (namelen-- > 0)
    putch (*name++);
  bioskey (0);
  window (1, 2, width, height + 1);
#endif	/* MSDOS */
#ifdef	DL_CURSES
  move (LINES - 1, 0);
  DL_TITLE ();
  ClearLine ();
  move (LINES - 1, 0);
  addch (' ');
  addstr (msg);
  while (namelen-- > 0)
    addch (*name++);
  addstr (" [Press any key]");
  move (LINES - 1, COLS - 1);
  refresh ();
  getchar ();
  SetBottomLine (bottomline);
#endif	/* DL_CURSES */
} /* ReportError */
