/* $ vesa.cpp 29/06/99 21:24 $
 * VESA VBE functions. Revision 1.0
 *
 * Copyright (C) 1999  Dmitry Uvarov <mit@pc2o205a.pnpi.spb.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <i86.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "..\include\pmode.h"
#include "..\include\vesa.h"

/* global variables */
dword VBEMemory;
word  VBEVersion;
word  VBE_MMIOSel = 0;
/* internal variables */
static ibool        haveRiva128;

static word         rmbuf_sel=0;      /* real mode memory block selector */
static word         rmbuf_seg=0;      /* real mode memory block segment  */
static char*        rmbuf_ptr=NULL;   /* pointer to real mode memory block */
static word         rmbuf_size=8192;  /* size of real mode memory block  */
static VBE_pmInfo   *pmInfo=NULL;

#ifdef __VBE__DEBUG__
static char*  errorstr = "You MUST call VBE_Init() before you can call the VESA.CPP module!";
static char*  errorstr2= "You MUST call VBE_initPMCode before attempt to get pointer to PM code";
#endif
/* Main procedures */
/***************************************************************************
*                                                                          *
* Function: _CorrectInternalPointer;                                       *
* Status: internal                                                         *
* Description: convert pointer based on old buffer to new buffer location  *
*                                                                          *
***************************************************************************/
void _CorrectInternalPointer(dword *p, const void *oldbuf, const void *newbuf, int size)
  {
     dword diff = *p - (unsigned)oldbuf;
     /* if pointer is outside the old buffer then just exit */
     if (diff > size) return;
     /* we must currect it... */
     *p = (unsigned)newbuf + diff;
  }

/***************************************************************************
*                                                                          *
* Function: _vbe_exit;                                                     *
* Status: internal                                                         *
* Description: deinitiates VBE library. Frees allocated real mode memory   *
*                                                                          *
***************************************************************************/
void _vbe_exit()
  {
     if (rmbuf_sel) PM_FreeDOSMemory(rmbuf_sel);
  }

/***************************************************************************
*                                                                          *
* Function: VBE_init();                                                    *
* Description: Initiates VESA module. Allocates real mode memory block     *
*              and init VBEInfo structure.                                 *
* Returns:  TRUE if initialization succeed. FALSE overwise                 *
*                                                                          *
***************************************************************************/
ibool  VBE_init()
  {
     /* return if library already initiated */
     if (rmbuf_sel) return TRUE;
     /* allocate 8K of DOS real memory */
     if (!PM_AllocateDOSMemory(rmbuf_size, &rmbuf_sel, &rmbuf_seg)) return FALSE;
     rmbuf_ptr = (char*) RMSegToFlat(rmbuf_seg);

     /* get VBE information */
     VBE_vbeInfo VBEInfo;
     if (!VBE_getVBEInfo(&VBEInfo)) {
        PM_FreeDOSMemory(rmbuf_sel); rmbuf_sel = 0;
#ifdef __VBE__DEBUG__
        printf("VBE_init(): VBE call failed. Possibly no VBE installed\n");
#endif
        return FALSE;
     }
     VBEMemory = VBEInfo.TotalMemory*65536;
     VBEVersion = VBEInfo.VbeVersion;
     /* Check for Riva128 based cards since they have broken triple buffering
      * and stereo support.
      */
     haveRiva128 = ((VBEVersion >= 0x300) &&
                    ((strstr(VBEInfo.OemStringPtr,"NVidia") == 0) ||
                     (strstr(VBEInfo.OemStringPtr,"Riva")   == 0))  );

     /* register deinitialization routine */
     atexit(_vbe_exit);
     return TRUE;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getRmBuf;                                                  *
* Description: returns size and pointer to allocated real mode memory block*
*              so it can be shared with other applications                 *
* Returns:  size - size of block in bytes                                  *
*           sel  - protected mode selector to the buffer                   *
*           seg  - real mode segment to the buffer                         *
*                                                                          *
***************************************************************************/
void   VBE_getRmBuf(long *size, word *sel, word *rseg)
  {
     *size = rmbuf_size;
     *sel  = rmbuf_sel;
     *rseg = rmbuf_seg;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getVBEInfo;                                                *
* Description: returns block of VESA BIOS information                      *
* Returns:  VBE info block                                                 *
*           TRUE if successful                                             *
*                                                                          *
***************************************************************************/
ibool  VBE_getVBEInfo(VBE_vbeInfo *vbeInfo)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     memset(rmbuf_ptr,0,sizeof(VBE_vbeInfo));
     /* we have only 32bit fields in RMREGS structure... */
     r.eax = vbeGetVBEInfo;
     r.es  = rmbuf_seg;
     /* force VBE to give VESA 2.0 extended info */
     *((dword*)rmbuf_ptr) = vbeVESA20_SIG;
     PM_RealModeInt(vbeInt, &r);
     memcpy(vbeInfo, rmbuf_ptr, sizeof(VBE_vbeInfo));

     if (((r.eax & 0xFFFF) != vbeSuccess) ||
        (*((dword*)&vbeInfo->VbeSignature) != vbeVESA_SIG)) return FALSE;

     /* convert all real mode pointers to flat pointers */
     vbeInfo->OemStringPtr =      (char*)RMToFlat(vbeInfo->OemStringPtr);
     vbeInfo->VideoModePtr =      (word*)RMToFlat(vbeInfo->VideoModePtr);
     vbeInfo->OemVendorNamePtr =  (char*)RMToFlat(vbeInfo->OemVendorNamePtr);
     vbeInfo->OemProductNamePtr = (char*)RMToFlat(vbeInfo->OemProductNamePtr);
     vbeInfo->OemProductRevPtr =  (char*)RMToFlat(vbeInfo->OemProductRevPtr);

     /* By VESA Specification extended data (such as vendor name) can be
     *  stored inside Vesa Bios or inside buffer with VBE information
     *  If so, we must re-order pointers to new location
     */

     _CorrectInternalPointer((dword*)&vbeInfo->OemStringPtr,      (void*)rmbuf_ptr, (void*)vbeInfo, sizeof(VBE_vbeInfo));
     _CorrectInternalPointer((dword*)&vbeInfo->VideoModePtr,      (void*)rmbuf_ptr, (void*)vbeInfo, sizeof(VBE_vbeInfo));
     _CorrectInternalPointer((dword*)&vbeInfo->OemVendorNamePtr,  (void*)rmbuf_ptr, (void*)vbeInfo, sizeof(VBE_vbeInfo));
     _CorrectInternalPointer((dword*)&vbeInfo->OemProductNamePtr, (void*)rmbuf_ptr, (void*)vbeInfo, sizeof(VBE_vbeInfo));
     _CorrectInternalPointer((dword*)&vbeInfo->OemProductRevPtr,  (void*)rmbuf_ptr, (void*)vbeInfo, sizeof(VBE_vbeInfo));

     /* The Boca Vortex VBE implementation returns a version number of
     *  2.2, but it's really only VBE 1.2 (2.2 is the version number of
     *  their VBE BIOS!)
     */
     if ((vbeInfo->VbeVersion > 0x200) &&
         (strcmp(vbeInfo->OemStringPtr,"VORTEX")==0))
            vbeInfo->VbeVersion = 0x102;

     return TRUE;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getModeInfo;                                               *
* Description: returns mode info block provided by VBE interface           *
* Returns:  VBE mode info block                                            *
*           TRUE if successful                                             *
* NOTE: WinFuncPtr does not mapped to flat memory.                         *
*                                                                          *
***************************************************************************/
ibool  VBE_getModeInfo(int mode, VBE_modeInfo *modeInfo)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     memset(rmbuf_ptr,0,sizeof(VBE_modeInfo));
     r.eax = vbeGetModeInfo;
     r.ecx = mode;
     r.es  = rmbuf_seg;
     PM_RealModeInt(vbeInt, &r);
     memcpy(modeInfo, rmbuf_ptr, sizeof(VBE_modeInfo));

     if (((r.eax & 0xFFFF) != vbeSuccess) ||
        ((modeInfo->ModeAttributes & vbeModeAvailable) == 0))
           return FALSE;

     /* Map out triple buffer and stereo flags for NVidia Riva128
      * chips.
      */
     if (haveRiva128) {
          modeInfo->ModeAttributes &= ~vbeModeTripleBuf;
          modeInfo->ModeAttributes &= ~vbeModeStereo;
     }

     /* Convert the 32k direct color modes of VBE 1.2+ BIOSes to
      * be recognised as 15 bits per pixel modes.
      */
     if ((modeInfo->BitsPerPixel == 16) && (modeInfo->RsvdMaskSize == 1))
          modeInfo->BitsPerPixel = 15;
     return TRUE;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_setVideoMode;                                              *
* Description: sets vesa video mode by calling vesa function               *
* Returns:  TRUE if successful                                             *
*                                                                          *
***************************************************************************/
ibool  VBE_setVideoMode(int mode)
  {
#ifdef __VBE__DEBUG__
     if (mode & vbeRefreshCtrl) {
        printf("VBE_setVideoMode(): modes with refresh control enabled must be set\n"
               "                    using VBE_setVideoModeExt()\n");
        exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     if ((mode & vbeModeMask) < 0x100) {
      /* VBE implementations barf terribly if you try to set non-VBE
       * video modes with the VBE set mode call.
       */
        r.eax = (mode & 0xFF) | ((mode & vbeDontClear) >> 8);
        PM_RealModeInt(0x10,&r);
        return TRUE;
     }
     /* set normal VBE mode */
     r.eax = vbeSetMode;
     r.ebx = mode;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getVideoMode;                                              *
* Description: returns current video mode                                  *
* Returns:  initiated video mode number                                    *
*                                                                          *
***************************************************************************/
int    VBE_getVideoMode()
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeGetMode;
     PM_RealModeInt(vbeInt, &r);
     return ((r.eax & 0xFFFF) == vbeSuccess) ? (r.ebx & 0xFFFF) : 0;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_saveState;                                                 *
* Description: save current Svga state and return pointer to allocated     *
*              state buffer. By default save all states                    *
* Returns:  pointer to state buffer. NULL if failed                        *
*                                                                          *
***************************************************************************/
void*  VBE_saveState(word states)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeSaveRestoreState;
     r.ecx = states;
     // get state buffer size first
     r.edx = vbeStGetSize;
     PM_RealModeInt(vbeInt, &r);
     if ((r.eax & 0xFFFF) != vbeSuccess) return NULL;
     int size = (r.ebx & 0xFFFF) * 64;  // get amount of bytes needed
     if (!size) return NULL; // this might never happen
     // now request state information
     r.eax = vbeSaveRestoreState;
     r.ecx = states;
     // get state buffer size first
     r.edx = vbeStStore;
     r.es  = rmbuf_seg;  // es:bx points to state buffer
     r.ebx = 0;
     PM_RealModeInt(vbeInt, &r);
     if ((r.eax & 0xFFFF) != vbeSuccess) return NULL;
     // now allocate memory for state buffer + one more word to store buffer
     // size
     char* buf = new char[size+2];
     if (!buf) return NULL;       // if allocation was failed...
     *((word*)buf) = size;        // store buffer size
     memcpy(buf+2,rmbuf_ptr,size);  // copy information to state buffer
     // return pointer to state buffer
     return (void*)buf;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_restoreState;                                              *
* Description: restore saved by function VBE_saveState Svga state from     *
*              state buffer. Also dispose allocated memory for state buffer*
* Returns:  TRUE if successful                                             *
*                                                                          *
***************************************************************************/
ibool  VBE_restoreState(void* buf)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
     if (!buf) {
          printf("VBE_restoreState(): you must specify non-zero pointer to state buffer\n");
          exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     // get buffer size
     int size = *((word*)(buf));
     // copy state buffer to real mode memory block
     memcpy(rmbuf_ptr,(word*)(buf)+1,size);
     r.eax = vbeSaveRestoreState;
     r.edx = vbeStRestore;
     r.es  = rmbuf_seg;
     r.ebx = 0;
     PM_RealModeInt(vbeInt, &r);
     if ((r.eax & 0xFFFF) != vbeSuccess) return FALSE;
     // now dispose memory block allocated for state buffer
     delete buf;
     return TRUE;
  };

/***************************************************************************
*                                                                          *
* Function: VBE_setBank;                                                   *
* Description: map specified bank to memory window. By default maps to     *
*              window A (if device support multiply I/O windows)           *
* Returns:  TRUE if successful                                             *
*                                                                          *
***************************************************************************/
/* exported to vesa.h */

/***************************************************************************
*                                                                          *
* Function: VBE_getBank;                                                   *
* Description: returns current window position. By default return position *
*              of window A (if device support multiply I/O windows)        *
* Returns:  current mapped window position                                 *
*                                                                          *
***************************************************************************/
/* exported to vesa.h */

/***************************************************************************
*                                                                          *
* Function: VBE_setPixelsPerScanline;                                      *
* Description: sets logical scanline width.                                *
* Returns:  TRUE if successful                                             *
*           number of pixels initiated for scanline                        *
*           number of bytes per scanline                                   *
*           maximum scan lines number                                      *
*                                                                          *
***************************************************************************/
ibool  VBE_setPixelsPerScanline(int pixelsPerLine, int *newBytes, int *newPixels, int *maxScanLines)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeSetLogicalSize;
     r.ebx = vbeSetInPixels;
     r.ecx = pixelsPerLine;
     PM_RealModeInt(vbeInt, &r);
     if (newBytes)  *newBytes  = r.ebx & 0xFFFF;
     if (newPixels) *newPixels = r.ecx & 0xFFFF;
     if (maxScanLines) *maxScanLines = r.edx & 0xFFFF;
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_setBytesPerScanline;                                       *
* Description: sets logical scanline width.                                *
* Returns:  TRUE if successful                                             *
*           number of pixels initiated for scanline                        *
*           number of bytes per scanline                                   *
*           maximum scan lines number                                      *
*                                                                          *
***************************************************************************/
ibool  VBE_setBytesPerScanline (int  bytesPerLine, int *newBytes, int *newPixels, int *maxScanLines)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeSetLogicalSize;
     r.ebx = vbeSetInBytes;
     r.ecx = bytesPerLine;
     PM_RealModeInt(vbeInt, &r);
     if (newBytes)  *newBytes  = r.ebx & 0xFFFF;
     if (newPixels) *newPixels = r.ecx & 0xFFFF;
     if (maxScanLines) *maxScanLines = r.edx & 0xFFFF;
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getScanlineLength;                                         *
* Description: returns scan line length.                                   *
* Returns:  TRUE if successful                                             *
*           number of pixels initiated for scanline                        *
*           number of bytes per scanline                                   *
*           maximum scan lines number                                      *
*                                                                          *
***************************************************************************/
ibool  VBE_getScanlineLength(int *bytesPerLine,int *pixelsPerLine, int *maxScanlines)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeSetLogicalSize;
     r.ebx = vbeGetLength;
     PM_RealModeInt(vbeInt, &r);
     if (bytesPerLine)  *bytesPerLine  = r.ebx & 0xFFFF;
     if (pixelsPerLine) *pixelsPerLine = r.ecx & 0xFFFF;
     if (maxScanlines) *maxScanlines = r.edx & 0xFFFF;
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getMaxScanLineLength;                                      *
* Description: returns maximum scan line length.                           *
* Returns:  TRUE if successful                                             *
*           maximum number of pixels initiated for scanline                *
*           maximum number of bytes per scanline                           *
*                                                                          *
***************************************************************************/
ibool  VBE_getMaxScanlineLength(int *maxBytes,int *maxPixels)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeSetLogicalSize;
     r.ebx = vbeGetMaxLength;
     PM_RealModeInt(vbeInt, &r);
     if (maxBytes)  *maxBytes  = r.ebx & 0xFFFF;
     if (maxPixels) *maxPixels = r.ecx & 0xFFFF;
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_setDisplayStart;                                           *
* Description: select pixel to be displayed at upper left corner of display*
* Returns:  TRUE if successful                                             *
*                                                                          *
***************************************************************************/
/* exported to vesa.h */
/*
ibool  VBE_setDisplayStart(int x, int y, ibool waitVR)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = waitVR ? vbeSetDispStartVR : vbeSetDispStart;
     r.ecx = x;
     r.edx = y;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess;
  }
*/
/***************************************************************************
*                                                                          *
* Function: VBE_getDisplayStart;                                           *
* Description: returns current upper left corner pixel                     *
* Returns:  TRUE if successful                                             *
*           x,y - coordinates of upper left pixel displayed                *
*                                                                          *
***************************************************************************/
/* exported to vesa.h */
/*
ibool  VBE_getDisplayStart(int *x, int *y)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = vbeGetDispStart;
     PM_RealModeInt(vbeInt, &r);
     *x = r.ecx & 0xFFFF;
     *y = r.edx & 0xFFFF;
     return (r.eax & 0xFFFF) == vbeSuccess;
  }
*/

/***************************************************************************
*     Ŀ             *
*       V E S A  2.0   E X T E N D E D   F U N C T I O N S               *
*                  *
***************************************************************************/

/***************************************************************************
*                                                                          *
* Function: VBE_setDACWidth;                                               *
* Description: set width of video card DAC. Note! will return failure in   *
*              direct color or YUV modes.                                  *
* Returns:  TRUE if successful                                             *
*          current DAC width if pointer specified                          *
*                                                                          *
***************************************************************************/
ibool  VBE_setDACWidth(int width, int *currwidth)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDACControl;
     r.ebx = vbeSetDAC + ((width & 0xFF) << 8);
     PM_RealModeInt(vbeInt, &r);
     if (currwidth) *currwidth = (r.ebx>>8) & 0xFF;
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getDACWidth;                                               *
* Description: return current DAC width                                    *
* Returns: current DAC width                                               *
*                                                                          *
***************************************************************************/
int    VBE_getDACWidth()
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDACControl;
     r.ebx = vbeGetDAC;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF)==vbeSuccess ? ((r.ebx>>8) & 0xFF) : 0;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_setPalette;                                                *
* Description: set block of palette registers using VESA 2.0 call          *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
ibool  VBE_setPalette(int start, int num, VBE_palette *pal, ibool waitVR)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbePaletteControl;
     r.ebx = waitVR ? vbeSetPaletteVR : vbeSetPalette;
     r.ecx = num;
     r.edx = start;
     r.es  = rmbuf_seg;
     /* copy palette data to real mode buffer */
     memcpy(rmbuf_ptr, pal, sizeof(VBE_palette)*num);
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF)==vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_setSecondaryPalette;                                       *
* Description: set block of palette registers using VESA 2.0 call          *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
ibool  VBE_setSecondaryPalette(int start, int num, VBE_palette *pal)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbePaletteControl;
     r.ebx = vbeSetSecPalette;
     r.ecx = num;
     r.edx = start;
     r.es  = rmbuf_seg;
     /* copy palette data to real mode buffer */
     memcpy(rmbuf_ptr, pal, sizeof(VBE_palette)*num);
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF)==vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getPalette;                                                *
* Description: reads block of palette registers using VESA 2.0 call        *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
ibool  VBE_getPalette(int start, int num, VBE_palette *pal)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbePaletteControl;
     r.ebx = vbeGetPalette;
     r.ecx = num;
     r.edx = start;
     r.es  = rmbuf_seg;
     PM_RealModeInt(vbeInt, &r);
     /* copy palette data from real mode buffer */
     memcpy(pal, rmbuf_ptr, sizeof(VBE_palette)*num);
     return (r.eax & 0xFFFF)==vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getSecPalette;                                             *
* Description: reads block of palette registers using VESA 2.0 call        *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
ibool  VBE_getSecondaryPalette(int start, int num, VBE_palette *pal)
  {
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbePaletteControl;
     r.ebx = vbeGetSecPalette;
     r.ecx = num;
     r.edx = start;
     r.es  = rmbuf_seg;
     PM_RealModeInt(vbeInt, &r);
     /* copy palette data from real mode buffer */
     memcpy(pal, rmbuf_ptr, sizeof(VBE_palette)*num);
     return (r.eax & 0xFFFF)==vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_getPMInfo;                                                 *
* Description: returns VESA 2.0 32bit PM code info block.                  *
* Returns: TRUE if successful                                              *
*          size - lenght of table with code (for copying purpose)          *
*                                                                          *
***************************************************************************/
ibool  VBE_getPMInfo(VBE_pmInfo *pmInfo)
  {
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbePMInterface;
     PM_RealModeInt(vbeInt, &r);
     if ((r.eax & 0xFFFF)!=vbeSuccess) return FALSE;
     /* get pointer to PM Interface table */
     void *pmTable = (void*)((unsigned)RMSegToFlat(r.es) + (r.edi & 0xFFFF));
     /* copy table to our buffer */
     memcpy(pmInfo, pmTable, sizeof(VBE_pmInfo));
     return TRUE;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_initPMCode;                                                *
* Description: initiates VESA 2.0 protected mode code. Reallocates VESA    *
*              into an internal memory block. The internal memory block    *
*              is allocated with malloc() and must be freed with           *
*              VBE_freePMCode procedure.                                   *
*              Note, that PM code *must* be reinitiated after each mode set*
*              as the routines will change depending on the underlying     *
*              video mode.                                                 *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
ibool  VBE_initPMCode()
  {
     if (pmInfo) {
#ifdef __VBE__DEBUG__
        printf("VBE_initPMCode(): PM code was already initiated. Reinit\n");
#endif
        /* if code was already initiated, reinit it */
        if (!VBE_freePMCode) return FALSE;
     }
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbePMInterface;
     PM_RealModeInt(vbeInt, &r);

     if ((r.eax & 0xFFFF)!=vbeSuccess) return FALSE;
     /* get pointer to PM Interface table */
     void *pmTable = (void*)((unsigned)RMSegToFlat(r.es) + (r.edi & 0xFFFF));
     int  size = r.ecx & 0xFFFF;
     /* allocate memory for pm code and info block */
     pmInfo = (VBE_pmInfo*)malloc(size);
     /* Memory allocation must never fail */
     assert(pmInfo);
     memcpy(pmInfo,pmTable,size);
        /* Read the IO priveledge info and determine if we need to
         * pass a selector to MMIO registers to the bank switch code.
         * Application code can simply check the value of VBE_MMIOSel
         * and if this is non-zero then you must pass this value in ES
         * when you call the bank switch code.
         */
     if (pmInfo->IOPrivInfo && !VBE_MMIOSel) {
         word *p = (word*)((char*)pmInfo + pmInfo->IOPrivInfo);
         /* skip all registers */
         while (*p != 0xFFFF)
             p++;
         p++;
         if (*p != 0xFFFF) {
             /* We have an memory mapped IO location listed, which
              * we need to use for mapping the memory mapped
              * registers. Note that we simply allocate a selector to
              * a 512kb region of memory, since the stupid VBE 2.0 spec
              * only defines a maximum length of 64Kb for the selector.
              * 512Kb should cover all known graphics hardware.
              */
              VBE_MMIOSel = PM_CreateSelector((dword)p,(512 * 1024L)-1);
         }
     }
     return TRUE;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_freePMCode;                                                *
* Description: frees memory allocated for PM code.                         *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
void   VBE_freePMCode()
  {
     if (pmInfo) {
         free(pmInfo);
         pmInfo = NULL;
         PM_FreeSelector(VBE_MMIOSel);
         VBE_MMIOSel = 0;
     }
  }

/***************************************************************************
*                                                                          *
* The following procedures provides 32bit protected mode interface with    *
* additional VBE 2.0 functions. All calls parameters stay the same as      *
* corresponding real mode functions.                                       *
*                                                                          *
* Note! Function 0x4F07 (set display start) has different format:          *
* ax = 0x4F07                                                              *
* bl = vbeSetDispStart or vbeSetDispStartVR                                *
* cx = bits 0-15 of display start address                                  *
* dx = bits 16-31 of display start address                                 *
*                                                                          *
***************************************************************************/
/***************************************************************************
*                                                                          *
* Function:     VBE_getSetBank                                             *
* Returns:      Pointer to the 32 bit VBE 2.0 bank switching routine.      *
*                                                                          *
***************************************************************************/
void  *VBE_getSetBank()
  {
#ifdef __VBE__DEBUG__
     if (!pmInfo) {
        printf(errorstr2); exit(1);
     }
#endif
     return (char*)pmInfo + pmInfo->setWindow;
  }

/***************************************************************************
*                                                                          *
* Function:     VBE_getSetDisplayStart                                     *
* Returns:      Pointer to the 32 bit VBE 2.0 CRT start address routine.   *
*                                                                          *
***************************************************************************/
void  *VBE_getSetDisplayStart()
  {
#ifdef __VBE__DEBUG__
     if (!pmInfo) {
        printf(errorstr2); exit(1);
     }
#endif
     return (char*)pmInfo + pmInfo->setDisplayStart;
  }

/***************************************************************************
*                                                                          *
* Function:     VBE_getSetPalette                                          *
* Returns:      Pointer to the 32 bit VBE 2.0 palette programming routine. *
*                                                                          *
***************************************************************************/
void  *VBE_getSetPalette()
  {
#ifdef __VBE__DEBUG__
     if (!pmInfo) {
        printf(errorstr2); exit(1);
     }
#endif
     return (char*)pmInfo + pmInfo->setPalette;
  }

/***************************************************************************
*     Ŀ             *
*       V E S A  3.0   E X T E N D E D   F U N C T I O N S               *
*                  *
***************************************************************************/

/***************************************************************************
*                                                                          *
* Function: VBE_setVideoModeExt;                                           *
* Description: attemts to set the specified video mode.                    *
* Returns: TRUE if successful                                              *
*                                                                          *
***************************************************************************/
ibool  VBE_setVideoModeExt(int mode,VBE_CRTCInfo *crtc)
  {
     /* only avaible on VBE/Core 3.0 with Refresh Control */
     if (VBEVersion < 0x300 && (mode & vbeRefreshCtrl)) return FALSE;
#ifdef __VBE__DEBUG__
     if (!rmbuf_sel) {
          printf(errorstr); exit(1);
     }
#endif
     RMREGS r;
     memset(&r,0,sizeof(RMREGS));
#ifdef __VBE__DEBUG__
     if (!(mode & vbeRefreshCtrl)) {
        printf("VBE_setVideoModeExt(): you must set vbeRefreshCtrl flag in mode number\n"
               "                       to initiate mode with refresh control enabled\n");
        exit(1);
     }
     if (!crtc) {
        printf("VBE_setVideoModeExt(): you must specify refresh control info block!\n");
        exit(1);
     }
#endif
     if ((mode & vbeRefreshCtrl) && crtc) {
        r.eax = vbeSetMode;
        r.ebx = mode;
        r.es = rmbuf_sel;
        memcpy(rmbuf_ptr, crtc, sizeof(VBE_CRTCInfo));
        PM_RealModeInt(vbeInt, &r);
        return (r.eax & 0xFFFF) == vbeSuccess;
     }
     return FALSE;
  }

/***************************************************************************
*                                                                          *
* Function: VBE_setDisplayStartAlt;                                        *
* Description: sets the new starting display position to the specified     *
*              32-bit display start address.                               *
* Returns:  TRUE if successful                                             *
*                                                                          *
***************************************************************************/
/* exported to vesa.h */
/*
ibool  VBE_setDisplayStartAlt(dword startAddr,ibool waitVRT)
  {
     if (VBEVersion <0x300) return FALSE;
     RMREGS  r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = waitVRT ? vbeSetDispStartAltVR : vbeSetDispStartAlt;
     r.ecx = startAddr;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess;
  }
*/
/***************************************************************************

*                                                                          *
* Function: VBE_getDisplayStartStatus                                      *
* Description:  Returns the status of the previous display start request.  *
*               If this function is supported the programmer can implement *
*               hardware triple buffering using this function.             *
* Returns:      0 if last flip not occurred, 1 if already flipped          *
*                               -1 if not supported                        *
*                                                                          *
***************************************************************************/
int    VBE_getDisplayStartStatus()
  {
     if (VBEVersion <0x300) return FALSE;
     RMREGS  r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = vbeGetDispStartStatus;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess ? ((r.ecx & 0xFFFF)!=0) : (-1);
  }

/***************************************************************************
*                                                                          *
* Function:     VBE_enableStereoMode                                       *
* Description:  Puts the system into hardware stereo mode for LC shutter   *
*               glasses, where the display swaps between two display start *
*               addresses every vertical retrace.                          *
* Returns:      True if stereo mode enabled, false if not supported.       *
*                                                                          *
***************************************************************************/
ibool  VBE_enableStereoMode()
  {
     if (VBEVersion <0x300) return FALSE;
     RMREGS  r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = vbeEnableStereoMode;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function:     VBE_disableStereoMode                                      *
* Description:  Puts the system back into normal, non-stereo display mode  *
*               after having stereo mode enabled.                          *
* Returns:      True if stereo mode disabled, false if not supported.      *
*                                                                          *
***************************************************************************/
ibool  VBE_disableStereoMode(void)
  {
     if (VBEVersion <0x300) return FALSE;
     RMREGS  r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = vbeDisableStereoMode;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function:     VBE_setStereoDisplayStart                                  *
* Parameters:   leftAddr        - 32-bit start address for left image      *
*               rightAddr       - 32-bit start address for right image     *
* Description:  Sets the new starting display position to the specified    *
*               32-bit display start address. Note that this function is   *
*               different the the version above, since it takes a 32-bit   *
*               byte offset in video memory as the starting address which  *
*               gives the programmer maximum control over the stat address.*
* Returns:      True if function was successful, false if not supported.   *
*                                                                          *
***************************************************************************/
ibool  VBE_setStereoDisplayStart(dword leftAddr,dword rightAddr,ibool waitVRT)
  {
     if (VBEVersion <0x300) return FALSE;
     RMREGS  r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeDispStartControl;
     r.ebx = waitVRT ? vbeSetStereoDispStartVR : vbeSetStereoDispStart;
     r.ecx = leftAddr;
     r.edx = rightAddr;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess;
  }

/***************************************************************************
*                                                                          *
* Function:     VBE_getClosestClock                                        *
* Parameters:   mode       - VBE mode to be used (include vbeLinearBuffer) *
*               pixelClock - Desired pixel clock                           *
* Returns:      Closest pixel clock to desired clock (-1 if not supported) *
* Description:  Calls the VBE/Core 3.0 interface to determine the closest  *
*               pixel clock to the requested value. The BIOS will always   *
*               search for a pixel clock that is no more than 1% below the *
*               requested clock or somewhere higher than the clock. If the *
*               clock is higher note that it may well be many Mhz higher   *
*               that requested and the application will have to check that *
*               the returned value is suitable for it's needs.This function*
*               returns the actual pixel clock that will be programmed by  *
*               the hardware.                                              *
*                                                                          *
*               Note that if the pixel clock will be used with a linear    *
*               framebuffer mode, make sure you pass in the linear         *
*               framebuffer flag to this function.                         *
*                                                                          *
***************************************************************************/
dword  VBE_getClosestClock(word mode,dword pixelClock)
  {
     if (VBEVersion <0x300) return FALSE;
     RMREGS  r;
     memset(&r,0,sizeof(RMREGS));
     r.eax = vbeGetClosestClock;
     r.ecx = pixelClock;
     r.edx = mode;
     PM_RealModeInt(vbeInt, &r);
     return (r.eax & 0xFFFF) == vbeSuccess ? r.eax : (-1);
  }
