/*
*******************************************************************************
FILE: 			MANDEL.CPP
AUTHOR:     	S.Fourmanoit 	fourman@CAM.ORG
DESCRIPTION:   	This program computes the Mandelbrot set in an acceptable
		resolution (320x200x256) at good speed, using a convergence
		acceleration method for iterated series.
		It also includes zooming capabilities precise enough to generate
		accurately the set in 1/100 by 1/100 regions of the complex plan.
MISCELLANIOUS: 	This code was optimised for an I86 processor with some floating
		point capabilities.  It was compiled DOS mode on Borland C4.5.

		This code is built in a way the Mandel class is fully independant
		of any video hardware.  However, you'll need to pass the adress
		of a SET_PIXELS_CALLBACK function to the constructor of the mandel
		class in order to use any other screen resolution or graphic card
		than a VGA adaptator, mode 13h.
DATE:		12/22/98
*******************************************************************************
*/

//--- Inclusions --------------------------------------------------------------
#include <iostream.h>
#include <dos.h>
#include <conio.h>

//--- Declarations statements -------------------------------------------------
#define MANDEL_LIMIT 100      //used by Complex::is_to_far_from_origin()
#define MAX_MANDEL_LOOP 100   //used by Mandel::belong_to_mandel_set()

#define SCREEN_WIDTH 320      //x resolution of the screen
#define SCREEN_HEIGHT 200     //y resolution of the screen

#define ARROW_STEP 10         //used by Arrow::move_arrow()

//--- Enumerations statements -------------------------------------------------
enum KEYS {up=72      , right=77      , down=80       , left=75,
 	   ctrl_up=141, ctrl_right=116, ctrl_down=145 , ctrl_left=115,
	   escape=27  , enter=13 , space=32, no_key = 0};
enum ARROW_TYPE { UP_LEFT_ARROW, DOWN_RIGHT_ARROW };
enum USER_STATE { OK, CANCEL };

//--- Type definitions --------------------------------------------------------
typedef unsigned char BYTE;
typedef unsigned int UINT;
typedef void (*SET_PIXELS_CALLBACK) (UINT x, UINT y, BYTE * color_array, UINT pixel_nbr);

//--- Class complex -----------------------------------------------------------
class complex {
	public:
		complex(void) {
			r=i=0;
		}
		complex(double newr,double newi) {
			r=newr;
			i=newi;
		}

		double r,i;

		int is_too_far_from_origin(void) {
			int int_r=(int) r;
			int int_i=(int) i;

			if ( (int_r*int_r)+(int_i*int_i)>=MANDEL_LIMIT)
				return 1;
			else
				return 0;
		}

		complex operator + (complex c2) {
			return complex(r+c2.r,i+c2.i);
		}

		complex operator * (complex c2) {
			return complex( (r*c2.r)-(i*c2.i) , (r*c2.i)+(i*c2.r) );
		}

		void operator = (complex c2) {
			r=c2.r;
			i=c2.i;
		}
};

//--- Class Screen ------------------------------------------------------------
class Screen {
	public:
		static void init_mode(BYTE mode) {
			REGS regs;

			regs.h.ah=0;
			regs.h.al=mode;
			int86(0x10,&regs,&regs);
		}

		static void set_pixel(UINT x, UINT y, BYTE color);
		static void set_pixels(UINT x, UINT y, BYTE * color_array, UINT pixel_nbr);

		static BYTE get_pixel(UINT x, UINT y);
		static void get_pixels(UINT x, UINT y, BYTE * color_array, UINT pixel_nbr);

		static void set_hline(UINT x, UINT y, BYTE color, int pixel_nbr);
		static void set_vline(UINT x, UINT y, BYTE color, int pixel_nbr);
		static void set_line(UINT x1, UINT y1, UINT x2, UINT y2, BYTE color);
	private:
};

void Screen::set_pixel(UINT x, UINT y, BYTE color) {
	BYTE far * lpcolor;
	*(lpcolor = (BYTE *) MK_FP(0xA000,(y*SCREEN_WIDTH)+x))=color;
}

void Screen::set_pixels(UINT x, UINT y, BYTE * color_array, UINT pixel_nbr) {
	BYTE far * lpcolor;
	BYTE far * color_array_backup=color_array;

	for ( lpcolor=(BYTE *) MK_FP(0xA000,(y*SCREEN_WIDTH)+x);
	      pixel_nbr;
	      pixel_nbr--,lpcolor++,color_array++)
		*lpcolor=*color_array;

	color_array=color_array_backup;
}

BYTE Screen::get_pixel(UINT x, UINT y) {
	BYTE far * lpcolor;
	return (*(lpcolor = (BYTE *) MK_FP(0xA000,(y*SCREEN_WIDTH)+x)));
}

void Screen::get_pixels(UINT x, UINT y, BYTE * color_array, UINT pixel_nbr) {
	BYTE far * lpcolor;
	BYTE far * color_array_backup=color_array;

	for ( lpcolor=(BYTE *) MK_FP(0xA000,(y*SCREEN_WIDTH)+x);
	      pixel_nbr;
	      pixel_nbr--,lpcolor++,color_array++)
		*color_array=*lpcolor;

	color_array=color_array_backup;
}

void Screen::set_hline(UINT x, UINT y, BYTE color, int pixel_nbr) {
	BYTE far * lpcolor=(BYTE *) MK_FP(0xA000,(y*SCREEN_WIDTH)+x);

	if (pixel_nbr<0) {
		lpcolor=(lpcolor+pixel_nbr+1);
		pixel_nbr=-pixel_nbr;
	}

	for (;pixel_nbr;pixel_nbr--,lpcolor++)
		*lpcolor=color;
}

void Screen::set_vline(UINT x, UINT y, BYTE color, int pixel_nbr) {
	BYTE far * lpcolor=(BYTE *) MK_FP(0xA000,(y*SCREEN_WIDTH)+x);

	if (pixel_nbr<0) {
		lpcolor=(lpcolor+(pixel_nbr+1)*SCREEN_WIDTH);
		pixel_nbr=-pixel_nbr;
	}

	for (;pixel_nbr;pixel_nbr--,lpcolor=lpcolor+SCREEN_WIDTH)
		*lpcolor=color;
}

//--- Class Mandel ------------------------------------------------------------
class Mandel {
		public:
			Mandel(SET_PIXELS_CALLBACK set_pixels) {
				lp_set_pixels = set_pixels;
				set_first_argand_plane(-SCREEN_WIDTH/double(SCREEN_HEIGHT)*3/2, 1,
							SCREEN_WIDTH/double(SCREEN_HEIGHT)/2, -1);
			}

			void set_next_argand_plane(UINT x1,UINT y1,UINT x2, UINT y2) {
				old_first_r = first_r;
				old_first_i = first_i;
				old_step_r  = step_r;
				old_step_i  = step_i;

				first_r=first_r+(x1*step_r);
				first_i=first_i+(y1*step_i);
				step_r*=(x2-x1)/double(SCREEN_WIDTH);
				step_i*=(y2-y1)/double(SCREEN_HEIGHT);

				plane_is_generated=0;
			}

			void set_previous_argand_plane(void) {
				if (!( first_r==old_first_r && first_i==old_first_i &&
				       step_r==old_step_r   && step_i==old_step_i      )) {

					first_r=old_first_r;
					first_i=old_first_i;
					step_r=old_step_r;
					step_i=old_step_i;

					plane_is_generated=0;
				}
				else plane_is_generated=1;
			}

			UINT generate_argand_plane (void );

		private:
			SET_PIXELS_CALLBACK lp_set_pixels;

			double first_r, first_i, step_r, step_i;
			double old_first_r, old_first_i, old_step_r, old_step_i;
			UINT plane_is_generated;

			void set_first_argand_plane (double top_r    , double top_i,
 						     double bottom_r , double bottom_i ) {
				
				old_first_r = (first_r = top_r);
				old_first_i = (first_i = top_i);
				old_step_r  = (step_r = (bottom_r-top_r)/SCREEN_WIDTH);
				old_step_i  = (step_i =(top_i-bottom_i)/-SCREEN_HEIGHT);

				plane_is_generated=0;
			}

			BYTE belong_to_mandel_set(complex c1);
};

BYTE Mandel::belong_to_mandel_set(complex c1) {
	UINT i=0;
	complex original_c;

	original_c=c1;

	while (i<=MAX_MANDEL_LOOP && !c1.is_too_far_from_origin()) {
		c1=c1*c1+original_c;
		i++;
	}

	if (c1.is_too_far_from_origin())
		return ((BYTE)i/10+7);        //Used for color rendition
	else
		return 0;
}

UINT Mandel::generate_argand_plane (void) {
	UINT x,y;
	BYTE color_array[SCREEN_WIDTH];
	complex current_complex(first_r,first_i);

	if (!plane_is_generated) {
		for (y=0; y<=SCREEN_HEIGHT; y++, current_complex.i+=step_i) {
			for (x=0, current_complex.r=first_r;
				  x<=319;
				  x++, current_complex.r+=step_r )
				color_array[x]=belong_to_mandel_set(current_complex);

			lp_set_pixels(0,y,(BYTE *) &color_array, SCREEN_WIDTH);
		}
	plane_is_generated=1;
	return 1;
	}
	else return 0;
}

//--- Class Arrow -------------------------------------------------------------
class Arrow {
	public:
		Arrow (ARROW_TYPE direction) {
			arrow_direction=direction;
			scale_flag=0;
			already_shown_flag=0;
		}

		void reset_arrow(void) {
			scale_flag=0;
			already_shown_flag=0;
		}

		UINT get_x(void) { return current_x; }
		UINT get_y(void) { return current_y; }
		USER_STATE move_arrow(UINT x , UINT y );

	private:
		BYTE        arrow_background[25];
		ARROW_TYPE  arrow_direction;
		UINT        already_shown_flag;
		UINT        scale_flag, scale_x  , scale_y;
		UINT        min_x     , min_y    ,
        		    current_x , current_y;

		void set_arrow( UINT x , UINT y ) {
			current_x = (min_x = x)+1;
			current_y = (min_y = y)+1;
			already_shown_flag=1;

			get_arrow_background(current_x, current_y);
			refresh_arrow(current_x, current_y);
		}

		KEYS get_key(void);
		void correct_scale(UINT &x, UINT &y);
		void refresh_arrow(UINT x, UINT y);
		void get_arrow_background(UINT x, UINT y );
		void set_arrow_background(UINT x, UINT y );

};

void Arrow::correct_scale(UINT &x, UINT &y) {
		if (scale_flag)
			x=min_x+((y-min_y)*SCREEN_WIDTH/SCREEN_HEIGHT);
		else
			y=min_y+((x-min_x)*SCREEN_HEIGHT/SCREEN_WIDTH);
}

KEYS Arrow::get_key(void) {
	int user_answer;
	KEYS user_answer_key;

	do {
		user_answer=getch();
		if (!user_answer) user_answer= getch();

		switch (user_answer) {
			case 72 : case 77 : case 80 : case 75 :
			case 141: case 116: case 145: case 115:
			case 27 : case 13 : case 32 :
    				 user_answer_key = (KEYS) user_answer; break;
			default: user_answer_key = no_key;
		}

	}
	while (user_answer_key==no_key);

	return user_answer_key;
}

void Arrow::get_arrow_background(UINT x, UINT y ) {
	int i;

	if (arrow_direction==DOWN_RIGHT_ARROW) {
		x=(int(x)-4<0)?0:x-=4;
		y=(int(y)-4<0)?0:y-=4;
	}

	for (i=0;i<=4;i++)
		Screen::get_pixels(x,y+i,(BYTE *) &arrow_background[i*5],5);
}

void Arrow::set_arrow_background(UINT x, UINT y ) {
	int i;

	if (arrow_direction==DOWN_RIGHT_ARROW) {
		x=(int(x)-4<0)?0:x-=4;
		y=(int(y)-4<0)?0:y-=4;
	}

	for (i=0;i<=4;i++)
		Screen::set_pixels(x,y+i,(BYTE *) &arrow_background[i*5],5);
}

void Arrow::refresh_arrow(UINT x, UINT y) {
	int length_x;
	int length_y;

	set_arrow_background(current_x, current_y);
	current_x=x; current_y=y;
	get_arrow_background(current_x,current_y);

	if (!arrow_direction) {
		length_x = (length_x = SCREEN_WIDTH - x)>5?5:length_x;
		length_y = (length_y = SCREEN_HEIGHT - y)>5?5:length_y;

		Screen::set_hline(x,y,12,length_x);
		Screen::set_vline(x,y,12,length_y);
	}
	else {
		length_x = (length_x = x)>5?5:length_x;
		length_y = (length_y = y)>5?5:length_y;

		Screen::set_hline(x,y,12,-length_x);
		Screen::set_vline(x,y,12,-length_y);
	}

}

USER_STATE Arrow::move_arrow(UINT x =0, UINT y = 0) {
	UINT next_x,
		  next_y;
	KEYS user_answer;

	if (!already_shown_flag)
		set_arrow(x,y);

	next_x=current_x;
	next_y=current_y;

	do {
		user_answer=get_key();
		switch (user_answer) {
			case up:         next_y-=1;break;
			case right:      next_x+=1;break;
			case down:       next_y+=1;break;
			case left:       next_x-=1;break;
			case ctrl_up:	 next_y-=ARROW_STEP;break;
			case ctrl_right: next_x+=ARROW_STEP;break;
			case ctrl_down:	 next_y+=ARROW_STEP;break;
			case ctrl_left:	 next_x-=ARROW_STEP;break;
			case space    :  if (arrow_direction==DOWN_RIGHT_ARROW) {
						if (!scale_flag) {
							scale_x=next_x; scale_y=next_y;
							correct_scale(next_x,next_y);
							scale_flag=1;
						}
						else {
							next_x=scale_x; next_y=scale_y;
							correct_scale(next_x,next_y);
							scale_flag=0;
						}
					 }
					 break;
		}

		if (user_answer!=space) scale_flag=0;
		if (!(next_x>min_x &&
		      next_x<SCREEN_WIDTH-((arrow_direction==UP_LEFT_ARROW)?1:0) &&
		      next_y>min_y &&
		      next_y<SCREEN_HEIGHT-((arrow_direction==UP_LEFT_ARROW)?1:0))) {
		      	next_x=current_x; next_y=current_y;
		}
		else
			refresh_arrow(next_x,next_y);
	}
	while (user_answer!=escape && user_answer!=enter);
	
	if (user_answer==escape) {
		set_arrow_background(current_x,current_y);
		already_shown_flag=0;
		return CANCEL;
	}
	else return OK;
}

//--- function welcome_message ------------------------------------------------
void welcome_message(void) {
	const char* message[20]
	= { "MANDEL.EXE - S.Fourmanoit, fourman@CAM.ORG, MCMXCVIII.",
		 "",
		 "This program computes the Mandelbrot set in an acceptable",
		 "resolution (320x200x256) at good speed, using a convergence",
		 "acceleration method for iterated series.",
		 "",
		 "It also includes zooming capabilities precise enough to generate",
		 "accurately the set in 1/100 by 1/100 regions of the complex plan.",
		 "",
		 "Use :",
		 "    ARROWS      to move de selection crosses ",
		 "    CRTL+ARROWS to move de selection crosses quicker",
		 "    SPACE       to set a 'square' selection",
		 "                (press once or two times when setting the second cross)",
		 "    ENTER       to validate de selection",
		 "    ESCAPE      to reset the last selection cross OR",
		 "                to go back to the previous zoom   OR",
		 "                to exit the program",
		 "",
		 "Press a key to begin"
	  };
	BYTE i;

	for (i=0; i<=19; i++)
		cout <<message[i]<<endl;
}

//--- Main function -----------------------------------------------------------
void main(void) {
	Mandel mandel(Screen::set_pixels);
	Arrow arrow1(UP_LEFT_ARROW), arrow2(DOWN_RIGHT_ARROW);
	USER_STATE arrow1_user_state, arrow2_user_state;
	UINT exit_flag=0;

	Screen::init_mode(0x03);
	welcome_message(); getch();
	Screen::init_mode(0x13);
	mandel.generate_argand_plane();

	do {
		do {
			do {
				if ((arrow1_user_state=arrow1.move_arrow())==CANCEL) {
					mandel.set_previous_argand_plane();
					if (!mandel.generate_argand_plane()) exit_flag=1 ;
				}
			}
			while (arrow1_user_state==CANCEL && !exit_flag);

			if (!exit_flag)
				arrow2_user_state=arrow2.move_arrow(arrow1.get_x(),arrow1.get_y());
		}
		while ((arrow1_user_state == CANCEL || arrow2_user_state == CANCEL) &&
				  !exit_flag);

		if (!exit_flag) {
			arrow1.reset_arrow(); arrow2.reset_arrow();
			mandel.set_next_argand_plane(arrow1.get_x(), arrow1.get_y(),
												  arrow2.get_x(), arrow2.get_y() );
			mandel.generate_argand_plane();
		}
	}
	while (!exit_flag);
	Screen::init_mode(0x03);
	cout << "Have a nice day!" << endl << endl;
}
