/* 
 *    Example program for the Allegro library, by Shawn Hargreaves.
 *
 *    This program demonstrates how to use the 3d matrix functions.
 *    It isn't a very elegant or efficient piece of code, but it does 
 *    show the stuff in action. I'll leave it to you to design a proper 
 *    model structure and rendering pipeline: after all, the best way to 
 *    do that sort of stuff varies hugely from one game to another.
 */


#include <stdlib.h>
#include <stdio.h>

#include <fastgl.h>
#include "3dview.h"


#define NUM_SHAPES         8     /* number of bouncing cubes */
#define NUM_VERTICES       8     /* a cube has eight corners */
#define NUM_FACES          6     /* a cube has six faces */

/* convert radians to degrees */
#define DEG(n)    ((n) * 180.0 / M_PI)

/* parameters controlling the camera and projection state */
int fov = 48;
float aspect = 1;
float xpos = 5;
float ypos = 2;
float zpos = 1;
float heading = .9;
float pitch = .6;
float roll = 0;

int draw_flag;


typedef struct QUAD              /* four vertices makes a quad */
{
   VTX *vtxlist;
   int v1, v2, v3, v4;
} QUAD;


typedef struct SHAPE             /* store position of a shape */
{
   float x, y, z;                /* x, y, z position */
   float rx, ry, rz;             /* rotations */
   float dz;                     /* speed of movement */
   float drx, dry, drz;          /* speed of rotation */
} SHAPE;

VTX points[] =                   /* a cube, centered on the origin */
{
   /* vertices of the cube */
   { -32 , -32, -32 },
   { -32 ,  32, -32 },
   {  32 ,  32, -32 },
   {  32 , -32, -32 },
   { -32 , -32,  32 },
   { -32 ,  32,  32 },
   {  32 ,  32,  32 },
   {  32 , -32,  32 },
};


QUAD faces[] =                   /* group the vertices into polygons */
{
   { points, 0, 3, 2, 1 },
   { points, 4, 5, 6, 7 },
   { points, 0, 1, 5, 4 },
   { points, 2, 3, 7, 6 },
   { points, 0, 4, 7, 3 },
   { points, 1, 2, 6, 5 }
};


SHAPE shapes[NUM_SHAPES];        /* a list of shapes */

/* somewhere to put translated vertices */
VTX output_points[NUM_VERTICES * NUM_SHAPES];
QUAD output_faces[NUM_FACES * NUM_SHAPES];

float xfront, yfront, zfront;
float xup, yup, zup;

enum { 
   wireframe, 
   flat_shaded, 
} render_mode = wireframe;

/* initialise shape positions */
void init_shapes()
{
   int c;

   for (c=0; c<NUM_SHAPES; c++)
   {
      shapes[c].x = (rand() & 0xFFF);
      shapes[c].y = (rand() & 0xFFF);
      shapes[c].z =	(rand() & 0xFFF);
      shapes[c].rx = 0;
      shapes[c].ry = 0;
      shapes[c].rz = 0;
      shapes[c].dz =  1;
      shapes[c].drx = 1;
      shapes[c].dry = 2;
      shapes[c].drz = 1;
   }
}


/* update shape positions */
void animate_shapes()
{
   int c;

   for (c=0; c<NUM_SHAPES; c++) {
      shapes[c].z += shapes[c].dz;

      if ((shapes[c].z > 1024) ||
	  (shapes[c].z < 192))
	 shapes[c].dz = -shapes[c].dz;

      shapes[c].rx += shapes[c].drx;
      shapes[c].ry += shapes[c].dry;
      shapes[c].rz += shapes[c].drz;
   }
}


void render(MATRIX_f *camera)
{
   MATRIX_f roller;

   /* calculate the in-front vector */
   xfront = sin(heading) * cos(pitch);
   yfront = sin(pitch);
   zfront = cos(heading) * cos(pitch);

   /* rotate the up vector around the in-front vector by the roll angle */
   get_vector_rotation_matrix_f(&roller, xfront, yfront, zfront, roll*128.0/M_PI);
   apply_matrix_f(&roller, 0, -1, 0, &xup, &yup, &zup);

   /* build the camera matrix */
   get_camera_matrix_f(camera,
		       xpos, ypos, zpos,        /* camera position */
		       xfront, yfront, zfront,  /* in-front vector */
		       xup, yup, zup,           /* up vector */
		       fov,                     /* field of view */
		       aspect);                 /* aspect ratio */
}

/* translate shapes from 3d world space to 2d screen space */
void translate_shapes(MATRIX_f& camera)
{
   int c, d;
   MATRIX_f matrix,output;
   VTX *outpoint = output_points;
   QUAD *outface = output_faces;

   for (c=0; c<NUM_SHAPES; c++)
   {
      /* build a transformation matrix */
      get_transformation_matrix_f(&matrix, 1,
				shapes[c].rx, shapes[c].ry, shapes[c].rz,
				shapes[c].x, shapes[c].y, shapes[c].z);

	 matrix_mul_f(&matrix,&camera,&output);
     for (d=0; d<NUM_VERTICES; d++)
     {
		 apply_matrix_f(&output, points[d].x, points[d].y, points[d].z, &outpoint[d].x, &outpoint[d].y, &outpoint[d].z);
		 persp_project_f(outpoint[d].x, outpoint[d].y, outpoint[d].z, &outpoint[d].x, &outpoint[d].y);
     }

     /* output the faces */
     for (d=0; d<NUM_FACES; d++) {
	 outface[d] = faces[d];
	 outface[d].vtxlist = outpoint;
     }

     outpoint += NUM_VERTICES;
     outface += NUM_FACES;
   }
}

/* draw a quad */
void fill_face(The3Dview *b, VTX *v1, VTX *v2, VTX *v3, VTX *v4, int mode, int color)
{
	if (polygon_z_normal_f(v1, v2, v3) < 0) return;
	set_fcolor(color);
	quad3d_f(b, mode, v1, v2, v3, v4);
}


/* callback for qsort() */
int quad_cmp(const void *e1, const void *e2)
{
   QUAD *q1 = (QUAD *)e1;
   QUAD *q2 = (QUAD *)e2;

   float d1 = q1->vtxlist[q1->v1].z + q1->vtxlist[q1->v2].z +
	      q1->vtxlist[q1->v3].z + q1->vtxlist[q1->v4].z;

   float d2 = q2->vtxlist[q2->v1].z + q2->vtxlist[q2->v2].z +
	      q2->vtxlist[q2->v3].z + q2->vtxlist[q2->v4].z;

   return (int)(d2 - d1);
}


/* draw the shapes calculated by translate_shapes() */
void draw_shapes(The3Dview *b)
{
   int c;
   QUAD *face = output_faces;
   VTX *v1, *v2, *v3, *v4;

   /* depth sort */
   qsort(output_faces, NUM_FACES * NUM_SHAPES, sizeof(QUAD), quad_cmp);

   for (c=0; c < NUM_FACES*NUM_SHAPES; c++)
   {
      /* find the vertices used by the face */
      v1 = face->vtxlist + face->v1;
      v2 = face->vtxlist + face->v2;
      v3 = face->vtxlist + face->v3;
      v3 = face->vtxlist + face->v3;
      v4 = face->vtxlist + face->v4;
      fill_face(b, v1, v2, v3, v4, 1, (c&15)+16);
      face++;
   }
}

The3Dview *buffer;
MATRIX_f camera;

void DelayProc(void)
{
//	if (draw_flag)
	{
		buffer->clear(0);
		render(&camera);
    	animate_shapes();
		translate_shapes(camera);
		draw_shapes(buffer);
		set_ppop(_GTRANSP);
		set_colors(CYELLOW,0);
		buffer->printf(0,00, "Field of view: %d (f/F changes)", fov);
		buffer->printf(0,16, "Aspect ratio: %.2f (a/A changes)", aspect);
		buffer->printf(0,32, "X position: %.2f (x/X changes)", xpos);
		buffer->printf(0,48, "Y position: %.2f (y/Y changes)", ypos);
		buffer->printf(0,64, "Z position: %.2f (z/Z changes)", zpos);
		buffer->printf(0,80, "Heading: %.2f deg (left/right changes)", DEG(heading));
		buffer->printf(0,96, "Pitch: %.2f deg (pgup/pgdn changes)", DEG(pitch));
		buffer->printf(0,112,"Roll: %.2f deg (r/R changes)", DEG(roll));
		buffer->printf(0,152,"Front vector: %.2f, %.2f, %.2f", xfront, yfront, zfront);
		buffer->printf(0,168,"Up vector: %.2f, %.2f, %.2f", xup, yup, zup);
		set_ppop(_GSET);
		buffer->redraw();
	}
}

int app(GuiEvent *e)
{
   int key=e->Key();

   if (e->Type()!=KEYEVENT) return 0;
   if (key==KLEFT)      heading -= 0.05;
   if (key==KRIGHT)      heading += 0.05;
   if (key==PGUP) if (pitch > -M_PI/4) pitch -= 0.05;
   if (key== PGDOWN) if (pitch < M_PI/4) pitch += 0.05;
   if (key=='r') { if (roll < M_PI/4 ) roll += 0.05; }
   if (key=='R') { if (roll > -M_PI/4) roll -= 0.05; }
   if (key=='a') { if (aspect>2) aspect =2; else aspect += 0.05; }
   if (key=='A') { if (aspect<.1) aspect =.1; else aspect -= 0.05;}
   if (key=='f') { if (fov>96) fov =96; else fov ++; }
   if (key=='F') { if (fov<16) fov =16; else fov --;}
   if (key=='x') { xpos += 5; }
   if (key=='X') { xpos -= 5; }
   if (key=='y') { ypos += 5; }
   if (key=='Y') { ypos -= 5; }
   if (key=='z') { zpos += 5; }
   if (key=='Z') { zpos -= 5; }
   if (key==KUP) { xpos += sin(heading) ; zpos += cos(heading) ; }
   if (key==KDOWN) { xpos -= sin(heading) ; zpos -= cos(heading) ; }
   draw_flag = 1;
   return 0;
}

int main(int argc, char **argv)
{
    App myApp(3,argc,argv,0);
    buffer = new The3Dview(0,0,X_width, Y_width);
	for(int i=16;i<32;i++) CreateColor((i-16)*4,(i-16)*4,(i-16)*4,i);
    /* initialise the bouncing shapes */
	init_shapes();
	myApp.SetDelayProc(DelayProc);
	VTX vtx= {1,2,3};
	buffer->AddToWorld(1, &vtx);
	myApp.Run(app);
	delete buffer;
	return 0;
}


