/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"


#define STACKDEPTH      50

/*
#define tolower(x)      AnsiLower ((LPSTR) (LONG) (BYTE) (x))
#define toupper(x)      AnsiUpper ((LPSTR) (LONG) (BYTE) (x))
*/
#define tolower(x)		(('A' <= (x) && (x) <= 'Z') ? (x) + 32 : (x))
#define toupper(x)		(('a' <= (x) && (x) <= 'a') ? (x) - 32 : (x))
#define islower(x)      ('a' <= (x) && (x) <= 'z')
#define isupper(x)      ('A' <= (x) && (x) <= 'Z')
#define isalpha(x)      (isupper(x) || islower(x))


static char Typed[10] = "";
static char Temp[10] = "";
static KANJI InputBuffer[10];
static KANJI KanjiBuffer[MAXLINELEN];
static KANJI LastConv[MAXLINELEN];
BYTE LastConvKey[256];
static int LastLen;

static struct {
    int offset;
    int length;
} GlossaryStack[STACKDEPTH];
static int Stackp = 0;
        

typedef struct {
    int state;
    char *input;
    int next;
} STATE;


/* The State Machine */

static STATE states[] = {
    { 0, "aiueo",    -1 },          /* Start */
    { 0, "hmzbpr",    1 },
    { 0, "kg",       30 },
    { 0, "y",        27 },
	{ 0, "n",         3 },
    { 0, "j",         4 },
	{ 0, "t",         5 },
    { 0, "d",         6 },
    { 0, "s",         7 },
    { 0, "c",         8 },
    { 0, "w",         9 },
    { 0, "v",        13 },
    { 0, "f",        14 },
    { 0, "+",        24 },
    { 0, "`",        -1 },
    { 0, "'",        -1 },
    { 0, "",         -1 },

	{ 1, "aiueo",    -1 },          /* {khmgjdbpr} */
    { 1, "y",         2 },

    { 2, "aueo",     -1 },          /* [...]y */

	{ 3, "y",         2 },          /* n */
	{ 3, "aiueo'",   -1 },

    { 4, "y",         2 },          /* j */
    { 4, "aiueo",    -1 },

	{ 5, "y",         2 },          /* t */
	{ 5, "aiueo",    -1 },
    { 5, "sz",       10 },
    { 5, "c",        28 },

    { 6, "aiueo",    -1 },          /* d */
    { 6, "y",        11 },

    { 7, "y",         2 },          /* s */
    { 7, "aiueo",    -1 },
    { 7, "h",        12 },

    { 8, "y",         2 },          /* c */
    { 8, "h",        12 },
    { 8, "i",        -1 },
    { 8, "/",        -1 },

    { 9, "aieo",     -1 },          /* w */

    { 10, "u",       -1 },          /* t{sz} */

    { 11, "ui",      -1 },          /* dy */

    { 12, "aiueo",   -1 },          /* ch, sh */

    { 13, "aiueo",   -1 },          /* v */

    { 14, "aiueo",   -1 },          /* f */
    { 14, "-",       -1 },

    { 22, "`",       -1 },          /* ` */

    { 23, "'",       -1 },          /* ' */

    { 24, "aiueo",   -1 },          /* + */
    { 24, "t",       25 },
    { 24, "y",        2 },
    { 24, "w",       29 },

    { 25, "u",       -1 },          /* +t */
    { 25, "sz",      26 },

    { 26, "u",       -1 },          /* +t{sz} */

    { 27, "aueo",    -1 },          /* y */
    { 27, "=",       -1 },

    { 28, "y",        2 },          /* tc */
    { 28, "h",       12 },
    { 28, "i",       -1 },

    { 29, "a",       -1 },          /* +w */

    { 30, "aiueo",   -1 },          /* {kg}.. */
    { 30, "y",        2 },
    { 30, "w",       31 },

    { 31, "aieo",    -1 },          /* {kg}{uw}.. */

    { -999, NULL,    -1 }
};


/* The Conversion Tables */

static struct {
    char ascii;
    KANJI kanji;
} JAsciiEquivalents[] = {
    { ' ', 0x2121 },
    { ',', 0x2122 },
    { '.', 0x2123 },
    { ',', 0x2124 },
    { '.', 0x2125 },
    { '.', 0x2126 },
    { ':', 0x2127 },
    { ';', 0x2128 },
    { '?', 0x2129 },
    { '!', 0x212a },
    { '\"', 0x212b },
    { '^', 0x2130 },
    { '_', 0x2132 },
    { '-', 0x213c },
    { '-', 0x213d },
    { '-', 0x213e },
    { '/', 0x213f },
    { '\\', 0x2140 },
    { '~', 0x2141 },
    { '|', 0x2143 },
    { '`', 0x2146 },
    { '\'', 0x2147 },
    { '\"', 0x2148 },
    { '\"', 0x2149 },
    { '(', 0x214a },
    { ')', 0x214b },
    { '(', 0x214c },
    { ')', 0x214d },
    { '[', 0x214e },
    { ']', 0x214f },
    { '{', 0x2150 },
    { '}', 0x2151 },
    { '<', 0x2152 },
    { '>', 0x2153 },
    { '[', 0x215a },
    { ']', 0x215b },
    { '+', 0x215c },
    { '-', 0x215d },
    { 'x', 0x215f },
    { '=', 0x2161 },
    { '$', 0x2170 },
    { '%', 0x2173 },
    { '#', 0x2174 },
    { '&', 0x2175 },
    { '*', 0x2176 },
    { '@', 0x2177 },
    { '*', 0x2179 },
    { '*', 0x217a },
    { '*', 0x2228 },
    { '=', 0x222e },
    { '0', 0x2330 },
    { '1', 0x2331 },
    { '2', 0x2332 },
    { '3', 0x2333 },
    { '4', 0x2334 },
    { '5', 0x2335 },
    { '6', 0x2336 },
    { '7', 0x2337 },
    { '8', 0x2338 },
    { '9', 0x2339 },
    { '\0', 0x0000 }
};

static struct {
	char *string;
    BYTE *real;
} ComposedEquivalents[] = {
    { "sha",  "\x37\x63" },
    { "shi",  "\x37" },
    { "shu",  "\x37\x65" },
    { "she",  "\x37\x27" },
    { "sho",  "\x37\x67" },

    { "ja",   "\x38\x63" },
    { "ji",   "\x38" },
    { "ju",   "\x38\x65" },
    { "je",   "\x38\x27" },
    { "jo",   "\x38\x67" },

    { "chi",  "\x41" },
    { "ci",   "\x41" },

    { "cha",  "\x41\x63" },
    { "chu",  "\x41\x65" },
    { "che",  "\x41\x27" },
    { "cho",  "\x41\x67" },

    { "tcha", "\x43\x41\x63" },
    { "tchi", "\x43\x41" },
    { "tci",  "\x43\x41" },
    { "tchu", "\x43\x41\x65" },
    { "tche", "\x43\x41\x27" },
    { "tcho", "\x43\x41\x67" },

    { "tsu",  "\x44" },
    { "tzu",  "\x44" },

    { "+tsu", "\x43" },
    { "+tzu", "\x43" },

    { "dyi",  "\x47\x23" },
    { "dyu",  "\x49\x25" },

    { "fa",   "\x55\x21" },
    { "fi",   "\x55\x23" },
    { "fu",   "\x55" },
    { "fe",   "\x55\x27" },
    { "fo",   "\x55\x29" },

    { "ye",   "\x24\x27" },

    { "kwa",  "\x2f\x21" },
    { "kwi",  "\x2f\x23" },
    { "kwe",  "\x2f\x27" },
    { "kwo",  "\x2f\x29" },

    { "kua",  "\x2f\x21" },
    { "kui",  "\x2f\x23" },
    { "kue",  "\x2f\x27" },
    { "kuo",  "\x2f\x29" },

    { "gwa",  "\x2f\x21" },
    { "gwi",  "\x2f\x23" },
    { "gwe",  "\x2f\x27" },
    { "gwo",  "\x2f\x29" },

    { "gua",  "\x2f\x21" },
    { "gui",  "\x2f\x23" },
    { "gue",  "\x2f\x27" },
    { "guo",  "\x2f\x29" },

    { "n'",   "\x73" },

    { "va",   "[\x25\x74\x25\x21]" },
    { "vi",   "[\x25\x74\x25\x23]" },
    { "vu",   "[\x25\x74]" },
    { "ve",   "[\x25\x74\x25\x27]" },
    { "vo",   "[\x25\x74\x25\x29]" },

    { "`",   "[\x21\x56]" },    /* These two are necessary because they */
    { "'",   "[\x21\x57]" },    /* translate to non-ASCII equivalents   */

    { "y=",  "[\x21\x6f]" },    /* These two start with a letter */
    { "f-",  "[\x21\x72]" },

    { NULL,  NULL }
};

static char *direct[] = {
    "+a", "a", "+i", "i", "+u", "u", "+e", "e", "+o", "o",
    "ka", "ga", "ki", "gi", "ku", "gu", "ke", "ge", "ko", "go",
    "sa", "za", "si", "zi", "su", "zu", "se", "ze", "so", "zo",
    "ta", "da", "ti", "di", "+tu", "tu", "du", "te", "de", "to", "do",
	"na", "ni", "nu", "ne", "no",
	"ha", "ba", "pa", "hi", "bi", "pi", "hu", "bu", "pu", "he", "be", "pe",
		"ho", "bo", "po",
	"ma", "mi", "mu", "me", "mo",
    "+ya", "ya", "+yu", "yu", "+yo", "yo",
	"ra", "ri", "ru", "re", "ro",
    "+wa", "wa", "wi", "we", "wo",
	"n",
	NULL
};



BOOL ConvertOK (FILEOPTIONS *f)
{
    if (strlen(Typed) == 1 && strchr("AEIOU", Typed[0]) != NULL) return (TRUE);
    if (SELPARA1(f) == NULL || SELPARA1(f) != SELPARA2(f)) return (FALSE);
    return (TRUE);
}



static int FindDirect (char *s)
{
	int i;

    for (i = 0; direct[i] != NULL; i++)
		if (!strcmp(direct[i], s)) return (i);

    return (-1);
}


KANJI TranslateJAscii (KANJI ch, BOOL JtoA)
{
    int i;
    char cch;

    if (JtoA) {
        ch &= 0x7f7f;
        for (i = 0; JAsciiEquivalents[i].ascii; i++) {
            if (JAsciiEquivalents[i].kanji == ch) return (JAsciiEquivalents[i].ascii);
        }
        if (/* A */ 0x2341 <= ch && ch <= 0x235a /* Z */) return ('A' + (ch - 0x2341));
        if (/* a */ 0x2361 <= ch && ch <= 0x237a /* z */) return ('A' + (ch - 0x2361));
    } else {
        cch = LOBYTE(ch);
        for (i = 0; JAsciiEquivalents[i].ascii; i++) {
            if (JAsciiEquivalents[i].ascii == cch) return (JAsciiEquivalents[i].kanji);
        }
        if ('A' <= cch && cch <= 'Z') return (0x2341 /* A */ + (cch - 'A'));
        if ('a' <= cch && cch <= 'z') return (0x2361 /* a */ + (cch - 'a'));
    }
    return (0);
}



char *ReverseKana (KANJI kana)
{
	int hi, lo;

    hi = HIBYTE(kana) & 0x7f;
    lo = LOBYTE(kana) & 0x7f;

    if (hi != 0x24 && hi != 0x25) return (NULL);
    if (lo < 0x21 && lo > 0x73) return (NULL);

    return (direct[lo - 0x21]);

}



void TurnOffSelection (FILEOPTIONS *f)
{
    if (SELPARA1(f) == NULL) return;

    FlipHighlight(f);
    SELPARA1(f) = SELPARA2(f) = NULL;
    SELPOS1(f) = SELPOS2(f) = 0;
    SELNOW(f) = FALSE;
    SELTYPE(f) = SEL_SELECTION;

    if (global.convsel >= 0 || kanji_list[0] != 0) {
        global.convsel = -1;
        kanji_list[0] = LastLen = LastConv[0] = 0;
        SendMessage(global.convhwnd, LB_RESETCONTENT, 0, NULL);
    }
}



void FlipHighlight (FILEOPTIONS *f)
{
	int i, j, k, x, w;
	int cgap, lgap;
    KANJI c1, c2, c3;
	BOOL insel = FALSE;
	HDC hdc, fhdc;
	POSITION p;


    /* Does it start in a selection? */

    insel = InSelection(f, f->top);

    HideCaret(f->hwnd);

    lgap = LINEGAP(f);
    cgap = CHARGAP(f);

    p = f->top;
    j = LINEGAP(f);

    if (PREVLINE(p)) j -= LINEOF(p)->height + PARAOF(p)->spacing;

    fhdc = f->hdc;
    f->hdc = hdc = GetDC(f->hwnd);

    do {
		if (j + lgap > f->height) break;
        j += LINEOF(p)->height;

        for (i = k = 0; k <= LINEOF(p)->length; k++) {
            if (k == 0 && PARAOF(p) == SELPARA2(f) && &UNITOF(p,0) > &SELCHAR2(f)) {
                ReleaseDC(f->hwnd, hdc);
                ShowCaret(f->hwnd);
                return;
            }

            if (LINEOF(p)->next != NULL && k == LINEOF(p)->length) break;

            x = i - f->startx + LEFTMARGIN(p) * BASEWIDTH(f);

            c1 = (k <= 0) ? 0 : CHAROF(p,k-1);
            c2 = CHAROF(p,k);
            c3 = (k < LINEOF(p)->length - 1) ? CHAROF(p,k+1) : 0;

            if (insel && (PARAOF(p) != SELPARA2(f) || LINEOF(p)->position + LINEOF(p)->length - 1 < SELPOS2(f))) {
                c3 = (LINEOF(p)->length > 0) ? CHAROF(p,LINEOF(p)->length - 1) : 0;
				if ((ISKANJI(c2) && c1 != '\t') || ISKANJI(c1) || (k <= 0 && LINEOF(p)->length > 0)) {
                    PatBlt(hdc, x - cgap, j - LINEOF(p)->height - lgap,
							LINEOF(p)->width - f->startx - x + (ISKANJI(c3) ? 0 : cgap),
                            LINEOF(p)->height + 2 * lgap, DSTINVERT);
                } else {
                    PatBlt(hdc, x, j - LINEOF(p)->height - lgap,
                            LINEOF(p)->width - f->startx - x,
                            LINEOF(p)->height + 2 * lgap, DSTINVERT);
                }
                break;
            } else if (!insel && PARAOF(p) == SELPARA1(f) && &UNITOF(p,k) >= &SELCHAR1(f)) {
                insel = TRUE;
                if (k >= LINEOF(p)->length) {
                    /* Nothing */
                } else if ((k <= 0 && LINEOF(p)->length > 0) || (ISKANJI(c2) && c1 != '\t') || ISKANJI(c1)) {
                    PatBlt(hdc, x - cgap, j - LINEOF(p)->height - lgap,
                           cgap, LINEOF(p)->height + 2 * lgap, DSTINVERT);
				}
            } else if (insel && k == 0 && LINEOF(p)->length > 0) {
                PatBlt(hdc, x - cgap, j - LINEOF(p)->height - lgap,
                       cgap, LINEOF(p)->height + 2 * lgap, DSTINVERT);
            } else if (insel && k > 0 && k < LINEOF(p)->length) {
                if ((ISKANJI(c2) && c1 != '\t') || ISKANJI(c1)) {
                    PatBlt(hdc, x - f->leading + cgap, j - LINEOF(p)->height - lgap,
                           f->leading - cgap, LINEOF(p)->height + 2 * lgap, DSTINVERT);
                }
            }

            POSOF(p) = k;

            w = LOWORD(GetDimension(f, p, i));  /* Char width */

            if (insel) {
                if (c2 == '\t') {
                    PatBlt(hdc, x, j - LINEOF(p)->height - lgap,
                            w, LINEOF(p)->height + 2 * lgap, DSTINVERT);
                } else if (ISKANJI(c2) || ISKANJI(c3)) {
                    PatBlt(hdc, x, j - LINEOF(p)->height - lgap,
                            w + cgap, LINEOF(p)->height + 2 * lgap, DSTINVERT);
                } else {
                    PatBlt(hdc, x, j - LINEOF(p)->height - lgap,
                            w, LINEOF(p)->height + 2 * lgap, DSTINVERT);
                }

                if (PARAOF(p) == SELPARA2(f) && &UNITOF(p,k) >= &SELCHAR2(f)) {
                    ReleaseDC(f->hwnd, hdc);
                    ShowCaret(f->hwnd);
					return;
                }
            }

            i += w;

            if (c2 != '\t') {
                if (ISKANJI(c2) || ISKANJI(c3)) i += f->leading;
            }
        }

        j += PARAOF(p)->spacing;
    } while (NEXTLINE(p));

    ReleaseDC(f->hwnd, hdc);
    ShowCaret(f->hwnd);

    f->hdc = fhdc;
}



void ReallocateText (POSITION p, unsigned int n)
{
	unsigned int len, size;


    /* The optimal block size */

    size = (n * sizeof(UNIT)) / TEXTBLOCKSIZE;
    size = (size + 1) * TEXTBLOCKSIZE;
    len = size / sizeof(UNIT);

    if (PARAOF(p)->textsize == len) return;


	/* Reallocate the block */

	PARAOF(p)->textsize = len;

    if (size >= C64K) {
        ErrorMessage(global.hwnd, "Paragraph too large (%u)!\nCannot reallocate!", PARAOF(p)->textsize);
		return;
    }

    PARAOF(p)->text = BlockRealloc(PARAOF(p)->text, size);
}



static BOOL InsideConversion (FILEOPTIONS *f, POSITION p)
{
    int abspos;

    if (SELPARA1(f) == NULL || SELPARA2(f) == NULL || SELTYPE(f) != SEL_CONVERSION)
        return (0);

    if (SELPARA1(f) != PARAOF(p) || SELPARA2(f) != PARAOF(p)) return (0);

    abspos = POS2ABS(p);

    if (SELPOS1(f) == abspos) return (1);
    if (SELPOS2(f) == abspos) return (3);
    if (SELPOS1(f) <= abspos && abspos <= SELPOS2(f)) return (2);
    if (abspos < SELPOS1(f)) return (0);
	if (abspos > SELPOS2(f)) return (4);
	return (0);
}



void InsertString (FILEOPTIONS *f, POSITION p, KANJI far *buf, int Options)
{
	int i;
	int len = kanjilen(buf);
	UNIT far *cp;
    ONELINE far *lp;

	if (len <= 0) return;

    if (!SELNOW(f)) {
        switch (InsideConversion(f, p)) {
            case 0: case 1: case 4: break;
            case 2: case 3:         TurnOffSelection(f); break;
        }
    }


    ReallocateText (p, unitlen(PARAOF(p)->text) + len + 1);

    cp = &UNITOF(f->current, CURCHAR(f));

    for (i = unitlen(PARAOF(p)->text) - LINEOF(p)->position; i >= POSOF(p); i--) {
		UNITOF(p,i+len) = UNITOF(p,i);

        if (cp == &UNITOF(p,i)) CURCHAR(f) += len;


        if (Options & OP_MOVESEL) {
            if (SELPARA1(f) == PARAOF(p) && &UNITOF(p,i) == &SELCHAR1(f))
				SELPOS1(f) += len;
            if (SELPARA2(f) == PARAOF(p) && &UNITOF(p,i) == &SELCHAR2(f))
                SELPOS2(f) += len;
        }
    }

	for (i = 0; i < len; i++) {
        CHAROF(p,POSOF(p)+i) = buf[i];
    }

    LINEOF(p)->length += len;
    f->nr_bytes += len;

    for (lp = LINEOF(p)->next; lp != NULL; lp = lp->next) lp->position += len;

    if (Options & OP_REFORMAT)
        ReformatParagraph(f, p, PARAOF(p)->next, Options);

    f->changed = TRUE;
}



void DeleteString (FILEOPTIONS *f, POSITION p, int len, int Options)
{
    int i;
	UNIT far *cp = &UNITOF(f->current, CURCHAR(f));
	ONELINE far *lp;
	POSITION tp;

	if (len <= 0) return;

    if (!SELNOW(f)) {
        tp = p;
        POSOF(tp) += (len - 1);

        switch (InsideConversion(f, p)) {
            case 0: case 4: break;
            case 1: case 2: case 3: TurnOffSelection(f); break;
        }
        switch (InsideConversion(f, tp)) {
            case 0: case 4: break;
            case 1: case 2: case 3: TurnOffSelection(f); break;
        }
        if (InsideConversion(f, p) == 0 && InsideConversion(f, tp) == 4)
            TurnOffSelection(f);
    }


    i = unitlen(&UNITOF(p,POSOF(p)));
	if (len > i) len = i;


    if (Options & OP_MOVESEL) {
        if (SELPARA1(f) == PARAOF(p) && &SELCHAR1(f) >= &UNITOF(p,POSOF(p)) &&
			&SELCHAR1(f) < &UNITOF(p,POSOF(p)+len))
                SELPOS1(f) = LINEOF(p)->position + POSOF(p) + len;

        if (SELPARA2(f) == PARAOF(p) && &SELCHAR2(f) >= &UNITOF(p,POSOF(p)) &&
            &SELCHAR2(f) < &UNITOF(p,POSOF(p)+len))
                SELPOS2(f) = LINEOF(p)->position + POSOF(p) - 1;

        if (SELPOS1(f) > SELPOS2(f)) {
			SELPARA1(f) = SELPARA2(f) = NULL;
            SELNOW(f) = FALSE;
        }
    }


	if (PARAOF(p) == CURPARA(f) && &UNITOF(p,POSOF(p)) <= cp && &UNITOF(p,POSOF(p)+len) > cp) {
        CURCHAR(f) = POSOF(p) + len;
		cp = &UNITOF(p,CURCHAR(f));
    }


	for (i = POSOF(p); ; i++) {
        UNITOF(p,i) = UNITOF(p,i+len);

        if (cp == &UNITOF(p,i+len)) {
            CURLINE(f) = LINEOF(p);
            CURCHAR(f) = i;
		}

		if (Options & OP_MOVESEL) {
            if (SELPARA1(f) == PARAOF(p) && &UNITOF(p,i+len) == &SELCHAR1(f))
				SELPOS1(f) -= len;
            if (SELPARA2(f) == PARAOF(p) && &UNITOF(p,i+len) == &SELCHAR2(f))
				SELPOS2(f) -= len;
		}

        if (!CHAROF(p,i)) break;
	}
	LINEOF(p)->length -= len;
	f->nr_bytes -= len;

    for (lp = LINEOF(p)->next; lp != NULL; lp = lp->next) lp->position -= len;

    ReallocateText (p, unitlen(PARAOF(p)->text));

    if (Options & OP_REFORMAT)
        ReformatParagraph(f, p, PARAOF(p)->next, Options);

    f->changed = TRUE;
}



void ReplaceString (FILEOPTIONS *f, POSITION p, int len, KANJI far *buf, int Options)
{
	int i, gap;
	POSITION p1;
	int buflen = kanjilen(buf);
	UNIT far *cp;

	if (buflen <= 0) {
		DeleteString (f, p, len, Options);
        return;
	} else if (len <= 0) {
		InsertString (f, p, buf, Options);
        return;
	}


	if (!SELNOW(f) && !(Options & OP_CHOOSEKANJI)) {
        p1 = p;
        POSOF(p1) += (len - 1);

        switch (InsideConversion(f, p)) {
            case 0: case 4: break;
            case 1: case 2: case 3: TurnOffSelection(f); break;
		}
        switch (InsideConversion(f, p1)) {
            case 0: case 4: break;
            case 1: case 2: case 3: TurnOffSelection(f); break;
        }
		if (InsideConversion(f, p) == 0 && InsideConversion(f, p1) == 4)
            TurnOffSelection(f);
    }


    gap = buflen - len;

    ReallocateText (p, unitlen(PARAOF(p)->text) + gap + 1);

    cp = &UNITOF(f->current, CURCHAR(f));


    if (gap > 0) {          /* Lengthen */

        for (i = unitlen(PARAOF(p)->text) - LINEOF(p)->position; i >= POSOF(p); i--) {
            UNITOF(p,i+gap) = UNITOF(p,i);

			if (cp == &UNITOF(p,i)) CURCHAR(f) += gap;

			if (Options & OP_MOVESEL) {
				if (SELPARA1(f) == PARAOF(p) && &UNITOF(p,i) == &SELCHAR1(f))
					SELPOS1(f) += gap;
                if (SELPARA2(f) == PARAOF(p) && &UNITOF(p,i) == &SELCHAR2(f))
					SELPOS2(f) += gap;
			}
        }
		LINEOF(p)->length += gap;
        f->nr_bytes += gap;
    } else {
        gap *= -1;

        for (i = POSOF(p); ; i++) {
            UNITOF(p,i) = UNITOF(p,i+gap);

			if (cp == &UNITOF(p,i+gap)) {
                CURLINE(f) = LINEOF(p);
                CURCHAR(f) = i;
            }

			if (Options & OP_MOVESEL) {
                if (SELPARA1(f) == PARAOF(p) && &UNITOF(p,i) == &SELCHAR1(f))
                    SELPOS1(f) -= gap;
                if (SELPARA2(f) == PARAOF(p) && &UNITOF(p,i) == &SELCHAR2(f))
                    SELPOS2(f) -= gap;
            }

            if (!CHAROF(p,i)) break;
		}
		LINEOF(p)->length -= gap;
		f->nr_bytes -= gap;
    }


    for (i = 0; i < buflen; i++) {
        CHAROF(p,POSOF(p)+i) = buf[i];
    }


    f->changed = TRUE;

	if (!(Options & OP_REFORMAT)) return;

    Options &= ~OP_MINIMAL;
    ReformatParagraph(f, p, PARAOF(p)->next, Options);
    return;
}



void ConvertNow (FILEOPTIONS *f, BOOL always)
{
    int i, j, r;
    int len, convlen, choice;
    POSITION p;
    BOOL Now;


    if (Typed[0]) CharInput(f, '\x07');     /* Bell */

    Now = SELNOW(f);

    if (!Now) {
        KANJI convbuf[MAXLINELEN];

        /* Convert the selected text */

        for (i = 0; i <= SELPOS2(f) - SELPOS1(f); i++) {
            KanjiBuffer[i] = SELPARA1(f)->text[SELPOS1(f)+i].kanji;
            if (HIBYTE(KanjiBuffer[i]) != 0x24) {
                MessageBeep(0);
                return;
            }
        }
        KanjiBuffer[i] = 0;

        r = FindConversion(KanjiBuffer, convbuf);

        if (r == 2 || r == 3) {         /* Exact match */
            LastLen = kanjilen(KanjiBuffer);
			kanjicpy(LastConv, convbuf);
        } else {
            MessageBeep(0);
            return;
        }
    }

    AbsoluteToPosition (SEL1(f), &p);

    convlen = SELPOS2(f) - SELPOS1(f) + 1;

    if (kanjilen(LastConv) <= 0 || (!always && convlen > LastLen)) {
		MessageBeep(0);
        return;

        if (Now) {
            SELPARA1(f) = SELPARA2(f) = NULL;
            SELPOS1(f) = SELPOS2(f) = 0;
            SELNOW(f) = FALSE;
            UndoFixConvert(f, p, convlen, 0);
            DeleteString(f, p, convlen, OP_REFORMAT | OP_UPDATE | OP_MOVETOEND | OP_CHOOSEKANJI);
        }
        return;
    }

    kanjicpy(kanji_list, LastConv);
    for (i = 0; i < convlen; i++) {
        KanjiBuffer[i] = SELPARA1(f)->text[SELPOS1(f) + i].kanji;
        LastConvKey[i] = LOBYTE(KanjiBuffer[i]);
    }
    KanjiBuffer[i] = 0;
    LastConvKey[LastLen] = 0;


	/* Is there a default? */

    choice = ConvCacheLookup(LastConvKey, kanji_list);


    /* Put in the kana reading to the end */

    len = kanjilen(kanji_list);
    if (len > 0 && kanji_list[len-1] != '/') {
        kanji_list[len++] = '/';
    }

    for (i = 0, j = len; i < LastLen; i++, j++) {
        kanji_list[j] = KanjiBuffer[i];
    }
    //kanji_list[j++] = '/';
    kanji_list[j] = 0;


    /* Update the conversion list */

    global.convsel = (choice >= 0) ? choice : 0;
    SendMessage(global.convhwnd, LB_RESETCONTENT, 0, (LONG) f);


    /* Find that kanji word */

    for (i = j = 0; j < global.convsel && kanji_list[i]; i++)
        if (kanji_list[i] == '/') j++;

    if (!kanji_list[i]) {
        for (i--; i >= 0 && kanji_list[i] != '/'; i--);
        i++;
	} else {
        if (kanji_list[i] == '/') i++;
    }

    for (j = 0; kanji_list[i] != 0; i++, j++) {
        if (kanji_list[i] == '/') break;
		KanjiBuffer[j] = kanji_list[i];
    }
    KanjiBuffer[j] = 0;

    SELPOS2(f) = SELPOS1(f) + kanjilen(KanjiBuffer) - 1;

    for (i = LastLen; i < convlen; i++)
		KanjiBuffer[j++] = SELPARA1(f)->text[SELPOS1(f)+i].kanji;
    KanjiBuffer[j] = 0;

    if (SELPOS2(f) < SELPOS1(f)) {
		SELPARA1(f) = SELPARA2(f) = NULL;
        SELPOS1(f) = SELPOS2(f) = 0;
    }

    if (Now) {
        UndoFixConvert(f, p, convlen, kanjilen(KanjiBuffer));
    } else {
        POSITION p1, p2;

        p1 = p;
        POSOF(p1) = POS2ABS(p1) + convlen;
        AbsoluteToPosition(p1, &p2);
        UndoAddReplace(f, p, p2);
        POSOF(p1) += (kanjilen(KanjiBuffer) - convlen);
        AbsoluteToPosition(p1, &p2);
        UndoFixReplace(f, p2);
    }

    ReplaceString(f, p, convlen, KanjiBuffer, OP_REFORMAT | OP_UPDATE | OP_MOVETOEND | OP_CHOOSEKANJI);

    SELNOW(f) = FALSE;
    SELTYPE(f) = SEL_CONVERSION;
}


static void TestForGlossary(FILEOPTIONS *f, POSITION p, KANJI *InputBuffer)
{
	int i, j, r;
    KANJI far *kp;
	KANJI buffer[MAXLINELEN];


    if (!global.dynamicglossary) return;

    for (i = 0; i <= Stackp; i++) {
        if (i == Stackp) {
            if (!InputBuffer[0]) break;
            kanjicpy(buffer, InputBuffer);
        } else {
            for (j = 0; j <= GlossaryStack[i].length; j++) {
                buffer[j] = CURPARA(f)->text[GlossaryStack[i].offset + j].kanji;
            }
			buffer[j] = 0;
        }

        r = SearchGlossary(buffer);

        if (r == -1) {          /* Partial match */
            if (i == Stackp) {
                GlossaryStack[Stackp].offset = POS2ABS(p);
                GlossaryStack[Stackp].length = 1;
                if (Stackp >= STACKDEPTH) {
                    ErrorMessage(global.hwnd, "The glossaries are too complicated!  "
                                                 "Stack overflow.  Some matchings may be lost.");
                } else {
					Stackp++;
                }
                break;
            } else {
                GlossaryStack[i].length++;
            }
        } else if (r == -2) {   /* No match */
            if (i != Stackp) {
                /* Delete this */
                for (j = i + 1; j < Stackp; j++) GlossaryStack[j-1] = GlossaryStack[j];
                i--;
                Stackp--;
            }
		} else {                /* Exact match! */

            p = f->current;
            POSOF(p) -= (GlossaryStack[i].length + 1);

            if (POSOF(p) < 0) {
                LINEOF(p) = LINEOF(p)->prev;
                POSOF(p) += LINEOF(p)->length;
            }

            kp = GetGlossary(r);

            ReplaceString(f, p, GlossaryStack[i].length + 1, kp,
                OP_REFORMAT | OP_UPDATE | OP_MOVETOEND/* | OP_CHOOSEKANJI*/);

			UndoFixConvert(f, p, GlossaryStack[i].length + 1, kanjilen(kp));
            Stackp = 0;
			break;
        }
    }
}



BOOL CharInput (FILEOPTIONS *f, WORD cch)
{
    int i, j, len, r;
    int option = OP_MOVETOEND;
    char ch;
    BOOL loopback = FALSE;
	BOOL allcaps;
    BOOL capitalized;
    POSITION p;
    KANJI convbuf[MAXLINELEN];
	UNIT far *cp;
    static int state = 0;



    ch = tolower(cch);

    /* Esc, Bell, Null, Backspace, Delete, Return */

    if (cch == 0 || cch == 0x07  /* Bell */) {
        Stackp = 0;
    } else if (Typed[0] && strchr("\x1b\b\x7f", cch) != NULL) {
        state = 0;
		Typed[0] = '\0';
        StatusMessage(Typed);
        return (FALSE);
	} else if ((cch == '\b' || cch == 127) && SELPARA1(f) != NULL && SELTYPE(f) != SEL_CONVERSION) {    /* Delete block */
        POSITION abs, start, stop;

        Stackp = 0;

        PARAOF(abs) = SELPARA1(f);
		POSOF(abs) = SELPOS1(f);

		AbsoluteToPosition(abs, &start);

		PARAOF(abs) = SELPARA2(f);
		POSOF(abs) = SELPOS2(f);

        AbsoluteToPosition(abs, &stop);

        /* Get rid of the selection */

        SELPARA1(f) = SELPARA2(f) = NULL;   SELPOS1(f) = SELPOS2(f) = 0;
        SELNOW(f) = FALSE;                  SELTYPE(f) = SEL_SELECTION;


        f->current = start;

        if (!FindCaret(f, FALSE)) {
            MoveIntoWindow(f);
			InvalidateRect(f->hwnd, NULL, TRUE);
			UpdateWindow(f->hwnd);
        }

        /* Move the cursor to the end */

        f->current = stop;
        if (CURCHAR(f) >= CURLINE(f)->length) {
            NEXTLINE(f->current);
        } else {
            CURCHAR(f)++;
        }

        UndoAddDelete(f, start, f->current);
        BlockDelete(f, start, stop);
	} else if (cch == '\b') {       /* Backspace */
        Stackp = 0;

        TakeCareOfThings(f, !SELNOW(f));

		if (CURCHAR(f) == 0 && CURLINE(f)->prev == NULL) {
            if (CURPARA(f)->prev != NULL) {
                CURPARA(f) = CURPARA(f)->prev;
				CURLINE(f) = CURPARA(f)->lastline;
                CURCHAR(f) = CURLINE(f)->length;

                /* Join two paragraphs */
                InputBuffer[0] = '\b';
                InputBuffer[1] = 0;
                UndoJoinLine(f, f->current);
                InsertString(f, f->current, InputBuffer,
						OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND | OP_MINIMAL);
			}
		} else {
            option = OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MINIMAL;

            if (CURCHAR(f) <= 0) {
				CURLINE(f) = CURLINE(f)->prev;
                CURCHAR(f) = CURLINE(f)->length;
                if (CHAROF(f->current,CURCHAR(f)) != '\n') CURCHAR(f)--;
                if (CURCHAR(f) < 0) CURCHAR(f) = 0;
            } else {
                CURCHAR(f)--;
            }

			cp = &UNITOF(f->current, CURCHAR(f));

            if (CURLINE(f)->prev != NULL && cp[-1].kanji != '\n' && cp[1].kanji == 0)
                option |= OP_MOVETOEND;

            UndoAddErase(f, f->current, 1);
            DeleteString(f, f->current, 1, option);
        }
    } else if (cch == 127) {        /* Delete */
        Stackp = 0;

        if (UNITOF(f->current,CURCHAR(f)).kanji == 0) {
            if (CURPARA(f)->next != NULL) {
                /* Join two paragraphs */
				InputBuffer[0] = '\b';
                InputBuffer[1] = 0;

                UndoJoinLine(f, f->current);
                InsertString(f, f->current, InputBuffer,
                    OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND | OP_MINIMAL);
            }
        } else {
            UndoAddErase(f, f->current, 1);
			DeleteString (f, f->current, 1,
                OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MINIMAL);
        }
    } else if (!Typed[0] && cch == '\r') {       /* Return */
        Stackp = 0;

        if (!(f->type & FN_NORMAL)) return (FALSE);       /* Not in a file */

        if (SELNOW(f)) {
            ConvertNow(f, TRUE);
            cch = '\0';
        } else {
            InputBuffer[0] = '\r';
            InputBuffer[1] = 0;
            UndoAddTyping(f, f->current, 1, TRUE);
            InsertString(f, f->current, InputBuffer,
                OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND | OP_MINIMAL);
        }
    } else if (!Typed[0] && cch == '\n') {       /* Soft return */
        Stackp = 0;

        if (!(f->type & FN_NORMAL)) return (FALSE);       /* Not in a file */

        if (SELNOW(f)) {
            ConvertNow(f, TRUE);
            cch = '\0';
        } else {
            InputBuffer[0] = '\n';
            InputBuffer[1] = 0;

            capitalized = allcaps = FALSE;
            option = 0;

            goto DoIns;
        }
	} else {
		/* Now check the allowed functions */

		if ((f->type & FN_EITHER) && f->nr_bytes > 0) {
			KANJI ch;

			ch = f->paragraph->text[0].kanji;

			switch (global.mode) {
				case M_KANA:    if (ISKANJI(ch)) break;
								MessageBeep(0);
								return (TRUE);

				case M_ASCII:   if (!ISKANJI(ch)) break;
								MessageBeep(0);
								return (TRUE);
			}
		}
	}


Loop:

    if (global.mode == M_ASCII && cch != '\t') {
        if ((cch >= ' ' && cch < 127) || (cch >= 128 + ' ' && cch < 255)) {
            InputBuffer[0] = cch;
            InputBuffer[1] = 0;
            p = f->current;

            if (SELPARA1(f) != NULL && SELTYPE(f) == SEL_SELECTION) {
                POSITION p1, p2;

                AbsoluteToPosition(SEL1(f), &p1);
                f->current = p1;
                if (!FindCaret(f, FALSE)) {
                    MoveIntoWindow(f);
                    InvalidateRect(f->hwnd, NULL, TRUE);
                    UpdateWindow(f->hwnd);
                }

				AbsoluteToPosition(SEL2(f), &p2);
				f->current = p2;
                if (CURCHAR(f) >= CURLINE(f)->length) {
                    NEXTLINE(f->current);
                } else {
                    CURCHAR(f)++;
                }
                TakeCareOfThings(f, FALSE);
                UndoAddReplace(f, p1, f->current);
                BlockReplace(f, p1, p2, InputBuffer);
                UndoFixReplace(f, f->current);
            } else {
                TakeCareOfThings(f, FALSE);
                UndoAddTyping(f, f->current, 1, FALSE);
                InsertString(f, f->current, InputBuffer,
                    OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND | OP_MINIMAL);
            }
            TestForGlossary(f, p, InputBuffer);
        }
        return (TRUE);
    }

    len = strlen(Typed);


	/* Get the next state */

	for (i = 0; states[i].state >= 0 && states[i].state != state; i++);

    for (; states[i].state == state; i++) {
        if (strchr(states[i].input, ch) != NULL) break;
	}

    if (states[i].state != state) {
        if (state == 0) {
            Typed[0] = '\0';

            if (cch == '\t') InputBuffer[0] = '\t';
            else {
                InputBuffer[0] = TranslateJAscii(cch, FALSE);
                if (!InputBuffer[0]) {
                    StatusMessage (Typed);
                    return (FALSE);
                }
            }
            InputBuffer[1] = 0;
            capitalized = allcaps = loopback = FALSE;
            goto DoIns;
        } else {
            state = 0;
			loopback = TRUE;
        }
    } else {
		if (cch >= ' ') Typed[len] = cch;
        Typed[++len] = '\0';

        state = states[i].next;
        if (state >= 0) {
            StatusMessage (Typed);
            return (FALSE);
        }

        loopback = FALSE;
	}

    /* Completed */

    if (strcmp(Typed, "N'")) {
        capitalized = isupper(Typed[0]);
        for (i = 0, allcaps = TRUE; Typed[i]; i++) {
            if (islower(Typed[i])) {
                allcaps = FALSE;
                break;
            }
        }
    } else {
        capitalized = FALSE;
        allcaps = TRUE;
    }


    StatusMessage(Typed);

    /* Translate the keystroke */

    if (loopback && len == 1 && Typed[0] == cch &&
        isalpha(Typed[0]) && strchr("AIUEOaiueo", cch) == NULL &&
        (!global.nnconvert || strchr("Nn", cch) == NULL)) {

        InputBuffer[0] = (allcaps ? 0x2521 : 0x2421) + FindDirect("+tu");
        InputBuffer[1] = 0;
    } else if (!loopback && len == 1 && strchr("AIUEO", Typed[0])) {
		state = -1;      /* Impossible state */
		return (FALSE);
    } else if (loopback && len == 1 && strchr("AIUEO",Typed[0]) != NULL) {
        AnsiLower(Typed);
        if (islower(cch) || cch == 0x07  /* Bell */) {
            capitalized = TRUE;
            allcaps = FALSE;
            InputBuffer[0] = 0x2421 + FindDirect(Typed);
        } else {
            /* An upper case follows capital A I U E O */
            loopback = TRUE;
            state = 0;
            InputBuffer[0] = 0x2521 + FindDirect(Typed);
            capitalized = FALSE;
			allcaps = TRUE;
        }
        InputBuffer[1] = 0;
    } else if (len == 3 && tolower(Typed[0]) != 'd' && Typed[0] != '+' && tolower(Typed[1]) == 'y') {
        AnsiLower(Typed);
        if (Typed[2] == 'e') {
            strcpy(Temp, "+e");
        } else {
            strcpy(Temp, Typed);
        }
        Temp[0] = '+';
        Typed[1] = 'i';
        Typed[2] = '\0';
		if (Typed[0] == 'z') Typed[0] = 'j';

        InputBuffer[0] = (allcaps ? 0x2521 : 0x2421) + FindDirect(Typed);
        InputBuffer[1] = (allcaps ? 0x2521 : 0x2421) + FindDirect(Temp);

        InputBuffer[2] = 0;
    } else {
        AnsiLower(Typed);
        for (i = 0; ComposedEquivalents[i].string != NULL; i++) {
            if (!strcmp(ComposedEquivalents[i].string, Typed)) break;
        }
        if (ComposedEquivalents[i].string != NULL) {
            if (ComposedEquivalents[i].real[0] != '[') {
                for (j = 0; ComposedEquivalents[i].real[j]; j++)
                    InputBuffer[j] = (allcaps ? 0x2500 : 0x2400) + ComposedEquivalents[i].real[j];
                InputBuffer[j] = 0;
            } else {
                int t;

                for (t = 0; ComposedEquivalents[i].real[t + 1] != ']'; t += 2) {
                    InputBuffer[t/2] = (ComposedEquivalents[i].real[t + 1] << 8) | ComposedEquivalents[i].real[t + 2];
                }
                InputBuffer[t/2] = 0;
            }
        } else {
            j = FindDirect(Typed);
            if (j < 0) {
                Typed[0] = '\0';
                if (loopback) goto Loop;
                return (FALSE);
			}
            InputBuffer[0] = (allcaps ? 0x2521 : 0x2421) + j;
            InputBuffer[1] = 0;
        }
	}


DoIns:

    state = 0;
    Typed[0] = '\0';

    if (SELNOW(f) && capitalized && !(f->type & FN_NOKANJI)) ConvertNow(f, TRUE);

    if (SELNOW(f)) {
        SELPOS2(f) += kanjilen(InputBuffer);
    } else if (capitalized && !allcaps && !(f->type & FN_NOKANJI)) {
        FlipHighlight(f);
        SELPARA1(f) = SELPARA2(f) = CURPARA(f);
        SELPOS1(f) = CURLINE(f)->position + CURCHAR(f);
        SELPOS2(f) = SELPOS1(f) + kanjilen(InputBuffer) - 1;
        SELNOW(f) = TRUE;
		SELTYPE(f) = SEL_CONVERSION;
        global.convsel = -1;
        kanji_list[0] = LastLen = LastConv[0] = 0;
        SendMessage(global.convhwnd, LB_RESETCONTENT, 0, NULL);
	}

    if (!f->caret) {
        MoveIntoWindow(f);
        InvalidateRect(f->hwnd, NULL, TRUE);
		UpdateWindow(f->hwnd);
    }


    /* may start as 0 or OP_MOVETOEND */

    option |= (OP_REFORMAT | OP_UPDATE | OP_MINIMAL);
    if (!SELNOW(f)) option |= OP_MOVESEL;

    p = f->current;

    /* Replace? */

    if (SELPARA1(f) != NULL && SELTYPE(f) == SEL_SELECTION) {
        POSITION p1, p2;

        AbsoluteToPosition(SEL1(f), &p1);
        f->current = p1;
        if (!FindCaret(f, FALSE)) {
            MoveIntoWindow(f);
            InvalidateRect(f->hwnd, NULL, TRUE);
            UpdateWindow(f->hwnd);
        }

        AbsoluteToPosition(SEL2(f), &p2);
        f->current = p2;
        if (CURCHAR(f) >= CURLINE(f)->length) {
            NEXTLINE(f->current);
        } else {
            CURCHAR(f)++;
        }
        TakeCareOfThings(f, FALSE);
        UndoAddReplace(f, p1, f->current);
        BlockReplace(f, p1, p2, InputBuffer);
        UndoFixReplace(f, f->current);
    } else {
        TakeCareOfThings(f, FALSE);
        UndoAddTyping(f, f->current, kanjilen(InputBuffer), FALSE);
        InsertString(f, f->current, InputBuffer, option);
    }

    if (SELNOW(f)) {
        /* Is kana? */

        if (HIBYTE(SELCHAR2(f).kanji) == 0x24) {
            for (i = 0; i <= SELPOS2(f) - SELPOS1(f); i++)
                KanjiBuffer[i] = SELPARA1(f)->text[SELPOS1(f)+i].kanji;
            KanjiBuffer[i] = 0;

            r = FindConversion(KanjiBuffer, convbuf);
        } else {
            r = 0;
        }

        if (r == 2 || r == 3) {         /* Exact match */
            LastLen = kanjilen(KanjiBuffer);
			kanjicpy(LastConv, convbuf);
        }

        if (r == 0 || r == 2) {         /* No future hope */
            ConvertNow(f, TRUE);
        } else if (global.dynamicconvert) {
            kanjicpy(kanji_list, convbuf);
            SendMessage(global.convhwnd, LB_RESETCONTENT, 0, (LONG) f);
        }
    } else {
        TestForGlossary(f, p, InputBuffer);
    }

    if (loopback) goto Loop;

    StatusMessage (Typed);

    return (TRUE);
}
