/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"

#include "idm.h"

#include <math.h>


#define CTRLCHAR    '^'
#define SHIFTCHAR   '+'

#define TTYPE_FONTTYPE      0x04
#define PSBLOCKSIZE         20

#define MAXUNDOLEVELS       32000
#define DEFUNDOLEVELS       50


typedef struct {
    BOOL ctrl;
    BOOL shift;
    char alpha;
    WORD scancode;
} HOTKEY;


typedef struct {
    char *desc;
    int menuid;
	HOTKEY hotkey;
} JWPFUNC;


#define ALPHASTART      4221
#define FUNCSTART       4251

static FARPROC OldStaticProc;
static int Selected;
static int CurrentFunction = 1;

static HOTKEY OldKey, CurrentKey;

static JWPFUNC jwpfunc[] = {
    { "Create New File",        IDM_FILENEW,            { 1, 0, 'N', 0 } },
    { "Open File",              IDM_FILEOPEN,           { 1, 0, 'O', 0 } },
    { "Close File",             IDM_FILECLOSE,          { 1, 0, 'X', 0 } },
    { "Save File",              IDM_FILESAVE,           { 0, 0, 0, VK_F11 } },
    { "Save as Something Else", IDM_FILESAVEAS,         { 0, 0, 0, VK_F12 } },
    { "Find File",              IDM_FILEFIND,           { 0, 0, 0, 0 } },
    { "Print File",             IDM_FILEPRINT,          { 1, 0, 'P', 0 } },
//  { "Print Preview",          IDM_FILEPREVIEW,        { 1, 0, 'V', 0 } },
    { "Printer Setup",          IDM_FILEPRINTER,        { 0, 0, 0, 0 } },
    { "Edit File Summary",      IDM_FILESUMMARY,        { 0, 0, 0, 0 } },
    { "Edit Header/Footer",     IDM_FILEHEADERS,        { 0, 0, 0, 0 } },
    { "Toggle Kana/Text mode",  IDM_TOGGLEMODE,         { 0, 0, 0, VK_F4 } },
    { "Toggle Glossary List",   IDM_VIEWGLOSSARY,       { 1, 0, 'G', 0 } },
    { "Toggle Draft View",      IDM_VIEWDRAFT,          { 1, 0, 0, VK_F2 } },
    { "Toggle Iconbar",         IDM_VIEWRIBBON,         { 0, 0, 0, 0 } },
    { "Toggle Status Bar",      IDM_VIEWSPECIAL,        { 0, 0, 0, 0 } },
    { "Toggle File Ruler",      IDM_VIEWSPECIAL,        { 0, 0, 0, 0 } },
    { "Toggle Special Symbols", IDM_VIEWSPECIAL,        { 0, 0, 0, 0 } },
    { "Toggle Dynamic Glossary",IDM_TOGGLEGLOSSARY,     { 0, 0, 0, 0 } },
    { "Search for Text",        IDM_EDITSEARCH,         { 1, 0, 'S', 0 } },
    { "Continue Search/Replace",IDM_CONTINUESEARCH,     { 0, 0, 0, 0 } },
    { "Replace Text",           IDM_EDITREPLACE,        { 1, 0, 'R', 0 } },
    { "Convert Kana to Kanji (L->R)", IDM_KANJICONVERT, { 0, 0, 0, VK_F2 } },
    { "Convert Kana to Kanji (R->L)", IDM_KANJILCONVERT,{ 0, 1, 0, VK_F2 } },
//  { "Revert Kanji to Kana",   IDM_KANJIREVERSE,       { 0, 0, 0, 0 } },
    { "Kanji Lookup",           IDM_KANJILOOKUP,        { 1, 0, 'L', 0 } },
    { "Kanji Info",             IDM_KANJIINFO,          { 1, 0, 'K', 0 } },
    { "JIS Input",              IDM_KANJIINPUT,         { 1, 0, 'J', 0 } },
    { "JIS Table",              IDM_KANJITABLE,         { 1, 0, 'T', 0 } },
    { "Format Paragraph",       IDM_FORMATPARAGRAPH,    { 0, 0, 0, 0 } },
    { "Page Setup",             IDM_FORMATPAGE,         { 0, 0, 0, 0 } },
	{ "Set Options",            IDM_UTILITIESOPTIONS,   { 0, 0, 0, 0 } },
    { "Change Hot Keys",        IDM_UTILITIESHOTKEYS,   { 0, 0, 0, 0 } },
    { "Maintain Kanji Fonts",   IDM_UTILITIESFONTS,     { 0, 0, 0, 0 } },
    { "Show Statistics",        IDM_UTILITIESSTAT,      { 0, 0, 0, 0 } },
    { "Dictionary",             IDM_UTILITIESDICT,      { 0, 0, 0, VK_F6 } },
    { "Edit Glossary",          IDM_UTILITIESGLOSSARY,  { 0, 0, 0, 0 } },
    { "Edit User Conversions",  IDM_UTILITIESCONVERT,   { 0, 0, 0, 0 } },
    { "Cascade File Windows",   IDM_WINDOWCASCADE,      { 0, 1, 0, VK_F4 } },
    { "Tile File Windows",      IDM_WINDOWTILE,         { 0, 1, 0, VK_F5 } },
    { "Close all File Windows", IDM_WINDOWCLOSE,        { 0, 0, 0, 0 } },
    { "Help",                   IDM_HELPINDEX,          { 0, 0, 0, VK_F1 } }
};

#define NRJWPFUNC   (sizeof(jwpfunc) / sizeof(JWPFUNC))

static HOTKEY HotKeysBackup[NRJWPFUNC];
BOOL OptionsChanged = FALSE;

static double far *PointSizes = NULL;

/* Option backup's */

static RECT glistsize, convbar;
static FILEFORMAT clipboard;
static BOOL DraftView, ShowSpecial, ShowRuler, NNConvert;
static BOOL ProgramMaximized;
static RECT ProgramPosition;
static int SysFontNum, BaseFontNum, PrintFontNum;
static char *AsciiFontName = NULL;
static double AsciiFontSize;
static MEASUREMENT Measure;
static PAGESETUP PageSetup;
static JUSTIFICATION LongJust, ShortJust;
static int UndoLevels, SaveUndoLevels;



WORD DecodeKeyStroke (WORD message, WORD wParam, BOOL shift, BOOL ctrl)
{
	int i;

	if (message == WM_KEYDOWN) {
		for (i = 0; i < NRJWPFUNC; i++) {
            if (jwpfunc[i].hotkey.alpha == 0 &&
                jwpfunc[i].hotkey.scancode == wParam &&
                jwpfunc[i].hotkey.shift == shift &&
                jwpfunc[i].hotkey.ctrl == ctrl)
                    return (jwpfunc[i].menuid);
		}
	} else if (message == WM_CHAR) {
		for (i = 0; i < NRJWPFUNC; i++) {
			if (jwpfunc[i].hotkey.alpha == wParam) return (jwpfunc[i].menuid);
		}
	}

	return (0);
}



static int ConvertKeyCode (HOTKEY key, char *buffer)
{
	buffer[0] = '\0';

    if (key.alpha == 0) {
        if (key.shift) strcat(buffer, "Shift+");
        if (key.ctrl) strcat(buffer, "Ctrl+");

        switch (key.scancode) {
			case VK_F1:  strcat(buffer, "F1"); return (FUNCSTART + 0);
			case VK_F2:  strcat(buffer, "F2"); return (FUNCSTART + 1);
			case VK_F3:  strcat(buffer, "F3"); return (FUNCSTART + 2);
			case VK_F4:  strcat(buffer, "F4"); return (FUNCSTART + 3);
			case VK_F5:  strcat(buffer, "F5"); return (FUNCSTART + 4);
			case VK_F6:  strcat(buffer, "F6"); return (FUNCSTART + 5);
			case VK_F7:  strcat(buffer, "F7"); return (FUNCSTART + 6);
			case VK_F8:  strcat(buffer, "F8"); return (FUNCSTART + 7);
			case VK_F9:  strcat(buffer, "F9"); return (FUNCSTART + 8);
			case VK_F10: strcat(buffer, "F10"); return (FUNCSTART + 9);
			case VK_F11: strcat(buffer, "F11"); return (FUNCSTART + 10);
			case VK_F12: strcat(buffer, "F12"); return (FUNCSTART + 11);
		}

        return (-1);
    }


    if (key.alpha < 'A' || key.alpha > 'Z') return (-1);

    sprintf(buffer, "Ctrl+%c", key.alpha);

    return (ALPHASTART + (key.alpha - 'A'));
}



static void IDToKeyCode (int id, HOTKEY *key)
{
    if (id >= FUNCSTART) {
        key->alpha = 0;
        switch (id - FUNCSTART) {
			case 0:  key->scancode = VK_F1; break;
            case 1:  key->scancode = VK_F2; break;
            case 2:  key->scancode = VK_F3; break;
            case 3:  key->scancode = VK_F4; break;
            case 4:  key->scancode = VK_F5; break;
            case 5:  key->scancode = VK_F6; break;
            case 6:  key->scancode = VK_F7; break;
            case 7:  key->scancode = VK_F8; break;
            case 8:  key->scancode = VK_F9; break;
            case 9:  key->scancode = VK_F10; break;
            case 10: key->scancode = VK_F11; break;
            case 11: key->scancode = VK_F12; break;
            default: key->scancode = VK_F1; break;
        }
    } else {
        key->alpha = 'A' + (id - ALPHASTART);
        key->scancode = 0;
    }
}


static BOOL HotKeyEqual (HOTKEY k1, HOTKEY k2)
{
    if (k1.alpha == 0) {
		return (k1.shift == k2.shift &&
				k1.ctrl == k2.ctrl &&
                k1.scancode == k2.scancode);
    } else {
        return (k1.alpha == k2.alpha);
    }
}



void ProcessMenuTexts (void)
{
    int i, j;
    WORD menuid;
    char buffer[MAXLINELEN], keytext[50];

    for (i = 0; i < NRJWPFUNC; i++) {
        menuid = jwpfunc[i].menuid;
        GetMenuString(hmenu, menuid, buffer, MAXLINELEN, MF_BYCOMMAND);

        for (j = 0; buffer[j] && buffer[j] != '\t'; j++);
        buffer[j] = '\0';

        if (ConvertKeyCode(jwpfunc[i].hotkey, keytext) >= 0) {
            strcat(buffer, "\t");
            strcat(buffer, keytext);
        }

        j = GetMenuState(hmenu, menuid, MF_BYCOMMAND);
        j |= (MF_BYCOMMAND | MF_STRING);

        ModifyMenu(hmenu, menuid, j, menuid, buffer);
    }

    /* Now check the menu items */

    if (global.draftview)
        CheckMenuItem(hmenu, IDM_VIEWDRAFT, MF_CHECKED | MF_BYCOMMAND);
    if (global.showruler)
        CheckMenuItem(hmenu, IDM_VIEWRULER, MF_CHECKED | MF_BYCOMMAND);
    if (global.tbhwnd != NULL)
        CheckMenuItem(hmenu, IDM_VIEWRIBBON, MF_CHECKED | MF_BYCOMMAND);
    if (global.statushwnd != NULL)
        CheckMenuItem(hmenu, IDM_VIEWSTATUS, MF_CHECKED | MF_BYCOMMAND);
    if (global.glisthwnd != NULL)
        CheckMenuItem(hmenu, IDM_VIEWGLOSSARY, MF_CHECKED | MF_BYCOMMAND);
    if (global.showspecial)
        CheckMenuItem(hmenu, IDM_VIEWSPECIAL, MF_CHECKED | MF_BYCOMMAND);
}



LONG FAR PASCAL HotKeyChoiceProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
        case WM_LBUTTONDOWN: {
            int i;
            HWND OldSelected;

            i = GetDlgCtrlID(hwnd);
            if (i == Selected) return (0);

            OldSelected = GetDlgItem(GetParent(hwnd), Selected);
            SendMessage(hwnd, WM_USER, 0, 0L);
            SendMessage(OldSelected, WM_USER, 0, 0L);

            Selected = i;
            IDToKeyCode(i, &CurrentKey);
            EnableWindow(GetDlgItem(GetParent(hwnd), 4211), (CurrentKey.alpha == 0));
            EnableWindow(GetDlgItem(GetParent(hwnd), 4212), (CurrentKey.alpha == 0));

            SendMessage(GetParent(hwnd), WM_USER, 0, 0L);
            return (0);
        }

        case WM_USER: {
            RECT rect;
            HDC hdc;

            GetClientRect(hwnd, &rect);
            hdc = GetDC(hwnd);
            PatBlt(hdc, 1, 1, rect.right - 2, rect.bottom - 2, DSTINVERT);
            ReleaseDC(hwnd, hdc);
            return (TRUE);
        }

        case WM_PAINT: {
            int i, width, height;
            DWORD dimensions;
            HDC hdc;
            PAINTSTRUCT ps;
            RECT rect;
            char buf[5];

            GetClientRect(hwnd, &rect);

            hdc = BeginPaint(hwnd, &ps);
            Create3DEffect (hdc, &rect, 1, 0);
            SetBkMode(hdc, TRANSPARENT);

            i = GetDlgCtrlID(hwnd);

            if (i >= FUNCSTART) {
                sprintf(buf, "F%d", i - FUNCSTART + 1);

                dimensions = GetTextExtent(hdc, buf, strlen(buf));
            } else {
                sprintf(buf, "%c", 'A' + (i - ALPHASTART));

                dimensions = GetTextExtent(hdc, buf, 1);
            }

            width = LOWORD(dimensions);
            height = HIWORD(dimensions);

            TextOut(hdc, (rect.right - width) / 2, (rect.bottom - height) / 2,
                    buf, strlen(buf));

            if (i == Selected)
                PatBlt(hdc, 1, 1, rect.right - 2, rect.bottom - 2, DSTINVERT);

            EndPaint(hwnd, &ps);
            return (0);
        }

        default: break;
    }

    return (CallWindowProc(OldStaticProc, hwnd, message, wParam, lParam));
}



BOOL FAR PASCAL ChangeHotKeyProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int i;
    POINT point;
	HWND child;
    char buffer[MAXLINELEN], keytext[50];

    switch (message) {
        case WM_INITDIALOG:
            OldStaticProc = (FARPROC) GetWindowLong(GetDlgItem(hwnd, ALPHASTART), GWL_WNDPROC);

            for (i = 0; i < 26; i++) {
                if (strchr("HIM", 'A' + i) != NULL) continue;
                SetWindowLong(GetDlgItem(hwnd, ALPHASTART + i), GWL_WNDPROC, (LONG) HotKeyChoiceProc);
            }

            for (i = 0; i < 12; i++)
                SetWindowLong(GetDlgItem(hwnd, FUNCSTART + i), GWL_WNDPROC, (LONG) HotKeyChoiceProc);

            Selected = ConvertKeyCode(OldKey, keytext);
            CurrentKey = OldKey;

            sprintf(buffer, "Hot Key for \"%s\"", jwpfunc[CurrentFunction].desc);
            SetDlgItemText(hwnd, 4201, buffer);

            if (Selected >= 0) {
                sprintf(buffer, "Old Hot Key:\t%s", keytext);
                SetDlgItemText(hwnd, 4202, buffer);
                sprintf(buffer, "New Hot Key:\t%s", keytext);
                SetDlgItemText(hwnd, 4203, buffer);
            } else {
                SetDlgItemText(hwnd, 4202, "Old Hot Key:\tNone");
                SetDlgItemText(hwnd, 4203, "New Hot Key:\tNone");
            }

            CheckDlgButton(hwnd, 4211, OldKey.ctrl);
            CheckDlgButton(hwnd, 4212, OldKey.shift);

            EnableWindow(GetDlgItem(hwnd, 4211), (CurrentKey.alpha == 0));
            EnableWindow(GetDlgItem(hwnd, 4212), (CurrentKey.alpha == 0));

            CenterDialogBox(hwnd);

            return (TRUE);

        case WM_LBUTTONDOWN:

            point.x = LOWORD(lParam);
            point.y = HIWORD(lParam);

            child = ChildWindowFromPoint(hwnd, point);

            if (child == hwnd || child == NULL) return (TRUE);

            for (i = 0; i < 26; i++) {
                if (child == GetDlgItem(hwnd, ALPHASTART + i)) {
                    SendDlgItemMessage(hwnd, ALPHASTART + i, WM_LBUTTONDOWN, wParam, lParam);
                    return (TRUE);
                }
            }
            for (i = 0; i < 12; i++) {
                if (child == GetDlgItem(hwnd, FUNCSTART + i)) {
                    SendDlgItemMessage(hwnd, FUNCSTART + i, WM_LBUTTONDOWN, wParam, lParam);
                    return (TRUE);
                }
            }

            return (TRUE);

        case WM_USER:
            if (ConvertKeyCode(CurrentKey, keytext) >= 0) {
                sprintf(buffer, "New Hot Key:\t%s", keytext);
                SetDlgItemText(hwnd, 4203, buffer);
            } else {
                SetDlgItemText(hwnd, 4203, "New Hot Key:\tNone");
            }
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case 4211:      /* Check boxes */
                case 4212:
                    CurrentKey.ctrl = IsDlgButtonChecked(hwnd, 4211);
                    CurrentKey.shift = IsDlgButtonChecked(hwnd, 4212);

                    if (ConvertKeyCode(CurrentKey, keytext) >= 0) {
                        sprintf(buffer, "New Hot Key:\t%s", keytext);
                        SetDlgItemText(hwnd, 4203, buffer);
                    }
                    return (TRUE);

                case IDOK:
                    if (CurrentKey.alpha != 0 && strchr("HIM", CurrentKey.alpha) != NULL) {
                        ConvertKeyCode(CurrentKey, keytext);
                        ErrorMessage(hwnd, "You cannot assign %s to anything", keytext);
                        return (TRUE);
                    }

                    for (i = 0; i < NRJWPFUNC; i++) {
                        if (i == CurrentFunction) continue;
                        if (HotKeyEqual(CurrentKey, jwpfunc[i].hotkey)) break;
					}

					if (i < NRJWPFUNC) {
                        if (ConvertKeyCode(CurrentKey, keytext) >= 0) {
                            ErrorMessage(hwnd, "The key %s is already assigned to\n"
                                                  "\"%s\"", keytext, jwpfunc[i].desc);
                            return (TRUE);
                        }
					}

                    if (CurrentKey.alpha != 0) CurrentKey.shift = CurrentKey.ctrl = FALSE;

					jwpfunc[CurrentFunction].hotkey = CurrentKey;

                    OptionsChanged = TRUE;

					ProcessMenuTexts();
					EndDialog(hwnd, FALSE);
					return (TRUE);

				case IDCANCEL:
					EndDialog(hwnd, FALSE);
                    return (TRUE);
            }
    }

    return (FALSE);
}



BOOL FAR PASCAL HotKeysProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int i;

    switch (message) {
        case WM_INITDIALOG: {
            char buffer[MAXLINELEN];

            for (i = 0; i < NRJWPFUNC; i++) {
                SendDlgItemMessage(hwnd, 4201, LB_ADDSTRING, 0, (LONG) jwpfunc[i].desc);
                HotKeysBackup[i] = jwpfunc[i].hotkey;
            }

            SendDlgItemMessage(hwnd, 4201, LB_SETCURSEL, 0, 0L);

            if (ConvertKeyCode(jwpfunc[0].hotkey, buffer) >= 0) {
                SetDlgItemText(hwnd, 4202, buffer);
            } else {
                SetDlgItemText(hwnd, 4202, "None");
            }

			CenterDialogBox(hwnd);

            return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case 4201:      /* List Box */
					switch (HIWORD(lParam)) {
						case LBN_SELCHANGE: {
							char buffer[MAXLINELEN];

							i = SendDlgItemMessage(hwnd, 4201, LB_GETCURSEL, 0, 0L);
							if (i == LB_ERR) {
								MessageBeep(0);
								return (TRUE);
							}
							if (ConvertKeyCode(jwpfunc[i].hotkey, buffer) >= 0) {
								SetDlgItemText(hwnd, 4202, buffer);
							} else {
								SetDlgItemText(hwnd, 4202, "None");
							}
							return (TRUE);
						}

						case LBN_DBLCLK:
							i = SendDlgItemMessage(hwnd, 4201, LB_GETCURSEL, 0, 0L);
							if (i == LB_ERR) {
                                MessageBeep(0);
                                return (TRUE);
                            }
                            SendMessage(hwnd, WM_COMMAND, 4211, 0L);
                            return (TRUE);
                    }
                    return (TRUE);

                case 4211: {    /* Change */
                    char buffer[MAXLINELEN];

                    i = SendDlgItemMessage(hwnd, 4201, LB_GETCURSEL, 0, 0L);
                    if (i == LB_ERR) {
                        MessageBeep(0);
                        return (TRUE);
                    }
                    CurrentFunction = i;
                    OldKey = CurrentKey = jwpfunc[i].hotkey;

                    DialogBox (hInstance, "ChangeHotKey", hwnd, ChangeHotKeyProc);

                    if (ConvertKeyCode(jwpfunc[i].hotkey, buffer) >= 0) {
                        SetDlgItemText(hwnd, 4202, buffer);
                    } else {
                        SetDlgItemText(hwnd, 4202, "None");
                    }

                    return (TRUE);
                }

                case 4212:      /* Delete */
                    i = SendDlgItemMessage(hwnd, 4201, LB_GETCURSEL, 0, 0L);
                    if (i == LB_ERR) {
                        MessageBeep(0);
                        return (TRUE);
                    }
                    CurrentKey.shift = CurrentKey.alpha = FALSE;
                    CurrentKey.alpha = CurrentKey.scancode = 0;
                    jwpfunc[i].hotkey = CurrentKey;
                    SetDlgItemText(hwnd, 4202, "None");
                    return (TRUE);

                case IDCANCEL:
                    for (i = 0; i < NRJWPFUNC; i++) jwpfunc[i].hotkey = HotKeysBackup[i];
					EndDialog(hwnd, FALSE);
                    return (TRUE);

                case IDOK:
                    for (i = 0; i < NRJWPFUNC; i++) {
						if (HotKeysBackup[i].alpha != jwpfunc[i].hotkey.alpha) {
                            OptionsChanged = TRUE;
                            break;
                        }
                        if (jwpfunc[i].hotkey.alpha == 0) {
                            if (jwpfunc[i].hotkey.shift != HotKeysBackup[i].shift ||
                                jwpfunc[i].hotkey.ctrl != HotKeysBackup[i].ctrl ||
                                jwpfunc[i].hotkey.scancode != HotKeysBackup[i].scancode)
                                    OptionsChanged = TRUE;
									break;
						}
					}

                    ProcessMenuTexts();
					EndDialog(hwnd, FALSE);
                    return (TRUE);
            }
    }

	return (FALSE);
}



char *DupString (char *s)
{
	char *cp;

	cp = MemAlloc(strlen(s) + 5);

	if (cp != NULL) strcpy(cp, s);

	return (cp);
}



BOOL ReadProgramSizeOptions (int *x, int *y, int *width, int *length)
{
    OFSTRUCT of;
    char buffer[MAXLINELEN];
	char OptionsFileName[MAXFILENAMELEN];

    if (OpenFile(PROGNAME ".INI", &of, OF_READ | OF_EXIST) == -1) {
		strcpy(OptionsFileName, PROGNAME ".INI");
	} else {
		strcpy(OptionsFileName, of.szPathName);
	}

    GetPrivateProfileString("Others", "Program Size", "500 400", buffer, MAXLINELEN, OptionsFileName);
    sscanf(buffer, "%d %d", width, length);

	ProgramPosition.right = *width;
	ProgramPosition.bottom = *length;

	GetPrivateProfileString("Others", "Program Position", "Maximized", buffer, MAXLINELEN, OptionsFileName);

	if (!stricmp(buffer, "Maximized")) {
        ProgramMaximized = TRUE;
        *x = *y = 0;
		return (TRUE);
	} else {
        ProgramMaximized = TRUE;
		sscanf(buffer, "%d %d", x, y);
		ProgramPosition.left = *x;
		ProgramPosition.top = *y;
		return (FALSE);
	}
}



BOOL Read3DDialogOption (void)
{
    OFSTRUCT of;
    char buffer[MAXLINELEN];
	char OptionsFileName[MAXFILENAMELEN];

    if (OpenFile(PROGNAME ".INI", &of, OF_READ | OF_EXIST) == -1) {
		strcpy(OptionsFileName, PROGNAME ".INI");
	} else {
		strcpy(OptionsFileName, of.szPathName);
	}

    GetPrivateProfileString("Others", "3D Dialog Boxes", "Yes", buffer, MAXLINELEN, OptionsFileName);
    return (buffer[0] == 'Y' || buffer[0] == 'y');
}



void ReadOptionsFile (void)
{
    int i, j;
    int fd;
    char ch;
    OFSTRUCT of;
    HOTKEY hotkey;
    char buffer[MAXLINELEN];
    char keyname[50];
    char OptionsFileName[MAXFILENAMELEN];
    char SectionName[50];


    GetWindowsDirectory(buffer, MAXLINELEN);
    i = strlen(buffer);
    if (i > 0 && buffer[i-1] != '\\') {
        buffer[i] = '\\';
        buffer[i+1] = '\0';
    }
    strcat(buffer, PROGNAME ".INI");

    if (OpenFile(buffer, &of, OF_READ | OF_EXIST) == -1) {
        ErrorMessage(global.hwnd, "Cannot read initialization file %s!", buffer);
        return;
    } else {
		strcpy(OptionsFileName, of.szPathName);
    }


    /* Path names */

    strcpy(SectionName, "File Names");
    GetPrivateProfileString(SectionName, PROGNAME " Directory", "C:\\" PROGNAME, buffer, MAXLINELEN, OptionsFileName);
    i = strlen(buffer);
    if (i > 0 && buffer[i-1] != '\\') strcat(buffer, "\\");
    global.jwppath = DupString(buffer);

    GetPrivateProfileString(SectionName, "Help File", PROGNAME ".HLP", buffer, MAXLINELEN, OptionsFileName);
    global.help = DupString(buffer);
	GetPrivateProfileString(SectionName, "Conversion Dictionary", "KANJIDCT.DCT", buffer, MAXLINELEN, OptionsFileName);
    global.convdict = DupString(buffer);
	GetPrivateProfileString(SectionName, "Conversion Dictionary Index", "KANJIDCT.IDX", buffer, MAXLINELEN, OptionsFileName);
	global.convidx = DupString(buffer);
    GetPrivateProfileString(SectionName, "User Conversion Dictionary", "USERDICT.DCT", buffer, MAXLINELEN, OptionsFileName);
	global.userdict = DupString(buffer);
    GetPrivateProfileString(SectionName, "Conversion Cache", "CNVCACHE.SAV", buffer, MAXLINELEN, OptionsFileName);
    global.convcache = DupString(buffer);
    GetPrivateProfileString(SectionName, "Japanese Dictionary", "EDICT", buffer, MAXLINELEN, OptionsFileName);
    global.jdict = DupString(buffer);
    GetPrivateProfileString(SectionName, "Japanese Dictionary Index", "EDICT.JDX", buffer, MAXLINELEN, OptionsFileName);
    global.jdictidx = DupString(buffer);
    GetPrivateProfileString(SectionName, "Kanji Info", "KINFO.DAT", buffer, MAXLINELEN, OptionsFileName);
    global.kinfo = DupString(buffer);
    GetPrivateProfileString(SectionName, "Kanji Info Index", "KINFO.IDX", buffer, MAXLINELEN, OptionsFileName);
    global.kinfoidx = DupString(buffer);
    GetPrivateProfileString(SectionName, "Global Glossary", "GLOSSARY.DAT", buffer, MAXLINELEN, OptionsFileName);
    global.glossary = DupString(buffer);

    /* Page Setup */

    strcpy(SectionName, "Page Setup");
    GetPrivateProfileString(SectionName, "Measurement Unit", "inches", buffer, MAXLINELEN, OptionsFileName);
    global.measure = Measure = FindMeasurement(buffer, -1);
    GetPrivateProfileString(SectionName, "Margins", "1.0  1.0  1.0  1.0", buffer, MAXLINELEN, OptionsFileName);
	sscanf(buffer, "%lf %lf %lf %lf", &(global.pagesetup.margins[0]), &(global.pagesetup.margins[1]),
				&(global.pagesetup.margins[2]), &(global.pagesetup.margins[3]));
    GetPrivateProfileString(SectionName, "Orientation", "Portrait", buffer, MAXLINELEN, OptionsFileName);

    if (!stricmp(buffer, "Portrait")) global.pagesetup.landscape = FALSE;
    else if (!stricmp(buffer, "Landscape")) global.pagesetup.landscape = TRUE;
    else global.pagesetup.landscape = FALSE;

    GetPrivateProfileString(SectionName, "Long Text Justification", "Justify", buffer, MAXLINELEN, OptionsFileName);
    if (!stricmp(buffer, "Justify")) global.longjust = J_JUSTIFY;
    else if (!strcmp(buffer, "Left")) global.longjust = J_LEFT;
    else global.longjust = J_JUSTIFY;

    LongJust = global.longjust;

    GetPrivateProfileString(SectionName, "Short Text Justification", "Center", buffer, MAXLINELEN, OptionsFileName);
    if (!stricmp(buffer, "Justify")) global.shortjust = J_JUSTIFY;
    else if (!stricmp(buffer, "Left")) global.shortjust = J_LEFT;
    else if (!stricmp(buffer, "Right")) global.shortjust = J_RIGHT;
    else if (!stricmp(buffer, "Center")) global.shortjust = J_CENTER;
    else global.shortjust = J_CENTER;

    ShortJust = global.shortjust;

    PageSetup = global.pagesetup;


    /* ASCII Fonts */

    strcpy(SectionName, "Base ASCII Font");
    GetPrivateProfileString(SectionName, "Face Name", "Times New Roman", buffer, MAXLINELEN, OptionsFileName);
    DefAsciiFont.facename = DupString(buffer);
    AsciiFontName = DupString(buffer);

    GetPrivateProfileString(SectionName, "Size", "12", buffer, MAXLINELEN, OptionsFileName);
    AsciiFontSize = DefAsciiFont.size = atof(buffer);

    /* Installed Kanji Fonts */

    strcpy(SectionName, "Kanji Fonts");
    global.nr_fonts = GetPrivateProfileInt(SectionName, "Number of Kanji Fonts", 0, OptionsFileName);

    SysFontNum = GetPrivateProfileInt(SectionName, "System Font", 1, OptionsFileName) - 1;
    BaseFontNum = GetPrivateProfileInt(SectionName, "Default Base Font", 1, OptionsFileName) - 1;
    PrintFontNum = GetPrivateProfileInt(SectionName, "Default Print Font", 1, OptionsFileName) - 1;

    GetPrivateProfileString(SectionName, "Default Print Scale", "1.0000", buffer, MAXLINELEN, OptionsFileName);
	sscanf(buffer, "%lf", &(global.printscale));

	InstalledFonts = MemAlloc(global.nr_fonts * sizeof(KANJIFONT));

    PRINTFONT = SYSFONT = BASEFONT = NULL;

    for (i = j = 0; i < global.nr_fonts; i++) {
        sprintf(keyname, "Font #%d", i+1);
        GetPrivateProfileString(SectionName, keyname, "", buffer, MAXLINELEN, OptionsFileName);

        if (!buffer[0]) {
            ErrorMessage(global.hwnd, "Cannot find entry for font #%d in JWP.INI!", i+1);
        } else {
            fd = OpenFont(buffer, &(InstalledFonts[j]));
            if (fd < 0) {
                ErrorMessage(global.hwnd, "Cannot open kanji font %s!", buffer);
				exit(-1);
			} else {
				close(fd);
                if (PrintFontNum == i) {
                    PRINTFONT = &(InstalledFonts[j]);
                    PrintFontNum = j;
                }
                if (SysFontNum == i) {
                    SYSFONT = &(InstalledFonts[j]);
                    SysFontNum = j;
                }
                if (BaseFontNum == i) {
                    BASEFONT = &(InstalledFonts[j]);
                    BaseFontNum = j;
                }
                j++;
            }
        }
    }

    if (j <= 0) {
        ErrorMessage(global.hwnd, "Cannot open any kanji font!");
        exit(-1);
    }

    if (PRINTFONT == NULL) {
        PRINTFONT = &(InstalledFonts[0]);
        PrintFontNum = 0;
    }
    if (SYSFONT == NULL) {
        SYSFONT = &(InstalledFonts[0]);
        SysFontNum = 0;
    }
    if (BASEFONT == NULL) {
        BASEFONT = &(InstalledFonts[0]);
        BaseFontNum = 0;
    }

    global.nr_fonts = j;

    if (global.nr_fonts <= 0) {
        ErrorMessage(global.hwnd, "No Installed fonts!\n\nProgram will die now.");
        exit(-1);                               /* Cannot go on */
    }

    /* Force even number spacing */

	if (SYSFONT->leading % 2 != 0) SYSFONT->leading++;
	if (SYSFONT->spacing % 2 != 0) SYSFONT->spacing++;

    /* Glossary Options */

    strcpy(SectionName, "Glossary");
    GetPrivateProfileString(SectionName, "Dynamic Glossary", "Yes", buffer, MAXLINELEN, OptionsFileName);
    global.dynamicglossary = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Glossary List Disappears", "No", buffer, MAXLINELEN, OptionsFileName);
    global.glistbehaviour = (buffer[0] == 'Y' || buffer[0] == 'y') ? 1 : 0;
    GetPrivateProfileString(SectionName, "Glossary List Visible", "No", buffer, MAXLINELEN, OptionsFileName);
    global.glistvisible = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Glossary List Position", "-1 -1", buffer, MAXLINELEN, OptionsFileName);
    sscanf(buffer, "%d %d", &global.glistsize.left, &global.glistsize.top);
    GetPrivateProfileString(SectionName, "Glossary List Size", "-1 -1", buffer, MAXLINELEN, OptionsFileName);
    sscanf(buffer, "%d %d", &global.glistsize.right, &global.glistsize.bottom);

    glistsize = global.glistsize;


    /* Undo Options */

    strcpy(SectionName, "Undo");
    global.undolevels = GetPrivateProfileInt(SectionName, "Number of Undo Levels", DEFUNDOLEVELS, OptionsFileName);
    UndoLevels = global.undolevels;
    GetPrivateProfileString(SectionName, "Undo Levels Saved", "All", buffer, MAXLINELEN, OptionsFileName);
    if (!stricmp(buffer, "All")) global.saveundolevels = -1;
    else {
        sscanf(buffer, "%d", &global.saveundolevels);
    }
    SaveUndoLevels = global.undolevels;


    /* Other Options */

    strcpy(SectionName, "Others");
    GetPrivateProfileString(SectionName, "Relax Margin for Punctuation", "Yes", buffer, MAXLINELEN, OptionsFileName);
    global.relaxmargin = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Dynamic Conversion", "No", buffer, MAXLINELEN, OptionsFileName);
    global.dynamicconvert = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Conversion Bar Position", "Bottom", buffer, MAXLINELEN, OptionsFileName);

    if (!stricmp(buffer, "Top")) global.listposition = 1;
    else if (!stricmp(buffer, "Bottom")) global.listposition = 2;
    else {
        global.listposition = 0;
        sscanf(buffer, "%d %d", &global.convbar.left, &global.convbar.top);
    }

    GetPrivateProfileString(SectionName, "Conversion Bar Size", "-1 -1", buffer, MAXLINELEN, OptionsFileName);
    sscanf(buffer, "%d %d", &global.convbar.right, &global.convbar.bottom);

    convbar = global.convbar;

    GetPrivateProfileString(SectionName, "Clipboard Format", "EUC", buffer, MAXLINELEN, OptionsFileName);
    for (i = FF_EUC; FileFormatNames[i] != NULL; i++) {
        if (!strcmp(FileFormatNames[i], buffer)) break;
    }
    if (FileFormatNames[i] != NULL) global.clipboard = i; else global.clipboard = FF_EUC;
    clipboard = global.clipboard;

    GetPrivateProfileString(SectionName, "Draft View", "No", buffer, MAXLINELEN, OptionsFileName);
    DraftView = global.draftview = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Show Special Symbols", "No", buffer, MAXLINELEN, OptionsFileName);
    ShowSpecial = global.showspecial = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Show Iconbar", "Yes", buffer, MAXLINELEN, OptionsFileName);
    global.showribbon = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Show Status Bar", "Yes", buffer, MAXLINELEN, OptionsFileName);
    global.showstatus = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "Show File Ruler", "Yes", buffer, MAXLINELEN, OptionsFileName);
    ShowRuler = global.showruler = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "3D Dialog Boxes", "Yes", buffer, MAXLINELEN, OptionsFileName);
    Dialogs3D = global.ctl3d = (buffer[0] == 'Y' || buffer[0] == 'y');
    GetPrivateProfileString(SectionName, "NN Conversion", "Yes", buffer, MAXLINELEN, OptionsFileName);
    NNConvert = global.nnconvert = (buffer[0] == 'Y' || buffer[0] == 'y');

    /* Quick files */

    strcpy(SectionName, "Quick Files");

    j = GetPrivateProfileInt(SectionName, "Number of Quick Files", 0, OptionsFileName);
    for (i = j; i > 0; i--) {
        sprintf(keyname, "File%d", i);
        GetPrivateProfileString(SectionName, keyname, "", buffer, MAXLINELEN, OptionsFileName);
        if (buffer[0]) PutQuickFile(buffer);
    }

    /* Hot keys */

    strcpy(SectionName, "Hot Keys");

    for (i = 0; i < NRJWPFUNC; i++) {
        GetPrivateProfileString(SectionName, jwpfunc[i].desc, "None", buffer, MAXLINELEN, OptionsFileName);
        if (!buffer[0] || !stricmp(buffer, "None")) continue;

        hotkey.ctrl = hotkey.shift = FALSE;
        hotkey.alpha = hotkey.scancode = 0;

        for (j = 0; buffer[j]; j++) {
            if (buffer[j] == SHIFTCHAR) {
                hotkey.shift = TRUE;
            } else if (buffer[j] == CTRLCHAR) {
                ch = buffer[j+1];
                if ('a' <= ch && ch <= 'z') ch -= 32;
                if ('A' <= ch && ch <= 'Z' && (ch != 'F' || !buffer[j+2])) {
                    hotkey.scancode = 0;
                    hotkey.alpha = ch;
					hotkey.ctrl = TRUE;
					hotkey.shift = FALSE;
					break;
				}
                hotkey.ctrl = TRUE;
			} else if (buffer[j] == 'F' || buffer[j] == 'f') {
				hotkey.alpha = 0;
				switch(atoi(buffer + j + 1)) {
					case 1: hotkey.scancode = VK_F1; break;
					case 2: hotkey.scancode = VK_F2; break;
					case 3: hotkey.scancode = VK_F3; break;
					case 4: hotkey.scancode = VK_F4; break;
					case 5: hotkey.scancode = VK_F5; break;
					case 6: hotkey.scancode = VK_F6; break;
					case 7: hotkey.scancode = VK_F7; break;
					case 8: hotkey.scancode = VK_F8; break;
					case 9: hotkey.scancode = VK_F9; break;
					case 10: hotkey.scancode = VK_F10; break;
					case 11: hotkey.scancode = VK_F11; break;
					case 12: hotkey.scancode = VK_F12; break;
				}
				break;
			}
		}

		if (hotkey.alpha != 0 || hotkey.scancode != 0) jwpfunc[i].hotkey = hotkey;
	}

	ProcessMenuTexts();

	OptionsChanged = FALSE;
}



void WriteOptionsFile (BOOL always)
{
	int i, fd;
	FILE *fp;
	OFSTRUCT of;
    char buffer[MAXLINELEN];
	BOOL DoIt = FALSE;
    RECT progrect;
    time_t t;


    GetWindowRect(global.hwnd, &progrect);

    if (always || OptionsChanged) {
        DoIt = TRUE;
    } else {
        /* Things that may change but do not affect OptionsChanged */

        if (global.glistvisible && global.glisthwnd == NULL) DoIt = TRUE;
        if (!global.glistvisible && global.glisthwnd != NULL) DoIt = TRUE;

        if (global.statushwnd == NULL && global.showstatus) DoIt = TRUE;
        if (global.statushwnd != NULL && !global.showstatus) DoIt = TRUE;

        if (global.tbhwnd == NULL && global.showribbon) DoIt = TRUE;
        if (global.tbhwnd != NULL && !global.showribbon) DoIt = TRUE;

        if (fabs(AsciiFontSize - DefAsciiFont.size) > 0.01) DoIt = TRUE;
        if (strcmp(AsciiFontName, DefAsciiFont.facename)) DoIt = TRUE;

        if (memcmp(&glistsize, &global.glistsize, sizeof(RECT)) != 0) DoIt = TRUE;
        if (memcmp(&convbar, &global.convbar, sizeof(RECT)) != 0) DoIt = TRUE;

        if (clipboard != global.clipboard) DoIt = TRUE;
        if (global.draftview != DraftView) DoIt = TRUE;
        if (global.showspecial != ShowSpecial) DoIt = TRUE;
        if (global.showruler != ShowRuler) DoIt = TRUE;
        if (global.measure != Measure) DoIt = TRUE;
        if (global.longjust != LongJust) DoIt = TRUE;
        if (global.shortjust != ShortJust) DoIt = TRUE;
        if (global.ctl3d != Dialogs3D) DoIt = TRUE;
        if (global.nnconvert != NNConvert) DoIt = TRUE;
        if (global.undolevels != UndoLevels) DoIt = TRUE;
        if (global.saveundolevels != SaveUndoLevels) DoIt = TRUE;

        if (fabs(global.pagesetup.margins[0] - PageSetup.margins[0]) > 0.01) DoIt = TRUE;
        if (fabs(global.pagesetup.margins[1] - PageSetup.margins[1]) > 0.01) DoIt = TRUE;
        if (fabs(global.pagesetup.margins[2] - PageSetup.margins[2]) > 0.01) DoIt = TRUE;
        if (fabs(global.pagesetup.margins[3] - PageSetup.margins[3]) > 0.01) DoIt = TRUE;
        if (global.pagesetup.landscape != PageSetup.landscape) DoIt = TRUE;

        if (IsZoomed(global.hwnd)) {
            if (!ProgramMaximized) DoIt = TRUE;
		} else {
            if (ProgramMaximized) DoIt = TRUE;
            if (ProgramPosition.left != progrect.left) DoIt = TRUE;
            if (ProgramPosition.right != progrect.right) DoIt = TRUE;
            if (ProgramPosition.top != progrect.top) DoIt = TRUE;
            if (ProgramPosition.bottom != progrect.bottom) DoIt = TRUE;
        }
    }

	if (!DoIt) return;


    GetWindowsDirectory(buffer, MAXLINELEN);
    i = strlen(buffer);
    if (i > 0 && buffer[i-1] != '\\') {
        buffer[i] = '\\';
        buffer[i+1] = '\0';
    }
    strcat(buffer, PROGNAME ".INI");

    fd = OpenFile(buffer, &of, OF_WRITE | OF_CREATE);
    if (fd < 0) {
        ErrorMessage(global.hwnd, "Cannot write initialization file %s!", buffer);
		return;
    }

    fp = fdopen(fd, "wt");
    if (fp == NULL) {
        ErrorMessage(global.hwnd, "Cannot create " PROGNAME " initialization file!");
		close(fd);
		return;
    }

    /* Information */

    fprintf(fp, "; JWP.INI - Initialization file for JWP\n");
    fprintf(fp, "; Version 1.1\n");
	time(&t);
    fprintf(fp, "; Last Modified %s", ctime(&t));

    /* Path names */

    fprintf(fp, "\n[File Names]\n");
    fprintf(fp, "JWP Directory=%s\n", global.jwppath);
    fprintf(fp, "Help File=%s\n", global.help);
    fprintf(fp, "Conversion Dictionary=%s\n", global.convdict);
    fprintf(fp, "Conversion Dictionary Index=%s\n", global.convidx);
	fprintf(fp, "User Conversion Dictionary=%s\n", global.userdict);
    fprintf(fp, "Conversion Cache=%s\n", global.convcache);
    fprintf(fp, "Japanese Dictionary=%s\n", global.jdict);
    fprintf(fp, "Japanese Dictionary Index=%s\n", global.jdictidx);
    fprintf(fp, "Kanji Info=%s\n", global.kinfo);
    fprintf(fp, "Kanji Info Index=%s\n", global.kinfoidx);
    fprintf(fp, "Global Glossary=%s\n", global.glossary);

    /* Page Setup */

    fprintf(fp, "\n[Page Setup]\n");
    FindMeasurement(buffer, global.measure);
    fprintf(fp, "Measurement Unit=%s\n", buffer);
    fprintf(fp, "Margins=%4.2lf %4.2lf %4.2lf %4.2lf\n",
                global.pagesetup.margins[0], global.pagesetup.margins[1],
                global.pagesetup.margins[1], global.pagesetup.margins[3]);
    fprintf(fp, "Orientation=%s\n", global.pagesetup.landscape ? "Landscape" : "Portrait");
    fprintf(fp, "Long Text Justification=");
    switch (global.longjust) {
        case J_LEFT:    fprintf(fp, "Left\n"); break;
        default:
        case J_JUSTIFY: fprintf(fp, "Justify\n"); break;
    }
    fprintf(fp, "Short Text Justification=");
    switch (global.shortjust) {
        case J_LEFT:    fprintf(fp, "Left\n"); break;
        case J_RIGHT:   fprintf(fp, "Right\n"); break;
        default:
        case J_CENTER:  fprintf(fp, "Center\n"); break;
        case J_JUSTIFY: fprintf(fp, "Justify\n"); break;
    }

    /* ASCII Font */

    fprintf(fp, "\n[Base ASCII Font]\n");
    fprintf(fp, "Face Name=%s\n", DefAsciiFont.facename);
    fprintf(fp, "Size=%3.1lf\n", DefAsciiFont.size);

    /* Kanji Fonts */

    fprintf(fp, "\n[Kanji Fonts]\n");
    fprintf(fp, "Number of Kanji Fonts=%d\n", global.nr_fonts);

    fprintf(fp, "System Font=%d\n", SysFontNum + 1);
    fprintf(fp, "Default Base Font=%d\n", BaseFontNum + 1);
    fprintf(fp, "Default Print Font=%d\n", PrintFontNum + 1);
    fprintf(fp, "Default Print Scale=%6.4lf\n", global.printscale);

    for(i = 0; i < global.nr_fonts; i++) {
		fprintf(fp, "Font #%d=%s\n", i+1, InstalledFonts[i].filename);
    }

    /* Glossary Options */

    fprintf(fp, "\n[Glossary]\n");
    fprintf(fp, "Dynamic Glossary=%s\n", (global.dynamicglossary ? "Yes" : "No"));
    fprintf(fp, "Glossary List Disappears=%s\n", (global.glistbehaviour ? "Yes" : "No"));
	fprintf(fp, "Glossary List Visible=%s\n", (global.glisthwnd != NULL) ? "Yes" : "No");
    fprintf(fp, "Glossary List Position=%d %d\n", global.glistsize.left, global.glistsize.top);
    fprintf(fp, "Glossary List Size=%d %d\n", global.glistsize.right, global.glistsize.bottom);

    /* Undo Options */

    fprintf(fp, "\n[Undo]\n");
    if (global.undolevels < 0) {
        fprintf(fp, "Number of Undo Levels=0\n");
    } else {
        fprintf(fp, "Number of Undo Levels=%d\n", global.undolevels);
    }
    if (global.saveundolevels < 0) {
        fprintf(fp, "Undo Levels Saved=All\n");
    } else {
        fprintf(fp, "Undo Levels Saved=%d\n", global.saveundolevels);
    }

    /* Other Options */

    fprintf(fp, "\n[Others]\n");

    if (IsZoomed(global.hwnd)) {
		fprintf(fp, "Program Position=Maximized\n");
    } else {
        fprintf(fp, "Program Position=%d %d\n", progrect.left, progrect.top);
        fprintf(fp, "Program Size=%d %d\n", progrect.right - progrect.left,
                                            progrect.bottom - progrect.top);
    }

    fprintf(fp, "Relax Margin for Punctuation=%s\n", (global.relaxmargin ? "Yes" : "No"));
    fprintf(fp, "Dynamic Conversion=%s\n", (global.dynamicconvert ? "Yes" : "No"));

    switch (global.listposition) {
        case 0: fprintf(fp, "Conversion Bar Position=%d %d\n", global.convbar.left, global.convbar.top);
                break;
        case 1: fprintf(fp, "Conversion Bar Position=Top\n");
				break;
        default:
        case 2: fprintf(fp, "Conversion Bar Position=Bottom\n");
                break;
    }

    fprintf(fp, "Conversion Bar Size=%d %d\n", global.convbar.right, global.convbar.bottom);
    fprintf(fp, "Clipboard Format=%s\n", FileFormatNames[global.clipboard]);
    fprintf(fp, "Draft View=%s\n", global.draftview ? "Yes" : "No");
    fprintf(fp, "Show Special Symbols=%s\n", global.showspecial ? "Yes" : "No");
    fprintf(fp, "Show Iconbar=%s\n", (global.tbhwnd != NULL) ? "Yes" : "No");
    fprintf(fp, "Show Status Bar=%s\n", (global.statushwnd != NULL) ? "Yes" : "No");
    fprintf(fp, "Show File Ruler=%s\n", global.showruler ? "Yes" : "No");
    fprintf(fp, "3D Dialog Boxes=%s\n", global.ctl3d ? "Yes" : "No");
    fprintf(fp, "NN Conversion=%s\n", global.nnconvert ? "Yes" : "No");

    /* Quick files */

    fprintf(fp, "\n[Quick Files]\n");

    for (i = 0; ; i++) {
        char *cp;
        extern char *GetQuickFile(int);

        cp = GetQuickFile(i);
        if (cp == NULL) break;
        fprintf(fp, "File%d=%s\n", i + 1, cp);
    }

    fprintf(fp, "Number of Quick Files=%d\n", i);

    /* Hot keys */

    fprintf(fp, "\n[Hot Keys]\n");

    for (i = 0; i < NRJWPFUNC; i++) {
		if (jwpfunc[i].hotkey.alpha == 0 && jwpfunc[i].hotkey.scancode == 0) {
            fprintf(fp, "%s=None\n", jwpfunc[i].desc);
            continue;
        }

        if (jwpfunc[i].hotkey.alpha != 0) {
            sprintf(buffer, "%c", CTRLCHAR);
            buffer[1] = jwpfunc[i].hotkey.alpha;
            buffer[2] = '\0';
        } else {
            if (jwpfunc[i].hotkey.ctrl) {
                if (jwpfunc[i].hotkey.shift) sprintf(buffer, "%c%cF", SHIFTCHAR, CTRLCHAR);
                else sprintf(buffer, "%cF", CTRLCHAR);
			} else {
                if (jwpfunc[i].hotkey.shift) sprintf(buffer, "%cF", SHIFTCHAR);
                else strcpy(buffer, "F");
            }

            switch (jwpfunc[i].hotkey.scancode) {
                case VK_F1:  strcat(buffer, "1"); break;
                case VK_F2:  strcat(buffer, "2"); break;
                case VK_F3:  strcat(buffer, "3"); break;
                case VK_F4:  strcat(buffer, "4"); break;
                case VK_F5:  strcat(buffer, "5"); break;
                case VK_F6:  strcat(buffer, "6"); break;
                case VK_F7:  strcat(buffer, "7"); break;
				case VK_F8:  strcat(buffer, "8"); break;
                case VK_F9:  strcat(buffer, "9"); break;
                case VK_F10: strcat(buffer, "10"); break;
                case VK_F11: strcat(buffer, "11"); break;
                case VK_F12: strcat(buffer, "12"); break;
                default:     strcat(buffer, "0"); break;
            }
        }

        fprintf(fp, "%s=%s\n", jwpfunc[i].desc, buffer);
    }

    fclose(fp);
}



BOOL FAR PASCAL OptionsProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    static FILEFORMAT OldCBFormat;

    switch (message) {
        case WM_INITDIALOG: {
            int i;
            char buffer[10];

            CheckDlgButton (hwnd, 4211, global.relaxmargin);
            CheckDlgButton (hwnd, 4212, global.dynamicconvert);

#ifdef CTL3D
            CheckDlgButton (hwnd, 4213, global.ctl3d);
#else CTL3D
            EnableWindow (GetDlgItem(hwnd, 4213), FALSE);
#endif CTL3D
            CheckDlgButton (hwnd, 4214, global.nnconvert);
            CheckDlgButton (hwnd, 4215, global.glistbehaviour);

            SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, FALSE, 0L);
            SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, FALSE, 0L);
            SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, FALSE, 0L);

            switch (global.listposition) {
                case 0: SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, TRUE, 0L);
                        break;
                case 1: SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, TRUE, 0L);
                        break;
                case 2: SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, TRUE, 0L);
                        break;
			}

            SendDlgItemMessage(hwnd, 4231, BM_SETCHECK, FALSE, 0L);
            SendDlgItemMessage(hwnd, 4232, BM_SETCHECK, FALSE, 0L);

            switch (global.glistbehaviour) {
                case 0: SendDlgItemMessage(hwnd, 4231, BM_SETCHECK, TRUE, 0L);
                        break;
                case 1: SendDlgItemMessage(hwnd, 4232, BM_SETCHECK, TRUE, 0L);
                        break;
            }

            /* We start from EUC, skipping Unknown and Normal */

            for (i = FF_EUC; FileFormatNames[i] != NULL; i++)
                SendDlgItemMessage(hwnd, 4241, CB_INSERTSTRING, -1, (LONG) ((LPSTR) FileFormatNames[i]));
            SendDlgItemMessage(hwnd, 4241, CB_SETCURSEL, global.clipboard - 2, 0L);

            OldCBFormat = global.clipboard;

            /* Levels of undo, several pre-set levels */

            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "No Undo");
            SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 0, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "1");
            if (global.undolevels >= 1) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 1, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "5");
            if (global.undolevels >= 5) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 2, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "10");
            if (global.undolevels >= 10) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 3, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "50");
            if (global.undolevels >= 50) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 4, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "100");
            if (global.undolevels >= 100) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 5, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "500");
            if (global.undolevels >= 500) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 6, 0L);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) "1000");
            if (global.undolevels >= 1000) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 7, 0L);
            sprintf(buffer, "%d", MAXUNDOLEVELS);
            SendDlgItemMessage(hwnd, 4242, CB_INSERTSTRING, -1, (LONG) buffer);
            if (global.undolevels >= MAXUNDOLEVELS) SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 8, 0L);

            return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case 4211:      /* Check boxes */
                case 4212:
                case 4213:
                case 4214:
					if (HIWORD(lParam) == BN_DOUBLECLICKED) SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
					return (TRUE);

				case 4221:      /* Top */
					SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, TRUE, 0L);
					SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, FALSE, 0L);
					SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, FALSE, 0L);
					if (HIWORD(lParam) == BN_DOUBLECLICKED) SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
					return (TRUE);

				case 4222:      /* Bottom */
					SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, TRUE, 0L);
					SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, FALSE, 0L);
					SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, FALSE, 0L);
					if (HIWORD(lParam) == BN_DOUBLECLICKED) SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
					return (TRUE);

				case 4223:      /* Floating */
					SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, TRUE, 0L);
					SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, FALSE, 0L);
					SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, FALSE, 0L);
					if (HIWORD(lParam) == BN_DOUBLECLICKED) SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
					return (TRUE);

                case 4241: {        /* Clipboard format */
                    int i;

                    switch (HIWORD(lParam)) {
                        case LBN_SELCHANGE:
                        case LBN_DBLCLK:
                            i = (WORD) SendMessage(GetDlgItem(hwnd, 4241), CB_GETCURSEL, 0, 0L);
							if (i == LB_ERR) {
								MessageBeep(0);
								return (TRUE);
							}
                            global.clipboard = i + FF_EUC;   /* We start from EUC (2) */

                            if (HIWORD(lParam) == LBN_DBLCLK)
                                SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                            return (TRUE);
                    }
                    break;
                }

                case 4242:          /* Undo levels */
                    switch (HIWORD(lParam)) {
                        case LBN_DBLCLK:
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                            return (TRUE);
                    }
                    break;

				case IDOK: {
					int i;
                    BOOL Redraw = FALSE;
					RECT rect;

                    i = IsDlgButtonChecked(hwnd, 4211);
                    if (i != global.relaxmargin) OptionsChanged = TRUE;
                    global.relaxmargin = i;

                    i = IsDlgButtonChecked(hwnd, 4212);
                    if (i != global.dynamicconvert) OptionsChanged = TRUE;
                    global.dynamicconvert = i;

#ifdef CTL3D
                    i = IsDlgButtonChecked(hwnd, 4213);
                    if (i != global.ctl3d) OptionsChanged = TRUE;
                    global.ctl3d = i;

                    /*
                    if (global.ctl3d != Dialogs3D) {
                        char buffer[200];

                        sprintf(buffer, "You must exit " PROGNAME " and re-run "
                                        "it for 3D dialog boxes to %s.",
                                        Dialogs3D ? "disappear" : "appear");
                        MessageBeep(0);
						MessageBox(hwnd, buffer, "A reminder...", MB_ICONASTERISK | MB_OK);
                    }
                    */
#endif CTL3D

                    i = IsDlgButtonChecked(hwnd, 4214);
                    if (i != global.nnconvert) OptionsChanged = TRUE;
                    global.nnconvert = i;

                    i = IsDlgButtonChecked(hwnd, 4215);
                    if (i != global.glistbehaviour) OptionsChanged = TRUE;
                    global.glistbehaviour = i;

                    if (IsDlgButtonChecked(hwnd, 4223)) {
                        if (global.listposition != 0) OptionsChanged = Redraw = TRUE;
                        global.listposition = 0;
                    } else if (IsDlgButtonChecked(hwnd, 4221)) {
                        if (global.listposition != 1) OptionsChanged = Redraw = TRUE;
                        global.listposition = 1;
                    } else {
                        if (global.listposition != 2) OptionsChanged = Redraw = TRUE;
                        global.listposition = 2;
                    }

                    i = (WORD) SendMessage(GetDlgItem(hwnd, 4242), CB_GETCURSEL, 0, 0L);
                    if (i != LB_ERR) {
                        switch (i) {
                            case 0: global.undolevels = 0; break;
                            case 1: global.undolevels = 1; break;
                            case 2: global.undolevels = 5; break;
                            case 3: global.undolevels = 10; break;
                            case 4: global.undolevels = 50; break;
                            case 5: global.undolevels = 100; break;
                            case 6: global.undolevels = 500; break;
                            case 7: global.undolevels = 1000; break;
                            case 8: global.undolevels = MAXUNDOLEVELS; break;
                        }
                    }

					EndDialog(hwnd, FALSE);

                    if (Redraw) {
                        GetClientRect(global.hwnd, &rect);
                        i = rect.bottom - rect.top;
                        PostMessage(global.hwnd, WM_SIZE, 0, MAKELONG(rect.right - rect.left, i));
                    }

                    if (OldCBFormat != global.clipboard)
                        ResetClipboard();

					return (TRUE);
				}

                case IDCANCEL:
					EndDialog(hwnd, FALSE);
					return (TRUE);
            }
    }

    return (FALSE);
}



static KANJI far *ConvertFontName (int id, LONG lParam, KANJI *buf)
{
    int i, j;
    KANJI *kp;
    char tempbuf[50];

    kp = InstalledFonts[lParam].facename;

    for (i = 0; kp[i]; i++) buf[i] = kp[i];

    sprintf(tempbuf, " (%d x %d)", InstalledFonts[lParam].width, InstalledFonts[lParam].height);

    for (j = 0; tempbuf[j]; j++) buf[i++] = tempbuf[j];
    buf[i] = 0;

	return (buf);
}



short FAR PASCAL AsciiFontEnumProc (LOGFONT far *lf, TEXTMETRIC far *tm, short ftype, LPSTR data)
{
    int i, len;
	HWND hwnd;
	char buffer[MAXLINELEN];
    char temp[MAXLINELEN];

	hwnd = (HWND) data;

	if (ftype & DEVICE_FONTTYPE) {
		buffer[0] = '*';
		_fstrcpy(buffer + 1, lf->lfFaceName);
	} else {
		_fstrcpy(buffer, lf->lfFaceName);
    }

    len = SendMessage(hwnd, CB_GETCOUNT, 0, 0L);

    for (i = 0; i < len; i++) {
        SendMessage(hwnd, CB_GETLBTEXT, i, (LONG) ((char far *) temp));
        if (!stricmp(buffer, temp)) return (1);
    }

    SendMessage(hwnd, CB_ADDSTRING, 0, (LONG) buffer);

    return (1);
}



short FAR PASCAL AsciiFontSizeEnumProc (LOGFONT far *lf, TEXTMETRIC far *tm, short ftype, LPSTR data)
{
	int i, j, len, dpi;
	double ps;
	HDC hdc;

    if (!(ftype & RASTER_FONTTYPE) || (ftype & TTYPE_FONTTYPE)) {
		for (i = 8, j = 0; i <= 24; i++) PointSizes[j++] = i;
		PointSizes[j++] = 36.0;
		PointSizes[j++] = 48.0;
		PointSizes[j] = 0.0;

		return (1);
	}

	len = SegHeapGetSize(PointSizes) / sizeof(double);

	hdc = (HDC) data;

	dpi = GetDeviceCaps(hdc, LOGPIXELSY);
    ps = ((double) tm->tmHeight - (double) tm->tmInternalLeading) * 72.0 / (double) dpi;

    for (i = 0; i < len; i++) {
        if (PointSizes[i] == 0.0) break;
        if (fabs(ps - PointSizes[i]) < 0.01) return (0);
    }

    if (i >= len) PointSizes = (double far *) BlockRealloc(PointSizes, (len + PSBLOCKSIZE) * sizeof(double));

    PointSizes[i++] = ps;
    PointSizes[i] = 0.0;

    return (0);
}



static void FillAuxStuff (HWND hwnd, int group, int window, int font, double scale)
{
    int len;
    double dpp, ps;
    char buffer[MAXLINELEN];

	/* Fill in some aux stuff */

	dpp = (double) global.resolution.y / 72.0;
    ps = ((double) InstalledFonts[font].height * scale) / dpp;
    sprintf(buffer, "Kanji Printing Size = %3.1lf point%s",
            ps, (ps > 1.10) ? "s" : "");
    SetDlgItemText(hwnd, group, buffer);
    sprintf(buffer, "%3.1lf", ps);
    len = strlen(buffer);
    if (len > 2 && buffer[len-1] == '0') buffer[len-2] = '\0';
	SetDlgItemText(hwnd, window, buffer);
}



static void FillPointSizes (HWND hwnd, int window, char *facename, double ps)
{
	int i, len;
	double itemps;
	char buffer[20];
	HDC hdc;
    FARPROC enumproc;

	SendDlgItemMessage(hwnd, window, WM_SETREDRAW, FALSE, 0L);
	SendDlgItemMessage(hwnd, window, CB_RESETCONTENT, 0, 0L);

	hdc = GetPrinterDC(TRUE, NULL);
	if (hdc == NULL) hdc = CreateIC("DISPLAY", NULL, NULL, NULL);

	if (*facename == '*') facename++;

	PointSizes = (double far *) BlockAlloc(sizeof(double) * PSBLOCKSIZE);
    PointSizes[0] = 0.0;

	enumproc = MakeProcInstance((FARPROC) AsciiFontSizeEnumProc, hInstance);
	EnumFonts (hdc, facename, enumproc, (LONG) hdc);
	FreeProcInstance(enumproc);

	for (i = 0; PointSizes[i] != 0.0; i++) {
		sprintf(buffer, "%3.1lf", PointSizes[i]);

		len = strlen(buffer);
		if (buffer[len - 1] == '0') buffer[len - 2] = '\0';

        SendDlgItemMessage(hwnd, window, CB_ADDSTRING, 0, (LONG) ((char far *) buffer));
	}
	FreeBlock(PointSizes);

    len = SendDlgItemMessage(hwnd, window, CB_GETCOUNT, 0, 0L);
    for (i = 0; i < len; i++) {
        SendDlgItemMessage(hwnd, window, CB_GETLBTEXT, i, (LONG) ((char far *) buffer));
        itemps = atof(buffer);
        if (fabs(itemps - ps) < 0.01) break;
    }

    if (i < len) SendDlgItemMessage(hwnd, window, CB_SETCURSEL, i, 0L);
    DeleteDC(hdc);

	SendDlgItemMessage(hwnd, window, WM_SETREDRAW, TRUE, 0L);
	InvalidateRect(GetDlgItem(hwnd, window), NULL, TRUE);
}



BOOL FAR PASCAL InstalledFontsProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	switch (message) {
		case WM_INITDIALOG: {
			int i, len;
			HDC hdc;
            FARPROC enumproc;
			char buffer[MAXLINELEN];

            SendDlgItemMessage(hwnd, 4231, EM_LIMITTEXT, 4, 0L);
            SendDlgItemMessage(hwnd, 4202, CB_LIMITTEXT, 4, 0L);

			/* Fill in the ASCII font box */

			hdc = GetPrinterDC(TRUE, NULL);
			if (hdc == NULL) hdc = CreateIC("DISPLAY", NULL, NULL, NULL);

			enumproc = MakeProcInstance((FARPROC) AsciiFontEnumProc, hInstance);
            EnumFonts (hdc, NULL, enumproc, (LONG) GetDlgItem(hwnd, 4201));
            FreeProcInstance(enumproc);

			len = SendDlgItemMessage(hwnd, 4201, CB_GETCOUNT, 0, 0L);
			for (i = 0; i < len; i++) {
				SendDlgItemMessage(hwnd, 4201, CB_GETLBTEXT, i, (LONG) ((char far *) buffer));
                if (buffer[0] == '*') {
                    if (!strcmp(buffer + 1, DefAsciiFont.facename)) break;
                } else {
                    if (!strcmp(buffer, DefAsciiFont.facename)) break;
                }
            }

            if (i < len) {
				SendDlgItemMessage(hwnd, 4201, CB_SETCURSEL, i, 0L);
				FillPointSizes(hwnd, 4202, buffer, DefAsciiFont.size);
            }
            DeleteDC(hdc);


			/* Fill in all the kanji font boxes */

			for (i = 0; i < global.nr_fonts; i++) {
				SendDlgItemMessage(hwnd, 4211, CB_ADDSTRING, 0, (LONG) i);
				SendDlgItemMessage(hwnd, 4221, CB_ADDSTRING, 0, (LONG) i);
			}

            /* Select the right items */

            for (i = 0; i < global.nr_fonts; i++) {
                if (SendDlgItemMessage(hwnd, 4211, CB_GETITEMDATA, i, 0L) == BaseFontNum) {
                    SendDlgItemMessage(hwnd, 4211, CB_SETCURSEL, i, 0L);
                }
                if (SendDlgItemMessage(hwnd, 4221, CB_GETITEMDATA, i, 0L) == PrintFontNum) {
					SendDlgItemMessage(hwnd, 4221, CB_SETCURSEL, i, 0L);
					FillAuxStuff(hwnd, 4232, 4231, PrintFontNum, global.printscale);
                }
            }

			SetFocus(GetDlgItem(hwnd, 4201));
            CenterDialogBox(hwnd);
            return (TRUE);
        }

		case WM_COMMAND: {
            int i, j;
            BOOL NeedReformat;
            double ratio, ps;
            char buffer[MAXLINELEN];

            switch (wParam) {
                case 4211:      /* Base font */
					switch (HIWORD(lParam)) {
                        case CBN_DBLCLK:
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                            return (TRUE);
                    }
                    return (TRUE);

                case 4201:      /* ASCII font */
					switch (HIWORD(lParam)) {
						case CBN_SELCHANGE:
                            i = SendDlgItemMessage(hwnd, 4201, CB_GETCURSEL, 0, 0L);

                            if (i != CB_ERR) {  /* Update the point size */
                                SendDlgItemMessage(hwnd, 4201, CB_GETLBTEXT, i, (LONG) ((char far *) buffer));
								FillPointSizes(hwnd, 4202, buffer, DefAsciiFont.size);
                            }
                            return (TRUE);

                        case CBN_DBLCLK:
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                            return (TRUE);
                    }
                    return (TRUE);

                case 4221:      /* Print font */
					switch (HIWORD(lParam)) {
						case CBN_SELCHANGE:
                            i = SendDlgItemMessage(hwnd, 4221, CB_GETCURSEL, 0, 0L);
							if (i != CB_ERR) {  /* Update the point size */
                                j = SendDlgItemMessage(hwnd, 4221, CB_GETITEMDATA, i, 0L);
								FillAuxStuff(hwnd, 4232, 4231, j, 1.0);
                            }
                            return (TRUE);

                        case CBN_DBLCLK:
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                            return (TRUE);
                    }
                    return (TRUE);

                case IDOK:
                    NeedReformat = FALSE;

                    /* Check the base font */

                    i = SendDlgItemMessage(hwnd, 4211, CB_GETCURSEL, 0, 0L);
                    if (i != CB_ERR) {
                        j = SendDlgItemMessage(hwnd, 4211, CB_GETITEMDATA, i, 0L);
                        if (j != BaseFontNum) {
                            BaseFontNum = j;
                            BASEFONT = &(InstalledFonts[j]);
                            NeedReformat = OptionsChanged = TRUE;
                        }
					}

					/* Check the printing font */

                    i = SendDlgItemMessage(hwnd, 4221, CB_GETCURSEL, 0, 0L);
                    if (i != CB_ERR) {
                        j = SendDlgItemMessage(hwnd, 4221, CB_GETITEMDATA, i, 0L);
                        if (j != PrintFontNum) {
                            PrintFontNum = j;
                            PRINTFONT = &(InstalledFonts[j]);
                            NeedReformat = OptionsChanged = TRUE;
                        }
                    }

					/* Is the aspect ratio OK? */

                    ratio = (double) InstalledFonts[BaseFontNum].width /
                                (double) InstalledFonts[BaseFontNum].height;
                    ratio /= ((double) InstalledFonts[PrintFontNum].width /
                                 (double) InstalledFonts[PrintFontNum].height);

                    if (fabs(ratio - 1.0) > 0.01) {
                        ErrorMessage(hwnd, "The kanji font for display and printing must "
                                              "have the same aspect ratio!");
                        return (TRUE);
                    }

					/* Check the printing scale */

                    GetDlgItemText(hwnd, 4231, buffer, 5);
                    ps = atof(buffer);       /* Printing point size */

                    GetDlgItemText(hwnd, 4202, buffer, 5);

                    if (atof(buffer) - ps > 0.01) {
                        ErrorMessage(hwnd, "WARNING!  You have chosen a text font that is larger "
                                              "than the kanji print font!");
                    }

                    j = (int) floor(ps * (double) global.resolution.y / 72.0 + 0.5);      /* dots */

                    ratio = (double) j / (double) InstalledFonts[PrintFontNum].height;

                    if (fabs(global.printscale - ratio) > 0.01) {
                        global.printscale = ratio;
                        NeedReformat = OptionsChanged = TRUE;
                    }

                    /* ASCII font */

                    GetDlgItemText(hwnd, 4201, buffer, MAXLINELEN);
                    if (strcmp(DefAsciiFont.facename, buffer)) {
						FreeMem(DefAsciiFont.facename);
                        if (buffer[0] == '*') {
                            DefAsciiFont.facename = DupString(buffer + 1);
                        } else {
                            DefAsciiFont.facename = DupString(buffer);
                        }
                        NeedReformat = OptionsChanged = TRUE;
                    }
                    GetDlgItemText(hwnd, 4202, buffer, 5);
                    ps = atof(buffer);
                    if (ps >= 0.01 && fabs(ps - DefAsciiFont.size) > 0.01) {
                        DefAsciiFont.size = ps;
                        NeedReformat = OptionsChanged = TRUE;
                    }

                    EndDialog(hwnd, NeedReformat);
                    return (TRUE);

                case IDCANCEL:
                    EndDialog(hwnd, FALSE);
                    return (TRUE);
			}
            return (FALSE);
        }

        case WM_COMPAREITEM:
        case WM_DELETEITEM:
        case WM_DRAWITEM:
        case WM_MEASUREITEM:
            return (JlistProc(hwnd, message, wParam, lParam, TRUE, ConvertFontName));
    }

    return (FALSE);
}
