/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/rect.c,v 2.30.1.2 1992/10/21 04:43:23 william Exp $";
#endif

#include <stdio.h>
#include <math.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "arc.e"
#include "poly.e"
#include "setup.e"
#include "spline.e"

int BBoxIntersect (Rect1, Rect2)
   struct BBRec	Rect1, Rect2;
{
   if (Rect1.ltx < Rect2.ltx)
      if (Rect1.lty < Rect2.lty)
         return (Rect1.rbx >= Rect2.ltx && Rect1.rby >= Rect2.lty);
      else
         return (Rect1.rbx >= Rect2.ltx && Rect1.lty <= Rect2.rby);
   else
      if (Rect1.lty < Rect2.lty)
         return (Rect1.ltx <= Rect2.rbx && Rect1.rby >= Rect2.lty);
      else
         return (Rect1.ltx <= Rect2.rbx && Rect1.lty <= Rect2.rby);
}

int PointInBBox (X, Y, Rect)
   int	X, Y;
   struct BBRec	Rect;
{
   return (X >= Rect.ltx && Y >= Rect.lty && X <= Rect.rbx && Y <= Rect.rby);
}

#define POLYGON_TOL (1.0e-5)

int PointInPolygon (X, Y, NumPts, V)
   int		X, Y, NumPts;
   XPoint	* V;
{
   register double	x1, x2, y1, y2;
   double		m, y_int;
   int			n, count = 0;

   x2 = (double)V[0].x;
   y2 = (double)V[0].y;
   for (n = 0; n < NumPts-1; n++)
   {
      x1 = x2;
      y1 = y2;
      x2 = (double)V[n+1].x;
      y2 = (double)V[n+1].y;
      if (x2 == x1)
      {
         if (X == x1 && Y >= min(y1,y2) && Y <= max(y1,y2)) count++;
         continue;
      }
      if (x2 > x1)
      {
         if (X >= x2 || X < x1) continue;
      }
      else
      {
         if (X > x1 || X <= x2) continue;
      }

      m = (y1 - y2) / (x1 - x2);
      y_int = m * X + (y1 - m * x1);
      if (Y <= y_int)
      {
         if (fabs(X-x1) < POLYGON_TOL)
         {
            double	x3 = (double)((n == 0) ? V[NumPts-2].x : V[n-1].x);

            if (x2 > X && X > x3 || x3 > X && X > x2) count++;
         }
         else
            count++;
      }
   }
   return (count & 0x1);
}

int PointOnPoly (X, Y, NumPts, V, W)
   int		X, Y, NumPts, W;
   XPoint	* V;
{
   register double	x1, x2, y1, y2;
   double		x_int, y_int, dx, dy, abs_dx, abs_dy, real_w;
   int			n, horizontal;

   x2 = (double)V[0].x;
   y2 = (double)V[0].y;
   for (n = 0; n < NumPts-1; n++)
   {
      x1 = x2;
      y1 = y2;
      x2 = (double)V[n+1].x;
      y2 = (double)V[n+1].y;

      if (V[n].x==V[n+1].x && V[n].y==V[n+1].y) continue;

      dx = x2 - x1; abs_dx = fabs(dx);
      dy = y2 - y1; abs_dy = fabs(dy);
      horizontal = (abs_dx >= abs_dy);

      if (horizontal)
      {
         if (x2 >= x1)
         {
            if (X < x1 || X > x2) continue;
         }
         else
         {
            if (X < x2 || X > x1) continue;
         }
         real_w = sqrt(abs_dx*abs_dx+abs_dy*abs_dy)*W/abs_dx;
      }
      else
      {
         if (y2 >= y1)
         {
            if (Y < y1 || Y > y2) continue;
         }
         else
         {
            if (Y < y2 || Y > y1) continue;
         }
         real_w = sqrt(abs_dx*abs_dx+abs_dy*abs_dy)*W/abs_dy;
      }

      if (abs_dx >= abs_dy)
      {  /* kind of a horizontal segment */
         y_int = y1+(((double)X)-x1)*dy/dx;
         if ((double)fabs((double)Y-y_int) <= (real_w+4)) return (TRUE);
      }
      else
      {  /* kind of a vertical segment */
         x_int = x1+(((double)Y)-y1)*dx/dy;
         if ((double)fabs((double)X-x_int) <= (real_w+4)) return (TRUE);
      }
   }
   return (FALSE);
}

int FindGoodXBm (XOff, YOff, XBmObj)
   int			XOff, YOff;
   struct ObjRec	* XBmObj;
   /* XOff and YOff are screen offsets */
{
   struct BBRec	bbox;

   bbox.ltx = OFFSET_X(XBmObj->obbox.ltx)-3;
   bbox.lty = OFFSET_Y(XBmObj->obbox.lty)-3;
   bbox.rbx = OFFSET_X(XBmObj->obbox.rbx)+3;
   bbox.rby = OFFSET_Y(XBmObj->obbox.rby)+3;
   return (PointInBBox (XOff, YOff, bbox));
}

int FindGoodXPm (XOff, YOff, XPmObj)
   int			XOff, YOff;
   struct ObjRec	* XPmObj;
   /* XOff and YOff are screen offsets */
{
   struct BBRec	bbox;

   bbox.ltx = OFFSET_X(XPmObj->obbox.ltx)-3;
   bbox.lty = OFFSET_Y(XPmObj->obbox.lty)-3;
   bbox.rbx = OFFSET_X(XPmObj->obbox.rbx)+3;
   bbox.rby = OFFSET_Y(XPmObj->obbox.rby)+3;
   return (PointInBBox (XOff, YOff, bbox));
}

int FindGoodBox (XOff, YOff, BoxObj)
   int			XOff, YOff;
   struct ObjRec	* BoxObj;
   /* XOff and YOff are screen offsets */
{
   struct BBRec	bbox;
   int		w;

   bbox.ltx = OFFSET_X(BoxObj->obbox.ltx)-3;
   bbox.lty = OFFSET_Y(BoxObj->obbox.lty)-3;
   bbox.rbx = OFFSET_X(BoxObj->obbox.rbx)+3;
   bbox.rby = OFFSET_Y(BoxObj->obbox.rby)+3;
   if (!PointInBBox (XOff, YOff, bbox)) return (FALSE);

   if (BoxObj->detail.b->fill != NONEPAT) return (TRUE);

   w = HALF_W(BoxObj->detail.b->width);
   bbox.ltx = OFFSET_X(BoxObj->obbox.ltx+w)+3;
   bbox.lty = OFFSET_Y(BoxObj->obbox.lty+w)+3;
   bbox.rbx = OFFSET_X(BoxObj->obbox.rbx-w)-3;
   bbox.rby = OFFSET_Y(BoxObj->obbox.rby-w)-3;
   return (!PointInBBox (XOff, YOff, bbox));
}

int FindGoodRCBox (XOff, YOff, RCBoxObj)
   int			XOff, YOff;
   struct ObjRec	* RCBoxObj;
   /* XOff and YOff are screen offsets */
{
   register struct BBRec	* obbox;
   register struct BBRec	bbox;
   int				w, r;

   obbox = &(RCBoxObj->obbox);

   bbox.ltx = OFFSET_X(obbox->ltx)-3;
   bbox.lty = OFFSET_Y(obbox->lty)-3;
   bbox.rbx = OFFSET_X(obbox->rbx)+3;
   bbox.rby = OFFSET_Y(obbox->rby)+3;
   if (!PointInBBox (XOff, YOff, bbox)) return (FALSE);

   if (RCBoxObj->detail.b->fill != NONEPAT) return (TRUE);

   r = RCBoxObj->detail.rcb->radius;
   w = HALF_W(RCBoxObj->detail.rcb->width);

   bbox.ltx = OFFSET_X(obbox->ltx+w)+3; bbox.lty = OFFSET_Y(obbox->lty+r)+3;
   bbox.rbx = OFFSET_X(obbox->rbx-w)-3; bbox.rby = OFFSET_Y(obbox->rby-r)-3;

   if (PointInBBox (XOff, YOff, bbox))
      return (FALSE);
   else
   {
      bbox.ltx = OFFSET_X(obbox->ltx+r)+3; bbox.lty = OFFSET_Y(obbox->lty+w)+3;
      bbox.rbx = OFFSET_X(obbox->rbx-r)-3; bbox.rby = OFFSET_Y(obbox->rby-w)-3;
      return (!PointInBBox (XOff, YOff, bbox));
   }
}

int FindGoodOval (XOff, YOff, OvalObj)
   int			XOff, YOff;
   struct ObjRec	* OvalObj;
   /* XOff and YOff are screen offsets */
{
   struct OvalRec	* oval_ptr = OvalObj->detail.o;
   int			w, ltx, lty, rbx, rby;
   double		cx, cy, rx, ry, tmp_x, tmp_y;
   double		x1 = 0.0, x2 = 0.0, y1 = 0.0, y2 = 0.0;
   int			fill = oval_ptr->fill;

   ltx = OFFSET_X(OvalObj->obbox.ltx); lty = OFFSET_Y(OvalObj->obbox.lty);
   rbx = OFFSET_X(OvalObj->obbox.rbx); rby = OFFSET_Y(OvalObj->obbox.rby);

   if (ltx==rbx && lty==rby) return (FALSE);

   cx = (ltx + rbx) / 2; cy = (lty + rby) / 2;
   rx = (rbx - ltx) / 2; ry = (rby - lty) / 2;

   if (rx >= ry)
   {  /* flat oval */
      tmp_y = sqrt (fabs ((double)(ry*ry*(1-(XOff-cx)*(XOff-cx)/rx/rx))));
      y1 = cy - tmp_y;
      y2 = cy + tmp_y;
   }
   else
   {  /* tall oval */
      tmp_x = sqrt (fabs ((double)(rx*rx*(1-(YOff-cy)*(YOff-cy)/ry/ry))));
      x1 = cx - tmp_x;
      x2 = cx + tmp_x;
   }

   if (fill != NONEPAT)
   {
      if (rx >= ry)
      {  /* flat oval */
         if (YOff >= y1-4 && y2+4 >= YOff) return (TRUE);
      }
      else
      {  /* tall oval */
         if (XOff >= x1-4 && x2+4 >= XOff) return (TRUE);
      }
   }
   w = ZOOMED_HALF_W(oval_ptr->width)+4;
   if (rx >= ry)
      return ((fabs((double)(YOff-y1))<=w) || (fabs((double)(YOff-y2))<=w));
   else
      return ((fabs((double)(XOff-x1))<=w) || (fabs((double)(XOff-x2))<=w));
}

int FindGoodPoly (XOff, YOff, PolyObj)
   int			XOff, YOff;
   struct ObjRec	* PolyObj;
   /* XOff and YOff are screen offsets */
{
   int			n, dx, dy;
   double		len, sin, cos, aw, ah;
   XPoint		* v, tmp_v[4];
   struct PolyRec	* poly_ptr = PolyObj->detail.p;

   if (poly_ptr->fill != NONEPAT)
   {
      n = poly_ptr->sn;
      v = poly_ptr->svlist;
      v[n].x = v[0].x; v[n].y = v[0].y;
      if (PointInPolygon (XOff, YOff, n+1, v)) return (TRUE);
   }
   if (poly_ptr->style == LS_PLAIN)
   {
      if (PointOnPoly (XOff, YOff, poly_ptr->sn, poly_ptr->svlist,
            ZOOMED_HALF_W(poly_ptr->width)))
         return (TRUE);
   }
   else
   {
      if (PointOnPoly (XOff, YOff, poly_ptr->asn, poly_ptr->asvlist,
            ZOOMED_HALF_W(poly_ptr->width)))
         return (TRUE);
   }
   v = poly_ptr->vlist;
   n = poly_ptr->n;
   aw = poly_ptr->aw;
   ah = poly_ptr->ah;

   dx = v[1].x - v[0].x;
   dy = v[1].y - v[0].y;
   if ((poly_ptr->style & LS_LEFT) && (dx != 0 || dy != 0))
   {
      len = (double) sqrt ((double)(dx*dx + dy*dy));
      sin = ((double)dy) / len;
      cos = ((double)dx) / len;

      tmp_v[0].x = tmp_v[3].x = OFFSET_X(v[0].x);
      tmp_v[0].y = tmp_v[3].y = OFFSET_Y(v[0].y);
      tmp_v[1].x = OFFSET_X(round(v[0].x+aw*cos-ah*sin));
      tmp_v[1].y = OFFSET_Y(round(v[0].y+aw*sin+ah*cos));
      tmp_v[2].x = OFFSET_X(round(v[0].x+aw*cos+ah*sin));
      tmp_v[2].y = OFFSET_Y(round(v[0].y+aw*sin-ah*cos));

      if (PointInPolygon (XOff, YOff, 4, tmp_v)) return (TRUE);
   }
   dx = v[n-1].x - v[n-2].x;
   dy = v[n-1].y - v[n-2].y;
   if ((poly_ptr->style & LS_RIGHT) && (dx != 0 || dy != 0))
   {
      len = (double) sqrt ((double)(dx*dx + dy*dy));
      sin = ((double)dy) / len;
      cos = ((double)dx) / len;

      tmp_v[0].x = tmp_v[3].x = OFFSET_X(v[n-1].x);
      tmp_v[0].y = tmp_v[3].y = OFFSET_Y(v[n-1].y);
      tmp_v[1].x = OFFSET_X(round(v[n-1].x-aw*cos+ah*sin));
      tmp_v[1].y = OFFSET_Y(round(v[n-1].y-aw*sin-ah*cos));
      tmp_v[2].x = OFFSET_X(round(v[n-1].x-aw*cos-ah*sin));
      tmp_v[2].y = OFFSET_Y(round(v[n-1].y-aw*sin+ah*cos));

      if (PointInPolygon (XOff, YOff, 4, tmp_v)) return (TRUE);
   }
   return (FALSE);
}

int FindGoodPolygon (XOff, YOff, PolygonObj)
   int			XOff, YOff;
   struct ObjRec	* PolygonObj;
   /* XOff and YOff are screen offsets */
{
   struct PolygonRec	* polygon_ptr = PolygonObj->detail.g;

   if (polygon_ptr->fill != NONEPAT)
      if (PointInPolygon (XOff, YOff, polygon_ptr->sn, polygon_ptr->svlist))
         return (TRUE);

   return (PointOnPoly (XOff, YOff, polygon_ptr->sn, polygon_ptr->svlist,
         ZOOMED_HALF_W(polygon_ptr->width)));
}

static
int PointInFlatPie (Y,dir,angle,ov_int_y1,ov_int_y2,rad_int_y1,rad_int_y2)
   int		Y, dir, angle;
   double	ov_int_y1, ov_int_y2, rad_int_y1, rad_int_y2;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y2 && ov_int_y2>=Y) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=ov_int_y1 && rad_int_y1>=Y) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=ov_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y1 && ov_int_y2>=Y) return (TRUE);
               }
               break;
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=ov_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=ov_int_y1 && rad_int_y1>=Y) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y2 && ov_int_y2>=Y) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y1 && ov_int_y2>=Y) return (TRUE);
               }
               break;
         }
         break;
   }
   return (FALSE);
}

static
int PointInTallPie (X,dir,angle,ov_int_x1,ov_int_x2,rad_int_x1,rad_int_x2)
   int		X, dir, angle;
   double	ov_int_x1, ov_int_x2, rad_int_x1, rad_int_x2;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x1 && ov_int_x2>=X) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x2 && ov_int_x2>=X) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=ov_int_x1 && rad_int_x1>=X) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=ov_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               break;
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x1 && ov_int_x2>=X) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=ov_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=ov_int_x1 && rad_int_x1>=X) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x2 && ov_int_x2>=X) return (TRUE);
               }
               break;
         }
         break;
   }
   return (FALSE);
}

static
int XInPieRange (X, dir, angle, cx, rx)
   int		X, dir, angle;
   double	cx, rx;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
            case 90: return (X>=cx && cx+rx>=X);

            case 180:
            case 270: return (X>=cx-rx && cx>=X);
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
            case 270: return (X>=cx && cx+rx>=X);

            case 90:
            case 180: return (X>=cx-rx && cx>=X);
         }
         break;
   }
   return (FALSE);
}

static
int YInPieRange (Y, dir, angle, cy, ry)
   int		Y, dir, angle;
   double	cy, ry;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
            case 270: return (Y>=cy && cy+ry>=Y);

            case 90:
            case 180: return (Y>=cy-ry && cy>=Y);
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
            case 90: return (Y>=cy-ry && cy>=Y);

            case 180:
            case 270: return (Y>=cy && cy+ry>=Y);
         }
         break;
   }
   return (FALSE);
}

#define ARC_TOL (1.0e-5)

int FindGoodArc (XOff, YOff, ArcObj)
   int			XOff, YOff;
   struct ObjRec	* ArcObj;
   /* XOff and YOff are screen offsets */
{
   struct ArcRec	* arc_ptr = ArcObj->detail.a;
   int			w, h, ltx, lty, rbx, rby, aw, ah, dx, dy, theta=0;
   double		cx, cy, rx, ry, tmp_x, tmp_y, x = 0.0, y = 0.0;
   double		ov_int_x1 = 0.0, ov_int_x2 = 0.0;
   double		ov_int_y1 = 0.0, ov_int_y2 = 0.0;
   double		rad_int_x1 = 0.0, rad_int_x2 = 0.0;
   double		rad_int_y1 = 0.0, rad_int_y2 = 0.0;
   double		len, sine, cosine;
   int			fill = arc_ptr->fill, angle1, angle2;
   int			arc_x1, arc_y1, arc_x2, arc_y2, theta1, theta2;
   int			pass_theta1, just_pass_theta1, angle, dir, full_circle;
   XPoint		tmp_v[4];

   ltx = OFFSET_X(arc_ptr->ltx);
   lty = OFFSET_Y(arc_ptr->lty);
   rbx = OFFSET_X(arc_ptr->ltx+arc_ptr->w);
   rby = OFFSET_Y(arc_ptr->lty+arc_ptr->h);

   if (ltx==rbx && lty==rby) return (FALSE);

   rx = (rbx-ltx)/2; ry = (rby-lty)/2;
   cx = (double)(OFFSET_X(arc_ptr->xc)); cy = (double)(OFFSET_Y(arc_ptr->yc));
   arc_x1 = OFFSET_X(arc_ptr->x1); arc_y1 = OFFSET_Y(arc_ptr->y1);
   dir = arc_ptr->dir;

   theta1 = (int)(arc_ptr->angle1)/64;
   theta2 = theta1 + (int)(arc_ptr->angle2)/64;

   ArcRealX2Y2 (arc_ptr, &arc_x2, &arc_y2);
   arc_x2 = OFFSET_X(arc_x2); arc_y2 = OFFSET_Y(arc_y2);

   if (theta2 < -180) theta2 += 360;
   if (theta2 > 180) theta2 -= 360;

   if (theta1 < 0) theta1 += 360;
   if (theta2 <= 0) theta2 += 360;

   angle1 = arc_ptr->angle1;
   angle2 = arc_ptr->angle2;

   full_circle = (abs(angle2) == 64*360);

   if (rx >= ry)
   {  /* flat oval */
      tmp_y = sqrt (fabs ((double)(ry*ry*(1-(XOff-cx)*(XOff-cx)/rx/rx))));
      ov_int_y1 = cy - tmp_y;
      ov_int_y2 = cy + tmp_y;
   }
   else
   {  /* tall oval */
      tmp_x = sqrt (fabs ((double)(rx*rx*(1-(YOff-cy)*(YOff-cy)/ry/ry))));
      ov_int_x1 = cx - tmp_x;
      ov_int_x2 = cx + tmp_x;
   }

   w = ZOOMED_HALF_W(arc_ptr->width)+4;
   if (rx >= ry)
   {  /* flat oval */
      if (fabs (arc_x1-cx) < ARC_TOL)
      {
         switch (theta1)
         {
            case 90: rad_int_y1 = cy - 1/ARC_TOL; break;
            case 270: rad_int_y1 = cy + 1/ARC_TOL; break;
            default: fprintf (stderr, "theta1 = %1d (flat)\n", theta1); break;
         }
      }
      else
         rad_int_y1 = cy + (XOff-cx)*(arc_y1-cy)/(arc_x1-cx);

      if (fabs (arc_x2-cx) < ARC_TOL)
      {
         switch (theta2)
         {
            case 90: rad_int_y2 = cy - 1/ARC_TOL; break;
            case 270: rad_int_y2 = cy + 1/ARC_TOL; break;
            default: fprintf (stderr, "theta2 = %1d (flat)\n", theta2); break;
         }
      }
      else
         rad_int_y2 = cy + (XOff-cx)*(arc_y2-cy)/(arc_x2-cx);
   }
   else
   {  /* tall oval */
      if (fabs (arc_y1-cy) < ARC_TOL)
      {
         switch (theta1)
         {
            case 0:
            case 360: rad_int_x1 = cx + 1/ARC_TOL; break;
            case 180: rad_int_x1 = cx - 1/ARC_TOL; break;
            default: fprintf (stderr, "theta1 = %1d (tall)\n", theta1); break;
         }
      }
      else
         rad_int_x1 = cx + (YOff-cy)*(arc_x1-cx)/(arc_y1-cy);

      if (fabs (arc_y2-cy) < ARC_TOL)
      {
         switch (theta2)
         {
            case 0:
            case 360: rad_int_x2 = cx + 1/ARC_TOL; break;
            case 180: rad_int_x2 = cx - 1/ARC_TOL; break;
            default: fprintf (stderr, "theta2 = %1d (tall)\n", theta2); break;
         }
      }
      else
         rad_int_x2 = cx + (YOff-cy)*(arc_x2-cx)/(arc_y2-cy);
   }
   if (dir == ARC_CCW)
   {
      angle = 0;
      pass_theta1 = FALSE;
      just_pass_theta1 = FALSE;
      while (angle < theta2 || !pass_theta1)
      {
         if (angle >= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            just_pass_theta1 = TRUE;
            if (theta2 > theta1 && angle >= theta2 && !full_circle)
            {  /* theta1 and theta2 are in the same quadrant */
               if (fill != NONEPAT)
               {
                  if (rx >= ry)
                  {  /* flat oval */
                     if (PointInFlatPie (YOff, dir, angle, ov_int_y1,
                           ov_int_y2, rad_int_y1, rad_int_y2))
                        return (TRUE);
                  }
                  else
                  {  /* tall oval */
                     if (PointInTallPie (XOff, dir, angle, ov_int_x1,
                           ov_int_x2, rad_int_x1, rad_int_x2))
                        return (TRUE);
                  }
               }
               if (rx >= ry)
               {  /* flat oval */
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        return (XOff>=arc_x1 && arc_x2>=XOff &&
                              (fabs ((double)(YOff-ov_int_y2)) <= w));
                     case 90:
                     case 180:
                        return (XOff>=arc_x2 && arc_x1>=XOff &&
                              (fabs ((double)(YOff-ov_int_y1)) <= w));
                  }
               }
               else
               {  /* tall oval */
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        return (YOff>=arc_y2 && arc_y1>=YOff &&
                              (fabs ((double)(XOff-ov_int_x2)) <= w));
                     case 180:
                     case 270:
                        return (YOff>=arc_y1 && arc_y2>=YOff &&
                              (fabs ((double)(XOff-ov_int_x1)) <= w));
                  }
               }
            }
            if (theta2 <= theta1) angle -= 360;
            if (angle > theta2) angle -= 360;
         }
         if (just_pass_theta1)
         {
            just_pass_theta1 = FALSE;
            if (rx >= ry)
            {  /* flat oval */
               switch ((angle+360)%360)
               {
                  case 0: y = cy; break;
                  case 90: y = cy-ry; break;
                  case 180: y = cy; break;
                  case 270: y = cy+ry; break;
               }
               if (fill != NONEPAT)
               {
                  if (XInPieRange (XOff, dir, angle, cx, rx) &&
                        PointInFlatPie (YOff, dir, angle, ov_int_y1,
                        ov_int_y2, rad_int_y1, y))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (XOff>=arc_x1 && cx+rx>=XOff &&
                           (fabs ((double)(YOff-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (XOff>=cx && arc_x1>=XOff &&
                           (fabs ((double)(YOff-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (XOff>=cx-rx && arc_x1>=XOff &&
                           (fabs ((double)(YOff-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (XOff>=arc_x1 && cx>=XOff &&
                           (fabs ((double)(YOff-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
               }
            }
            else
            {  /* tall oval */
               switch ((angle+360)%360)
               {
                  case 0: x = cx+rx; break;
                  case 90: x = cx; break;
                  case 180: x = cx-rx; break;
                  case 270: x = cx; break;
               }
               if (fill != NONEPAT)
               {
                  if (YInPieRange (YOff, dir, angle, cy, ry) &&
                        PointInTallPie (XOff, dir, angle, ov_int_x1,
                        ov_int_x2, rad_int_x1, x))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (YOff>=cy && arc_y1>=YOff &&
                           (fabs ((double)(XOff-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (YOff>=cy-ry && arc_y1>=YOff &&
                           (fabs ((double)(XOff-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (YOff>=arc_y1 && cy>=YOff &&
                           (fabs ((double)(XOff-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (YOff>=arc_y1 && cy+ry>=YOff &&
                           (fabs ((double)(XOff-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
               }
            }
         }
         else if (pass_theta1)
         {  /* see if point is in the quadrant */
            if (rx >= ry)
            {  /* flat oval */
               if (XInPieRange (XOff, dir, angle, cx, rx))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 90:
                        case 180:
                           if (YOff>=ov_int_y1 && cy>=YOff) return (TRUE);
                           break;

                        case 0:
                        case 270:
                           if (YOff>=cy && ov_int_y2>=YOff) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        if (fabs ((double)(YOff-ov_int_y2)) <= w) return (TRUE);
                        break;
                     case 90:
                     case 180:
                        if (fabs ((double)(YOff-ov_int_y1)) <= w) return (TRUE);
                        break;
                  }
               }
            }
            else
            {  /* tall oval */
               if (YInPieRange (YOff, dir, angle, cy, ry))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 0:
                        case 90:
                           if (XOff>=cx && ov_int_x2>=XOff) return (TRUE);
                           break;

                        case 180:
                        case 270:
                           if (XOff>=ov_int_x1 && cx>=XOff) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        if (fabs ((double)(XOff-ov_int_x2)) <= w) return (TRUE);
                        break;

                     case 180:
                     case 270:
                        if (fabs ((double)(XOff-ov_int_x1)) <= w) return (TRUE);
                        break;
                  }
               }
            }
         }
         angle = (angle == 360) ? 0 : (angle+90);
      }
      if (rx >= ry)
      {  /* flat oval */
         switch ((angle+360)%360)
         {
            case 0: y = cy+ry; break;
            case 180: y = cy-ry; break;

            case 90:
            case 270: y = cy; break;
         }
         if (fill != NONEPAT)
         {
            if (XInPieRange (XOff, dir, angle, cx, rx) &&
                  PointInFlatPie (YOff, dir, angle, ov_int_y1, ov_int_y2,
                  y, rad_int_y2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (XOff>=cx && arc_x2>=XOff &&
                     (fabs ((double)(YOff-ov_int_y2)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (XOff>=arc_x2 && cx+rx>=XOff &&
                     (fabs ((double)(YOff-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (XOff>=arc_x2 && cx>=XOff &&
                     (fabs ((double)(YOff-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (XOff>=cx-rx && arc_x2>=XOff &&
                     (fabs ((double)(YOff-ov_int_y2)) <= w))
                  return (TRUE);
               break;
         }
      }
      else
      {  /* tall oval */
         switch ((angle+360)%360)
         {
            case 0:
            case 180: x = cx; break;

            case 90: x = cx+rx; break;
            case 270: x = cx-rx; break;
         }
         if (fill != NONEPAT)
         {
            if (YInPieRange (YOff, dir, angle, cy, ry) &&
                  PointInTallPie (XOff, dir, angle, ov_int_x1, ov_int_x2,
                  x, rad_int_x2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (YOff>=arc_y2 && cy+ry>=YOff &&
                     (fabs ((double)(XOff-ov_int_x2)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (YOff>=arc_y2 && cy>=YOff &&
                     (fabs ((double)(XOff-ov_int_x2)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (YOff>=cy-ry && arc_y2>=YOff &&
                     (fabs ((double)(XOff-ov_int_x1)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (YOff>=cy && arc_y2>=YOff &&
                     (fabs ((double)(XOff-ov_int_x1)) <= w))
                  return (TRUE);
               break;
         }
      }
   }
   else
   {
      angle = 360;
      pass_theta1 = FALSE;
      just_pass_theta1 = FALSE;
      while (angle > theta2 || !pass_theta1)
      {
         if (angle <= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            just_pass_theta1 = TRUE;
            if (theta2 < theta1 && angle <= theta2 && !full_circle)
            {  /* theta1 and theta2 are in the same quadrant */
               if (fill != NONEPAT)
               {
                  if (rx >= ry)
                  {
                     if (PointInFlatPie (YOff, dir, angle, ov_int_y1,
                           ov_int_y2, rad_int_y1, rad_int_y2))
                        return (TRUE);
                  }
                  else
                  {
                     if (PointInTallPie (XOff, dir, angle, ov_int_x1,
                           ov_int_x2, rad_int_x1, rad_int_x2))
                        return (TRUE);
                  }
               }
               if (rx >= ry)
               {  /* flat oval */
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        return (XOff>=arc_x1 && arc_x2>=XOff &&
                              (fabs ((double)(YOff-ov_int_y1)) <= w));
                     case 180:
                     case 270:
                        return (XOff>=arc_x2 && arc_x1>=XOff &&
                              (fabs ((double)(YOff-ov_int_y2)) <= w));
                  }
               }
               else
               {  /* tall oval */
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        return (YOff>=arc_y1 && arc_y2>=YOff &&
                              (fabs ((double)(XOff-ov_int_x2)) <= w));
                     case 90:
                     case 180:
                        return (YOff>=arc_y2 && arc_y1>=YOff &&
                              (fabs ((double)(XOff-ov_int_x1)) <= w));
                  }
               }
            }
            if (theta2 >= theta1) angle += 360;
            if (angle <= theta2) angle += 360;
         }
         if (just_pass_theta1)
         {
            just_pass_theta1 = FALSE;
            if (rx >= ry)
            {  /* flat oval */
               switch ((angle+360)%360)
               {
                  case 0: y = cy; break;
                  case 90: y = cy-ry; break;
                  case 180: y = cy; break;
                  case 270: y = cy+ry; break;
               }
               if (fill != NONEPAT)
               {
                  if (XInPieRange (XOff, dir, angle, cx, rx) &&
                        PointInFlatPie (YOff, dir, angle, ov_int_y1,
                        ov_int_y2, rad_int_y1, y))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (XOff>=arc_x1 && cx+rx>=XOff &&
                           (fabs ((double)(YOff-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (XOff>=arc_x1 && cx>=XOff &&
                           (fabs ((double)(YOff-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (XOff>=cx-rx && arc_x1>=XOff &&
                           (fabs ((double)(YOff-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (XOff>=cx && arc_x1>=XOff &&
                           (fabs ((double)(YOff-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
               }
            }
            else
            {  /* tall oval */
               switch ((angle+360)%360)
               {
                  case 0: x = cx+rx; break;
                  case 90: x = cx; break;
                  case 180: x = cx-rx; break;
                  case 270: x = cx; break;
               }
               if (fill != NONEPAT)
               {
                  if (YInPieRange (YOff, dir, angle, cy, ry) &&
                        PointInTallPie (XOff, dir, angle, ov_int_x1,
                        ov_int_x2, rad_int_x1, x))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (YOff>=arc_y1 && cy>=YOff &&
                           (fabs ((double)(XOff-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (YOff>=cy-ry && arc_y1>=YOff &&
                           (fabs ((double)(XOff-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (YOff>=cy && arc_y1>=YOff &&
                           (fabs ((double)(XOff-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (YOff>=arc_y1 && cy+ry>=YOff &&
                           (fabs ((double)(XOff-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
               }
            }
         }
         else if (pass_theta1)
         {  /* see if point is in the quadrant */
            if (rx >= ry)
            {  /* flat oval */
               if (XInPieRange (XOff, dir, angle, cx, rx))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 0:
                        case 90:
                           if (YOff>=ov_int_y1 && cy>=YOff) return (TRUE);
                           break;
      
                        case 180:
                        case 270:
                           if (YOff>=cy && ov_int_y2>=YOff) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        if (fabs ((double)(YOff-ov_int_y1)) <= w) return (TRUE);
                        break;
                     case 180:
                     case 270:
                        if (fabs ((double)(YOff-ov_int_y2)) <= w) return (TRUE);
                        break;
                  }
               }
            }
            else
            {  /* tall oval */
               if (YInPieRange (YOff, dir, angle, cy, ry))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 0:
                        case 270:
                           if (XOff>=cx && ov_int_x2>=XOff) return (TRUE);
                           break;
                        case 90:
                        case 180: 
                           if (XOff>=ov_int_x1 && cx>=XOff) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        if (fabs ((double)(XOff-ov_int_x2)) <= w) return (TRUE);
                        break;
                     case 90:
                     case 180:
                        if (fabs ((double)(XOff-ov_int_x1)) <= w) return (TRUE);
                        break;
                  }
               }
            }
         }
         angle = (angle == 0) ? 360 : (angle-90);
      }
      if (rx >= ry)
      {  /* flat oval */
         switch ((angle+360)%360)
         {
            case 0: y = cy-ry; break;
            case 180: y = cy+ry; break;

            case 90:
            case 270: y = cy; break;
         }
         if (fill != NONEPAT)
         {
            if (XInPieRange (XOff, dir, angle, cx, rx) &&
                  PointInFlatPie (YOff, dir, angle, ov_int_y1, ov_int_y2,
                  y, rad_int_y2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (XOff>=cx && arc_x2>=XOff &&
                     (fabs ((double)(YOff-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (XOff>=cx-rx && arc_x2>=XOff &&
                     (fabs ((double)(YOff-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (XOff>=arc_x2 && cx>=XOff &&
                     (fabs ((double)(YOff-ov_int_y2)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (XOff>=arc_x2 && cx+rx>=XOff &&
                     (fabs ((double)(YOff-ov_int_y2)) <= w))
                  return (TRUE);
               break;
         }
      }
      else
      {  /* tall oval */
         switch ((angle+360)%360)
         {
            case 0:
            case 180: x = cx; break;

            case 90: x = cx-rx; break;
            case 270: x = cx+rx; break;
         }
         if (fill != NONEPAT)
         {
            if (YInPieRange (YOff, dir, angle, cy, ry) &&
                  PointInTallPie (XOff, dir, angle, ov_int_x1, ov_int_x2,
                  x, rad_int_x2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (YOff>=cy-ry && arc_y2>=YOff &&
                     (fabs ((double)(XOff-ov_int_x2)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (YOff>=arc_y2 && cy>=YOff &&
                     (fabs ((double)(XOff-ov_int_x1)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (YOff>=arc_y2 && cy+ry>=YOff &&
                     (fabs ((double)(XOff-ov_int_x1)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (YOff>=cy && arc_y2>=YOff &&
                     (fabs ((double)(XOff-ov_int_x2)) <= w))
                  return (TRUE);
               break;
         }
      }
   }

   w = ZOOMED_SIZE(arc_ptr->w);
   h = ZOOMED_SIZE(arc_ptr->h);
   aw = ZOOMED_SIZE(arc_ptr->aw); if (aw == 0) aw = 1;
   ah = ZOOMED_SIZE(arc_ptr->ah); if (ah == 0) ah = 1;

   if ((arc_ptr->style & LS_LEFT) && (angle2%(360*64) != 0))
   {  /* the arrow should appear at angle1 */
      switch (dir)
      {
         case ARC_CCW: theta = (int)(angle1/64)-90; break;
         case ARC_CW: theta = (int)(angle1/64)+90; break;
      }
      dx = -round(w*cos(((double)theta)*M_PI/180.0));
      dy = round(h*sin(((double)theta)*M_PI/180.0));
      if (dx == 0 && dy == 0)
      {
         sine = cosine = ((double)0.0);

         tmp_v[0].x = tmp_v[1].x = tmp_v[2].x = tmp_v[3].x = arc_x1;
         tmp_v[0].y = tmp_v[1].y = tmp_v[2].y = tmp_v[3].y = arc_y1;
      }
      else
      {
         len = (double) sqrt ((double)(dx*dx+dy*dy));
         sine = dy/len;
         cosine = dx/len;

         tmp_v[0].x = tmp_v[3].x = arc_x1;
         tmp_v[0].y = tmp_v[3].y = arc_y1;
         tmp_v[1].x = round(arc_x1 + aw*cosine - ah*sine);
         tmp_v[1].y = round(arc_y1 + aw*sine + ah*cosine);
         tmp_v[2].x = round(arc_x1 + aw*cosine + ah*sine);
         tmp_v[2].y = round(arc_y1 + aw*sine - ah*cosine);
      }

      if (PointInPolygon (XOff, YOff, 4, tmp_v)) return (TRUE);
   }
   if ((arc_ptr->style & LS_RIGHT) && (angle2%(360*64) != 0))
   {
      switch (dir)
      {
         case ARC_CCW: theta = (int)((angle1+angle2)/64)-90; break;
         case ARC_CW: theta = (int)((angle1+angle2)/64)+90; break;
      }
      dx = -round(w*cos(((double)theta)*M_PI/180.0));
      dy = round(h*sin(((double)theta)*M_PI/180.0));
      if (dx == 0 && dy == 0)
      {
         sine = cosine = ((double)0.0);

         tmp_v[0].x = tmp_v[1].x = tmp_v[2].x = tmp_v[3].x = arc_x2;
         tmp_v[0].y = tmp_v[1].y = tmp_v[2].y = tmp_v[3].y = arc_y2;
      }
      else
      {
         len = (double) sqrt ((double)(dx*dx+dy*dy));
         sine = dy/len;
         cosine = dx/len;

         tmp_v[0].x = tmp_v[3].x = arc_x2;
         tmp_v[0].y = tmp_v[3].y = arc_y2;
         tmp_v[1].x = round(arc_x2 - aw*cosine + ah*sine);
         tmp_v[1].y = round(arc_y2 - aw*sine - ah*cosine);
         tmp_v[2].x = round(arc_x2 - aw*cosine - ah*sine);
         tmp_v[2].y = round(arc_y2 - aw*sine + ah*cosine);
      }

      if (PointInPolygon (XOff, YOff, 4, tmp_v)) return (TRUE);
   }
   return (FALSE);
}

int FindGoodObj (XOff, YOff, FirstObj, SubObj)
   int			XOff, YOff;
   struct ObjRec	* FirstObj, * * SubObj;
   /* XOff and YOff are screen offsets */
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;

   *SubObj = NULL;

   for (obj_ptr = FirstObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
         if (attr_ptr->shown &&
               XOff >= OFFSET_X(attr_ptr->obj->bbox.ltx)-3 &&
               YOff >= OFFSET_Y(attr_ptr->obj->bbox.lty)-3 &&
               XOff <= OFFSET_X(attr_ptr->obj->bbox.rbx)+3 &&
               YOff <= OFFSET_Y(attr_ptr->obj->bbox.rby)+3)
         {
            *SubObj = attr_ptr->obj;
            return (TRUE);
         }

      if (XOff >= OFFSET_X(obj_ptr->bbox.ltx)-3 &&
            YOff >= OFFSET_Y(obj_ptr->bbox.lty)-3 &&
            XOff <= OFFSET_X(obj_ptr->bbox.rbx)+3 &&
            YOff <= OFFSET_Y(obj_ptr->bbox.rby)+3)
      {
         switch (obj_ptr->type)
         {
            case OBJ_TEXT: return (TRUE);
            case OBJ_XBM:
               if (FindGoodXBm (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_XPM:
               if (FindGoodXPm (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_BOX:
               if (FindGoodBox (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_RCBOX:
               if (FindGoodRCBox (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_OVAL:
               if (FindGoodOval (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_POLY:
               if (FindGoodPoly (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_POLYGON:
               if (FindGoodPolygon (XOff, YOff, obj_ptr)) return (TRUE); break;
            case OBJ_ARC:
               if (FindGoodArc (XOff, YOff, obj_ptr)) return (TRUE); break;

            case OBJ_GROUP:
            case OBJ_SYM:
            case OBJ_ICON:
               if (FindGoodObj (XOff, YOff, obj_ptr->detail.r->first, SubObj))
                  return (TRUE);
               break;
         }
      }
   }
   return (FALSE);
}
