/* POPD - Program to change to a "pushed" directory */
/* PUSHD - Program to save current working directory in environment, and
    change to new directory */
/* Copyright Peter Barker, 1992. Version 1.0

Usage:
pushd <path> - will save current directory, and change dir to <path>
popd         - return to previous directory.
<path> may optionally include a drive name (followed by :). If a
    drive name only is specified, will change to default directory
    on drive. (e.g. pushd d: will change to previous default directory
    on drive d:, but pushd d:\ will change to root directory).
At least one of the environment variables TEMP or TMP must be set to a
directory which can be used for temporary files before using pushd or popd.


These programs operate by opening a file called "PUSH$77.00n" in the
directory pointed by your TMP or TEMP environment variables. The name of
the current working directory is saved in this file by PUSHD, and read
from this file by POPD. The first use of PUSHD will use n=1, the second
use n=2 etc. POPD looks for the highest n first, so the two programs
work as if they were pushing an popping directory names from a stack.
The file PUSHPOP.C should be compiled with Microsoft C v6 or Borland
C 3.1 (should work with other versions and C compilers), and PUSHPOP.EXE
copied to PUSHD.EXE and POPD.EXE. The program operates as PUSH or POP
by checking the name it was called by, or, in old versions of DOS where
this is not possible, by checking if the user specified a <path>. It also
uses this method if it renamed to something else not beginning with
PUSHD or POPD.
At least one of the environment variables TEMP or TMP must
be set before running the programs. */

#include <stdio.h>
#include <direct.h>
#include <stdlib.h>
#include <dos.h>
#include <string.h>

#define COPYRIGHT "PUSHD/POPD version 1.0 Copyright P.Barker 1992\n"

/* Tempory filename */
#define MYTEMP  "PUSHD$77.001"

/*==========================================================================*
 *                                 del_file                                 *
 *==========================================================================*/

void del_file( char *path, int handle  )
{
    /* Delete file from "stack", as no further use to us */
    _dos_close( handle );
    unlink( path );
}   /* del_file */

/*==========================================================================*
 *                                 get_temp                                 *
 *==========================================================================*/

void get_temp( char *path  )
{
    struct find_t finfo;
    char *ptr;
    int i;
    /* Form temporary file name */
    if( (ptr = getenv( "TEMP" )) == NULL )
        ptr = getenv( "TMP" );
    if( ptr == NULL ) {
        printf( "Must have environment variable TEMP or TMP set\n" );
        exit( 5 );
    }
    strcpy( path, ptr );
    if( path[ (i = strlen( path ) - 1) ] != '\\' ) {
        path[ ++i ] = '\\';
        path[ ++i ] = '\0';
    }
    strcat( path, MYTEMP );
    i = strlen( path ) - 1;
    /* Make sure last file in "stack" */
    while( _dos_findfirst( path, _A_NORMAL, &finfo ) == 0 )
        /* file already exists */
        path[ i ]++;
}   /* get_temp */

/*==========================================================================*
 *                                 pp_chdir                                 *
 *==========================================================================*/

int pp_chdir( char *dir  )
{
    unsigned i;
    /* Allow for user specifying drive with no trailing \ - will just change
        default drive */
    if( (i = strlen( dir )) != 2 || dir[ 1 ] != ':' ) {
        if( chdir( dir ) != 0 ) {
            /* May have terminating \, test */
            if( dir[ --i ] == '\\' ) {
                /* try without \ */
                dir[ i ] = '\0';
                if( chdir( dir ) != 0 ) {
                    dir[ i ] = '\\';
                    return( -1 );
                }
            } else {
                return( -1 );
            }
        }
    }
    
    /* Change drive */
    if( dir[ 1 ] == ':' ) {
        i = dir[ 0 ] - 'A';
        if( i > 26 )
            i = dir[ 0 ] - 'a';
        _dos_setdrive( i + 1, &i );
    }
    return( 0 );
}   /* pp_chdir */

/*==========================================================================*
 *                                  usage                                   *
 *==========================================================================*/

void usage( char *caller )
{
    printf( COPYRIGHT );
    printf( "\nUsage:\n\tpushd pathname\nStores name of "
        "current working directory, and changes directory to \"pathname\".\n"
        "\tpopd\nChanges to directory previously \"pushed\" by pushd.\n\n"
        "A valid path must be assigned to either TMP or TEMP in your "
        "environment to use\nthese programs, as the pathname is \"pushed\" "
        "to a file in this directory. At\n"
        "least one of them should be set in your AUTOEXEC.BAT file by a "
        "SET statement,\nfor example:\n\tSET TMP=C:\\TEMPDIR\n" );
}   /* usage */

/*==========================================================================*
 *                                   popd                                   *
 *==========================================================================*/

void popd( int argc, char **argv  )
{
    char tmppath[ FILENAME_MAX + 1 ];
    char newdir[ FILENAME_MAX + 1 ];
    int handle;
    unsigned i;

    /* Check if user wants help, or has not used correctly */
    if( argc > 1 ) {
        usage( "popd" );
        exit( 0 );
    }
    /* Construct temporary filename */
    get_temp( tmppath );
    /* Now change to highest numbered existing file (top of "stack") */
    i = strlen( tmppath ) - 1;
    tmppath[ i ]--;
    
    /* Open temporary file */
    if( _dos_open( tmppath, _A_NORMAL, &handle ) != 0 ) {
        printf( "No pathname pushed\n" );
        exit( 6 );
    }
    if( _dos_read( handle, newdir, FILENAME_MAX, &i ) != 0 ) {
        printf( "Unable to read file %s\n", tmppath );
        del_file( tmppath, handle );
        exit( 3 );
    }
    
    /* Change directory */
    newdir[ i ] = '\0';
    if( pp_chdir( newdir ) != 0 ) {
        printf( "Unable to change to directory %s\n", newdir );
        del_file( tmppath, handle );
        exit( 4 );
    }

    /* Delete file from "stack" */
    del_file( tmppath, handle );
}   /* popd */

/*==========================================================================*
 *                                  pushd                                   *
 *==========================================================================*/

void pushd( int argc, char **argv  )
{
    char dirname[ FILENAME_MAX + 1 ];
    char tmppath[ FILENAME_MAX + 1 ];
    int handle;
    unsigned i;

    /* Check if user wants help, or has not used correctly */
    if( argc != 2 || ( i = *(argv[ 1 ]) ) == '-' || i == '/' ) {
        usage( "pushd" );
        exit( 0 );
    }
    
    /* Get current directory */
    if( getcwd( dirname, FILENAME_MAX ) == NULL ) {
        printf( "No current directory!!!!!\n" );
        exit( 2 );
    }
    
    /* Find pathname for temporary file */
    get_temp( tmppath );
    
    /* Change directory */
    if( pp_chdir( argv[ 1 ] ) != 0 ) {
        printf( "Unable to change to directory %s\n", argv[ 1 ] );
        exit( 7 );
    }
    
    /* Create temporary file and save previous path */
    if( _dos_creat( tmppath, _A_NORMAL, &handle ) != 0 ) {
        printf( "Unable to create file %s\n", tmppath );
        exit( 6 );
    }
    if( _dos_write( handle, dirname, strlen( dirname ), &i ) != 0 ) {
        printf( "Unable to save directory name in file %s\n", tmppath );
        exit( 3 );
    }
}   /* pushd */

/*==========================================================================*
 *                                   main                                   *
 *                                   PUSHPOP                                *
 *==========================================================================*/

void main( int argc, char **argv )
{
    char *ptr;
    ptr = argv[ 0 ];
    do {
        while( *ptr && *ptr != 'p' && *ptr != 'P' )
            ptr++;
        if( strnicmp( ptr, "popd", 4 ) == 0 ) {
            popd( argc, argv );
            exit( 0 );
        } else if( strnicmp( ptr, "pushd", 5 ) == 0 ) {
            pushd( argc, argv );
            exit( 0 );
        }
        if( *ptr )
            ptr++;
    } while( *ptr );
    /* Allow for another name, and base choice on number of arguments */
    if( argc == 1 ) {
        popd( argc, argv );
        exit( 0 );
    } else if( argc == 2 ) {
        pushd( argc, argv );
        exit( 0 );
    } else
        usage( "pushpop" );
}
