*-------------------------------------------------------------------------------
*-- Program...: TIME.PRG
*-- Programmer: Ken Mayer (KENMAYER)
*-- Date......: 06/25/1992
*-- Notes.....: These are a series of routines that deal with time strings,
*--             and so on. Very useful. See README.TXT for more details on
*--             the use of this library file.
*-------------------------------------------------------------------------------

FUNCTION Delay
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Delay Loop.  Returns .T. after lapse of given number of 
*--               seconds. Accurate only to within one second.
*--               <some published ways to do this do not work, because they 
*--               ignore the possibility that the interval may start at 59 
*--               seconds, and thus that the ending number of seconds will be 
*--               smaller.>
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: TIME2SEC()           Function in TIME.PRG
*-- Called by...: Any
*-- Usage.......: Delay(<nSeconds>)
*-- Example.....: lX= Delay(10)
*-- Returns.....: Logical
*-- Parameters..: nSeconds = number of seconds to delay
*-------------------------------------------------------------------------------

	parameters nSeconds         && up to 86400, one day
	private nTimeout
	nTimeout = mod( Time2Sec( time() ) + nSeconds, 86400 )
   do while Time2Sec( time() ) # nTimeout
	   *-- Nothing to do ...
	enddo

RETURN .T.
*-- EoF: Delay()

FUNCTION Time2Sec
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Convert HH:MM:SS or HH:MM:SS.SS string to seconds.
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Time2Sec("<cTime>")
*-- Example.....: ?Time2Sec("01:24:15")
*-- Returns.....: Numeric
*-- Parameters..: cTime = Time string in format HH:MM:SS or HH:MM:SS.SS
*-------------------------------------------------------------------------------
	
	parameters cTime
	private cTemp, nSecs
	cTemp = cTime
	nSecs = 3600 * val( cTemp )
	cTemp = substr( cTemp, at( ":", cTemp ) + 1 )
	nSecs = nSecs + 60 * val( cTemp )
	
RETURN nSecs + val( substr( cTemp, at( ":", cTemp ) + 1 ) )
*-- EoF: Time2Sec()

FUNCTION Sec2Time
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Convert number of seconds to time string in format of
*--               HH:MM:SS or HH:MM:SS.SS.
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Sec2Time("<cTime>")
*-- Example.....: ?Sec2Time(30001.3)
*-- Returns.....: Character String
*-- Parameters..: nSeconds = Seconds to be converted 
*-------------------------------------------------------------------------------

	parameters nSeconds
	private nHrs, nMins , nSecs, cTemp
	nSecs = mod( nseconds, 86400 )
	nHrs  = int( nSecs / 3600 )
	nSecs = nSecs - nHrs * 3600
	nMins = int( nSecs / 60 )
	nSecs = nSecs - nMins * 60
	cTemp = transform( nHrs, "@L 99" ) + ":" + transform( nMins, "@L 99" ) ;
	  + ":"

RETURN cTemp+iif(nSecs=int(nSecs),transform(nSecs,"@L 99"),;
                 transform(nSecs,"@L 99.99"))
*-- EoF: Sec2Time()

FUNCTION DiffTime
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Calculate difference between two times given as HH:MM:SS 
*--               strings. If second time is smaller than first, assumes 
*--               midnight passage. Returns HH:MM:SS string, or HH:MM:SS.SS 
*--               string if fractional seconds passed.
*-- Rev. History: None
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: TIME2SEC()           Function in TIME.PRG
*--               SEC2TIME()           Function in TIME.PRG
*-- Called by...: Any
*-- Usage.......: DiffTime("<cTime1>","<cTime2>")
*-- Example.....: ?DiffTime("2:03:24","5:12:33")
*-- Returns.....: Character String
*-- Parameters..: cTime1 = Time to subtract from cTime2
*--               cTime2 = Time to subtract from (larger value, unless
*--                            after midnite)
*-------------------------------------------------------------------------------

	parameters cTime1, cTime2

RETURN Sec2Time( 86400 + Time2Sec( cTime2 ) - Time2Sec( cTime1 ) )
*-- EoF: DiffTime()

FUNCTION Civ2Mil
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Converts string like "12:59 a.m." to standard 24-hour 
*--               HH:MM:SS. If the string contains neither "a" nor "p", the 
*--               hours will not be converted.
*-- Rev. History: None
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Civ2Mil("<cCivilTime>")
*-- Example.....: ?Civ2Mil("2:03:24 a.m.")
*-- Returns.....: Character String
*-- Parameters..: cCivilTime = Time to convert to 24 hour time.
*-------------------------------------------------------------------------------

	parameters cCiviltime
	private cTstring, nTime
	cTstring =  trim( lower( cCiviltime ) )
	nTime = val( cTstring )
	do case
	  case "p" $ cTstring
	    cTstring = left( cTstring, at( "p", cTstring ) - 1 )
            nTime = mod( nTime, 12 ) + 12
	  case "a" $ cTstring
	    cTstring = left( cTstring, at( "a", cTstring ) - 1 )
	    nTime = mod( nTime, 12 )
	endcase
	cTstring = transform( nTime, "@L 99" ) ;
	  + trim( substr( cTstring, at( ":", cTstring ) ) )
	
RETURN cTstring + iif( len( cTstring ) = 5, ":00", "" )
*-- EoF: Civ2Mil()

FUNCTION Mil2Civ
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Converts HH:MM:SS 24-hour string to a.m or p.m. notation.
*-- Rev. History: None
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Mil2Civ("<cMilTime>")
*-- Example.....: ?Mil2Civ("14:03:24")
*-- Returns.....: Character String
*-- Parameters..: cMilTime = Time to convert to 24 hour time.
*-------------------------------------------------------------------------------

	parameters cMiltime
	private cCivtime, nHours, cMins
	cCivtime = ltrim( trim( cMiltime ) )
	nHours = val( cCivtime )
	cMins = substr( cCivtime, at( ":", cCivtime ) ) + " " ;
	  + iif( nHours > 11, "p.m.", "a.m." )
	
RETURN ltrim( str( mod( nHours + 11, 12 ) + 1 ) ) + cMins
*-- EoF: Mil2Civ()

*-------------------------------------------------------------------------------
*-- EoP: TIME.PRG
*-------------------------------------------------------------------------------
