/* Splint:
 * a data compression program
 *
 * This program refers the Pascal source codes of
 * a splay-prefix encoding program in an article of
 *
 * Jones, Douglas. W,:
 * Application of Splay Trees to Data Compression,
 * Communications ACM, Vol. 31, No. 8, pp. 996 - 1007. (August 1988)
 *
 * Written by Kenji Rikitake and Naoshi Wakatabe.
 * Copyright (C)1988, 1989 by Kenji Rikitake and Naoshi Wakatabe.
 * All rights reserved.
 * Permission to copy this program without fee all or part of this
 * material is granted, provided that the copies are not made or
 * distributed for direct commercial advantage.
 *
 * If you have any questions and/or suggestions, Contact
 *
 * Kenji Rikitake
 * 4-19-11-102, Sakurajousui, Setagaya-ku,
 * Tokyo 156 Japan
 * JUNET: rikitake%wadalab.t.u-tokyo.JUNET@relay.cs.net
 */
/* $Header: splint.cv  2.1  89/04/12 21:36:22  kenji  Exp $
 * $Log: RCS/splint.cv $
 * Revision 2.1  89/04/12 21:36:22  kenji
 * BSD filename bug fixed
 * 
 * Revision 1.6  89/03/01 14:09:36  kenji
 * Kenji welcome Naoshi one of the authors
 * 
 * Revision 1.5  89/02/22 20:39:10  kenji
 * Modified for Microsoft C 5.1 Small Model by Naoshi Wakatabe
 * 
 * Revision 1.3  89/01/07 14:51:16  kenji
 * unlink source file after successful conversion
 * 
 * Revision 1.2  89/01/07 11:48:44  kenji
 * now display RCS header
 * decoding file existence check included
 * 
 * Revision 1.1  89/01/06 23:47:40  kenji
 * Initial revision
 * 
 */

static char *identfield = "$Header: splint.cv  2.1  89/04/12 21:36:22  kenji  Exp $";


/* splint.c : encoding/decoding main program */

#include "splint.h"
#include "getopt.h"

#define MODE_UNDEF 0
#define DECODE_MODE 1
#define ENCODE_MODE 2

#define PWDLEN 64
#define FNAMLEN 32
#define MAGIC_1 0x93
#define MAGIC_2 0xB9

extern void spl_encode();
extern void spl_decode();

/* global variables */

static int encmode, verbose;
char spl_passwd[PWDLEN] = "";

/* encoding main routine */

void enc_file(orgname)
char *orgname;
{
	static char splfile[FNAMLEN + 3], orgfile[FNAMLEN];
#if defined(MSDOS)
	static char o_ext[_MAX_EXT];
	static char dummy[_MAX_PATH];
#endif
#if defined(__TURBOC__)
	static char o_ext[MAXEXT];
#endif
	int i, j;
	FILE *orgp, *splp;
	struct stat statbuf;

	/* make splayed filename in cwd */

#ifdef BSD
	char *p;
	if ((p = strrchr(orgname, '/')) != NULL) {
		strcpy(orgfile, p);
		}
	else {
		strcpy(orgfile, orgname);
		}
	/* add extension ".SP" if UNIX */
	i = strlen(orgfile);
	if (i > FNAMLEN) {
		i = FNAMLEN;
		}
	for (j = 0; j < i; j++) {
		splfile[j] = orgfile[j];
		}
	splfile[i] = '.';
	splfile[i + 1] = 'S';
	splfile[i + 2] = 'P';
	splfile[i + 3] = '\0';
#ifdef DEBUG
	fprintf(stderr, "%s, %s, %s\n", splfile, orgfile, orgname);
#endif
#endif

#ifdef __TURBOC__
	/* change extension to ".spl" if MS-DOS */
	fnsplit(orgname, NULL, NULL, orgfile, o_ext);
	strcpy(splfile, orgfile);
	strcat(orgfile, o_ext);
	strcat(splfile, ".spl");
#endif
#ifdef MSDOS
	/* change extension to ".spl" if MS-DOS(MSC) */
	_splitpath(orgname, dummy, dummy, orgfile, o_ext);
	strcpy(splfile, orgfile);
	strcat(orgfile, o_ext);
	strcat(splfile, ".spl");
#endif

	/* check existence */
	if (stat(splfile, &statbuf) == 0) {
		char response[2];
		response[0] = 'n';
		fprintf(stderr, "splint: splayed file %s exists.\n", splfile);
		fprintf(stderr, "Do you want to overwrite %s(Y/N)[N]? ", splfile);
		fflush(stderr);
		read(2, response, 2);
		while (response[1] != '\n') {
			if (read(2, response+1, 1) < 0) {
				perror("stderr"); break;
				}
			}
		if (response[0] != 'y') {
			fprintf(stderr, "splint: %s not overwritten\n", splfile);
			return;
			}
		}

	/* open files */
#ifdef BSD
	if((orgp = fopen(orgname, "r")) == NULL) {
		fprintf(stderr, "splint: cannot open input file \"%s\" \n", orgname);
		exit(-1);
		}
	if((splp = fopen(splfile, "w")) == NULL) {
		fprintf(stderr, "splint: cannot open splay file \"%s\" \n", splfile);
		exit(-1);
		}
#endif
#if defined(__TURBOC__) || defined(MSDOS)
	if((orgp = fopen(orgname, "rb")) == NULL) {
		fprintf(stderr, "splint: cannot open input file \"%s\" \n", orgname);
		exit(-1);
		}
	if((splp = fopen(splfile, "wb")) == NULL) {
		fprintf(stderr, "splint: cannot open splay file \"%s\" \n", splfile);
		exit(-1);
		}
#endif

	/* making a header */
	putc(MAGIC_1, splp);
	putc(MAGIC_2, splp);
	putc(STYLE_ID, splp);
	stat(orgname, &statbuf);
	putc(statbuf.st_size & 0xff, splp);
	putc((statbuf.st_size >> 8) & 0xff, splp);
	putc((statbuf.st_size >> 16) & 0xff, splp);
	putc((statbuf.st_size >> 24) & 0xff, splp);
	fputs(orgfile, splp);
	putc('\0', splp);
	
	/* encoding */
	rewind(orgp);
	spl_encode(orgp, splp);
	
	/* close files */
	fclose(orgp);
	fclose(splp);
	/* delete original file */
	unlink(orgname);

}

/* encoding main routine */

void dec_file(splname)
char *splname;
{
	static char orgfile[FNAMLEN + 1];
	int c, c1, c2, c3, i;
	FILE *orgp, *splp;
	struct stat statbuf;
	long orgfsize;

	/* open splayed file */

#ifdef BSD
	if((splp = fopen(splname, "rb")) == NULL) {
		fprintf(stderr, "splint: cannot open splay input file \"%s\" \n", splname);
		exit(-1);
		}
#endif
#if defined(__TURBOC__) || defined(MSDOS)
	if((splp = fopen(splname, "rb")) == NULL) {
		fprintf(stderr, "splint: cannot open splay input file \"%s\" \n", splname);
		exit(-1);
		}
#endif

	/* check MAGIC numbers */
	rewind(splp);
	c1 = getc(splp);
	c2 = getc(splp);
	c3 = getc(splp);
	if (c1 != MAGIC_1 || c2 != MAGIC_2 || c3 != STYLE_ID) {
		fprintf(stderr, "splint: bad file - magic number error\n");
		exit(-1);
		}
	
	/* get orgfile size */
	c = getc(splp);
	c1 = getc(splp);
	c2 = getc(splp);
	c3 = getc(splp);
	orgfsize = ((long)c3 << 24) | ((long)c2 << 16) | 
	           ((long)c1 << 8) | (long)c;
	
	/* get original filename */
	i = 0;
	while (1){
		c = getc(splp);
		orgfile[i] = c;
		if (c == '\0') break;
		i++;
		if (i > FNAMLEN) {
			fprintf(stderr, "splint: bad file - filename error\n");
			exit(-1);
			}
		}

#ifdef DEBUG
	fprintf(stderr, "filename = %s, length = %ld\n", orgfile, orgfsize);
#endif

	/* check existence */
	if (stat(orgfile, &statbuf) == 0) {
		char response[2];
		response[0] = 'n';
		fprintf(stderr, "splint: original file %s exists.\n", orgfile);
		fprintf(stderr, "Do you want to overwrite %s(Y/N)[N]? ", orgfile);
		fflush(stderr);
		read(2, response, 2);
		while (response[1] != '\n') {
			if (read(2, response+1, 1) < 0) {
				perror("stderr"); break;
				}
			}
		if (response[0] != 'y') {
			fprintf(stderr, "splint: %s not overwritten\n", orgfile);
			fclose(splp);
			return;
			}
		}

	/* open new original file */

#ifdef BSD
	if((orgp = fopen(orgfile, "wb")) == NULL) {
		fprintf(stderr, "splint: cannot open output file \"%s\" \n", orgfile);
		exit(-1);
		}
#endif
#if defined(__TURBOC__) || defined(MSDOS)
	if((orgp = fopen(orgfile, "wb")) == NULL) {
		fprintf(stderr, "splint: cannot open output file \"%s\" \n", orgfile);
		exit(-1);
		}
#endif

	/* decoding */
	spl_decode(splp, orgp);
	
	/* close files */
	fclose(orgp);
	fclose(splp);
	
	/* check file size */
	stat(orgfile, &statbuf);
	if (orgfsize == statbuf.st_size) {
		unlink(splname);
		}
	else {
		fprintf(stderr, "splint: decoded file %s has a wrong size, deleted\n", orgfile);
		unlink(orgfile);
		}
}

/* display usage */

static void usage()
{
	fprintf(stderr,"Usage: splint -[deh] -g<passwd> filename\n");
	fprintf(stderr,"-h to help\n");
}

/* display help & copyright message */

static void disp_help()
{
	fprintf(stderr,"Splint: a splay-prefix data compression program\n");
	fprintf(stderr,"Beta release version\n");
	fprintf(stderr,"$Header: splint.cv  2.1  89/04/12 21:36:22  kenji  Exp $\n");
	fprintf(stderr,"Copyright (c) 1988, 1989\n");
	fprintf(stderr,"by Kenji Rikitake and Naoshi Wakatabe.\n");
	fprintf(stderr,"All Rights Reserved.\n");
	putc('\n', stderr);
	usage();
	fprintf(stderr,"   -d: decode from splayed file\n");
	fprintf(stderr,"   -e: encode to splayed file\n");
	fprintf(stderr,"   -g: encrypt/decrypt with following password\n");
	fprintf(stderr,"   -h: display this help message\n");
	putc('\n', stderr);
}

/* main routine */

int main(argc, argv)
int argc;
char *argv[];
{
	int option;

	encmode = MODE_UNDEF;

	/* stdin = non-buffering keyboard */
	setbuf(stdin, NULL);

	/* parsing options */
	while ((option = getopt(argc, argv, "deg:hv")) != EOF)
		switch (option) {
		case 'd':
			encmode = DECODE_MODE;
			break;
		case 'e':
			encmode = ENCODE_MODE;
			break;
		case 'g':
			if (strlen(optarg) <= PWDLEN) {
				strcpy(spl_passwd, optarg);
				}
			else {
				fprintf(stderr, "splint: encrypt password too long\n");
				usage();
				exit(-1);
				}
			break;
		case 'h':
			disp_help();
			exit(0);
			break;
		case '?':
			usage();
			exit(-1);
			break;
		default:
			break;
		}

	if (encmode == MODE_UNDEF) {
		fprintf(stderr, "splint: tell me what should I do\n");
		usage();
		exit(-1);
		}

	/* solve wildcarded filenames */
	
	while (optind < argc) {
		switch (encmode) {
			case DECODE_MODE:
				fprintf(stderr, "Decoding %s:\n", argv[optind]);
				dec_file(argv[optind]);
				break;
			case ENCODE_MODE:
				fprintf(stderr, "Encoding %s:\n", argv[optind]);
				enc_file(argv[optind]);
				break;
			default:
				fprintf(stderr, "splint: internal error in enc/dec loop\n");
				exit(-1);
				break;
			}
		optind++;
		}
	
	exit(0);
}
