(*--------------------------------------------------------------------------*)
(*          Edit_String  -- Edit a string using Wordstar commands           *)
(*--------------------------------------------------------------------------*)

FUNCTION Edit_String( VAR In_Str      : AnyStr;
                          Buffer_Len  : INTEGER;
                          Start_X     : INTEGER;
                          Y           : INTEGER;
                          Force_Case  : BOOLEAN  ) : CHAR;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*     Function:  Edit_String                                               *)
(*                                                                          *)
(*     Purpose:   Provides for editing a string using Wordstar commands.    *)
(*                                                                          *)
(*     Callling Sequence:                                                   *)
(*                                                                          *)
(*        Edited_String = Edit_String( VAR  In_Str     : AnyStr;            *)
(*                                          Buffer_Len : INTEGER;           *)
(*                                          Start_X    : INTEGER;           *)
(*                                          Y          : INTEGER;           *)
(*                                          Force_Case : BOOLEAN ) : CHAR;  *)
(*                                                                          *)
(*           In_Str      --- String to be edited                            *)
(*           Buffer_Len  --- Maximum length allowed for In_Str              *)
(*           Start_X     --- Column to display string                       *)
(*           Y           --- Row to display string                          *)
(*           Force_Case  --- TRUE to force input to upper case              *)
(*                                                                          *)
(*     Calls:    DUPL                                                       *)
(*               GoToXY                                                     *)
(*               UpCase                                                     *)
(*               KeyPressed                                                 *)
(*               COPY                                                       *)
(*               INSERT                                                     *)
(*               DELETE                                                     *)
(*               Edit_Help                                                  *)
(*                                                                          *)
(*     Remarks:                                                             *)
(*                                                                          *)
(*        In addition to strict WordStar commands, the keys on the keypad   *)
(*        can also be used.                                                 *)
(*                                                                          *)
(*        Insert mode is on by default.                                     *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

Var
  Insert_Mode  : BOOLEAN           (* TRUE = insert mode, FALSE = overwrite *);
  Done         : BOOLEAN           (* TRUE if editing finished              *);
  Current_Char : CHAR              (* Current input editing character       *);
  X            : BYTE              (* Current column being edited           *);
  Escape       : BOOLEAN           (* TRUE if escape sequence read          *);
  Current      : CHAR              (* Current input character               *);
  In_string    : AnyStr            (* Working copy of string to be edited   *);

BEGIN (* Edit_String *)

                                   (* Initialize *)
  Done         := FALSE;
  Insert_Mode  := TRUE;
  X            := Start_X;
                                   (* Display the string to be edited *)
  In_String := In_str;
  GoToXY(X,Y);
  WRITE(In_String);
  GoToXY(X,Y);

  REPEAT                           (* Begin main edit/input loop *)

     IF (X - Start_X) = Buffer_Len THEN
        Current_Char := ^M         (* Terminate input if buffer is full *)
     ELSE
        READ( Kbd , Current_Char );    (* Get a character *)

     IF Force_Case THEN            (* Force upper case if requested *)
        Current_Char := UPCASE(Current_Char);

     REPEAT

        Escape := FALSE;

        CASE Current_Char of       (* Act on the current input *)

           ^[        : IF KeyPressed THEN
                          BEGIN

                             READ( Kbd , Current_Char );

                             Escape := TRUE;

                                   (* Translate escape sequences to *)
                                   (* WordStar commands             *)

                             CASE Current_Char OF

                                'H' : Current_Char := ^E;
                                'P' : Current_Char := ^X;
                                'K' : Current_Char := ^S;
                                'M' : Current_Char := ^D;
                                'S' : Current_Char := ^G;
                                'R' : Current_Char := ^V;
                                'O' : Current_Char := ^F;
                                'G' : Current_Char := ^A;
                                '<' : Current_Char := ^R;
                                's' : Current_Char := ^A;
                                't' : Current_Char := ^F;

                                ';' : BEGIN
                                         Edit_Help;
                                         Current_Char := ^@;
                                      END;

                                'D' : BEGIN
                                         Done   := TRUE;
                                         Escape := FALSE;
                                      End;

                                'I' : BEGIN
                                         Done   := TRUE;
                                         Escape := FALSE;
                                      End;

                                'Q' : BEGIN
                                         Done   := TRUE;
                                         Escape := FALSE;
                                      End;

                             END (* CASE *);

                          END (* Escape found *);

           ^E        : Done := TRUE;

           ^X        : Done := TRUE;

                                   (* Move to end of string *)
           ^F        : X := Start_x + LENGTH(In_string);

                                   (* Move to beginning of string *)
           ^A        : X := Start_x;

           ^R        : BEGIN
                          In_string := In_str;
                          GoToXY( Start_X , Y );
                          WRITE( In_string );
                       END;

                                   (* Toggle Insert/Overwrite Mode *)
           ^V        : Insert_Mode := NOT Insert_Mode;

                                   (* Non-destructive backspace *)
           ^S        : IF X > Start_X THEN
                          X := X - 1;

                                   (* Destructive backspace *)
           ^H,#127   : IF X > Start_X THEN
                          BEGIN
                             DELETE( In_String, X - Start_X, 1);
                             GoToXY( Start_X , Y );
                             WRITE( In_String , ' ' );
                             X := X - 1;
                          END;

                                    (* Move 1 column to right *)
           ^D        : IF (X - Start_X) < Buffer_Len THEN
                          IF (X - Start_X) < LENGTH( In_String ) THEN
                             X := X + 1;

                                    (* Delete character under cursor *)
           ^G        : BEGIN
                          DELETE( In_String, X - Start_X + 1, 1 );
                          GoToXY( Start_X , Y );
                          WRITE( In_String, ' ' );
                       END;

           ^M        : Done := TRUE;

           ^J        : Done := TRUE;

           ' '..'~'  : IF ( X - Start_X ) >= LENGTH( In_String ) THEN
                         BEGIN
                            In_String := In_String + Current_Char;
                            GoToXY( X , Y );
                            WRITE( Current_Char );
                            IF ( X - Start_X ) < Buffer_Len THEN
                               X := X + 1;
                         END

                       ELSE        (* Ordinary character *)

                                   (* If insert mode ... *)
                         IF Insert_Mode THEN
                            BEGIN

                               INSERT( Current_Char, In_String,
                                       X - Start_X + 1 );

                               In_String := COPY( In_String, 1, Buffer_Len );

                               GoToXY( Start_X, Y );
                               WRITE( In_String );

                               IF ( X - Start_X ) < Buffer_Len THEN
                                  X := X + 1;

                               GoToXY( X , Y );

                            END
                         ELSE
                           BEGIN   (* If Overwrite mode ... *)

                              In_String[ X - Start_X + 1 ] := Current_Char;

                              GoToXY( X , Y );
                              WRITE( Current_Char );

                              IF ( X - Start_X ) < Buffer_Len THEN
                                 X := X + 1;

                           END;

            ELSE

         END (* CASE *);

    UNTIL ( NOT Escape );

    GoToXY( X , Y );

  UNTIL Done;

  Edit_String := Current_Char;     (* Return the terminator *)
  In_str      := In_string;        (* Return updated string *)

END   (* Edit_String *);