/* eac.c              Easter-date calculations    ( terry stancliffe 1999
                                                    email: tpat@cwcom.net)
   usage:   eac yyyy
   input:   - a year number (AD)
   outputs: - dates of Easter Sunday(s), New Style and Old Style
            - current calendar difference, NS-OS (days)

   functions   easg()   and   easj()

*/

/* These Easter algorithms use checkable quantities (Golden Number,
Epact, Dominical Letter, paschal full moon date, etc, for direct
comparison with traditional Easter tables -- both for the Gregorian
(New Style) calendar and its Easter cycle, and for the Julian (Old
Style) calendar and its corresponding Easter cycle. */

/* compiles & runs ok using compact model in TC++ 1.01 */

/* - include <stdlib.h> */
/* - include <stdio.h> */
/* - include <string.h> */
/* - include <conio.h> */
/* - include <math.h> */

/* ------------------------------------------------------------------- */
/* test code --- */

void easg(), easj();

void main (argc, argv)
     int argc;
     char **argv;
{

char mnthn[10][4]=
  {"Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
int  mnthl[10] =  { 31,30,31,30,31,31,30,31,30,31 };

int  yyr, mmm, ddd, cc, caldiff, esd, esm;

char mnth[7];
char m3[]= {"March"};
char m4[]= {"April"};

/* begin main */

if (argc < 2)
   {
   printf("usage:   eac yyyy\n\n");
   printf("input:   - a year number (AD)\n");
   printf("outputs: - dates of Easter Sunday(s), New Style and Old Style\n");
   printf("         - current calendar difference, NS-OS (days)\n\n");
   exit (1);
   }
               /* get year number from command line */
sscanf (argv[1], "%d", &yyr);

if (yyr<-800)
   printf("Sorry! I can\'t go back as far as %d\n\n", yyr);
else
{
            /* get Gregorian Easter and show result */
easg(yyr, &mmm, &ddd, &esd);
if (mmm==3)  strcpy(mnth, m3);  else  strcpy(mnth, m4);
printf("Gregorian/New Style Easter %4d: %2d %s.\n", yyr, ddd, mnth);

            /* get Julian Easter */
easj(yyr, &mmm, &ddd, &esd);
if (mmm==3)  strcpy(mnth, m3);  else  strcpy(mnth, m4);

            /* translate OS Julian Easter date to NS equivalent */
if (yyr>=0)  cc= yyr/100;  else cc=  (yyr+1)/100 - 1;
caldiff= cc - (cc+8)/4;
esm= 3;  esd= esd + caldiff;
while (esd>mnthl[esm-3])  { esd= esd-mnthl[esm-3];  esm= esm+1; }

            /* show Julian Easter result with NS equivalent */
printf
("Julian /  Old Style Easter %4d: %2d %s (OS)",yyr,ddd,mnth);
printf
(" ( = %d %s (NS)).\n", esd, mnthn[esm-3]);

            /* show difference in calendars for the year */
printf("(Difference in Mar/Apr %4d",yyr);
printf(" between NS and OS calendars: %d days.)\n", caldiff);
}
} /*main*/

/* ------------------------------------------------------------------- */

/* function easg

   Easter calculation for Gregorian calendar --
   checks ok with a variety of tabular data from ESAE 1961 etc. */

void easg(yr, mmp, ddp, esp)
     int yr, *mmp, *ddp, *esp;

  /* input:    yr  -- year number AD
     outputs: *mmp -- 3 or 4 (for March or April Easter)
              *ddp -- day of month of Easter Sunday
              *esp -- Easter Sunday in days forward from 0 March */

{
int   gn,              /* Golden Number */
      ci, kk, adj,     /* Gregorian centurial adjustment of epact */
      ep,              /* Epact */
      pfm,
      ymod,
      dl,              /* Dominical Letter */
      mm, dd, es;

/* begin easg */

/* --> Golden number */                     /* gn in [1..19] */

/* Year 0 (1 BC) is given GN =1: year AD 18 had GN =19; all other years
repeat the same sequence. */

if (yr>=0)  gn=  1  + (yr % 19);
      else  gn=  19 + ((yr+1) % 19 );


/* --> Gregorian centurial adjustment of epact */

/* See rules in notes below and in eas.doc */

if (yr>=0)     ci= yr / 100;
         else  ci= ((yr+1) / 100 ) -1;  /* ci 19 is for 19xx, etc */

kk= ((ci+8) / 25) - 1;

adj=  (ci + 5 - ((ci+8) / 4) - ((ci+9-kk) / 3) );

     /* adj +7 is for 15xx-16xx; adj +8 is for 17xx-18xx; adj +9 is for
        19xx-21xx, etc (with a +25-cy repeat-period for the increments,
        and eventually to be reduced modulo 30), these values are to be
        SUBTRACTed from the Julian epact value (not explicitly
        calculated here, but obtainable by putting adj=0 in the formula
        below), but adj is to be ADDed to the 19gn term in the epact
        formula below.

        Comparison with traditional sources:  adj here is the row-
        difference (in the expanded epact table reproduced, in Coyne
        et al 1983 pp 212-3, from Clavius 1612 pp 110-111, see also
        file eas.doc) between the top row of epacts (Julian)
        (adj=0) and the row for the required century (adj= number
        of rows below top row). In the Calendar Act 1751 and the
        Book of Common Prayer, the adjustments are given relative
        to the values for 15xx-16xx, (which were in the 7th row
        below the Julian row in the Clavius table) setting that as
        zero, i.e. an offset of 7.  Thus, in the BCP and Calendar
        Act, 19xx-21xx has an adjustment of 2, -- here 2+7 =9, and
        so on.   */


/* --> Epact -- including centurial adjustment */   /* ep in [0..29] */

/* Tabular age of moon on 1 January in given year, minus 1 day. */

ep=  29 - ( 19*gn + adj + 2 ) % 30 ;


/* --> paschal (tabular) full moon date */          /* pfm in [21..49] */

/* Full moon is defined as 14th day counting tabular new moon as
   day 1, and paschal f.m. is first full-moon date falling on or
   after 21 March. */

pfm= 21 + ((53-ep) % 30);
   if ( (ep==24) || ((ep==25) && (gn>11) && (gn<20)) ) pfm= pfm-1;

                      /* pfm is expressed as an offset from 0 March */
                      /* pfm includes effect of special rules for
                         epact 24 and second epact 25 */


/* --> Dominical (Sunday) letter */

/* Letters A-G are written in repeating sequence against all days of the
calendar year (except 29 Feb in a leap year) starting with A on 1 January:
the Sunday letter used in the Easter calculation is then the letter that
falls against all the Sundays in March and April. In the Gregorian calendar
the DL has a 400-year cycle. */

if (yr>=0)   ymod=  yr - 400* (yr / 400);
       else  ymod=  yr - 400*((yr+1) / 400) + 400;
dl=   7 - (ymod + (ymod / 4) - (ymod / 100) + 6) % 7;

                                            /* dl in 1..7 for A..G */


/* --> Easter date */

es= 4 + dl + ( ((pfm-4-dl) / 7) +1)*7;
                /* es in [22..56], meaning 22-March through 25-April */

mm= 3 + (es-1) / 31;   dd= 1 + (es-1) % 31;
                                          /* sorts out month and day */


*mmp = mm; *ddp = dd; *esp = es;

/* for debugging or data-checking, unbracket next lines to print
   intermediates:
printf
("y:%5d  gn:%2d  adj:%2d  ep: %2d  dl: %c  pfm: %2d   es:%2d  m:%1d  d:%2d\n",
       yr, gn, adj, ep, (char)(dl+64), pfm, es, mm, dd);
*/

return;
} /* end easg */

/* ------------------------------------------------------------------- */

/* function easj

   Easter calculation for Julian (OS) calendar --
   checks ok with a variety of tabular data from ESAE 1961 etc. */

void easj(yr, mmp, ddp, esp)
int yr, *mmp, *ddp, *esp;
   /* input:    yr -- year number AD
      outputs: *mm -- 3 or 4 (for March or April Easter)
               *dd -- day of month of Easter Sunday
               *es -- Easter Sunday in days forward from 0 March */

{
int   gn,              /* Golden Number */
      ep,              /* Epact */
      pfm,
      ymod, dl,        /* Dominical Letter */
      mm, dd, es;

/* Easter dates are given by function easj (a) according to the
   traditional pre-1582 Easter cycle as recorded by Dionysius
   Exiguus in AD 525 (see Coyne et al 1983 and ESAE 1961) and (b)
   expressed in the Julian calendar of days and months, which is
   offset from the Gregorian calendar by 10 days in 15xx-16xx, 11
   days in 17xx, 12 days in 18xx, 13 days in 19xx-20xx, and
   generally in yr AD by
           ( (yr / 100) - (yr / 400) - 2 ) days
   for 1 March onwards (January and February have the same
   offset as the preceding year if that was different). */

/* begin easj */

/* --> Golden number */                     /* gn [1..19] */

if (yr>=0)  gn=  1  + (yr % 19);
      else  gn=  19 + ((yr+1) % 19 );


/* --> Epact (OS) */                        /* ep in [0..29] */

ep=  29 - ( 19*gn + 2 ) % 30 ;


/* --> paschal full moon date (OS) */       /* pfm in [21..49] */

pfm= 21 + ((53-ep) % 30);
                      /* pfm is expressed as an offset from 0 March */


/* --> Dominical (Sunday) letter (OS) */

if (yr>=0)   ymod=  yr - 28* (yr / 28);
      else   ymod=  yr - 28*((yr+1) / 28) + 28;
dl=  7 - (ymod +(ymod / 4) + 4) % 7 ;
                                            /* dl 1..7 for A..G */


/* --> Easter date */

es= 4 + dl + ( ((pfm-4-dl) / 7) +1)*7;
                /* es in [22..56], meaning 22-March through 25-April */

mm= 3 + (es-1) / 31;   dd= 1 + (es-1) % 31;
                                          /* sorts out month and day */

*mmp = mm; *ddp = dd; *esp = es;

/* for debugging or data-checking, unbracket next lines to print
   intermediates:
printf
("y:%5d  gn:%2d          ep:%2d  dl: %c  pfm: %2d   es:%2d  m:%d  d:%2d\n",
        yr, gn, ep, (char)(dl+64), pfm, es, mm, dd);
*/

return;
} /* end easj */

/* ------------------------------------------------------------------- */

