/*************************************************************************
** interpcom-1.1 (command interpreter - tutorial)                        **
** bessel.c : Manipulation of Bessel transforms                          **
**                                                                       **
** Copyright (C) 1998  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**									 **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details. 			 **
**									 **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free		 **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      Aile 45-55                                                       **
**      2, place Jussieu                                                 **
**      75251 Paris Cedex 05                                             **
**      France								 **
**                                                                       **
 *************************************************************************/


#include "interp.h"
#include "funct.h"


/*--------------------------------------------------------------------------
    Computation of the Bessel transform of a real function, from its
    Fourier transform
--------------------------------------------------------------------------*/
void
Bessel_tr(Tfourier *tr, double *x_r, double *f, int n, int nb, double *sin_c,
    double *tri_c)
{
    double	    z,
		    ct;
    int		    i,
		    ind,
		    j;
    double	    (*F)();

    ct = 1. / (double) nb / 3.;
    ind = (int) sin_c[0];
    ind = ind % 2;
    if (ind == 0)
	F = TransR;
    else
	F = TransI;

    for (j = 1; j <= n; j++) {
	z = F(tr, x_r[j] * sin_c[0]) * tri_c[1] +
	    F(tr, x_r[j] * sin_c[nb]) * tri_c[nb + 1];
	for (i = 2; i < nb - 1; i += 2) 
            z += 4. * F(tr, x_r[j] * sin_c[i]) * tri_c[i] +
		2. * F(tr, x_r[j] * sin_c[i + 1]) * tri_c[i + 1];
        f[j] = z * ct;
    }
}
/*------------------------------------------------------------------------*/





/*--------------------------------------------------------------------------
    Definition of the parameters of a Bessel transform
    The name of the associated command is def_bes_par
--------------------------------------------------------------------------*/
int		
def_bessel_param(int argc, char *argv[])
{
    int		    ind,
		    nb,
	 	    i0,
		    iw,
		    i,
		    i_par;
    char	    h[100],
		   *k[3];
    double	  **a,
		    t,
		    dt;

    ind = convert_int(argv[2]);
    if (ind < 0) {
	error_mess(_FUNC_MESS + 2);
        return 1;
    }
    nb = convert_int(argv[3]);
    if (nb < 2) {
	error_mess(_FUNC_MESS + 2);
        return 1;
    }
    iw = sketch_obj(argv[1], &i0);
    if (iw != 0) {
	error_mess(_FUNC_MESS + 0);
        return 1;
    }
    memset(h, 0, 100);
    sprintf(h, "nbes_par=%d", nb + 1);
    S_convert_int(h);
    k[0] = ch_copy("objdef");
    k[1] = ch_copy_int(_BESS_PAR - 1);
    k[2] = argv[1];
    i0 = obj_create(3, k);
    if (i0 == -1) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    a = (double **) Obj[_BESS_PAR - 1][i0].adresse;
    a[0][0] = (double) ind;
    a[1][0] = (double) nb;
    i_par = ind % 2;
    dt = PI / 2. / (double) nb; 

    for (i = 0; i <= nb; i++) {
	t = dt * i;
        a[0][i + 1] = sin(t);
        if (i_par == 0)
            a[1][i + 1] = cos(ind * t);
        else
	    a[1][i + 1] = sin(ind * t);
    }

    free(k[0]);
    free(k[1]);
    return 0;
}
/*------------------------------------------------------------------------*/





/*--------------------------------------------------------------------------
    Computation of the Bessel transform of a real function
    The name of the associated command is trans_bessel
--------------------------------------------------------------------------*/
int		
trans_bessel(int argc, char *argv[])
{
    int		    i00,
		    i01,
		    i02,
		    iw0,
		    iw1,
		    iw2,
		    i,
		    n;
    double	   *f_d,
		   *x_rd,
		  **b_par;
    float	   *x_rf;
    funct_f	   *a_f;
    funct_d	   *a_d;
    char	  **c;
    Transf	   *Tra;

    iw0 = sketch_obj_restr(argv[1], &i00, _FOUR_TR);
    if (iw0 != _FOUR_TR) {
	error_mess(_FUNC_MESS + 16);
	return 1;
    }
    c = (char **) Obj[iw0 - 1][i00].adresse;
    Tra = (Transf *) c[0];
    if (Tra->type == 1) {
	error_mess(_FUNC_MESS + 16);
	return 1;
    }
    iw1 = sketch_obj_restr(argv[2], &i01, _BESS_PAR);
    if (iw1 != _BESS_PAR) {
	error_mess(_FUNC_MESS + 18);
	return 1;
    }
    iw2 = sketch_obj_restr(argv[3], &i02, _RFUNC_F);
    if (iw2 != _RFUNC_F) {
	iw2 = sketch_obj_restr(argv[3], &i02, _RFUNC_D);
	if (iw2 != _RFUNC_D) {
	    error_mess(_FUNC_MESS + 17);
	    return 1;
  	}
    }
    a_f = NULL;
    a_d = NULL;
    c = (char **) Obj[iw2 - 1][i02].adresse;
    if (iw2 == _RFUNC_F) {
	a_f = (funct_f *) c[0];
	f_fill(a_f, &x_rf);
        n = a_f->nb;
        x_rd = double_alloc1(n);
        f_d = double_alloc1(n);
        for (i = 0; i <= n; i++)
	    x_rd[i] = (double) x_rf[i];
    }
    else {
	a_d = (funct_d *) c[0];
	d_fill(a_d, &x_rd);
        n = a_d->nb;
        f_d = a_d->f;
    }
    b_par = (double **) Obj[iw1 - 1][i01].adresse;
    
    Bessel_tr((Tfourier *) Tra->tr, x_rd, f_d, n, (int) b_par[1][0], 
 	b_par[0], b_par[1]);

    if (iw2 == _RFUNC_F) {
	for (i = 0; i <= n; i++) 
	    a_f->f[i] = (float) f_d[i];
 	XFREE(x_rd);
	XFREE(f_d);
	if (a_f->type > 0)
	    XFREE(x_rf);
    }
    else {
	if (a_d->type > 0)
	    XFREE(x_rd);
    }

    return 0;
}
/*------------------------------------------------------------------------*/
