/* example 4: show how to use the VESA 2.00+ linear framebuffer to access
   the video memory

   If you don't now what "linear frambuffer" means, read ex4.txt for some
   basic explanations. The plasma code used in this example is taken from a
   public domain source written by Phil Inch.
 */

#include <dos.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "../flat.h"
#include "ex4.h"

/* You should increase the stack size, if the program crashes. */
#ifdef __TURBOC__
extern unsigned _stklen = 2048;
#endif

/* the palette used */
char pal[768];
char *errmsg[] =
{
  NULL,
  "80386 or better CPU not detected",
  "CPU is in V86-mode",
  "Extended memory couldn't be allocated",
  "Unable to lock extended memory"
};

int
main ()
{
  int foo;

  em_max = 0;

  foo = init_flat ();

  if (foo)
    errexit (errmsg[foo]);

  /* try to read the VESA version number */
  foo = get_vesa_info ();
  if (0 > foo)
    errexit ("VESA API not supported or unaccessable");

  /* display it */
  printf ("VESA version: %d.%02d\n", foo >> 8, foo & 0xff);

  /* abort if less than two */
  if (2 > foo >> 8)
    errexit ("VESA v2.00 or better required");

  /* read video mode information */
  foo = get_vesa_mode_info (VIDEO_MODE, &vmi);
  if (foo)
    errexit ("Cannot set desired video mode");

  /* display some stuff */
  printf ("Video mode information\n"
          "**********************\n"
          "Width: %d\n"
          "Height: %d\n"
          "Linear framebuffer supported: %s\n"
          "Video memory base adress: 0x%lx\n",
          vmi.width, vmi.height, vmi.framebuf_avail ? "yes" : "NO",
          vmi.vidmem
    );

  /* abort if linear framebuffer unavailable */
  if (!vmi.framebuf_avail)
    errexit ("Linear framebuffer not supported");

  printf ("\n"
          "Press a key to see the plasma...\n");
  bget_key ();

  /* try to set the video mode with *enabled* linear framebuffer */
  if (set_vesa_mode (VIDEO_MODE | VESA2_LINEAR))
    errexit ("Cannot set VESA mode");

  /* create a nice looking palette */
  setup_pal ();

  /* seed the random number generator, not a good way, but it works */
  srand (time (NULL));

  /* draw the plasma */
  init_plasma ();
  split_box (0, 0, vmi.width - 1, vmi.height - 1);

  /* rotate the palette until a key is pressed */
  while (!bkey_pressed ())
    rotate_pal ();
  bget_key ();

  set_video_mode (0x03);
  exit_flat ();
  return (0);
}

/* init_plasma -- draw the first points in the corners of the plasma */
void
init_plasma (void)
{
  put_pixel (0, 0, 1 + (rand () % MAX_PAL));
  put_pixel (vmi.width - 1, 0, 1 + (rand () % MAX_PAL));
  put_pixel (0, vmi.height - 1, 1 + (rand () % MAX_PAL));
  put_pixel (vmi.width - 1, vmi.height - 1, 1 + (rand () % MAX_PAL));
}

/* adjust -- interpolate points */
void
adjust (int xa, int ya, int x, int y, int xb, int yb)
{
  unsigned char a, b;
  int d;
  double r, v;

  a = get_pixel (x, y);
  if (a)
    return;

  r = (double) rand () / RAND_MAX - 0.5;
  d = abs (xa - xb) + abs (ya - yb);

  a = get_pixel (xa, ya);
  b = get_pixel (xb, yb);

  v = (a + b) / 2 + d * r * DENSITY;

  if (v < 1)
    v = 1;
  if (v > MAX_PAL)
    v = MAX_PAL;

  put_pixel (x, y, v);
}

/* split_box -- recursively draw the plasma, interpolating points */
void
split_box (int x1, int y1, int x2, int y2)
{
  unsigned char c1, c2, c3, c4;
  int diffx, diffy;
  int x, y;

  diffx = x2 - x1;
  diffy = y2 - y1;

  if ((diffx < 2) && (diffy < 2))
    return;

  x = (x1 + x2) / 2;
  y = (y1 + y2) / 2;

  adjust (x1, y1, x, y1, x2, y1);
  adjust (x2, y1, x2, y, x2, y2);
  adjust (x1, y2, x, y2, x2, y2);
  adjust (x1, y1, x1, y, x1, y2);

  c1 = get_pixel (x, y);

  if (!c1)
    {
      c1 = get_pixel (x1, y1);
      c2 = get_pixel (x2, y1);
      c3 = get_pixel (x1, y2);
      c4 = get_pixel (x2, y2);
      put_pixel (x, y, (c1 + c2 + c3 + c4) / 4);
    }

  split_box (x1, y1, x, y);
  split_box (x, y1, x2, y);
  split_box (x, y, x2, y2);
  split_box (x1, y, x, y2);
}

/* setup_pal -- create a palette with gradient color changes */
void
setup_pal (void)
{
  char *p = pal;
  int r, g, b;

  /* 0 - black */
  *p++ = *p++ = *p++ = 0;

  /* 1-64 dark red - greenish yellow */
  for (r = 31, g = 0, b = 0; g < 64; ++g)
    {
      *p++ = r;
      *p++ = g;
      *p++ = b;
    }

  /* 65-128 greenish yellow - blue */
  for (r = 31, g = 63, b = 0; b < 64; r -= b & 1, ++b, --g)
    {
      *p++ = r;
      *p++ = g;
      *p++ = b;
    }

  /* 129-192 blue - green */
  for (r = 0, g = 0, b = 63; b; ++g, --b)
    {
      *p++ = r;
      *p++ = g;
      *p++ = b;
    }

  /* 193 - 255 green - reddish yellow */
  for (r = 0, g = 63, b = 0; r < 63; ++r, g -= r & 1)
    {
      *p++ = r;
      *p++ = g;
      *p++ = b;
    }

  set_pal (pal);
}

void
errexit (const char *fmt,...)
{
  va_list v;
  char s[513];
  va_start (v, fmt);

  vsprintf (s, fmt, v);
  fprintf (stderr, "%s\n", s);
  exit_flat ();
  exit (1);
}

/* rotate_pal -- rotate the palette up */
void
rotate_pal (void)
{
  int r, g, b;
  r = pal[3];
  g = pal[4];
  b = pal[5];
  memcpy (pal + 3, pal + 6, 763);
  pal[765] = r;
  pal[766] = g;
  pal[767] = b;
  vsync ();
  set_pal (pal);
}

/* vsync -- wait for the vertical retrace */
void
vsync (void)
{
  while (!(inportb (0x3da) & 0x08))
    ;
  while (inportb (0x3da) & 0x08)
    ;
}

void
set_video_mode (int m)
{
  union REGS r;

  r.x.ax = m;
  int86 (0x10, &r, &r);
}

int
bget_key (void)
{
  union REGS r;

  r.h.ah = 0;
  return (int86 (0x16, &r, &r));
}

int
bkey_pressed (void)
{
  union REGS r;
  r.h.ah = 1;
  int86 (0x16, &r, &r);

  /* return (!zero_flag); */
  return !(r.x.flags & 0x40);
}

/* put_pixel -- write a pixel to the screen */
void
put_pixel (int x, int y, int c)
{
  pokeb32 (vmi.vidmem + (long) y * vmi.width + (long) x, c);
}

int
get_pixel (int x, int y)
{
  return (peekb32 (vmi.vidmem + (long) y * vmi.width + (long) x));
}

/* get_vesa_info -- try to read the VESA API version number
   Returns the versopn (low byte major, high byte minor number) or -1 on
   error (i.e. either no VESA API available or some other kind of error).
 */
int
get_vesa_info (void)
{
  union REGS r;
  struct SREGS s;
  char buf[512];

  r.x.ax = 0x4f00;
  r.x.di = FP_OFF (buf);
  s.es = FP_SEG (buf);
  int86x (0x10, &r, &r, &s);

  if (0x4f == r.h.al)
    {
      if (r.h.ah)
        return (-r.h.ah);
      return (*(int *) (buf + 4));
    }
  else
    return (-1);
}

/* get_vesa_mode_info -- read information about a VESA video mode */
int
get_vesa_mode_info (int mode, VESA_MODE_INFO * vmip)
{
  union REGS r;
  struct SREGS s;
  char buf[256];

  r.x.ax = 0x4f01;
  r.x.cx = mode;
  r.x.di = FP_OFF (buf);
  s.es = FP_SEG (buf);
  int86x (0x10, &r, &r, &s);

  if (0x4f == r.h.al)
    {
      if (r.h.ah)
        return (-r.h.ah);

      vmip->framebuf_avail = (*(int *) (buf) & 0x80) ? -1 : 0;
      vmip->width = *(int *) (buf + 0x12);
      vmip->height = *(int *) (buf + 0x14);
      vmip->vidmem = *(flatptr *) (buf + 0x28);

      return (0);
    }
  else
    return (-1);
}

int
set_vesa_mode (int mode)
{
  union REGS r;

  r.x.ax = 0x4f02;
  r.x.bx = mode;
  int86 (0x10, &r, &r);
  if (0x4f == r.h.al)
    return (r.h.ah);
  return (-1);
}

void
set_pal (char *p)
{
  int foo;

  outportb (0x3c8, 0);
  for (foo = 0; foo < 768; ++foo)
    outportb (0x3c9, *p++);
}
