/*
   Module:  inout.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides the input/output routines for the ADI
            converter.  It provides the routines needed by the driver
            to open input and output files, and read data.  It also
            provides the output funtion that can be used by the device
            dependant routines to send their output to the file opened
            by the driver.

   Functions Provided:

            openin
            openout
            closein
            closeout
            getstr
            putarr

   Functions Required:

            none

*/




#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <stdlib.h>
#include <malloc.h>
#include <io.h>
#include "retcodes.h"
#include "inout.h"


/*
   Function: openin
   Purpose:  Allocate the input buffer and open the input file for reading.

   Pre: name is a pointer to an ascii string containing the name of the 
        input file.
        pltfpp is a pointer to storage for a pointer to a PLTFILE.

   Post:  The input buffer is allocated.
          The input file is opened for reading.
          If the buffer can't be allocated, *pltfpp is set to NULL and
          NOBUFF is returned.
          If the input file can't be opened, *pltfpp is set to NULL and
          BADOPEN is returned.
          Otherwise, TRUE is returned.
*/

int openin (name,pltfpp)
   char *name;
   PLTFILE **pltfpp;
{
   /* first try to allocate a PLTFILE */
   *pltfpp = (PLTFILE *) malloc (sizeof (PLTFILE));
   if (*pltfpp == (PLTFILE *)NULL)
      return (NOBUFF);

   /* now try to allocate the buffer */
   (*pltfpp)->buf = (char *) malloc ((size_t)(BUFFSIZE*sizeof(char)));  
   if ((*pltfpp)->buf == (char *)NULL) {
      free ((void *)*pltfpp);
      *pltfpp = (PLTFILE *)NULL;
      return (NOBUFF);
   }

   /* now attempt to open the input file */
   (*pltfpp)->fd = open (name,O_RDONLY|O_TEXT);
   if ((*pltfpp)->fd < 0) {
      free ((void *)(*pltfpp)->buf);
      free ((void *)*pltfpp);
      *pltfpp = (PLTFILE *)NULL;
      return (BADOPEN);
   }

   /* now set buf_ctr to zero so the first read will cause the buffer to be 
      filled from the input file
   */
   (*pltfpp)->buf_ctr = 0;

   (*pltfpp)->buf_ptr = (*pltfpp)->buf;
   (*pltfpp)->eof=FALSE;
   return (TRUE);
}



/*
   Function: openout
   Purpose:  Allocate the output buffer, and open the output file for
             writing.

   Pre: name is a pointer to an ascii string containing the name of the
        output file.
        pltfpp is a pointer to storage for a pointer to a PLTFILE.

   Post:  The output buffer is allocated.
          The output file is opened for writing.
          If the output buffer can't be allocated, *pltfpp is set to NULL and
          NOBUFF is returned.
          If the output file can't be opened, *pltfpp is set to NULL and
          BADOPEN is returned.
          Otherwise, TRUE is returned.
*/

int openout (name,pltfpp)
   char *name;
   PLTFILE **pltfpp;
{
   /* first try to allocate a PLTFILE */
   *pltfpp = (PLTFILE *) malloc (sizeof (PLTFILE));
   if (*pltfpp == (PLTFILE *)NULL)
      return (NOBUFF);

   /* now try to allocate the buffer */
   (*pltfpp)->buf = (char *) malloc ((size_t) (BUFFSIZE*sizeof(char)));  
   if ((*pltfpp)->buf == (char *)NULL) {
      free ((void *)*pltfpp);
      *pltfpp = (PLTFILE *)NULL;
      return (NOBUFF);
   }

   /* now attempt to open the output file */
   (*pltfpp)->fd = open (name,O_CREAT|O_BINARY|O_WRONLY,S_IREAD|S_IWRITE);
   if ((*pltfpp)->fd < 0) {
      free ((void *)(*pltfpp)->buf);
      free ((void*)*pltfpp);
      *pltfpp = (PLTFILE *)NULL;
      return (BADOPEN);
   }
   
   /* now set buf_ptr to point to the beginning of the buffer so the first 
      write will go into the buffer.
   */
   (*pltfpp)->buf_ptr = (*pltfpp)->buf;

   (*pltfpp)->buf_ctr = 0;
   (*pltfpp)->eof = FALSE;
   return (TRUE);
}



/*
   Function: closein
   Purpose:  Discard the input buffer, close the input file, and deallocate 
             the input buffer.

   Pre:  pltfpp is a pointer to a pointer to a PLTFILE.
         If *pltfpp does not point to an open PLTFILE, it is NULL.

   Post:  If *pltfpp = NULL, TRUE is returned.
          Otherwise, the contents of the input buffer are discarded, the 
          input file is closed, and the input buffer is deallocated.
          If an error occurs during the above process, FALSE is returned.
*/

int closein(pltfpp)
   PLTFILE **pltfpp;
{
   if (*pltfpp == (PLTFILE *)NULL)
      return (TRUE);
   free ((void *)(*pltfpp)->buf);
   if (close((*pltfpp)->fd) < 0) {
       free ((void *)*pltfpp);
       return (FALSE);
   }
   free ((void *)*pltfpp);
   return (TRUE);
}



/*
   Function: closeout
   Purpose:  Flush the output buffer, close the output file, and deallocate 
             the output buffer.

   Pre:  pltfpp is a pointer to a pointer to a PLTFILE.
         If *pltfpp does not point to an open PLTFILE, it is NULL.

   Post:  If *pltfpp = NULL, TRUE is returned.
          Otherwise, the output buffer is flushed, the output file is closed, 
          and the output buffer is deallocated.
          If an error occurs during the above process, FALSE is returned.
*/

int closeout(pltfpp)
   PLTFILE **pltfpp;
{
   if (*pltfpp == (PLTFILE *)NULL)
      return (TRUE);
   if (write((*pltfpp)->fd,(*pltfpp)->buf,(unsigned)(*pltfpp)->buf_ctr) != 
       (*pltfpp)->buf_ctr) {
      free ((void *)(*pltfpp)->buf);
      (void) close ((*pltfpp)->fd);
      free ((void *)*pltfpp);
      return (FALSE);
   }
   free ((void *)(*pltfpp)->buf);
   if (close((*pltfpp)->fd) < 0) {
       free ((void *)*pltfpp);
       return (FALSE);
   }
   free ((void *)*pltfpp);
   return (TRUE);
}



/*
   Function: getstr
   Purpose:  Read a string from the input buffer, possibly refilling the
             buffer from the input file.  Data is read up to the next
             line feed character.

   Pre:  pltfp is a pointer to a PLTFILE that has been opened for reading.
         max is the maximum number of chars to be stored in string.
         string is a pointer to storage for the input string.
         There is enough storage available in string to hold "max"+1 chars.

   Post: An attempt is made to read a string from the input buffer and store
         it in "string".
         Reading stops at eof or the first line feed.  The line feed,
         although included in the byte count, is replaced with a NULL.
         If the input buffer is emptied during the process, the input file is 
         read to refill the buffer.
         If "max" chars are read without encountering a line feed, the extra
         chars are discarded.
         The number of bytes read from the input buffer is returned.
         If end_of_file has already been reached, 0 is returned.
         If an error occurs, -1 is returned.
*/

int getstr (pltfp, max, string)
   PLTFILE *pltfp;
   int max;
   char *string;
{
   int cnt=0, done=FALSE;

   if (pltfp->eof == TRUE)
      return (0);
   if (pltfp->buf_ptr >= (pltfp->buf + pltfp->buf_ctr)) {
      /* the buffer is empty, refill it from the input file */
      switch (pltfp->buf_ctr = read(pltfp->fd,pltfp->buf,
                                    (unsigned)BUFFSIZE)) {
         case 0:
            pltfp->eof = TRUE;
            return (0);
            break;
         case -1:
            return (-1);
            break;
         default:
            pltfp->buf_ptr = pltfp->buf;
            /* pltfp->buf_ctr was set by the read funtion */
            break;
      }
   }
   while (!done && cnt < max) {
      if (pltfp->buf_ptr >= (pltfp->buf + pltfp->buf_ctr)) {
         /* the buffer is empty, refill it from the input file */
         switch (pltfp->buf_ctr = read(pltfp->fd,pltfp->buf,
                                       (unsigned)BUFFSIZE)) {
            case 0:
               done = pltfp->eof = TRUE;
               break;
            case -1:
               return (-1);
               break;
            default:
               pltfp->buf_ptr = pltfp->buf;
               /* pltfp->buf_ctr was set by the read funtion */
               break;
         }
      }
      if (!pltfp->eof) {
         if ((*string = *(pltfp->buf_ptr)) == '\n') {
            *string = (char)NULL;
            cnt++;
            pltfp->buf_ptr++;
            done = TRUE;
         } else {
            string++;
            cnt++;
            pltfp->buf_ptr++;
         }
      }
   }
   if (!done) {
      /* we have read max characters without finding a line feed.  Read until
         line feed or eof, but don't store in string.
      */
      while (!done) {
         if (pltfp->buf_ptr >= (pltfp->buf + pltfp->buf_ctr)) {
            /* the buffer is empty, refill it from the input file */
            switch (pltfp->buf_ctr = read(pltfp->fd,pltfp->buf,
                                          (unsigned)BUFFSIZE)) {
               case 0:
                  done = pltfp->eof = TRUE;
                  break;
               case -1:
                  return (-1);
                  break;
               default:
                  pltfp->buf_ptr = pltfp->buf;
                  /* pltfp->buf_ctr was set by the read funtion */
                  break;
           }
         }
         if (!pltfp->eof) {
            if (*(pltfp->buf_ptr) == '\n')
               done = TRUE;
            cnt++;
            pltfp->buf_ptr++;
         }
      }
   }
   return (cnt);
}



/*
   Function: putarr
   Purpose:  Write a char array to the output buffer, possibly causing the 
             buffer to be written to the output file.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.
        num is the number of chars to be written to the output buffer.
        chrarr is a pointer to the char array to be written.

   Post: An attempt is made to copy num chars from chrarr to the output 
         buffer.
         If the buffer becomes full during the process, an attempt is made
         to write the buffer to the output file.
         If an error occurs, FALSE is returned, otherwise TRUE is returned.
*/

int putarr (pltfp, num, chrarr)
   PLTFILE *pltfp;
   int num;
   char *chrarr;
{
   int cnt=0;

   if (pltfp->buf_ptr >= pltfp->buf+BUFFSIZE) {
      if (write(pltfp->fd,pltfp->buf,(unsigned)pltfp->buf_ctr) != pltfp->buf_ctr)
         return (FALSE);
      pltfp->buf_ptr = pltfp->buf;
      pltfp->buf_ctr = 0;
   }
   while (cnt < num) {
      if (pltfp->buf_ptr >= pltfp->buf+BUFFSIZE) {
         if (write(pltfp->fd,pltfp->buf,(unsigned)pltfp->buf_ctr) != pltfp->buf_ctr)
            return (FALSE);
         pltfp->buf_ptr = pltfp->buf;
         pltfp->buf_ctr = 0;
      }
      *(pltfp->buf_ptr++) = *chrarr++;
      cnt++;
      pltfp->buf_ctr++;
   }
   return (TRUE);
}
