/****************************************************************

	gr_x.c		X Windows Implementation
			of Bywater Graphics Interface Standard

			Tested on DecStation 2100 and 3100

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

****************************************************************/

#include "stdio.h"
#ifdef __STDC__
#include "stdlib.h"
#endif

#include "X11/Xlib.h"
#include "X11/Xutil.h"
#include "X11/keysym.h"
#include "X11/keysymdef.h"
#include "X11/cursorfont.h"

#include "bw.h"
#include "kb.h"
#include "gr.h"

#define IMAGES                  64
#define KEY_NEEDED              FALSE
#define	MAX_SCREEN_HEIGHT	500
#define MAX_SCREEN_WIDTH        800
#define X_OFFSET                 30
#define Y_OFFSET                 30
#define REQ_BORDER_WIDTH          3
#define	EVENT_MASK              ( ButtonPressMask   | \
                                  ButtonReleaseMask | \
                                  KeyPressMask )
#define KEYBUF_LENGTH            64

/****************************************************************

   The following globals are defined in the gr specification 
   and are accessible to users of gr-based programs.  

****************************************************************/

int   gr_screens = 2;
int   gr_colors = 2;
int   gr_pxsize = 30;
int   gr_pysize = 30;
int   gr_ismouse = TRUE;
int   gr_blitting = TRUE;
int   gr_clipping = FALSE;		/* Not yet */
int   gr_saving = TRUE;

/****************************************************************

   The following globals are specific to the X Windows implementaion
   of the gr standard and should not be accessible to users of 
   gr-based programs.  

****************************************************************/

struct gr_window      *grwind;          /* primary gr window */
Display               *x_display;	/* primary X Windows display */
Colormap              x_colormap;	/* system color map */
XColor                rgbcolor, hardwarecolor;
int                   x_screen;		/* which screen is in use */
XSetWindowAttributes  x_windattributes; /* send requested attributes */
XSizeHints            x_sizehints;      /* send size "hints" */
unsigned long         x_windmask;       /* send window mask */
Window                x_primary;        /* the principal X Window, used 
					   when GR_PRIMARY is specified */
Pixmap                x_hidden;         /* a second X Window, used 
					   when GR_HIDDEN is specified */
Drawable              x_selected;       /* X Drawable variable, which may be set
					   to either x_primary or x_hidden */
int                   x_mousevent;      /* is mouse event pending? */
int                   x_keybevent;      /* is keyboard event pending? */
int                   x_moxpos;         /* mouse x position pending */
int                   x_moypos;         /* mouse y position pending */
int                   x_keypending;     /* key pending */
int                   x_depth;		/* depth (color planes ) */
XFontStruct           *x_font10; 	/* font structure: 10 y */
XFontStruct           *x_font13; 	/* font structure: 13 y */
XFontStruct           *x_font15; 	/* font structure: 15 y */
XFontStruct           *x_font20; 	/* font structure: 20 y */
XFontStruct           *x_font31; 	/* font structure: 31 y */
XFontStruct           *x_curfont;       /* font struct in use */
XGCValues             x_gcvals;         /* graphics content values */
GC                    x_fontgc;   	/* graphics context for font */
GC                    x_drawgc;   	/* graphics context for drawing */
Cursor                x_cursor;         /* cursor to use for this X Window */
Pixmap                x_stippm;         /* stipple pixmap */

char                  x_stipple[] = { 170, 85, 170, 85, 170, 85, 170, 85 };
unsigned long         x_colchart[ 16 ];	/* chart of sixteen colors */
extern unsigned long  x_color();	/* see function below */
Pixmap                x_images[ IMAGES ]; /* images */

/****************************************************************

   gr_init()

****************************************************************/

gr_init( window, path )
   struct gr_window *window;
   char *path;
   {
   int x_xsize, x_ysize;
   int req_xsize, req_ysize;
   register int i;
   
   /*	Part 1 of the Initialization:
   **
   **	Open and X Windows Display and Window and set system parameters
   **	based on it.
   */

   grwind = window;

   /* Open an X Windows display */ 

   x_display = XOpenDisplay( NULL );
   
   /* If the display is null, return */

   if ( x_display == NULL )
      {
      return FALSE;
      }

   /* get the screen currently in use and some basic data */

   x_screen = DefaultScreen( x_display );
   x_depth = DefaultDepth( x_display, x_screen );
   x_colormap = DefaultColormap( x_display, x_screen );
   x_xsize = DisplayWidth( x_display, x_screen );
   x_ysize = DisplayHeight( x_display, x_screen );
   
   /* determine requested height and width of X Window */

   if ( x_xsize >= ( MAX_SCREEN_WIDTH + X_OFFSET ))
      {
      req_xsize = MAX_SCREEN_WIDTH;
      }
   else
      {
      req_xsize = x_xsize;
      }

   if ( x_ysize >= ( MAX_SCREEN_HEIGHT + Y_OFFSET ))
      {
      req_ysize = MAX_SCREEN_HEIGHT;
      }
   else
      {
      req_ysize = x_ysize;
      }

   /*   Set a cursor for this application */

   x_cursor = XCreateFontCursor( x_display, XC_left_ptr );

   /* set requested window attributes in structure to be passed */

   x_windattributes.border_pixel =
      WhitePixel( x_display, x_screen );
   x_windattributes.background_pixel =
      BlackPixel( x_display, x_screen );
   x_windattributes.cursor = x_cursor;
   x_windmask = ( CWCursor | CWBackPixel | CWBorderPixel 
      | CWOverrideRedirect );

   /* now create the window and hidden pixmap */

   x_primary = XCreateWindow( x_display, RootWindow( x_display, x_screen ),
      X_OFFSET, Y_OFFSET, req_xsize, req_ysize, REQ_BORDER_WIDTH, x_depth,
      InputOutput, CopyFromParent, x_windmask, &x_windattributes );

   x_hidden = XCreatePixmap( x_display, x_primary,
      req_xsize, req_ysize, x_depth );

   x_selected = x_primary;

   /* Send size hints to the window manager */

   x_sizehints.flags  = USPosition | USSize;
   x_sizehints.x      = X_OFFSET;
   x_sizehints.y      = Y_OFFSET;
   x_sizehints.width  = req_xsize;
   x_sizehints.height = req_ysize;
   XSetNormalHints( x_display, x_primary, &x_sizehints );

   /* Display the X Window */

   XMapRaised( x_display, x_primary );

   /* "Flush" the display to send all output to the screen */

   XFlush( x_display );

   /* Establish a graphics content for drawing */

   x_drawgc = XCreateGC( x_display, x_primary, (unsigned long) 0,
      &x_gcvals );

   /* Set the stipple pattern for filling */

   x_stippm = XCreateBitmapFromData( x_display, x_primary, x_stipple, 8, 8 );
   XSetStipple( x_display, x_drawgc, x_stippm );

   /* Set global gr variables based on this window */
   
   window->xmax = req_xsize;
   window->ymax = req_ysize;

   /*	Part 2 of the Initialization:
   **
   **	Set a default font.
   */

   x_font10 = XLoadQueryFont( x_display, "6x10" );
   x_font13 = XLoadQueryFont( x_display, "8x13" );
   x_font15 = XLoadQueryFont( x_display, "9x15" );
   x_font20 = XLoadQueryFont( x_display, "vr-20" );
   x_font31 = XLoadQueryFont( x_display, "vr-31" );
   x_fontgc = XCreateGC( x_display, x_primary, (unsigned long) 0,
      &x_gcvals );
   XSetForeground( x_display, x_fontgc, WhitePixel( x_display, x_screen ));
   XSetBackground( x_display, x_fontgc, BlackPixel( x_display, x_screen ));
   XSetFont( x_display, x_fontgc, x_font13->fid );

   x_curfont = x_font13;
   grwind->fysize = x_font13->ascent + x_font13->descent;
   grwind->fxsize = XTextWidth( x_font13, "m", 1 );

   x_initcolor();		/* initialize color */

   /*   clear the screen */

   gr_cls( GR_PRIMARY );
   gr_cls( GR_HIDDEN );
   XFlush( x_display );

   /*	Part 3 of the Initialization:
   **
   **	Initialize X events so the keyboard and mouse imput can
   **	be processed. 
   */

   /* select input from key press, button press, and button release */   

   XSelectInput( x_display, x_primary, EVENT_MASK );

   /* set image buffer pointers to NULL */

   for ( i = 0; i < IMAGES; ++i )
      {
      x_images[ i ] = NULL;
      }

   /* start out at null state for keyboard and mouse entries */

   x_mousevent = FALSE;
   x_keybevent = FALSE;
   x_pollevent();

   window->initialized = grwind->initialized = TRUE;

#if	KEY_NEEDED
   kb_rx();
#endif
   return TRUE;
   }

/****************************************************************

   gr_deinit()

****************************************************************/

gr_deinit( screen )
   int screen;
   {

   /* remove the X Window */

   XDestroyWindow( x_display, x_primary );
   XFreePixmap( x_display, x_hidden );
   
   /* close the X Display */

   XCloseDisplay( x_display );

   }

/****************************************************************

   gr_cls()

****************************************************************/

gr_cls( screen )
   int screen;
   {
   if ( screen == GR_PRIMARY )
      {
      XClearWindow( x_display, x_primary );
      }
   else
      {
      gr_rectangle( GR_HIDDEN, 0, 0, grwind->xmax - 1, grwind->ymax - 1,
         BLACK, SOLID );
      }
   XFlush( x_display );
   }


/****************************************************************

   gr_pixel()

****************************************************************/

gr_pixel( screen, x, y, color )
   int screen;
   int x, y;
   int color;
   {

#ifdef  DEBUG
   if ( ( x < 0 ) || ( x > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_pixel(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_pixel(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   x_setscreen( screen );
   XSetForeground( x_display, x_drawgc, x_color( color ) );
   XDrawPoint( x_display, x_selected, x_drawgc, x, grwind->ymax - y );
   XFlush( x_display );
   }
         
/****************************************************************

   gr_line()

****************************************************************/

gr_line( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {

#ifdef  DEBUG
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   x_setscreen( screen );
   XSetLineAttributes( x_display, x_drawgc, (unsigned) 1, 
      x_linestyle( style ), 
      CapNotLast, JoinMiter );
   XSetForeground( x_display, x_drawgc, x_color( color ) );
   XDrawLine( x_display, x_selected, x_drawgc, x1, grwind->ymax - y1, 
      x2, grwind->ymax - y2 );
   XFlush( x_display );
   }

/****************************************************************

   gr_text()

****************************************************************/

gr_text( screen, x, y, string, foreground, background )
   int screen;
   int x, y;
   int foreground, background;
   char *string;
   {

#ifdef  DEBUG
   if ( ( x < 0 ) || ( x > grwind->xmax ))
      {
      fprintf( stderr, "ERROR: [pr:] gr_text(): x value is %d", x );
      kb_rx();
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > grwind->ymax ))
      {
      fprintf( stderr, "ERROR: [pr:] gr_text(): y value is %d", y );
      kb_rx();
      return BW_ERROR;
      }
#endif

   x_setscreen( screen );

   XSetForeground( x_display, x_fontgc, x_color( foreground ) );
   XSetBackground( x_display, x_fontgc, x_color( background ) );

   XDrawImageString( x_display, x_selected, x_fontgc, x, 
      grwind->ymax - ( y + x_curfont->descent ), string,
      strlen( string) );
   XFlush( x_display );
   }

/****************************************************************

   gr_strlen()

****************************************************************/

unsigned int
gr_strlen( string )
   char *string;
   {
   return XTextWidth( x_curfont, string, strlen( string ) );
   }

/****************************************************************

   gr_rectangle()

****************************************************************/

gr_rectangle( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {

#ifdef  DEBUG
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   x_setscreen( screen );

   if ( style == HOLLOW )
      {
      XSetLineAttributes( x_display, x_drawgc, (unsigned int) 1,
         LineSolid, CapNotLast, JoinMiter );
      XSetForeground( x_display, x_drawgc, x_color( color ) );
      XDrawRectangle( x_display, x_selected, x_drawgc, 
         x1, grwind->ymax - y2, x2 - x1, y2 - y1 );
      }
   else
      {
      XSetFillRule( x_display, x_drawgc, EvenOddRule );
      XSetFillStyle( x_display, x_drawgc, x_fillstyle( style ) );
      XSetForeground( x_display, x_drawgc, x_color( color ) );
      XSetBackground( x_display, x_drawgc, x_color( BLACK ) );
      XFillRectangle( x_display, x_selected, x_drawgc, 
         x1, grwind->ymax - y2, x2 - x1, y2 - y1 );
      }
   XFlush( x_display );
   }

/****************************************************************

   gr_circle()

****************************************************************/

gr_circle( screen, x, y, radius, color, style )
   int screen;
   int x, y, radius;
   int color, style;
   {

#ifdef  DEBUG
   if ( ( x < 0 ) || ( x > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_circle(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_circle(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   x_setscreen( screen );

   if ( style == HOLLOW )
      {
      XSetLineAttributes( x_display, x_drawgc, (unsigned int) 1,
         LineSolid, CapNotLast, JoinMiter );
      XSetForeground( x_display, x_drawgc, x_color( color ) );
      XDrawArc( x_display, x_selected, x_drawgc, 
         x - radius, ( grwind->ymax - y ) - radius, radius * 2, radius * 2,
         0, 360 * 64 );
      }
   else
      {
      XSetFillRule( x_display, x_drawgc, EvenOddRule );
      XSetFillStyle( x_display, x_drawgc, x_fillstyle( style ) );
      XSetForeground( x_display, x_drawgc, x_color( color ) );
      XSetBackground( x_display, x_drawgc, x_color( BLACK ) ); 
      XFillArc( x_display, x_selected, x_drawgc, 
         x - radius, ( grwind->ymax - y ) - radius, radius * 2, radius * 2,
         0, 360 * 64 );
      }
   XFlush( x_display );
   }

/****************************************************************

   gr_ellipse()

****************************************************************/

gr_ellipse( screen, x, y, x_radius, y_radius, mode, color, style )
   int screen;
   int x, y, x_radius, y_radius;
   int mode, color, style;
   {
   }
         
/****************************************************************

   gr_clip()

****************************************************************/

gr_clip( screen, mode, x1, y1, x2, y2 )
   int screen;
   int mode;
   int x1, y1, x2, y2;
   {

#ifdef  DEBUG
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   }

/****************************************************************

   gr_font()

****************************************************************/

gr_font( screen, type, rq_height )
   int screen;
   int type, rq_height;
   {
   XFontStruct *oldfont;

   oldfont = x_curfont;

   if ( rq_height > 25 )
      {
      x_curfont = x_font31;
      }
   else if ( rq_height > 17 )
      {
      x_curfont = x_font20;
      }
   else if ( rq_height > 14 )
      {
      x_curfont = x_font15;
      }
   else if ( rq_height > 11 )
      {
      x_curfont = x_font13;
      }
   else 
      {
      x_curfont = x_font10;
      }
      
   if ( ( x_curfont == NULL ) || ( x_curfont->fid == 0 ))
      {
      x_curfont = oldfont;
      }

   XSetFont( x_display, x_fontgc, x_curfont->fid );
   grwind->fysize = x_curfont->ascent + x_curfont->descent;
   grwind->fxsize = XTextWidth( x_curfont, "m", 1 );

   }

/****************************************************************

	gr_blit()

****************************************************************/

gr_blit( src, dst, x1, y1, x2, y2 )
   int src, dst;
   int x1, y1, x2, y2;
   {

#ifdef  DEBUG
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( src == dst )
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): src == dst" );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   if ( src == GR_HIDDEN )
      {
      XCopyArea( x_display, x_hidden, x_primary, x_drawgc,
         x1, grwind->ymax - y2, x2 - x1, y2 - y1, x1, grwind->ymax - y2 );
      }

   else
      {
      XCopyArea( x_display, x_primary, x_hidden, x_drawgc,
         x1, grwind->ymax - y2, x2 - x1, y2 - y1, x1, grwind->ymax - y2 );
      }

   }

/****************************************************************

   gr_imsave()

****************************************************************/

gr_imsave( screen, mode, x1, y1, x2, y2, image )
   int screen; 
   int mode, x1, y1, x2, y2;
   int *image;
   {
   register int r;
   int carry_on;

#ifdef  DEBUG
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   x_setscreen( screen );

   /* mode == TRUE, save the area */

   if ( mode == TRUE )
      {

      /* find an available slot */

      carry_on = TRUE;
      r = 0;
      while( ( carry_on == TRUE ) && ( r < IMAGES ) )
	 {
	 if ( x_images[ r ] == NULL )
	    {
	    carry_on = FALSE;
	    }
	 else
	    {
	    ++r;
	    }
	 }

      if ( r >= IMAGES )
	 {
	 bw_error( "No more slots for image storage" );
	 return FALSE;
	 }

      *image = r;

      /* create a new pixmap */

      x_images[ *image ] = XCreatePixmap( x_display, x_selected,
	 ( x2 - x1 ) + 1, ( y2 - y1 ) + 1, x_depth );

      /* copy the area */

      XCopyArea( x_display, x_selected, x_images[ *image ], x_drawgc,
         x1, grwind->ymax - y2, x2 - x1, y2 - y1, 0, 0 );
      }

   /* mode == FALSE, restore the area */

   else
      {
      XCopyArea( x_display, x_images[ *image ], x_selected, x_drawgc,
         0, 0, x2 - x1, y2 - y1, x1, grwind->ymax - y2 );
      }
   }

/****************************************************************

   gr_imfree()

****************************************************************/

gr_imfree( image )
   int image;
   {
   XFreePixmap( x_display, x_images[ image ] );
   x_images[ image ] = NULL;
   }

/****************************************************************

   gr_mouse()

****************************************************************/

gr_mouse( mode, x, y, buttons )
   int mode;
   int *x, *y;
   int *buttons;
   {

   switch ( mode )
      {
      case STATUS:
      case SAMPLE:
         if ( x_mousevent == TRUE )
            {
            *x = x_moxpos;
            *y = grwind->ymax - x_moypos;
            return TRUE;
            }
         x_pollevent();
         if ( x_mousevent == TRUE )
            {
            *x = x_moxpos;
            *y = grwind->ymax - x_moypos;
            return TRUE;
            }
         return FALSE;
         break;
      
      case WAIT:
         while( x_mousevent == FALSE )
            {
            x_pollevent();
            }
         x_mousevent = FALSE;
         *x = x_moxpos;
         *y = grwind->ymax - x_moypos;
         break;
         
      }

   }

/****************************************************************

   x_pollevent()

****************************************************************/

x_pollevent()
   {
   static XEvent x_event;
   static XComposeStatus x_cstatus;
   static KeySym x_keysym;
   static int x_keybufmaxlen = KEYBUF_LENGTH - 1;
   static char x_keybuf[ KEYBUF_LENGTH ];
   int length;
  
#ifdef OLD_DEBUG
   fprintf( stderr, "Polling...\n" );
#endif

   /* poll X Window to see if there is an event */

   if ( XCheckMaskEvent( x_display, (unsigned long) EVENT_MASK, &x_event ) 
      == True )
      {
      switch ( x_event.type )
         {
         case ButtonPress:
         case ButtonRelease:
#ifdef OLD_DEBUG
            fprintf( stderr, "Mouse event...\n" );
#endif
            x_mousevent = TRUE;
            x_moxpos    = x_event.xbutton.x;
            x_moypos    = x_event.xbutton.y;
            break;
         case KeyPress:
#ifdef OLD_DEBUG
            fprintf( stderr, "Keyboard event...\n" );
#endif
            x_keybevent  = TRUE;
            length = XLookupString( &x_event, x_keybuf, x_keybufmaxlen, 
               &x_keysym,
               &x_cstatus );
            if ( ( x_keysym >= ' ' ) && ( x_keysym <= '~' )
               && ( length > 0 ))
               {
               x_keypending = x_keysym;
               }
            else
               {
               switch( x_keysym )
                  {
                  case XK_Return:
                     x_keypending = '\r';
                     break;
                  case XK_BackSpace:
                     x_keypending = '\010';
                     break;
                  case XK_Escape:
                     x_keypending = '\033';
                     break;
                  case XK_Delete:
                     x_keypending = KB_DELETE;
                     break;
                  case XK_Up:
                     x_keypending = KB_UP;
                     break;
                  case XK_Down:
                     x_keypending = KB_DOWN;
                     break;
                  case XK_Right:
                     x_keypending = KB_RIGHT;
                     break;
                  case XK_Left:
                     x_keypending = KB_LEFT;
                     break;
                  case XK_F1:
                     x_keypending = KB_FK1;
                     break;
                  case XK_F2:
                     x_keypending = KB_FK2;
                     break;
                  case XK_F3:
                     x_keypending = KB_FK3;
                     break;
                  default:
                     x_keypending = 0;
                     break;
                  }
               }
            break;
         }      
      return TRUE;
      }

   return FALSE;
      
   }

/****************************************************************

   x_initcolor()

****************************************************************/

x_initcolor()
   {
   
   if ( x_depth <= 1 )
      {
      return FALSE;
      }

   gr_colors = 16;

   XLookupColor( x_display, x_colormap, "Black", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ BLACK ] = hardwarecolor.pixel;
   
   XLookupColor( x_display, x_colormap, "White", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ WHITE ] = hardwarecolor.pixel;
   
   XLookupColor( x_display, x_colormap, "Red", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ LIGHT_RED ] = hardwarecolor.pixel;
   x_colchart[ DARK_RED ] = hardwarecolor.pixel;
   
   XLookupColor( x_display, x_colormap, "Green", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ LIGHT_GREEN ] = hardwarecolor.pixel;
   x_colchart[ DARK_GREEN ] = hardwarecolor.pixel;

   XLookupColor( x_display, x_colormap, "Blue", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ LIGHT_BLUE ] = hardwarecolor.pixel;
   x_colchart[ DARK_BLUE ] = hardwarecolor.pixel;

   XLookupColor( x_display, x_colormap, "Yellow", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ LIGHT_YELLOW ] = hardwarecolor.pixel;
   x_colchart[ DARK_YELLOW ] = hardwarecolor.pixel;

   XLookupColor( x_display, x_colormap, "Cyan", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ LIGHT_CYAN ] = hardwarecolor.pixel;
   x_colchart[ DARK_CYAN ] = hardwarecolor.pixel;

   XLookupColor( x_display, x_colormap, "Magenta", &rgbcolor, &hardwarecolor );
   XAllocColor( x_display, x_colormap, &hardwarecolor );
   x_colchart[ LIGHT_MAGENTA ] = hardwarecolor.pixel;
   x_colchart[ DARK_MAGENTA ] = hardwarecolor.pixel;

   return TRUE;
   }

/****************************************************************

   x_color()

****************************************************************/

unsigned long
x_color( gr_color )
   int gr_color;
   {


   if ( x_depth <= 1 )
      {
      switch ( gr_color )
         {
         case WHITE:
            return WhitePixel( x_display, x_screen );
            break;
         case BLACK:
         default:
            return BlackPixel( x_display, x_screen );
            break;
         }
      }

   else
      {
      return x_colchart[ gr_color ];
      }
   }

/****************************************************************

   x_fillstyle()

****************************************************************/

x_fillstyle( gr_fillstyle )
   int gr_fillstyle;
   {
   switch( gr_fillstyle )
      {
      case GRID:
         return FillStippled;
         break;
      case HATCH:
         return FillTiled;
         break;
      case HOLLOW:
      case SOLID:
      default:
         return FillSolid;
         break;
      } 
   }

/****************************************************************

   x_linestyle()

****************************************************************/

x_linestyle( gr_linestyle )
   int gr_linestyle;
   {
   switch( gr_linestyle )
      {
      case GRID:
         return LineOnOffDash;
         break;
      case SOLID:
      case HOLLOW:
      default:
         return LineSolid;
         break;
      }
   }

/****************************************************************

   x_setscreen()

****************************************************************/

x_setscreen( screen )
   int screen;
   {
   switch( screen )
      {
      case GR_PRIMARY:
         x_selected = (Drawable) x_primary;
         break;
      case GR_HIDDEN:
         x_selected = (Drawable) x_hidden;
         break;
      default:
#ifdef DEBUG
         sprintf( bw_ebuf, "[pr:] x_screen received %d", screen );
         bw_error( bw_ebuf );
#endif
         return BW_ERROR;
         break;
      }
   return TRUE;
   }


