/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains functions for drawing text.

*/
#ifndef s_text_c
#define s_text_c

#include "s_text.h"

#include "s_common.h"
#include "s_string.h"
#include "s_math.h"
#include "s_gfx.h"

#include <stdarg.h>



/****************************************
****                                 ****
**** textout_right, textprintf_right ****
****                                 ****
*****************************************
The functios that Allegro missed. x is the right edge of the text instead of
the left edge. Returns the position of the beginning of the text.
*/
int textout_right(BITMAP *bmp,FONT *f,uchar *text,int x,int y,int color)
{
int length=text_length(font,text);
textout(bmp,f,text,x-length,y,color);
return x-length;
}
//This function was copied from Allegro. I added some more bytes to the
//buffer, to be a little safer.
int textprintf_right(BITMAP *bmp,FONT *f,int x,int y,int color,uchar *format,...)
{
char buf[1024];

va_list ap;
va_start(ap, format);
vsprintf(buf, format, ap);
va_end(ap);

return textout_right(bmp, f, buf, x, y, color);
}



/*******************
****            ****
**** textout_au ****
****            ****
********************
A textout function that interprets '&' as underline on the next character.
Allegro's function gui_textout doesn't work very good with my gui (for
example, it can only use Allegro's default gui font), so I had to do an own
function.
"au" means "ampersands underline".
*/
void textout_au_align(BITMAP *bmp,FONT *f,uchar *text,int x,int y,int color,int alignment)
{
int textlength=strlen(text),newtextlength=0;
uchar *newtext=xxmalloc(textlength+1);
int newtextwidth=0;
int underlinex=-1,underlinew=0;//Must initialise underlinew to avoid warning.
int i;
for(i=0;i<(textlength-1);i++){
	if(text[i]=='&'){
		if(text[i+1]=='&'){
			newtext[newtextlength++]='&';
			newtextwidth+=char_width(f,'&');
			i++;
		}else{
			underlinex=newtextwidth;
			underlinew=char_width(f,text[i+1]);
		}
	}else
		newtextwidth+=char_width(f,newtext[newtextlength++]=text[i]);
}
//Add last character.
newtext[newtextlength++]=text[i];
newtext[newtextlength]=0;
//Compensate for alignment.
if(alignment==al_center)
	x-=newtextwidth/2;
else if(alignment==al_right)
	x-=newtextwidth;
//Draw text.
textout(bmp,font,newtext,x,y,color);
free(newtext);
//Draw underline.
if(underlinex>=0)
	hline(bmp,x+underlinex,y+text_height(f)-gui_font_baseline,x+underlinex+underlinew-1,color);
}



/*******************
****            ****
**** char_width ****
****            ****
********************
Returns the width of one single character of a font. Allegro missed this
function; It only has one for strings.
*/
int char_width(FONT *f,int ascii)
{
if(!f)
	return 1;
if(f->height > 0)
	return f->height;
if(ascii < 32)
	return 0;
if((ascii-=32) >= FONT_SIZE)
	return 0;
return f->dat.dat_prop->dat[ascii]->w;
}
/*******************
****            ****
**** text_width ****
****            ****
********************
Returns the width of one single character of a font. Allegro missed this
function; It only has one for strings.
*/
int text_width(FONT *f,uchar *text)
{
if(!f)
	return 1;
return text_length(f,text);
}



/**********************
****               ****
**** charwrap_line ****
****               ****
***********************
Finds the last character that fits on a line of the specified pixelwidth in a
text editor.
*/
TLINEWIDTH *charwrap_line(uchar *text, long textstart, FONT *f, long pixelwidth)
{
TLINEWIDTH *ret=xxmalloc(sizeof(TLINEWIDTH));
ret->start=ret->end=textstart;
ret->pixelwidth=0;
ret->endofparagraph=ret->endofstring=FALSE;
for(; text[ret->end] && (text[ret->end]!=13) && (text[ret->end]!=10);
 ret->end++){
	ret->pixelwidth+=char_width(f,text[ret->end]);
	if(ret->pixelwidth>pixelwidth){
		ret->pixelwidth-=char_width(f,text[ret->end]);
		ret->nextlinestart=ret->end;
		return ret;
	}
}
ret->nextlinestart=ret->end+1;
if(text[ret->end])
	ret->endofparagraph=TRUE;
else
	ret->endofstring=TRUE;
return ret;
}



/**********************
****               ****
**** wordwrap_line ****
****               ****
***********************
Finds the number of characters that fits on a line of the specified
pixelwidth in a text editor. Word-wrapping is made if it is possible, which
means that whole words are kept together on the same line. The last character
of this substring is a non-space (>32). This means that a function that calls
this function must remove spaces in the beginning of lines manually.
*/
TLINEWIDTH *wordwrap_line(uchar *text, long textstart, FONT *f, long pixelwidth)
{
TLINEWIDTH *ret=charwrap_line(text,textstart,f,pixelwidth);
int lastpixelwidth=ret->pixelwidth;
int spacelength=char_width(f,' ');
if(ret->endofparagraph || ret->endofstring || (ret->end==ret->start))
	return ret;
//Find last word.
while((ret->end!=ret->start) && (text[ret->end]>32))
	ret->pixelwidth-=char_width(f,text[--(ret->end)]);
//If there were no spaces to break line at, we break in middle of the word.
if(ret->end==ret->start){
	ret->end=ret->nextlinestart;
	ret->pixelwidth=lastpixelwidth;
	return ret;
}
ret->nextlinestart=ret->end;
//Skip spaces at the end.
while((ret->end!=ret->start) && (text[ret->end-1]<=32)){
	ret->end--;
	ret->pixelwidth-=spacelength;
}
//Finding start of next line.
while(text[ret->nextlinestart]==' ')
	ret->nextlinestart++;
return ret;
}



/**********************
****               ****
**** textline_left ****
****               ****
***********************
Draws a line of text using left justification.
*/
void textline_left(BITMAP *bmp, FONT *f,uchar *text,
 TPOINT pos,TLINEWIDTH *line, int color)
{
draw_substr(bmp,f,text,pos,color,line->start,line->end-line->start);
}



/***********************
****                ****
**** textline_right ****
****                ****
************************
Draws a line of text using right justification.
*/
void textline_right(BITMAP *bmp, FONT *f,uchar *text,
 TPOINT pos,TLINEWIDTH *line, long spacetoadd, int color)
{
pos.x+=spacetoadd;
draw_substr(bmp,f,text,pos,color,line->start,line->end-line->start);
}



/************************
****                 ****
**** textline_center ****
****                 ****
*************************
Draws a line of text using left justification.
*/
void textline_center(BITMAP *bmp, FONT *f,uchar *text,
 TPOINT pos,TLINEWIDTH *line, long spacetoadd, int color)
{
pos.x+=d2(spacetoadd);
draw_substr(bmp,f,text,pos,color,line->start,line->end-line->start);
}



/*****************************
****                      ****
**** textline_charjustify ****
****                      ****
******************************
Draws a line of text using word justification.
Parameters:
	bmp, f      Bitmap and font to draw on and with.
	text        The text to print.
	pos         The position on the bitmap to draw on.
	line        Return value from wordwrap, containing data about length of
					line in characters and pixels.
	spacetoadd  Numbers of overflowing pixels on the line (those that should
					have been between the right edge of the text and the marginal
					if left justification would have been used).
	color       Foreground color of text, or -1 for font's original color.*/
void textline_charjustify(BITMAP *bmp, FONT *f,uchar *text,
 TPOINT pos,TLINEWIDTH *line, long spacetoadd, int color)
{
int i;
int numshort;//Number of characters one pixel shorter than the others.
int additionalwidth;
int firstchar;
int length=line->end-line->start;

if(line->endofstring||line->endofparagraph)
	textline_left(bmp,f,text,pos,line,color);
else{
	//Ignoring leading spaces. This makes it possible to have indentation in
	//the beginning of paragraphs that doesn't stretch.
	for(firstchar=line->start;text[firstchar]==' ';firstchar++)
		;//Do nothing
	pos.x+=char_width(f,' ')*(firstchar-line->start);
	numshort=length-spacetoadd%length;
	additionalwidth=spacetoadd/length;
	//Drawing text.
	for(i=firstchar;i<line->end;i++){
		charout(bmp,f,text[i],pos.x,pos.y,color);
		pos.x+=additionalwidth+char_width(f,text[i]);
		if(numshort){
			numshort--;
			if(!numshort)
				additionalwidth++;
		}
	}
}
}



/*****************************
****                      ****
**** textline_wordjustify ****
****                      ****
******************************
Draws a line of text using word justification.
Parameters:
	bmp, f      Bitmap and font to draw on and with.
	text        The text to print.
	pos         The position on the bitmap to draw on.
	line        Return value from wordwrap, containing data about length of
					line in characters and pixels.
	spacetoadd  Numbers of overflowing pixels on the line (those that should
					have been between the right edge of the text and the marginal
					if left justification would have been used).
	color       Foreground color of text, or -1 for font's original color.*/
void textline_wordjustify(BITMAP *bmp, FONT *f,uchar *text,
 TPOINT pos,TLINEWIDTH *line, long spacetoadd, int color)
{
int i;
int numspaces=0;
int numshort;//Number of spaces one pixel shorter than the others.
int spacewidth=char_width(f,' ');
int firstchar;
if(line->endofstring||line->endofparagraph)
	textline_left(bmp,f,text,pos,line,color);
else{
	//Ignoring leading spaces (Those will not be stretched.). This makes it
	//possible to have indentation in the beginning of paragraphs that doesn't
	//stretch.
	for(firstchar=line->start;text[firstchar]==' ';firstchar++)
		;//Do nothing
	pos.x+=spacewidth*(firstchar-line->start);
	//Counting spaces.
	for(i=firstchar;i<line->end;i++)
		if(text[i]==' ')
			numspaces++;
	if(numspaces){
		numshort=numspaces-spacetoadd%numspaces;
		spacewidth+=spacetoadd/numspaces;
		//Drawing text.
		for(i=firstchar;i<line->end;i++){
			if(text[i]==' '){
				pos.x+=spacewidth;
				if(numshort){
					numshort--;
					if(!numshort)
						spacewidth++;
				}
			}else{
				charout(bmp,f,text[i],pos.x,pos.y,color);
				pos.x+=char_width(f,text[i]);
			}
		}
	//If there were no spaces on the line, we draw left justified instead.
	}else
		textline_left(bmp,f,text,pos,line,color);
}
}



/*********************
****              ****
**** calc_textbox ****
****              ****
**********************
Calculates and returns a TTEXTBOX.*/
TTEXTBOX *calc_textbox(FONT *f,uchar *text,TRECT box,int xalignment,
 int yalignment)
{
int nummallocedlines;
TTEXTBOX *ret=xxmalloc(sizeof(TTEXTBOX));
long textstart=0;

ret->text=text;
ret->f=f;
ret->box=box;
ret->xalignment=xalignment;
ret->yalignment=yalignment;
ret->numlines=0;
ret->lines=xxmalloc(sizeof(TLINEWIDTH*)*(nummallocedlines=64));
ret->widestline=0;

//Calculating lines.
do{
	if(++ret->numlines>nummallocedlines)
		//Realloc 64 items at a time.
		ret->lines=xxrealloc(ret->lines,sizeof(TLINEWIDTH*)*(nummallocedlines+=64));
	ret->lines[ret->numlines-1]=wordwrap_line(ret->text,textstart,ret->f,
	 ret->box.w);
	textstart=ret->lines[ret->numlines-1]->nextlinestart;
	if(ret->lines[ret->numlines-1]->pixelwidth>ret->widestline)
		ret->widestline=ret->lines[ret->numlines-1]->pixelwidth;
}while(!ret->lines[ret->numlines-1]->endofstring);
//Removing unused lines at the end.
ret->lines=xxrealloc(ret->lines,sizeof(TLINEWIDTH*)*ret->numlines);
return ret;
}



/*********************
****              ****
**** draw_textbox ****
****              ****
**********************
Draws a TTEXTBOX.
*/
void draw_textbox(BITMAP *bmp,FONT *f,TTEXTBOX *tb,long startline,int color)
{
int lineheight=text_height(f),numshort=0;//Set to 0 to avoid uninitialized warning.
int height=0;                             //  -||-
int i;
TRECT cliprect=get_clip_asrect(bmp);
int clipmode=bmp->clip;//Store these so that we can reset afterwards.
TPOINT pos;
long marginw;

//Set clipping.
add_clip(bmp,tb->box);
//We may want to change margins.
switch(tb->xalignment){
	case al_wordjustifyleft:
	case al_charjustifyleft:
		marginw=tb->widestline;
		pos.x=tb->box.x;
	break;
	case al_wordjustifycenter:
	case al_charjustifycenter:
		marginw=tb->widestline;
		pos.x=tb->box.x+d2(tb->box.w-tb->widestline);
	break;
	case al_wordjustifyright:
	case al_charjustifyright:
		marginw=tb->widestline;
		pos.x=tb->box.x+tb->box.w-tb->widestline;
	break;
	default://Left,center,right,normal justify.
		marginw=tb->box.w;
		pos.x=tb->box.x;
	break;
}
//Decide where to start vertically.
switch(tb->yalignment){
	case al_top:
		pos.y=tb->box.y;
	break;
	case al_bottom:
		pos.y=tb->box.h-lineheight*tb->numlines;
	break;
	case al_center:
		pos.y=tb->box.y+d2(tb->box.h-lineheight*tb->numlines);
	break;
	case al_justify:
		//If only one line, we center it.
		if(tb->numlines==1)
			pos.y=tb->box.y+d2(tb->box.h-lineheight);
		else{
			pos.y=tb->box.y;
			height=lineheight*tb->numlines;
			if(height<tb->box.h){//Shouldn't shrink line distance.
				numshort=tb->numlines-(tb->box.h-lineheight)%(tb->numlines-1);
				lineheight=(tb->box.h-lineheight)/(tb->numlines-1);
			}
		}
	break;
}
//Drawing lines.
for(i=startline;(i < tb->numlines) && (pos.y < (tb->box.y + (slong)tb->box.h));
 i++){
	switch(tb->xalignment){
		case al_left:
			textline_left(bmp,f,tb->text,pos,tb->lines[i],color);
		break;
		case al_center:
			textline_center(bmp,f,tb->text,pos,tb->lines[i],tb->box.w-tb->lines[i]->pixelwidth,color);
		break;
		case al_right:
			textline_right(bmp,f,tb->text,pos,tb->lines[i],tb->box.w-tb->lines[i]->pixelwidth,color);
		break;
		case al_wordjustify:
		case al_wordjustifyleft:
		case al_wordjustifycenter:
		case al_wordjustifyright:
			textline_wordjustify(bmp,f,tb->text,pos,tb->lines[i],marginw-tb->lines[i]->pixelwidth,color);
		break;
		case al_charjustify:
		case al_charjustifyleft:
		case al_charjustifycenter:
		case al_charjustifyright:
			textline_charjustify(bmp,f,tb->text,pos,tb->lines[i],marginw-tb->lines[i]->pixelwidth,color);
		break;
	}
	switch(tb->yalignment){
		case al_top:
		case al_center:
		case al_bottom:
			pos.y+=lineheight;
		break;
		case al_justify:
			if(height<tb->box.h){
				pos.y+=lineheight;
				if(numshort){
					numshort--;
					if(!numshort)
						lineheight++;
				}
			//If not all lines fit, we top justify them instead.
			}else
				pos.y+=lineheight;
		break;
	}
}
bmp->clip=clipmode;
set_clip_fromrect(bmp,cliprect);
}



/****************
****         ****
**** textbox ****
****         ****
*****************
Prints text in a cool way on a bitmap.*/
void textbox(BITMAP *bmp,FONT *f,uchar *text,TRECT box,
 int xalignment,int yalignment,int color)
{
TTEXTBOX *tb=calc_textbox(f,text,box,xalignment,yalignment);
draw_textbox(bmp,f,tb,0,color);
free_textbox(tb);
}



/****************
****         ****
**** charout ****
****         ****
*****************
Draws a character on a bitmap. This is the same as Allegro's textout, but for
one character.
*/
void charout(BITMAP *bmp, FONT *f, uchar ascii, int x, int y, int color)
{
uchar tmpstring[2]={ascii,0};
textout(bmp,f,tmpstring,x,y,color);
}



/********************
****             ****
**** draw_substr ****
****             ****
*********************
Draws part of a string.
*/
void draw_substr(BITMAP *bmp, FONT *f, uchar *text, TPOINT pos, int color,
 int start, int length)
{
schar atend;
if(length<0)
	textout(bmp,f,text,pos.x,pos.y,color);
else if(length!=0){
	atend=(text+=start)[length];
	text[length]=0;
	textout(bmp,f,text,pos.x,pos.y,color);
	text[length]=atend;
}
}



/*********************
****              ****
**** free_textbox ****
****              ****
**********************
Frees a textbox structure.
*/
void free_textbox(TTEXTBOX *tb)
{
while(tb->numlines)
	free(tb->lines[--tb->numlines]);
free(tb->lines);
free(tb);
}

#endif
