/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains different functions for saving and loading to files.

*/
#ifndef s_fileio_c
#define s_fileio_c

#include "s_fileio.h"

#include "s_string.h"
#include "s_alloc.h"
#include "s_conv.h"

#include <allegro.h>



/*****************
****          ****
**** openread ****
****          ****
******************
Opens a file for reading. x function exits on error.*/
FILE *xopenread(const uchar *filename)
{
FILE *f;
if(!(f=fopen(filename,"rb")))
	exitfail(mergestrs_new("In function `xopenread()': `fopen()' failed (returned `NULL'). Couldn't open\n\r\"",
	 filename,"\" for reading.",NULL));
return f;
}
FILE *openread(const uchar *filename)
{
return fopen(filename,"rb");
}
/******************
****           ****
**** openwrite ****
****           ****
*******************
Opens a file for writing. x function exits on error.*/
FILE *xopenwrite(const uchar *filename)
{
FILE *f;
if(!(f=fopen(filename,"wb")))
	exitfail(mergestrs_new("In function `xopenwrite()': `fopen()' failed (returned `NULL'). Couldn't open\n\r\"",
	 filename,"\" for writing.",NULL));
return f;
}
FILE *openwrite(const uchar *filename)
{
return fopen(filename,"wb");
}





/***************
****        ****
**** xfread ****
****        ****
****************
The same as fread, but exiting on error. Also without that stupid size/num
system, only number of bytes is specified.*/
void xfread(void *buf,ulong length,FILE *f)
{
if(fread(buf,length,1,f)==-1)
	exitfail("In function `xfread()': `fread()' failed (returned `EOF'). Couldn't read from\n\r"
	"file.");
}
/********************
****             ****
**** xpack_fread ****
****             ****
*********************
The same as pack_read, but exiting on error.*/
void xpack_fread(void *buf,ulong length,PACKFILE *f)
{
if(pack_fread(buf,length,f)==-1)
	exitfail(mergestrs_new("In function `xpack_fread()': `pack_fread()' failed (returned `EOF'). Couldn't\n\r"
	"read from file \"", f->filename, "\".",NULL));
}





/****************
****         ****
**** savestr ****
****         ****
*****************
Saves a string in my super-format.*/
int savestr(FILE *f,uchar *text)
{
ulong length=strlen(text);
if(fwrite(&length,sizeof(ulong),1,f)==-1)
	return -1;
if(fwrite(text,sizeof(uchar),length,f)==-1)
	return -2;
return 0;
}
//Saves a string in my super-format, exiting on error.
void xsavestr(FILE *f,uchar *text)
{
switch(savestr(f,text)){
	case -1:
		exitfail("In function `xsavestr()': `fwrite()' in `savestr()' failed (returned `EOF').\n\r"
		 "Couldn't write string length to file.");
	break;
	case -2:
		exitfail("In function `xsavestr()': `fwrite()' in `savestr()' failed (returned `EOF').\n\r"
		 "Couldn't write string's characters to file.");
	break;
}
}
/****************
****         ****
**** loadstr ****
****         ****
*****************
Loads a c-string from my super-format.*/
uchar *loadstr_d(FILE *f,uchar *text)
{
ulong length;
int shallmalloc=!text;
if(fread(&length,sizeof(ulong),1,f)==-1)
	return NULL;
_ifstrxmalloc(text,length);
if(fread(text,sizeof(uchar),length,f)==-1){
	if(shallmalloc)
		free(text);
	return NULL;
}
return text;
}
//Loads a c-string in my super-format, exiting on error.
uchar *xloadstr_d(FILE *f,uchar *text)
{
ulong length;
int shallmalloc=!text;
if(fread(&length,sizeof(ulong),1,f)==-1)
	exitfail("In function `xloadstr()': `fread()' failed (returned `EOF'). Couldn't read\n\r"
	 "string length from file.");
_ifstrxmalloc(text,length);
if(fread(text,sizeof(uchar),length,f)==-1){
	if(shallmalloc)
		free(text);
	exitfail("In function `xloadstr()': `fread()' failed (returned `EOF'). Couldn't read\n\r"
	 "string's characters from file.");
}
return text;
}
/****************
****         ****
**** seekstr ****
****         ****
*****************
Skips a superstr in a file; doesn't load it into memory.*/
int seekstr(FILE *f)
{
ulong length;
if(fread(&length,sizeof(ulong),1,f)==-1)
	return -1;
if(fseek(f,length,SEEK_CUR))
	return -2;
return 0;
}
//Skips a superstr in a file; doesn't load it into memory. Exits on error.
void xseekstr(FILE *f)
{
switch(seekstr(f)){
	case -1:
		exitfail("In function `xseekstr()': `fread()' in `seekstr()' failed (returned `EOF').\n\r"
		 "Couldn't read string length from file.");
	break;
	case -2:
		exitfail("In function `xseekstr()': `fseek()' in `seekstr()' failed (returned nonzero).\n\r"
		 "Couldn't seek in file.");
	break;
}
}





/********************
****             ****
**** loadstrlist ****
****             ****
*********************
Saves a list of strings. Returns the 0 on success, and negative on error.
*/
int loadstrlist(FILE *f, TSTRLIST *sl)
{
int i;
int numstrings;
uchar *current;
if(fread(&numstrings,sizeof(int),1,f)==-1)
	return -1;
for(i=0;i<numstrings;i++){
	if(!(current=loadstr_new(f)))
		return -2-i;
	add_strlist_item(sl,current);
}
return 0;
}
void xloadstrlist(FILE *f, TSTRLIST *sl)
{
int ret=loadstrlist(f,sl);
if(ret==-1)
	exitfail("In function `xloadstrlist()': `fread()' in `loadstrlist()' failed (returned\n\r"
	 "`EOF'). Couldn't read number of strings from file.");
else if(ret<-1)
	exitfail(mergestrs_new(
	 "In function `xloadstrlist_new()': `loadstr_new()' in `loadstrlist()' failed\n\r"
	 "(returned `NULL'). Couldn't load string #", ul2str(-2-ret,NULL)," from file.",NULL));
}
/********************
****             ****
**** savestrlist ****
****             ****
*********************
Saves a list of strings. Returns the 0 on success, and -1 on error.
*/
int savestrlist(FILE *f, TSTRLIST *sl)
{
int i;
if(fwrite(&sl->num_strs,sizeof(int),1,f)==-1)
	return -1;
for(i=0;i<sl->num_strs;i++)
	if(savestr(f,sl->list[i]))
		return -2-i;
return 0;
}
void xsavestrlist(FILE *f, TSTRLIST *sl)
{
int ret=savestrlist(f,sl);
if(ret==-1)
	exitfail("In function `xsavestrlist()': `fwrite()' failed in `savestrlist()' (returned\n\r"
	 "`EOF'). Couldn't write number of strings to file.");
else if(ret<-1)
	exitfail(mergestrs_new("In function `xsavestrlist()': `savestr()' in `savestrlist()' failed (returned\n\r"
	"-1). Couldn't write string #", ul2str(-2-ret,NULL)," to file.",NULL));
}





/*********************
****              ****
**** pack_savestr ****
****              ****
**********************
Saves a c-string in my super-format.*/
int pack_savestr(PACKFILE *f,uchar *text)
{
ulong length=strlen(text);
if(pack_fwrite(&length,sizeof(ulong),f)==-1)
	return -1;
if(pack_fwrite(text,length,f)==-1)
	return -2;
return 0;
}
//Saves a c-string in my super-format, exiting on error.
void xpack_savestr(PACKFILE *f,uchar *text)
{
switch(pack_savestr(f,text)){
	case -1:
		exitfail(mergestrs_new("In function `xpack_savestr()': `pack_fwrite()' in `pack_savestr()' failed\n\r"
		 "(returned `EOF'). Couldn't write string length to file\n\r"
		 "\"",f->filename,"\".",NULL));
	break;
	case -2:
		exitfail(mergestrs_new("In function `xpack_savestr(): `pack_fwrite()' in `pack_savestr()' failed\n\r"
		 "(returned EOF). Couldn't write string characters to file\n\r"
		 "\"",f->filename,"\".",NULL));
	break;
}
}
/*********************
****              ****
**** pack_loadstr ****
****              ****
**********************
Saves a c-string in my super-format.*/
uchar *pack_loadstr_d(PACKFILE *f,uchar *text)
{
ulong length;
int shallmalloc=!text;
if(pack_fread(&length,sizeof(ulong),f)==-1)
	return NULL;
_ifstrxmalloc(text,length);
if(pack_fread(text,length,f)==-1){
	if(shallmalloc)
		free(text);
	return NULL;
}
return text;
}
//Loads a c-string in my super-format, exiting on error.
uchar *xpack_loadstr_d(PACKFILE *f,uchar *text)
{
ulong length;
int shallmalloc=!text;
if(pack_fread(&length,sizeof(ulong),f)==-1)
	exitfail(mergestrs_new("In function `xpack_loadstr()': `pack_fread()' failed (returned `EOF'). Couldn't\n\r"
	 "read string length from file \"",f->filename,"\".",NULL));
_ifstrxmalloc(text,length);
if(pack_fread(text,length,f)==-1){
	if(shallmalloc)
		free(text);
	exitfail(mergestrs_new("In function `xpack_loadstr()': `pack_fread()' failed (returned `EOF'). Couldn't\n\r"
	 "read string characters from file \"",f->filename,"\".",NULL));
}
return text;
}
/*********************
****              ****
**** pack_seekstr ****
****              ****
**********************
Skips a superstr in a file; doesn't load it into memory.*/
int pack_seekstr(PACKFILE *f)
{
ulong length;
if(pack_fread(&length,sizeof(ulong),f)==-1)
	return -1;
if(pack_fseek(f,length))
	return -2;
return 0;
}
//Skips a superstr in a file; doesn't load it into memory. Exit on error.
void xpack_seekstr(PACKFILE *f)
{
switch(pack_seekstr(f)){
	case -1:
		exitfail(mergestrs_new("In function `xpack_seekstr()': `pack_fread()' in `pack_seekstr()' failed\n\r"
		 "(returned `EOF'). Couldn't read string length from file\n\r"
		 "\"",f->filename,"\".",NULL));
	break;
	case -2:
		exitfail(mergestrs_new("In function `xpack_seekstr(): `pack_fseek()' in `pack_seekstr()' failed\n\r"
		 "(returned `EOF'). Couldn't seek in file\n\r"
		 "\"",f->filename,"\".",NULL));
	break;
}
}





/*************************
****                  ****
**** pack_loadstrlist ****
****                  ****
**************************
Loads `TSTRLIST' from file. The string list should already be created.
Returns -1 on error, and 0 on success.
*/
int pack_loadstrlist(PACKFILE *f, TSTRLIST *sl)
{
int i;
int numstrings;
uchar *current;
if(pack_fread(&numstrings,sizeof(int),f)==-1)
	return -1;
for(i=0;i<numstrings;i++){
	if(!(current=pack_loadstr_new(f)))
		return -2-i;
	add_strlist_item(sl,current);
}
return 0;
}
void xpack_loadstrlist(PACKFILE *f, TSTRLIST *sl)
{
int ret=pack_loadstrlist(f,sl);
if(ret==-1)
	exitfail(mergestrs_new(
	 "In function `xpack_loadstrlist_new()': `fread()' in `pack_loadstrlist()' failed\n\r"
	 "(returned `EOF'). Couldn't read number of strings from file\n\r"
	 "\"", f->filename, "\".", NULL));
else if(ret<-1)
	exitfail(mergestrs_new(
	 "In function `xpack_loadstrlist_new()': `pack_loadstr_new()' in\n\r"
	 "`pack_loadstrlist()'failed (returned `NULL'). Couldn't load string #", ul2str(-2-ret,NULL)," from\n\r"
	 "file \"", f->filename, "\".", NULL));
}
/*************************
****                  ****
**** pack_savestrlist ****
****                  ****
**************************
Saves a list of strings. Returns the 0 on success, and -1 on error.
Parameter `num' is number of string in the list.
*/
int pack_savestrlist(PACKFILE *f, TSTRLIST *sl)
{
int i;
if(pack_fwrite(&sl->num_strs,sizeof(int),f)==-1)
	return -1;
for(i=0;i<sl->num_strs;i++)
	if(pack_savestr(f,sl->list[i]))
		return -2-i;
return 0;
}
void xpack_savestrlist(PACKFILE *f, TSTRLIST *sl)
{
int ret=pack_savestrlist(f,sl);
if(ret==-1)
	exitfail(mergestrs_new("In function `xpack_savestrlist()': `fwrite()' in `pack_savestrlist()' failed\n\r"
	 "(returned `EOF'). Couldn't write number of strings to file\n\r"
	 "\"", f->filename, "\".", NULL));
else if(ret<-1)
	exitfail(mergestrs_new("In function `xpack_savestrlist()': `pack_savestr()' in `pack_savestrlist()'\n\r"
	 "failed (returned -1). Couldn't write string #", ul2str(-2-ret,NULL)," to file\n\r"
	 "\"", f->filename, "\".", NULL));
}







/*****************
****          ****
**** savecstr ****
****          ****
******************
Saves a null-terminated c-string to a file.*/
int savecstr(FILE *f,uchar *text)
{
return fwrite(text, sizeof(uchar), strlen(text)+1, f);
}
void xsavecstr(FILE *f,uchar *text)
{
if(savecstr(f,text))
	exitfail("In function `xsavecstr()': `fwrite()' in `savecstr()' failed (returned `EOF').\n\r"
	"Couldn't write string to file.");
}
/*****************
****          ****
**** loadcstr ****
****          ****
******************
Loads a null-terminated c-string from a file.*/
uchar *loadcstr_de(FILE *f,uchar *text)
{
int ascii=1;
ulong i;
for(i=0;ascii;i++){
	if((ascii=fgetc(f))==-1)
		return NULL;
	text[i]=ascii;
}
return text;
}
//Loads a null-terminated c-string from a file, exiting on error.*/
uchar *xloadcstr_de(FILE *f,uchar *text)
{
if(!loadcstr_de(f,text))
	exitfail("In function `xloadcstr()': `fgetc()' in `loadcstr()' failed (returned `EOF').\n\r"
	 "Couldn't read string from file.");
return text;
}
#define malloc_string_size    (64)
/*********************
****              ****
**** loadcstr_new ****
****              ****
**********************
Loads a c-string from a file, mallocing all memory needed.*/
uchar *loadcstr_new(FILE *f)
{
int ascii=1;
uchar *ret=xxmalloc(malloc_string_size);
ulong size=0;
ulong malloced_size=malloc_string_size;
//Mallocs 256 bytes at a time in order to save time. 256 could be any number;
//change it if you think another value will be faster.
while(1){
	for(;ascii&&(size<malloced_size);size++){
		if((ascii=fgetc(f))==-1){
			free(ret);
			return NULL;
		}
		ret[size]=ascii;
	}
	if(ascii==0)
		break;
	ret=xxrealloc(ret,malloced_size=(size+malloc_string_size));
}
ret=xxrealloc(ret,size);
return ret;
}
//Loads a c-string from a file, mallocing and exiting on error.
uchar *xloadcstr_new(FILE *f)
{
uchar *ret=loadcstr_new(f);
if(!ret)
	exitfail("In function `xloadcstr_new()': `fgetc()' in `loadcstr_new()' failed (returned\n\r"
	"`EOF'). Couldn't read string from file.");
return ret;
}





/**********************
****               ****
**** pack_savecstr ****
****               ****
***********************
Saves a null-terminated c-string to a packfile.*/
int pack_savecstr(PACKFILE *f,uchar *text)
{
return pack_fwrite(text, sizeof(uchar) * strlen(text)+1, f);
}
void xpack_savecstr(PACKFILE *f,uchar *text)
{
if(pack_fwrite(text, sizeof(uchar) * strlen(text)+1, f)==-1)
	exitfail(mergestrs_new("In function `xsavecstr()': `fwrite()' failed. Couldn't write string to file\n\r",
	 "\"",f->filename,"\".",NULL));
}
/**********************
****               ****
**** pack_loadcstr ****
****               ****
***********************
Loads a c-string from an allegro file.*/
uchar *pack_loadcstr(PACKFILE *f,uchar *text)
{
int ascii=1;
ulong i;
for(i=0;ascii;i++){
	if((ascii=pack_getc(f))==-1)
		return NULL;
	text[i]=ascii;
}
return text;
}
//Loads a c-string from an allegro file, exiting on error.
uchar *xpack_loadcstr(PACKFILE *f,uchar *text)
{
if(!pack_loadcstr(f,text))
	exitfail(mergestrs_new("In function `xpack_loadcstr()': `pack_getc()' in `pack_loadcstr()' failed\n\r"
	 "(returned `EOF'). Couldn't read from file\n\r"
	 "\"",f->filename,"\".",NULL));
return text;
}
/**************************
****                   ****
**** pack_loadcstr_new ****
****                   ****
***************************
Loads a c-string from an allegro file, mallocing and exiting on error.*/
uchar *pack_loadcstr_new(PACKFILE *f)
{
int ascii=1;
uchar *ret=xxmalloc(malloc_string_size);
ulong size=0;
ulong malloced_size=malloc_string_size;
//Mallocs 256 bytes at a time in order to save time. 256 could be any number;
//change it if you think another value will be faster.
while(1){
	for(;ascii&&(size<malloced_size);size++){
		if((ascii=pack_getc(f))==-1){
			free(ret);
			return NULL;
		}
		ret[size]=ascii;
	}
	if(ascii==0)
		break;
	ret=xxrealloc(ret,malloced_size=(size+malloc_string_size));
}
ret=xxrealloc(ret,size);
return ret;
}
#undef malloc_string_size
//Loads a c-string from an allegro file, mallocing and exiting on error.
uchar *xpack_loadcstr_new(PACKFILE *f)
{
uchar *ret=pack_loadcstr_new(f);
if(!ret)
	exitfail(mergestrs_new("In function `xpack_loadcstr_new()': `pack_getc()' in `pack_loadcstr_new()'"
	"failed (returned `EOF'). Couldn't read from file\n\r"
	"\"",f->filename,"\".",NULL));
return ret;
}





#endif
