/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains fading functions.

*/
#ifndef s_fade_c
#define s_fade_c

#include "s_fade.h"
#include "s_gfx.h"
//#include "s_random.h"
#include "s_timer.h"
#include "s_common.h"
#include <allegro.h>


/********************************
****                         ****
**** Simple palette routines ****
****                         ****
********************************/
//Fills a palette with the color.
void fill_palette(PALETTE pal,RGB color)
{
int i;
for(i=0;i<256;i++)
	pal[i]=color;
}

/******************************
****                       ****
**** Simple color routines ****
****                       ****
******************************/
//Returns an RGB struct containing the values in the parameters.
RGB v2rgb(char r,char g,char b)
{
RGB ret={r,g,b,0};
return ret;
}
//Returns a color that doesn't look like the parameter color.
RGB anothercolor(RGB color)
{
RGB ret={(color.r+32)&63,(color.g+32)&63,(color.b+32)&63};
return ret;
}
//Returns the color inverted.
RGB invertcolor(RGB color)
{
RGB ret={63-color.r,63-color.g,63-color.b};
return ret;
}
//Returns the grey color that has the same luminance as the parameter.
RGB greycolor(RGB color)
{
int value=63*luminancergb(color)/max_luminance;
return v2rgb(value,value,value);
}
//Converts from hsv to an RGB struct.
RGB hsv2rgb(float h,float s,float v)
{
int r,g,b;
hsv_to_rgb(h,s,v,&r,&g,&b);
return v2rgb(r>>2,g>>2,b>>2);
}
//Returns a randomized color.
RGB random_color(void)
{
return v2rgb(random()%64,random()%64,random()%64);
}
//Returns a color with random hue, but decided saturation and brightness.
RGB random_hue(float s,float v)
{
float h = (((double)(random() & (max_slong >> 12))) * 360.0) /
 (double)(max_slong >> 12);
return hsv2rgb(h,s,v);
}



/*********************
****              ****
**** luminancergb ****
****              ****
**********************
Returns luminance for an RGB struct. The same as the macro, but as a function.
*/
int luminancergb(RGB color)
{
return _luminancergb(color);
}
/*****************
****          ****
**** luminate ****
****          ****
******************
Returns the color faded so that it has the luminance newlum. Luminance ranges
from 0 to 6300, where 0 is black and 6300 is white.
*/
RGB luminate(RGB color,int newlum)
{
int colorlum=luminancergb(color);
if(newlum<colorlum){
	color.r=( ((int)color.r) * newlum ) / colorlum;
	color.g=( ((int)color.g) * newlum ) / colorlum;
	color.b=( ((int)color.b) * newlum ) / colorlum;
}else if(colorlum!=6300){
	color.r=( (int)color.r) + ((63-(int)color.r) * (newlum-colorlum) ) / (max_luminance-colorlum);
	color.g=( (int)color.g) + ((63-(int)color.g) * (newlum-colorlum) ) / (max_luminance-colorlum);
	color.b=( (int)color.b) + ((63-(int)color.b) * (newlum-colorlum) ) / (max_luminance-colorlum);
/* If colorlum is 6300, it would only cause a div by zero. If it is, we know
	the color is white, and the new color should therefore be a greyscale.
*/
}else{
	color.r=(newlum*63)/max_luminance;
	color.g=(newlum*63)/max_luminance;
	color.b=(newlum*63)/max_luminance;
}
return color;
}



/*********************
****              ****
**** change_color ****
****              ****
**********************
Returns a color on the way between two colors. `progress' ranges from 0 to
256, inclusive. If `progress' is 0, the `so' color will be returned. If
`progress' is 256, the `de' color will be returned, and between the result
will be interpolated.*/
RGB change_color(RGB so,RGB de,long progress)
{
de.r = ( progress * ((long)de.r) + (256-progress) * ((long)so.r) ) >> 8;
de.g = ( progress * ((long)de.g) + (256-progress) * ((long)so.g) ) >> 8;
de.b = ( progress * ((long)de.b) + (256-progress) * ((long)so.b) ) >> 8;
return de;
}
/*******************
****            ****
**** change_hue ****
****            ****
********************
This is rather much like the `change_color' function (above). The difference
is that this doesn't change the luminance. The return color will always have
the same luminance as the `so' color. For example, if `so' is light red and
`de' is black, we won't fade from light red towards black, but from light red
towards a lighter nuance of black (i.e. grey), that has the same luminance as
the light red color.
*/
RGB change_hue(RGB so,RGB de,long progress)
{
de=luminate(de,luminancergb(so));
de.r = ( progress*((long)de.r) + (256-progress)*((long)so.r) ) >> 8;
de.g = ( progress*((long)de.g) + (256-progress)*((long)so.g) ) >> 8;
de.b = ( progress*((long)de.b) + (256-progress)*((long)so.b) ) >> 8;
return de;
}




/********************************
****                         ****
**** make_nonlinear_progress ****
****                         ****
*********************************
Converts from linear progress to non-linear. This is rather complex and it's
maybe hard to understand what use it can have, but it is cool for fading.
Normally, when you fade, you have some kind of value telling how big part of
the fading is done. For example, you know that the fading will take 1000
milliseconds and you know that it was 241 milliseconds since start. Then the
progress is 241/1000, and all colors will be at 241/1000 of the way between
the source color and the destination color. When using this nonlinear
progress, however, a new progress is returned. That progress is not 241/1000
when input progress is 241/1000. If the parameter `displacement' is negative,
the progress will increase slow in the beginning and fast in the end, and if
it is positive it will increase fast in the beginning and slow in the end.
An easier way to understand this than reading my strange description is to
set the global variable `fade_curve' to -256 and do a slow outfading.

Parameters:

You should normally pass some integers to this function. The reason why the
data type is double is to protect from overflow and rounding errors in the
function.
	`displacement' How big part of [output max progress] that [output progress]
						is when [input progress] = [max input progress] / 2. This is
						measured in parts of 256.
	`maxi'         Max value for input progress.
	`maxo'         Max value for output progress.
	`x'            Current progress.

A proof of the function (for debugging purposes etc):

I want constant acceleration on the speed. This means that output speed as a
function of input speed is a second grade equation:

f(x) = ax^2 + bx + c

We know three points in the equation, and can therefore decide how it looks.
The first point is when input progress is 0. Then output progress should also
be 0. When input progress is max input progress, i.e. when the whole thing
is completed, the output progress should be max output progress. When input
progress is half way between 0 and max, the output progress should be at
(displacement+512)/1024 of the max output progress. (Since the value at half
the time can't be less than 1/4 of the max value or greater than 3/4 of the
max value).

f(0) = 0 <=>
 c = 0 <=>
 f(x) = ax^2 + bx
f(maxi) = maxo = a*maxi^2 + b*maxi
f(maxi/2) = maxo*(displacement+512)/1024 = a*maxi^2/4 + b*maxi/2
f(maxi)/2-f(maxi/2) = a*maxi^2/4 = maxo/2 - maxo*(displacement+512)/1024
a = (maxo/2 - maxo*(displacement+512/1024) / (maxi^2/4) =
  = (maxo*2 - maxo*(displacement+512)/256) / maxi^2 =
  = (maxo*2 - maxo*(displacement/256 + 2)) / maxi^2 =
  = (maxo * (2-(displacement/256 + 2))) / maxi^2 =
  = -(maxo * displacement/256) / maxi^2
f(maxi)= -(maxo * displacement/256) / maxi^2 )*maxi^2 + b*maxi =
		 = maxo
b = (maxo - -(maxo* displacement/256) / maxi^2)*maxi^2)/maxi =
  = (maxo + maxo*(displacement/256))/maxi =
  = maxo / maxi * (displacement/256 + 1)
f(x) = ax^2 + bx =
 = ( -(maxo * displacement/256) / maxi^2 ) * x^2 +
 + ( maxo / maxi * (displacement/256 + 1) ) * x

Testing the function:
f(0) = a*0^2 + b*0 = 0  -->  OK
f(maxi) = ( -(maxo * displacement/256) / maxi^2 ) * maxi^2 +
 + ( maxo / maxi * (displacement/256 + 1) ) * maxi =
 = -(maxo * displacement/256) + maxo * (displacement/256 + 1) =
 = maxo  -->  OK
f(maxi/2) = ( -(maxo * displacement/256) / maxi^2 ) * maxi^2 / 4 +
 + ( maxo / maxi * (displacement/256 + 1) ) * maxi / 2 =
 = ( -maxo * displacement/256 ) / 4 + (maxo * (displacement/256 + 1)) / 2 =
 = ( maxo * displacement / 512 + maxo / 2 - maxo * displacement / 1024 =
 = maxo / 2 + maxo * displacement / 1024 =
 = maxo * 512 / 1024 + maxo * displacement / 1024 =
 = maxo * (512 + displacement) / 1024  -->  OK


Optimizing the function:
f(x) = -(maxo * displacement/256) /maxi^2) * x*x +
 + (maxo /maxi * (displacement/256 + 1) ) *x =
 = (maxo * x) * -(displacement/256/(maxi*maxi)) * x +
 + (maxo * x) * (displacement/256 + 1)/maxi =
 = maxo * x * (-x*displacement/256/(maxi*maxi) + (displacement/256 + 1)/maxi =
 = maxo * x / maxi * (-x*displacement/256/maxi + displacement/256 + 1) =
 = maxo * x / maxi * ((displacement / 256) * (-x/maxi + 1) + 1)

Testing the optimized function:
f(0) = maxo * 0 / maxi * ... = 0  -->  OK
f(maxi) = maxo * maxi / maxi * ((displacement / 256) * (-maxi/maxi + 1) + 1) =
 = maxo * ((displacement / 256) * 0 + 1) =
 = maxo  -->  OK
f(maxi/2) = maxo* maxi/2 / maxi * ((displacement/256) * (-maxi/2/maxi + 1)+1)=
 = maxo/2 * ((displacement/256) * (1/2) + 1) =
 = maxo/2 * (displacement/512 + 1) =
 = maxo * (displacement / 1024 + 1/2) =
 = maxo * (displacement + 512) / 1024  -->  OK

*/
int make_nonlinear_progress(double x, double maxi, double maxo,
 double displacement)
{
//Avoid div by zero.
if(maxi == 0)
	return maxo;
//maxi is only used as a denominator and could therefore be inverted to save
//time (* is much faster than /).
maxi=1.0/maxi;
return maxo * x * maxi *
 ((displacement / 256) * (1 - x * maxi) + 1);
}
/**************************
****                   ****
**** make_lum_progress ****
****                   ****
***************************
Calculates a progress depending on the luminance of a color. This means that
dark colors (or light, depending on the `amount' parameter) will become black
faster when fading out. This is useful when fading out, because it gives a
special effect (that I can't describe very good). You should try this by
setting the global variable `fade_lum' to -256 and do an outfading. Using a
negative number is preferable when you fade towards darker colors and
positive are better when you fade towards lighter colors. If you do the
opposite, the image will appear inverted at the end of fading.

Parameters:
You should normally pass some integers to this function. The reason why the
data type is double is to protect from overflow and rounding errors in the
function.
	`amount'    Could be any value, but the normal should be something between
					-256 and 256. -256 means that dark colors reach max progress
					before light colors. 0 means that all colors reach max
					progress at the same time. 256 means that light colors reach
					max progress before dark colors.
	`maxi'      The maximum for input progress.
	`maxo'      The maximum for output progress.
	`lum'       Luminance of the color, ranging from 0 to `max_progress'.
	`progress'  The current progress, 0..`maxi'.

Return value:
	The new progress.

Dark colors (or light if amount<0) should faster reach max progress. This is
made so that their max progress is a bit higher than it should have been else.
If, for example, their max progress is the double value of what it should have
been otherways, the progress will reach max after half the time. When the
progress is calculated to be more than max progress, it is set to max
progress instead.

f(L) = maxprogress as a function of the luminance.
if(amount>0)
	f(max_luminance) = maxo
	f(0) = maxo * (1 + amount/256)
if(amount<0)
	Light colors should faster reach max progress.
	f(0) = maxo
	f(max_luminance) = maxo * (1 + abs(amount/256)) = maxo * (1 - amount/256)

f(L) could be a first grade equation, since we know two points of it.

if(amount>0)
	f(L) = k*L + m
	f(max_luminance) = maxo = k*max_luminance + m
	f(0) = maxo * (1 + amount/256) = k*0 + m
	m = maxo * (1 + amount/256)
	k = (maxo - maxo * (1 + amount/256)) / max_luminance =
	  = maxo * (1 - 1 - amount/256) / max_luminance =
	  = -maxo * amount / (256 * max_luminance)
	f(L) = -maxo * L * amount / (256*max_luminance) + maxo * (1 + amount/256) =
	  = (maxo*(1 + amount/256)*256*max_luminance - maxo*L*amount) /
	  / (256*max_luminance) =
	  = (maxo*(256 + amount)*max_luminance - maxo*L*amount) /
	  / (256*max_luminance) =
	  = maxo * (256*max_luminance + amount*max_luminance - L*amount) /
	  / (256*max_luminance) =
	  = maxo * (256*max_luminance+amount*(max_luminance-L)) /
	  / (256*max_luminance)
if(amount<0)
	f(L) = k*L + m
	f(0) = maxo = k*0 + m = m
	f(max_luminance) = maxo * (1 - amount/256) = k * max_luminance + maxo
	k = (maxo * (1 - amount/256) - maxo) / max_luminance =
	  = (maxo * ((1 - amount/256) - 1) / max_luminance =
	  = -maxo * amount / (256*max_luminance)
	f(L) = -maxo * amount * L / (256*max_luminance) + maxo =
	  = (-maxo * amount * L + maxo * 256*max_luminance) / (256*max_luminance)=
	  = maxo * (256*maxo_progress - amount * L) / (256*max_luminance)

g(x) = progress as a function of the linear progress
g(x) = x/maxi * f(L)
if(amount>0)
	g(x) = x * maxo * (256*max_luminance+amount*(max_luminance-L)) /
	  / (256*max_luminance*maxi)
if(amount<0)
	g(x) = x * maxo * (256*max_luminance - amount*L) /
	  / (256*max_luminance*maxi)

if(amount==0)
	g(x) = x/maxi * maxo

*/
int make_lum_progress(double progress, double maxi, double maxo,
 double lum, double amount)
{
int ret;
//Avoid div by zero.
if(maxi == 0)
	return maxo;
if(amount==0)
	ret=progress*maxo / maxi;
else if(amount>0)
	ret=progress * maxo * (256*max_luminance + amount*(max_luminance-lum)) /
	 (256 * max_luminance * maxi);
else
	ret=progress * maxo * (256*max_luminance - amount*lum) /
	 (256 * max_luminance * maxi);
if(ret>maxo)
	ret=maxo;
return ret;
}



//These variables are used internally when fading.
static PALETTE fade_pal_so,fade_pal_de,fade_pal_now;
static RGB fade_col;
static int fade_milliseconds;
static int fade_index_first,fade_index_last;
static int fade_order_inverted;

//These are external and alter the speed and some relating things of fading.
int fade_curviness=-128, fade_lum=0, fade_amount=256, fade_via_amount=128;

#define init_fade_variables()          \
	fade_milliseconds = milliseconds;   \
	fade_index_first = fromindex;       \
	fade_index_last = toindex;          \
	fade_order_inverted = FALSE

/**********************
****               ****
**** Normal fading ****
****               ****
**********************/
//Callback for normal fading.
void fade_callback(int progress)
{
int i;
if(fade_order_inverted)
	progress=fade_milliseconds-progress;
progress=make_nonlinear_progress(
 progress*fade_amount, fade_milliseconds*256, 32768, fade_curviness);
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_now[i]=change_color(fade_pal_so[i],fade_pal_de[i],
	 make_lum_progress(progress,32768,256,
	 luminancergb(fade_pal_so[i]),fade_lum));
set_palette(fade_pal_now);
}

//Fades out to black.
void fadeout_range(int milliseconds, int fromindex, int toindex)
{
init_fade_variables();
get_palette(fade_pal_so);
fill_palette(fade_pal_de,black);
callback_time(fade_callback,milliseconds);
}
//Fades in from black.
void fadein_range(int milliseconds, int fromindex, int toindex, PALETTE pal)
{
init_fade_variables();
fill_palette(fade_pal_de,black);
palcopy(pal,fade_pal_so);
fade_order_inverted=TRUE;
callback_time(fade_callback,milliseconds);
}
//Fades to `pal' from current palette.
void fadeto_range(int milliseconds, int fromindex, int toindex, PALETTE pal)
{
init_fade_variables();
get_palette(fade_pal_so);
palcopy(pal,fade_pal_de);
callback_time(fade_callback,milliseconds);
}
//Fades between two palettes.
void fadebetween_range(int milliseconds, int fromindex, int toindex, PALETTE so, PALETTE de)
{
init_fade_variables();
palcopy(so,fade_pal_so);
palcopy(de,fade_pal_de);
callback_time(fade_callback,milliseconds);
}

//Fades from current palette to `color'.
void fadeto_color_range(int milliseconds, int fromindex, int toindex, RGB color)
{
init_fade_variables();
get_palette(fade_pal_so);
fill_palette(fade_pal_de,color);
callback_time(fade_callback,milliseconds);
}
//Fades from current palette to `color', without changing hue.
void fadeto_hue_range(int milliseconds, int fromindex, int toindex, RGB color)
{
int i;
init_fade_variables();
get_palette(fade_pal_so);
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_de[i]=luminate(color,luminancergb(fade_pal_so[i]));
callback_time(fade_callback,milliseconds);
}



/*************************
****                  ****
**** Via color fading ****
****                  ****
*************************/
//Callback function for fading via a color.
void fade_viacolor_callback(int progress)
{
int i;
int prog,viaprog;
if(fade_order_inverted)
	progress=fade_milliseconds-progress;
progress=make_nonlinear_progress(
 progress*fade_amount, fade_milliseconds*256, 32768, fade_curviness);
for(i=fade_index_first;i<=fade_index_last;i++){
	prog=make_lum_progress(
	 progress, 32768, 256, luminancergb(fade_pal_so[i]), fade_lum);
	viaprog=prog*fade_via_amount/256;
	fade_pal_now[i]=change_color(
	 change_color(fade_pal_so[i],fade_col,viaprog),//Mix of so and via cols.
	 fade_pal_de[i],                                //Dest color.
	 prog);
}
set_palette(fade_pal_now);
}

//Fades out to black via `color'.
void fadeout_viacolor_range(int milliseconds, int fromindex, int toindex, RGB color)
{
init_fade_variables();
get_palette(fade_pal_so);
fill_palette(fade_pal_de,black);
fade_col=color;
callback_time(fade_viacolor_callback,milliseconds);
}
//Fades in from black via `color'.
void fadein_viacolor_range(int milliseconds, int fromindex, int toindex, PALETTE pal, RGB color)
{
init_fade_variables();
fill_palette(fade_pal_de,black);
palcopy(pal,fade_pal_so);
fade_col=color;
fade_order_inverted=TRUE;
callback_time(fade_viacolor_callback,milliseconds);
}
//Fades from current palette to `pal' via `color'.
void fadeto_viacolor_range(int milliseconds, int fromindex, int toindex, PALETTE pal, RGB color)
{
init_fade_variables();
get_palette(fade_pal_so);
palcopy(pal,fade_pal_de);
fade_col=color;
callback_time(fade_viacolor_callback,milliseconds);
}
//Fades from `so' to `de' via `color'.
void fadebetween_viacolor_range(int milliseconds, int fromindex, int toindex, PALETTE so, PALETTE de, RGB color)
{
init_fade_variables();
palcopy(so,fade_pal_so);
palcopy(de,fade_pal_de);
fade_col=color;
callback_time(fade_viacolor_callback,milliseconds);
}




/***********************
****                ****
**** Via hue fading ****
****                ****
***********************/
//Callback function for fading via a hue.
void fade_viahue_callback(int progress)
{
int i;
int prog,viaprog;
if(fade_order_inverted)
	progress=fade_milliseconds-progress;
progress=make_nonlinear_progress(
 progress*fade_amount, fade_milliseconds*256, 32768, fade_curviness);
for(i=fade_index_first;i<=fade_index_last;i++){
	prog=make_lum_progress(
	 progress, 32768, 256, luminancergb(fade_pal_so[i]), fade_lum);
	viaprog=prog*fade_via_amount/256;
	fade_pal_now[i]=change_color(
	 change_hue(fade_pal_so[i],fade_col,viaprog),//Mix of so and via cols.
	 fade_pal_de[i],                             //Dest color.
	 prog);
}
set_palette(fade_pal_now);
}

//Fades out to black via the hue of `color'.
void fadeout_viahue_range(int milliseconds, int fromindex, int toindex, RGB color)
{
init_fade_variables();
get_palette(fade_pal_so);
fill_palette(fade_pal_de,black);
fade_col=color;
callback_time(fade_viahue_callback,milliseconds);
}
//Fades in from black via the hue of `color'.
void fadein_viahue_range(int milliseconds, int fromindex, int toindex, PALETTE pal, RGB color)
{
init_fade_variables();
fill_palette(fade_pal_de,black);
palcopy(pal,fade_pal_so);
fade_col=color;
fade_order_inverted=TRUE;
callback_time(fade_viahue_callback,milliseconds);
}
//Fades from current palette via the hue of `color' to `pal'.
void fadeto_viahue_range(int milliseconds, int fromindex, int toindex, PALETTE pal, RGB color)
{
init_fade_variables();
get_palette(fade_pal_so);
palcopy(pal,fade_pal_de);
fade_col=color;
callback_time(fade_viahue_callback,milliseconds);
}
//Fades from `so' to `de' via the hue of `color'.
void fadebetween_viahue_range(int milliseconds, int fromindex, int toindex, PALETTE so, PALETTE de, RGB color)
{
init_fade_variables();
palcopy(so,fade_pal_so);
palcopy(de,fade_pal_de);
fade_col=color;
callback_time(fade_viahue_callback,milliseconds);
}

#undef init_fade_variables






/************************
****                 ****
**** Special fadings ****
****                 ****
************************/




/*************************
****                  ****
**** fadein_millenium ****
****                  ****
**************************
Does a cool infading like in the TV-serie Millenium.
*/
//Fades from a white palette to an inverted grey-scale palette.
static void __UNUSED__ fade_white2invertedgrey_callback(int progress)
{
int i;
progress=(256*progress)/fade_milliseconds;
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_de[i]=change_color(white,invertcolor(greycolor(fade_pal_so[i])),progress);
set_palette(fade_pal_de);
}
//Fades from a white palette to an inverted palette.
static void __UNUSED__ fade_white2inverted_callback(int progress)
{
int i;
progress=(256*progress)/fade_milliseconds;
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_de[i]=change_color(white,invertcolor(fade_pal_so[i]),progress);
set_palette(fade_pal_de);
}
//Fades from an inverted grey-scale palette to the palette.
static void __UNUSED__ fadein_invertedgrey_callback(int progress)
{
int i;
progress=(256*progress)/fade_milliseconds;
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_de[i]=change_color(invertcolor(greycolor(fade_pal_so[i])),fade_pal_so[i],progress);
set_palette(fade_pal_de);
}
//Fades from an inverted palette to the palette.
static void __UNUSED__ fadein_inverted_callback(int progress)
{
int i;
progress=(256*progress)/fade_milliseconds;
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_de[i]=change_color(invertcolor(fade_pal_so[i]),fade_pal_so[i],progress);
set_palette(fade_pal_de);
}
//Fades a white palette to the palette, using a millenium-style.
static void fadein_millenium_callback(int progress)
{
int i;
progress=(256*progress)/fade_milliseconds;
for (i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_de[i]=change_color(change_color(white,invertcolor(greycolor(fade_pal_so[i])),progress),fade_pal_so[i],progress);
set_palette(fade_pal_de);
}
//The real function fadein_millenium.
void fadein_millenium(PALETTE pal,int milliseconds)
{
int i;
for(i=fade_index_first;i<=fade_index_last;i++)
	fade_pal_so[i]=pal[i];
fade_milliseconds=milliseconds/2;
//callback_time(fade_white2invertedgrey_callback,fade_milliseconds);
//callback_time(fadein_invertedgrey_callback,fade_milliseconds);
//callback_time(fade_white2inverted_callback,fade_milliseconds);
//callback_time(fadein_inverted_callback,fade_milliseconds);
fade_milliseconds=milliseconds;
callback_time(fadein_millenium_callback,fade_milliseconds);
}




/************************
****                 ****
**** set_fade_method ****
****                 ****
*************************
Sets the current fading method. Fading method is a rather interesting thing:
If you set it to `fade_linear', fading is made in a normal way. If you set it
to `fade_slowfirst', the fading will start slowly (as if the fading time
would have been longer) and then accellerate so that it is much faster at the
end. `fade_fastfirst' will make fading faster than normally in the beginning,
but it slows down so that it is slower at the end. The most usual way of
fading is the linear way. I've never seen anything else except for this file.
However, there's a point in making fading slow at the beginning: The dark
colors are often so dark that you can't see the difference between them
some time before they are zero. This means there is a short while in the
fading when it seems like the computer is just waiting, since you can't see
that the colors are changing. By fading slow in the beginning and fast at the
end, this time is made as small as possible. The `fade_fastfirst' is not
really as useful as `fade_slowfirst', I think, but I put it there to be
consequent. Also, you may find it more useful than I do, depending of how you
like fadings to be.
*/


#endif
