/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains functions for converting between text and numbers. (More
flexible than the standard functions.)

*/
#ifndef s_conv_c
#define s_conv_c

#include "s_conv.h"

//#include "s_common.h"
#include "s_defs.h"
//#include "s_string.h"
#include "s_math.h"
#include "s_char.h"
#include "s_alloc.h"

#include <string.h>
#include <math.h>
#include <stdlib.h>



uchar char14=0xBC, char12=0xBD, char34=0xBE,
 minussign='-', zerosign=0, plussign=0, decimalsep='.', thousandsep=' ';

uchar *exppositivestring="e+", *expnegativestring="e-";

int hexuppercase=TRUE;

int hex_sep_distance=2, dec_sep_distance=3, oct_sep_distance=2,
 bin_sep_distance=8;



/******************
****           ****
**** void2text ****
****           ****
*******************
Converts from binary bytes into a string containing only uppercase characters.
(The first 16, i.e. 'A'..'P'.) The string is exactly the double size of the
original data. It can be converted back by using the text2void function.
*/
uchar *void2text_d(void *so,uchar *de,int length)
{
int i,dei=0;
_ifstrxmalloc(de,m2(length));
for(i=0;i<length;i++){
	de[dei++]='A'+(((uchar*)so)[i] & 0xF);
	de[dei++]='A'+((((uchar*)so)[i] >> 4) & 0xF);
}
return de;
}
/******************
****           ****
**** text2void ****
****           ****
*******************
Converts from the type in void2text back to binary data.
*/
uchar *text2void_d(uchar *so,void *de,int length)
{
int i,soi=0;
_ifxmalloc(de,length);
for(i=0;i<length;i++){
	((uchar*)de)[i] = so[soi]-'A' + ((so[soi+1]-'A') << 4);
	soi+=2;
}
return de;
}



/***********************
****                ****
**** l2str_extended ****
****                ****
************************
	Returns a string containing a long.
	This is an extremly flexible function.
	Parameters:
	===========
	 number              The number.
	 minlength           The minimum number of digits returned.
								If the number has less digits than minlength, it will
								be padded with zeros to the left. Note that if
								minlength is zero, 0 is converted to a zero-length
								string. If you want it to look like "0", set
								minlength to 1.
								Examples:
								"32 767" would be "000 032 767" if you specified 9 as
								minlenght.
								"-32768" would be "-000032768" with 9 as
								minlength.
								"0" would be "0" if you specified 1 as minlength, and
								nothing ("") if you specified 0 as minlength.
	 sepdistance         The distance between separators.
								You could make them million separators with this
								parameter if you want, but its main purpose is that
								you might want byte separators between the bytes in a
								hexadecimal or binary number. For thousand separators,
								pass 3. This means there will be three digits between
								separators.
	 base                Can be any number, and is the base with which the
								number is written.
								For example, pass 16 for hexadecimal numbers, 10 for
								normal decimal numbers or 23 if you prefer that number
								system.
	Returns:
	========
	A newly allocated string containing the number or `NULL' if you passed
	strange parameters.
*/
uchar *l2str_extended(slong number,uchar *text,
 int minlength, int sepdistance, int base)
{
uchar sign;
int numdigits=0,numseps,length=0,firstpos,textpos,
 seppos;
uchar digitvalue;
slong i;
//Checking parameters.
if((base<2)||(base>36))
	return NULL;
//Find out if any signs should be used.
if(number>0)
	sign=plussign;
else if(number<0){
	number=-number;
	sign=minussign;
}else
	sign=zerosign;
//Find first position where digits should be.
if(sign)
	firstpos=1;
else
	firstpos=0;

//Count length of number.
for(i=number;i;i/=base)
	numdigits++;
//Pretends that the number contains minlength number of digits. Luckily, this
//causes no bad side-effects.
if(numdigits<minlength)
	numdigits=minlength;
//With separators.
if(thousandsep&&sepdistance){
	numseps=_num_separators(numdigits,sepdistance);
	length=numdigits+firstpos+numseps;
	_ifstrxmalloc(text,length);
	textpos=length;
	seppos=sepdistance;
	//Filling the string with digits from right to left.
	while(textpos>firstpos){
		textpos--;
		//Count down and check separator counter.
		if(!(seppos--)){
			seppos=sepdistance-1;
			text[textpos]=thousandsep;
			textpos--;
		}
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
//Without separators.
}else{
	length=numdigits+firstpos;
	//mallocating.
	_ifstrxmalloc(text,length);
	textpos=length;
	//Filling text with digits from right to left.
	while(textpos>firstpos){
		textpos--;
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
}
if(sign)
	text[0]=sign;
return text;
}


/************************
****                 ****
**** ul2str_extended ****
****                 ****
*************************

	Returns a string containing an unsigned long.

	This is exactly the same function as l2str_extended, except that this
	one takes an unsigned long. Due to that, some parameters that handle the
	sign of the number (plussign, zerosign, minussign) aren't needed, and therefore I've
	removed them.
*/
uchar *ul2str_extended(ulong number,uchar *text,
 int minlength, int sepdistance, int base)
{
int numdigits=0,numseps,length=0,textpos,
 seppos;
uchar digitvalue;
ulong i;
//Checking parameters.
if((base<2)||(base>36))
	return NULL;

//Count length of number.
for(i=number;i;i/=base)
	numdigits++;
//Pretends that the number contains minlength number of digits. Luckily, this
//causes no bad side-effects.
if(numdigits<minlength)
	numdigits=minlength;
//With separators.
if(thousandsep&&sepdistance){
	numseps=_num_separators(numdigits,sepdistance);
	length=numdigits+numseps;
	_ifstrxmalloc(text,length);
	textpos=length;
	seppos=sepdistance;
	//Filling the string with digits from right to left.
	while(textpos){
		textpos--;
		//Count down and check separator counter.
		if(!(seppos--)){
			seppos=sepdistance-1;
			text[textpos]=thousandsep;
			textpos--;
		}
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
//Without separators.
}else{
	length=numdigits;
	//mallocating.
	_ifstrxmalloc(text,length);
	textpos=length;
	//Filling text with digits from right to left.
	while(textpos){
		textpos--;
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
}
return text;
}



/************************
****                 ****
**** ll2str_extended ****
****                 ****
*************************
	Returns a string containing a longlong.
	This is an extremly flexible function.
	Parameters:
	===========
	 number              The number.
	 minlength           The minimum number of digits returned.
								If the number has less digits than minlength, it will
								be padded with zeros to the left. Note that if
								minlength is zero, 0 is converted to a zero-length
								string. If you want it to look like "0", set
								minlength to 1.
								Examples:
								"32 767" would be "000 032 767" if you specified 9 as
								minlenght.
								"-32768" would be "-000032768" with 9 as
								minlength.
								"0" would be "0" if you specified 1 as minlength, and
								nothing ("") if you specified 0 as minlength.
	 sepdistance         The distance between separators.
								You could make them million separators with this
								parameter if you want, but its main purpose is that
								you might want byte separators between the bytes in a
								hexadecimal or binary number. For thousand separators,
								pass 3. This means there will be three digits between
								separators.
	 base                Can be any number, and is the base with which the
								number is written.
								For example, pass 16 for hexadecimal numbers, 10 for
								normal decimal numbers or 23 if you prefer that number
								system.
	Returns:
	========
	A newly allocated string containing the number or `NULL' if you passed
	strange parameters.
*/
uchar *ll2str_extended(slonglong number,uchar *text,
 int minlength, int sepdistance, int base)
{
uchar sign;
int numdigits=0,numseps,length=0,firstpos,textpos,
 seppos;
uchar digitvalue;
slonglong i;
//Checking parameters.
if((base<2)||(base>36))
	return NULL;
//Find out if any signs should be used.
if(number>0)
	sign=plussign;
else if(number<0){
	number=-number;
	sign=minussign;
}else
	sign=zerosign;
//Find first position where digits should be.
if(sign)
	firstpos=1;
else
	firstpos=0;

//Count length of number.
for(i=number;i;i/=base)
	numdigits++;
//Pretends that the number contains minlength number of digits. Luckily, this
//causes no bad side-effects.
if(numdigits<minlength)
	numdigits=minlength;
//With separators.
if(thousandsep&&sepdistance){
	numseps=_num_separators(numdigits,sepdistance);
	length=numdigits+firstpos+numseps;
	_ifstrxmalloc(text,length);
	textpos=length;
	seppos=sepdistance;
	//Filling the string with digits from right to left.
	while(textpos>firstpos){
		textpos--;
		//Count down and check separator counter.
		if(!(seppos--)){
			seppos=sepdistance-1;
			text[textpos]=thousandsep;
			textpos--;
		}
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
//Without separators.
}else{
	length=numdigits+firstpos;
	//mallocating.
	_ifstrxmalloc(text,length);
	textpos=length;
	//Filling text with digits from right to left.
	while(textpos>firstpos){
		textpos--;
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
}
if(sign)
	text[0]=sign;
return text;
}


/*************************
****                  ****
**** ull2str_extended ****
****                  ****
**************************

	Returns a string containing an unsigned longlong.

	This is exactly the same function as ll2str_extended, except that
	this one takes an unsigned longlong. Due to that, some parameters that
	handle the sign of the number (plussign, zerosign, minussign) aren't
	needed, and therefore I've removed them.
*/
uchar *ull2str_extended(ulonglong number,uchar *text,
 int minlength, int sepdistance, int base)
{
int numdigits=0,numseps,length=0,textpos,
 seppos;
uchar digitvalue;
ulonglong i;
//Checking parameters.
if((base<2)||(base>36))
	return NULL;

//Count length of number.
for(i=number;i;i/=base)
	numdigits++;
//Pretends that the number contains minlength number of digits. Luckily, this
//causes no bad side-effects.
if(numdigits<minlength)
	numdigits=minlength;
//With separators.
if(thousandsep&&sepdistance){
	numseps=_num_separators(numdigits,sepdistance);
	length=numdigits+numseps;
	_ifstrxmalloc(text,length);
	textpos=length;
	seppos=sepdistance;
	//Filling the string with digits from right to left.
	while(textpos){
		textpos--;
		//Count down and check separator counter.
		if(!(seppos--)){
			seppos=sepdistance-1;
			text[textpos]=thousandsep;
			textpos--;
		}
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
//Without separators.
}else{
	length=numdigits;
	//mallocating.
	_ifstrxmalloc(text,length);
	textpos=length;
	//Filling text with digits from right to left.
	while(textpos){
		textpos--;
		digitvalue=number%base;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		number/=base;
	}
}
return text;
}



/***************
****        ****
**** _d2str ****
****        ****
****************
Helper function that fills a string with a decimal number. This is used when
converting double numbers to strings. Returns length of string. The number
must be positive, and the eventual signs must be there already. `text' must
point to the first character after the sign, and must be big enough for the
worst possible event, i.e. the number of decimals is `maxdecimals'. All
parameters to this function must be valid and fixed up like in
d2str_extended.
*/
static int _d2str(uchar *text,double number,
 ulong decimalpoint, slong mindecimals, slong maxdecimals,
 uchar sepdistance, uchar base)
{
int textpos,length=decimalpoint;
int numdecimals=0;
double tmpnumber;
uchar digitvalue;
int seppos;
int onlyzeros=TRUE;

//With separators.
if(thousandsep&&sepdistance){
	textpos=decimalpoint;
	tmpnumber=number;
	seppos=sepdistance;
	//Filling text with digits from decimal point to left.
	while(textpos){
		textpos--;
		if(!(seppos--)){
			seppos=sepdistance-1;
			text[textpos]=thousandsep;
			textpos--;
		}
		digitvalue=fmod(tmpnumber,base);
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		tmpnumber/=base;
		if(digitvalue)
			onlyzeros=FALSE;
	}
	length=decimalpoint/*+_num_separators(decimalpoint,sepdistance)*/+1+mindecimals;
	seppos=sepdistance;
	tmpnumber=number;
	//Finding decimals from decimal point to right.
	for(textpos=decimalpoint+1;
	 numdecimals<maxdecimals;
	 textpos++){
		if(!(seppos--)){
			seppos=sepdistance-1;
			text[textpos]=thousandsep;
			textpos++;
		}
		digitvalue=tmpnumber=fmod(tmpnumber*base,base);
		//Increase the length of the number.
		if(digitvalue||(numdecimals<mindecimals))
			length=textpos+1;
		numdecimals++;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		if(digitvalue)
			onlyzeros=FALSE;
	}
	if(length>=(decimalpoint+_num_separators(decimalpoint,sepdistance)))
		text[decimalpoint]=decimalsep;
}else{
//Without separators.
	textpos=decimalpoint;
	tmpnumber=number;
	//Filling text with digits from decimal point to left.
	while(textpos){
		textpos--;
		digitvalue=fmod(tmpnumber,(double)base);
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		tmpnumber/=base;
		if(digitvalue)
			onlyzeros=FALSE;
	}
	length=decimalpoint+1+mindecimals;
	tmpnumber=number;
	//Finding decimals from decimal point to right.
	for(textpos=decimalpoint+1;numdecimals<maxdecimals;textpos++){
		tmpnumber*=base;
		digitvalue=tmpnumber=fmod(tmpnumber,base);
		if(digitvalue||(numdecimals<mindecimals))
			length=textpos+1;
		numdecimals++;
		text[textpos]=_digit2char(digitvalue,hexuppercase);
		if(digitvalue)
			onlyzeros=FALSE;
	}
	if(length>=decimalpoint)
		text[decimalpoint]=decimalsep;
}
if(onlyzeros)
	//If length is bigger than 16384, we know that the number contains only
	//zeros, and that we must use the zero sign.
	return length+16384;
else
	return length;
}
/***********************
****                ****
**** d2str_extended ****
****                ****
************************
	Returns a string containing a double.
	This is an extremly flexible function, where you can change everything.
	That also makes it necessary to give it 24 parameters, which is not always
	what you want. Therefore, I have defined some macros that take fewer
	parameters. If these aren't sufficient, you could always make your own
	macros. I also have plans to make a function that takes format strings
	and calls this function.

	Parameters:
	===========
	 number              The number.
	 decimaltype         The type of decimals used.
								Any of the following:
									`ct_nodecimals'The result will look like an
														integer.
									`ct_decimals'  The result will have decimals.
									`ct_ifexponent'The result will have decimals or
														be in exponent form, depending
														on the number and expifgreater/
														expifsmaller parameters.
									`ct_exponent'  The result will have the format
														of an exponent, i.e. like "1.2 e+24"
									`ct_halves'    Decimals will be represented by
														'1/2' (Not these three characters,
														but the character you specify in
														the parameter char12. This could
														maybe be different in different
														ascci tables.) or nothing.
									`ct_quarters'  Decimals will be represented by
														'1/4', '1/2', '3/4' or nothing. As
														when decimaltype is `ct_halves',
														it's not these strings  that
														represent a quarter, a half and
														three quarters, but the characters
														you specify in char14, char12 and
														char14 external variables, that will
														be used.
	 expifgreater,
	  expifsmaller       These only mean something if decimaltype is
								`ct_ifexponent'. If the number is greater than or
								equal to [base raised to expifgreater], or if number
								is smaller than [base raised to expifsmaller], the
								text will be in exponential form. Otherwise, the text
								will be in normal form with decimals. Expifsmaller
								should of course always be negative,   so that
								numbers with more than for example 5 zeros to the
								right of the decimal point will be in exponential
								form.
	 minlength           The minimum number of digits returned to the left of
								the decimal point.
								If the number has less digits than minlength, it will
								be padded with zeros to the left. Only the digits,
								not any thousand separators are counted. The normal
								for this is 1, since you mostly want zeros to appear
								as "0" and not as an empty string.
								Passing a negative value to this has a special
								meaning if decimaltype is `ct_halves' or
								`ct_quarters':
								Normally, minlength will then be treated as
								-minlength, but if the number is only zero and a half
								(or zero and a quarter etc. or the same negative),
								no zeros will be written before the half sign.
								Examples:
								"32 767" would be "000 032 767" if you specified 9 as
								minlenght.
								"-32768" would be "-000032768" with 9 as
								minlength.
								"0" would be "0" if you specified 1 as minlength, and
								nothing if you specified 0 as minlength.
	 mindecimals,
	 maxdecimals         The limits for number of decimals.
								The number of decimals in the text returned will
								always have at least mindecimals. The number has up
								to maxdecimals decimals if not the rightmost digit is
								zero. If maxdecimals is smaller than or equal to
								mindecimals, the number will have exactly mindecimals
								decimals. If mindecimals is 0, a number without
								decimals will end with the decimal separator. If
								mindecimals is -1, a number without decimals will
								end without the decimal separator.
								 Some examples:
								  mindecimals = 4 and maxdecimals = 7
									12             would look like "12.0000"
									12.34567       would look like "12.34567"
									12.34567898765 would look like "12.345679"
									12.345678912   would look like "12.3456789"
								  mindecimals = 0 and maxdecimals = 7
									12             would look like "12."
								  mindecimals = -1 and maxdecimals = 7
									12             would look like "12"
									12.1           would look like "12.1"
								  mindecimals = 3 and maxdecimals <= 3
									12             would look like "12.000"
									1234.9999      would look like "1235.000"
	 expmindecimals,
	  expmaxdecimals     The same as mindecmials and maxdecimals, only that
								these are for the coefficient in an exponential
								number.
	 expminlength        The minlength of the exponent.
								The exponent is the number to the right of "E".
								For example, if expminlength is 3, "2.34 e+7" will
								look like "2.34 e+007".
	 sepdistance         The distance between separators.
								You could make them million separators with this
								parameter if you want, but its main purpose is that
								you might want byte separators between the bytes in a
								hexadecimal or binary number. For thousand separators,
								pass 3. This means there will be three digits between
								separators.
	 base                The base with which the number is written.
								Can be any number between 2 and 36. For example, pass
								16 for hexadecimal numbers, 10 for normal decimal
								numbers or 23 if you prefer that number system.
	 decimaltypechosen   This is a pointer to an integer that tells which
								decimaltype the return has. Normally, this is of
								course the same as the parameter decimaltype, but if
								decimaltype is ifdecimals, this is either
								`ct_decimals' or `ct_exponent'. You may pass `NULL'
								to this if you don't care. Then I won't care either.
	Returns:
	========
	A newly allocated string containing the number or `NULL' if you passed
	strange parameters.
*/
uchar *d2str_extended(double number,uchar *text,
 int decimaltype,ulong expifgreater,ulong expifsmaller,
 slong minlength,
 slong mindecimals, slong maxdecimals,
 slong expmindecimals, slong expmaxdecimals, ulong expminlength,
 uchar sepdistance,
 uchar base, uchar *decimaltypechosen)
{
int numdigitsleft=0,maxlength,explength=0;
int textpos;
double xcoefficient;
int xexp=0;
int coefficientlength=0,firstpos;
uchar digitvalue;
uchar sign;
int expnegative;
uchar *expstring;
int expstringlength;
double tmpnumber;
double roundnumber;
int length;
int i;
double fractionalpart;
int lengthleft;
int nozeroifhalf=FALSE;
int shallrealloc=!!text;

switch(decimaltype){
	case ct_exponent:
		roundnumber=0;
	break;
	case ct_nodecimals:
		roundnumber=.5;
	break;
	case ct_halves:
		roundnumber=.25;
	break;
	case ct_quarters:
		roundnumber=.125;
	break;
	default:
// case ct_decimals:
// case ct_ifexponent:
		roundnumber=pow(base,-maxdecimals)/2;
	break;
}
//Checking parameters.
if((base<2)||(base>36))
	return NULL;
if(mindecimals>maxdecimals)
	maxdecimals=mindecimals;
if(minlength<0){
	minlength=-minlength;
	nozeroifhalf=TRUE;
}
//Find out if any signs should be used.
if(number>0)
	sign=plussign;
else if(number<0){
	number=-number;
	sign=minussign;
}else
	sign=zerosign;
//Find first position where digits should be.
if(sign)
	firstpos=1;
else
	firstpos=0;
//Count length of number.
for(tmpnumber=number+roundnumber;tmpnumber>=1;tmpnumber/=base)
	numdigitsleft++;

//If decimaltype is ct_ifexponent, we set it to `ct_exponent' or `ct_decimals'.
if(decimaltype==ct_ifexponent){
	if(number>=1){
		if(numdigitsleft>=expifgreater)
			decimaltype=ct_exponent;
		else
			decimaltype=ct_decimals;
	}else if(!number){
		if(decimaltype==ct_ifexponent)
			decimaltype=ct_decimals;
	}else{
		if(xexp<expifsmaller)
			decimaltype=ct_exponent;
		else
			decimaltype=ct_decimals;
	}
}

switch (decimaltype){
	case ct_exponent:
		if(number>=1)
			xexp=numdigitsleft-1;
		else if(number>0){
			//Counting zeros to the right of the decimal point.
			for(tmpnumber=number;tmpnumber<1;tmpnumber*=base)
				xexp--;
		}else{
			//Just returning with "0" or maybe "0".
			if(sign){
				_ifstrxmalloc(text,2);
				text[0]=sign;
				text[1]='0';
			}else{
				_ifstrxmalloc(text,1);
				text[0]='0';
			}
			return text;
		}
		//Creating the coefficient.
		xcoefficient=number/pow(base,xexp);
		//Add this in order to round instead of truncate.
		xcoefficient+=pow(base,-expmaxdecimals)/2;
		//After previous line, things may have changed a little.
		if(number>base){
			number/=base;
			xexp++;
		}
		if(number>=1){
			expnegative=FALSE;
			expstring=exppositivestring;
		}else if(number>0){
			expnegative=TRUE;
			expstring=expnegativestring;
			//Now we don't need the negative exponent any more.
			xexp=-xexp;
		}else{
			//Just returning with "0" or maybe "0".
			if(sign){
				_ifstrxmalloc(text,2);
				text[0]=sign;
				text[1]='0';
			}else{
				_ifstrxmalloc(text,1);
				text[0]='0';
			}
			return text;
		}
		expstringlength=strlen(expstring);
		//Counting length of exponent.
		for(i=xexp;i;i/=base)
			explength++;
		if(explength<expminlength)
			explength=expminlength;
		//Counting max length of the whole number.
		maxlength=firstpos+2+expmaxdecimals+
		 expstringlength+explength;
		//mallocating
		_ifstrxmalloc(text,maxlength);
		//Filling the result with coefficient.
		coefficientlength=firstpos+_d2str(
		 text+firstpos,xcoefficient,
		 1,expmindecimals,expmaxdecimals,
		 0,base);
		//Now we know the final size of the number, so we can reallocate.
		length=coefficientlength+expstringlength+explength;
		if(shallrealloc)
			_strxrealloc(text,length);
		//Filling the result with " e+" or " e-".
		memcpy(text+coefficientlength,expstring,expstringlength);
		//Filling text with exponent from right to left.
		textpos=length;
		while(textpos>(coefficientlength+expstringlength)){
			textpos--;
			digitvalue=xexp%base;
			text[textpos]=_digit2char(digitvalue,hexuppercase);
			xexp/=base;
		}
	break;
	case ct_nodecimals:
		//Protects from returning things like "-0" or "+0".
		if(number<.5){
			sign=zerosign;
			if(sign)
				firstpos=1;
			else
				firstpos=0;
		}
		if(numdigitsleft<minlength)
			numdigitsleft=minlength;
		if(thousandsep&&sepdistance){
			lengthleft=numdigitsleft+_num_separators(numdigitsleft,sepdistance);
			maxlength=firstpos+lengthleft;
		}else{
			lengthleft=numdigitsleft;
			maxlength=firstpos+lengthleft;
		}
		_ifstrxmalloc(text,maxlength);
		_d2str(
		 text+firstpos,number+.5,
		 numdigitsleft,-1,-1,
		 sepdistance,base);
	break;
	case ct_halves:
		//Protects from returning things like "-0" or "+0".
		if(number<.25){
			sign=zerosign;
			if(sign)
				firstpos=1;
			else
				firstpos=0;
		}
		if(numdigitsleft<minlength)
			numdigitsleft=minlength;
		if(thousandsep&&sepdistance){
			lengthleft=numdigitsleft+_num_separators(numdigitsleft,sepdistance);
			maxlength=firstpos+lengthleft;
		}else{
			lengthleft=numdigitsleft;
			maxlength=firstpos+lengthleft;
		}
		number+=.25;
		if(number-floor(number)<.5){
			_ifstrxmalloc(text,maxlength);
			_d2str(
			 text+firstpos,number,
			 lengthleft,-1,-1,
			 sepdistance,base);
		}else{
			//If number should have looked like "0", we create an own one
			//looking like "" only.
			if(nozeroifhalf&&number<1){
				maxlength=firstpos;
				_ifstrxmalloc(text,maxlength+1);
			}else{
				_ifstrxmalloc(text,maxlength+1);
				_d2str(
				 text+firstpos,number,
				 lengthleft,-1,-1,
				 sepdistance,base);
			}
			text[maxlength]=char12;
		}
	break;
	case ct_quarters:
		//Protects from returning things like "-0" or "+0".
		if(number<.125){
			sign=zerosign;
			if(sign)
				firstpos=1;
			else
				firstpos=0;
		}
		if(numdigitsleft<minlength)
			numdigitsleft=minlength;
		if(thousandsep&&sepdistance){
			lengthleft=numdigitsleft+_num_separators(numdigitsleft,sepdistance);
			maxlength=firstpos+lengthleft;
		}else{
			lengthleft=numdigitsleft;
			maxlength=firstpos+lengthleft;
		}
		_ifstrxmalloc(text,maxlength);
		number+=.125;
		if( (fractionalpart=(number-floor(number))) <.25){
			_ifstrxmalloc(text,maxlength);
			_d2str(
			 text+firstpos,number,
			 lengthleft,-1,-1,
			 sepdistance,base);
		}else{
			//If number should have looked like "0", we create an own one
			//looking like "" only.
			if(nozeroifhalf&&number<1){
				maxlength=firstpos;
				_ifstrxmalloc(text,maxlength+1);
			}else{
				_ifstrxmalloc(text,maxlength+1);
				_d2str(
				 text+firstpos,number,
				 lengthleft,-1,-1,
				 sepdistance,base);
			}
			if(fractionalpart<.5)
				text[maxlength]=char14;
			else if(fractionalpart<.75)
				text[maxlength]=char12;
			else
				text[maxlength]=char34;
		}
	break;
	default:
// case ct_decimals:
		if(numdigitsleft<minlength)
			numdigitsleft=minlength;
		if(thousandsep&&sepdistance){
			lengthleft=numdigitsleft+_num_separators(numdigitsleft,sepdistance);
			maxlength=firstpos+lengthleft+1+
			 maxdecimals+_num_separators(maxdecimals,sepdistance);
		}else{
			lengthleft=numdigitsleft;
			maxlength=firstpos+lengthleft+1+maxdecimals;
		}
		_ifstrxmalloc(text,maxlength);
		length=firstpos+_d2str(
		 text+firstpos,number+roundnumber,
		 lengthleft,mindecimals,maxdecimals,
		 sepdistance,base);
		//Protects from returning things like "-0" or "+0".
		if(length>16384){
			length-=16384;
			if(zerosign&&(!sign))
				memmove(text+1,text,length++);
			else if((!zerosign)&&sign)
				memmove(text,text+1,length--);
			if(shallrealloc)
				_strxrealloc(text,length);
			sign=zerosign;
		}else if(shallrealloc)
			_strxrealloc(text,length);
	break;
}
if(sign)
	text[0]=sign;
if(decimaltypechosen)
	*decimaltypechosen=decimaltype;
return text;
}


/***********************
****                ****
**** str2l_extended ****
****                ****
************************
	Returns a signed long number found in a text. This function does the same
	as str2d_extended, but with integer numbers.
	This function implies that the text contains a number and only a number.
	It won't fail if the text contains other characters, so if you for example
	have a string containing two numbers, both will be counted. If the string
	contains minus signs, all minus signs are toggle plus/minus on and off, no
	matter where in the string they are placed. An example:
		str2int_extended("Dummy 3 and 2. Foo 5 tmp , dummy 9 and so on",
		 '.','-',10)
	will return the number -3295 . (The point is used as decimal separator.
	The minus toggles plus/minus to minus.)

	However, doing this way makes it possible to include thousand separators
	without any unexpected side-effects.

	Parameters:
	===========
	 text             The text to take the number from.
	 base             The base with which the number is represented. Must be
							in range 2..36.
							For example 10 for decimal, 2 for binary and 16 for
							hexadecimal numbers.
	 valid            If the number in text was too big or too small for the
							datatype of the number, this points to FALSE. Else, it
							points to TRUE. Set this to NULL if you don't care.
							THIS IS NOT IMPLEMENTED YET!
*/
slong str2l_extended(uchar *text,int base,int *valid)
{
slong ret=0;
int negative=FALSE;
int i;

for(i=0;text[i];i++){
	if(charisdigit_base(text[i],base))
		ret=ret*base+_char2digit(text[i]);
	else if(text[i]==minussign)
		negative=!(negative);
}

return negative ? -ret : ret;
}


/************************
****                 ****
**** str2ul_extended ****
****                 ****
*************************
The same as str2l_extended, but unsigned. */
ulong str2ul_extended(uchar *text, int base, int *valid)
{
ulong ret=0;
int i;
if(valid){
	*valid=TRUE;
	for(i=0;text[i];i++)
		if(charisdigit_base(text[i],base))
			ret=ret*base+_char2digit(text[i]);
		else if(text[i]==minussign)
			*valid=FALSE;
}else{
	for(i=0;text[i];i++)
		if(charisdigit_base(text[i],base))
			ret=ret*base+_char2digit(text[i]);
}
return ret;
}




/************************
****                 ****
**** str2ll_extended ****
****                 ****
*************************
	Returns a signed longlong number found in a text. This function does the
	same as str2d_extended, but with integer numbers.
	This function implies that the text contains a number and only a number.
	It won't fail if the text contains other characters, so if you for example
	have a string containing two numbers, both will be counted. If the string
	contains minus signs, all minus signs are toggle plus/minus on and off, no
	matter where in the string they are placed. An example:
		str2int_extended("Dummy 3 and 2. Foo 5 tmp , dummy 9 and so on",
		 '.','-',10)
	will return the number -3295 . (The point is used as decimal separator.
	The minus toggles plus/minus to minus.)

	However, doing this way makes it possible to include thousand separators
	without any unexpected side-effects.

	Parameters:
	===========
	 text             The text to take the number from.
	 base             The base with which the number is represented. Must be
							in range 2..36.
							For example 10 for decimal, 2 for binary and 16 for
							hexadecimal numbers.
	 valid            If the number in text was too big or too small for the
							datatype of the number, this points to FALSE. Else, it
							points to TRUE. Set this to NULL if you don't care.
							THIS IS NOT IMPLEMENTED YET!
*/
slonglong str2ll_extended(uchar *text,int base,int *valid)
{
slonglong ret=0;
int negative=FALSE;
int i;

for(i=0;text[i];i++){
	if(charisdigit_base(text[i],base))
		ret=ret*base+_char2digit(text[i]);
	else if(text[i]==minussign)
		negative=!(negative);
}

return negative ? -ret : ret;
}


/*************************
****                  ****
**** str2ull_extended ****
****                  ****
**************************
The same as str2ll_extended, but unsigned. */
ulonglong str2ull_extended(uchar *text,int base,
 int *valid)
{
ulonglong ret=0;
int i;
if(valid){
	*valid=TRUE;
	for(i=0;text[i];i++)
		if(charisdigit_base(text[i],base))
			ret=ret*base+_char2digit(text[i]);
		else if(text[i]==minussign)
			*valid=FALSE;
}else{
	for(i=0;text[i];i++)
		if(charisdigit_base(text[i],base))
			ret=ret*base+_char2digit(text[i]);
}
return ret;
}


/***********************
****                ****
**** str2d_extended ****
****                ****
************************
	Returns a floating-point (double) number found in a text. This function
	implies that the text contains a number and only a number. It won't fail
	if the text contains other characters, so if you for example have a string
	containing two numbers, both will be counted. If the string contains minus
	signs, all minus signs toggle plus/minus on and off, no matter where in
	the string they are placed.
	An example:
		str2d_extended("Dummy, -4-7- foo. Dummy 1.1...",
		 ct_decimals, '.', '', '', '', '-', 10, NULL)
	will return the number -47.11 . (The first point is used as decimal
	separator. The first minus toggles plus/minus to minus, the second back to
	plus and the third toggles to minus again.)

	However, doing this way makes it possible to include thousand separators
	without any unexpected side-effects.

	An exception to this is when your decimaltype is `ct_exponent'. In
	exponents, you can't have thousand separators. The first non-digit is
	counted as the space between the coefficient and the exponent.

	Parameters:
	===========
	 text             The text to take the number from.
	 decimaltype      The type of decimals used.
							Any of the following:
								`ct_nodecimals'Ignores any decimals.
								`ct_decimals'  Takes care of any decimals found.
								`ct_exponent'  Treats the text as an exponential
													number. Note: exponential numbers can
													not have thousand separators. (And
													d2str never gives them thousand
													separators either.)
								`ct_halves'    Ignores all decimals, but takes care
													of '' as a half.
								`ct_quarters'  Ignores any decimals, but takes care
													of '', '' and ''.
								Any number greater than or equal to 1 indicates
								that the the number will be represented by
								exponents if it has more digits than decimaltype.
	 base             The base with which the number is represented. Must be
							in range 2..36.
							For example 10 for decimal, 2 for binary and 16 for
							hexadecimal numbers.
	 valid            If the number found in text was too big or too small for
							the datatype of the number, this points to FALSE. Else,
							it points to TRUE. Set this to NULL if you don't care.
							THIS IS NOT IMPLEMENTED YET!
*/
double str2d_extended(uchar *text,int decimaltype,
 int base,int *valid)
{
double ret=0;
int negative=FALSE;
double divide=base;
int raisedtominus=FALSE;
double xcoefficient=0,xexp=0;
int i;

//Checking parameters.
if((base<2)||(base>36))
	return NULL;
switch(decimaltype){
	case ct_exponent:
		//First we loop up to the first digit while we check for minuses.
		for(i=0;text[i];i++){
			if(text[i]==minussign)
				negative=!(negative);
			else if(charisdigit_base(text[i],base))
				break;
		}
		//The first digits will represent the exponent.
		for(;text[i];i++){
			if(charisdigit_base(text[i],base))
				xcoefficient=xcoefficient*base+_char2digit(text[i]);
			else if(text[i]==decimalsep){
				//Fractional part of exponent.
				divide=base;
				for(i++;text[i];i++){
					if(charisdigit_base(text[i],base)){
						xcoefficient+=_char2digit(text[i])/divide;
						divide*=base;
					}else
						break;
				}
				break;
			}else
				break;
		}
		//Between the coefficient and the exponent, we search for minuses.
		for(;text[i];i++){
			if(text[i]==minussign)
				raisedtominus=!raisedtominus;
			else if(charisdigit_base(text[i],base))
				break;
		}
		//The exponent.
		for(;text[i];i++){
			if(charisdigit_base(text[i],base))
				xexp=xexp*base+_char2digit(text[i]);
			else if(text[i]==decimalsep){
				//Looking for decimals.
				divide=base;
				for(i++;text[i];i++){
					if(charisdigit_base(text[i],base)){
						xexp+=(_char2digit(text[i])/divide);
						divide*=base;
					}
				}
				break;
			}
		}
		if(raisedtominus)
			ret=xcoefficient*pow(base,-xexp);
		else
			ret=xcoefficient*pow(base,xexp);
	break;
	case ct_decimals:
		for(i=0;text[i];i++){
			if(charisdigit_base(text[i],base))
				ret=ret*base+_char2digit(text[i]);
			else if(text[i]==decimalsep){
				//Looking for decimals.
				divide=base;
				for(i++;text[i];i++){
					if(charisdigit_base(text[i],base)){
						ret+=(_char2digit(text[i])/divide);
						divide*=base;
					}else if(text[i]==minussign)
						negative=!(negative);
				}
				break;
			}else if(text[i]==minussign)
				negative=!(negative);
		}
	break;
	case ct_nodecimals:
		for(i=0;text[i];i++){
			if(charisdigit_base(text[i],base))
				ret=ret*base+_char2digit(text[i]);
			else if(text[i]==minussign)
				negative=!(negative);
		}
	break;
	case ct_halves:
		for(i=0;text[i];i++){
			if(charisdigit_base(text[i],base))
				ret=ret*base+_char2digit(text[i]);
			else if(text[i]==char12)
				ret=ret+.5;
			else if(text[i]==minussign)
				negative=!(negative);
		}
	break;
	case ct_quarters:
		for(i=0;text[i];i++){
			if(charisdigit_base(text[i],base))
				ret=ret*base+_char2digit(text[i]);
			else if(text[i]==char14)
				ret+=.25;
			else if(text[i]==char12)
				ret+=.5;
			else if(text[i]==char34)
				ret+=.75;
			else if(text[i]==minussign)
				negative=!(negative);
		}
	break;
}
return negative ? -ret : ret;
}



int char2digit(uchar ascii)
{
return _char2digit(ascii);
}

int digit2char(int number,int uppercase)
{
return _digit2char(number,uppercase);
}

/* getdigitwithbase
	With the binary number system, you can extract one of the bits by anding
	the number and a number with that bit set. But with any number system, for
	example with the base 10, you can't do that by anding. Instead, call this
	macro. NOTE: this is only for integer numbers. It won't work on floating
	point numbers. (This macro is not used by my functions.)
	Parameters:
	 n    The number.
	 b    The base (10 for decimal numbers, 16 for hex, 2 for binary and so on.
	 p    [The base] raised to [the position of the digit], i.e. if you want to
			extract the fourth digit from right, you pass base^4. Note: First
			digit is #1. For example, the 7 in 4711 is at position 3.
*/
int get_digit_with_base(ulonglong number, int base, ulonglong position)
{
return _get_digit_with_base(number,base,position);
}

/* numseparators
	Returns the number of thousand separators needed in a number that has
	`length' digits and the distance `sepdistance' between separators.
*/
int num_separators(int length,int sepdistance)
{
return _num_separators(length,sepdistance);
}



#endif
